#!/bin/bash

MKE2FS=/home/jack/source/e2fsprogs/misc/mke2fs
DEBUGFS=/home/jack/source/e2fsprogs/debugfs/debugfs
E2FSCK=/home/jack/source/e2fsprogs/e2fsck/e2fsck
E2IMAGE=/home/jack/source/e2fsprogs/misc/e2image
IMAGE=test-image

test_correct()
{
  if diff $1.out $1.ok | grep "^[<>]" | grep -v "^[<>] [acm]time:" >/dev/null; then
    return 1
  fi
  return 0
}

dd if=/dev/zero of=$IMAGE bs=65536 seek=16383 count=1 2>/dev/null
touch empty_file

$MKE2FS -F -b 65536 -O dir_index test-image >/dev/null 2>&1

#
# Check whether lost+found has 2 blocks
#
$DEBUGFS -w $IMAGE <<EOF >test1.out 2>&1
stat lost+found
EOF
if test_correct test1; then
  echo "test1 passed"
else
  echo "test1 failed"
  exit
fi

#
# Check that detection of directory with wrong mode works
# Check that directory with rec_len not reaching end of block works
# Check that directory with name_len > rec_len, rec_len % 4 != 0,
#   rec_len exceeding block size works
# Check that directory with missing . and .. works
#
$DEBUGFS -w $IMAGE <<EOF >test2-debugfs.out 2>&1
mkdir corrupted_mode_dir
sif corrupted_mode_dir mode 010755
mkdir short_rec_len
mkdir other_corrupted_rec_len
cd other_corrupted_rec_len
write empty_file file
ln file entry_namelen_big
ln file entry_reclen_not_multiple
ln file entry_reclen_exceeds_block
cd ..
mkdir missing_._..
stat short_rec_len
stat other_corrupted_rec_len
ls other_corrupted_rec_len
stat missing_._..
EOF
if ! test_correct test2-debugfs; then
  echo "test2: Debugfs output differs from an expected one!"
  exit
fi
# Rewrite rec_len in short_rec_len
xxd -r <<EOF | dd of=$IMAGE bs=1 conv=notrunc seek=$((41*65536+12+4)) 2>/dev/null
0xf000
EOF
# Rewrite rec_len in other_corrupted_rec_len/entry_namelen_big
xxd -r <<EOF | dd of=$IMAGE bs=1 conv=notrunc seek=$((42*65536+12+12+12+6)) 2>/dev/null
0xfe
EOF
# Rewrite rec_len in other_corrupted_rec_len/entry_reclen_not_multiple
xxd -r <<EOF | dd of=$IMAGE bs=1 conv=notrunc seek=$((42*65536+12+12+12+28+4)) 2>/dev/null
0x2500
EOF
# Rewrite rec_len in other_corrupted_rec_len/entry_reclen_exceeds_block
xxd -r <<EOF | dd of=$IMAGE bs=1 conv=notrunc seek=$((42*65536+12+12+12+28+36+4)) 2>/dev/null
0xffff
EOF
# Rewrite directory missing_._..
xxd -r <<EOF | dd of=$IMAGE bs=1 conv=notrunc seek=$((43*65536)) 2>/dev/null
0x00000000
0xffff
EOF
# Check how debugfs behaves on corrupted blocks
$DEBUGFS $IMAGE <<EOF >test2-debugfs-corrupted.out 2>&1
ls short_rec_len
ls other_corrupted_rec_len
ls missing_._..
EOF
if test_correct test2-debugfs-corrupted; then
  echo "test2-debugfs passed"
else
  echo "test2-debugfs failed"
  exit
fi
# Check that e2image works for the file
$E2IMAGE -rs $IMAGE e2image-tmp-image >test2-e2image.out 2>&1
if test_correct test2-e2image; then
  echo "test2-e2image passed"
else
  echo "test2-e2image failed"
fi
# Check how e2fsck behaves on corrupted blocks
$E2FSCK -y -f $IMAGE >test2-fsck.out 2>&1
if test_correct test2-fsck; then
  echo "test2-e2fsck passed"
else
  echo "test2-e2fsck failed"
  exit
fi

#
# Check that linking and unlinking works
#
$DEBUGFS -w $IMAGE <<EOF >test3.out 2>&1
mkdir test3
cd test3
write empty_file short_file
ln short_file file_with_a_longer_file_name
sif short_file links_count 2
ls
rm short_file
ls
ln file_with_a_longer_file_name file_with_an_even_much_much_longer_file_name_abcdefgh_1
ls
ln file_with_a_longer_file_name file_with_an_even_much_much_longer_file_name_abcdefgh_2
sif file_with_a_longer_file_name links_count 3
ls
rm file_with_an_even_much_much_longer_file_name_abcdefgh_1
ls
ln file_with_a_longer_file_name vsf
ls
ln file_with_a_longer_file_name a_longer_file
ls
ln file_with_a_longer_file_name b_longer_file
sif file_with_a_longer_file_name links_count 5
ls
rm a_longer_file
ls
rm file_with_a_longer_file_name
ls
EOF
if test_correct test3; then
  echo "test3 passed"
else
  echo "test3 failed"
  exit
fi

#
# Check that htree compression and directory listing works
#
$DEBUGFS -w $IMAGE <<EOF >test4-debugfs.out 2>&1
mkdir test4
expand_dir test4
expand_dir test4
expand_dir test4
cd test4
write empty_file a_file_with_length_something_00000
sif a_file_with_length_something_00000 links_count 4001
EOF
for (( i = 0; i < 4000; i++ )); do
  $DEBUGFS -w $IMAGE <<EOF >>test4-debugfs.out 2>&1
cd test4
ln a_file_with_length_something_00000 a_file_with_length_something_$i
EOF
done
if test_correct test4-debugfs; then
  echo "test4-debugfs passed"
else
  echo "test4-debugfs failed"
  exit
fi
$E2FSCK -y -D -f $IMAGE >test4-fsck.out 2>&1
if test_correct test4-fsck; then
  echo "test4-e2fsck passed"
else
  echo "test4-e2fsck failed"
  exit
fi

#
# Check how e2image works
#
$E2IMAGE -rs $IMAGE e2image-tmp-image >test4-e2image.out 2>&1
if test_correct test4-e2image; then
  echo "test4-e2image passed"
else
  echo "test4-e2image failed"
  exit
fi
$DEBUGFS e2image-tmp-image <<EOF >test4-e2image-debugfs-1.out 2>&1
stat BAAAA
EOF
if test_correct test4-e2image-debugfs-1; then
  echo "test4-e2image-debugfs-1 passed"
else
  echo "test4-e2image-debugfs-1 failed"
  exit
fi
$DEBUGFS e2image-tmp-image <<EOF >test4-e2image-debugfs-2.out 2>&1
dirsearch BAAAA UAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
dirsearch BAAAA LBBAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
EOF
if [ `grep "^Entry found at" test4-e2image-debugfs-2.out | wc -l` -eq 2 ]; then
  echo "test4-e2image-debugfs-2 passed"
else
  echo "test4-e2image-debugfs-2 failed"
  exit
fi

$DEBUGFS $IMAGE <<EOF >test4-fsck-debugfs-1.out 2>&1
ls -l test4
EOF
if tr -s ' ' <test4-fsck-debugfs-1.out | cut -d ' ' -f 10 | sort | diff - test4-fsck-debugfs-1.ok >/dev/null; then
  echo "test4-fsck-debugfs-1 passed"
else
  echo "test4-fsck-debugfs-1 failed"
  exit
fi
$DEBUGFS $IMAGE <<EOF >test4-fsck-debugfs-2.out 2>&1
htree_dump test4
EOF
if head -10 test4-fsck-debugfs-2.out | diff - test4-fsck-debugfs-2a.ok >/dev/null; then
  if grep "^20" test4-fsck-debugfs-2.out | cut -d ' ' -f 4 | sort | diff - test4-fsck-debugfs-2b.ok >/dev/null; then
    echo "test4-fsck-debugfs-2 passed"
  else
    echo "test4-fsck-debugfs-2b failed"
    exit
  fi
else
  echo "test4-fsck-debugfs-2a failed"
  exit
fi

rm empty_file
rm e2image-tmp-image
