/* For pread/pwrite */
#define _XOPEN_SOURCE 500

#include <stdio.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <signal.h>
#include <sys/wait.h>
#include <sys/ptrace.h>
#include <asm/ptrace_offsets.h>
#include <unistd.h>
#include "stopper.h"

#define TEST_OK		0
#define TEST_FAILED	1
#define TEST_ABORTED	2

/* Some header files are broken... */
#ifndef PTRACE_PEEKUSR
# define PTRACE_PEEKUSR  PTRACE_PEEKUSER
#endif

int run_child()
{
	my_kill(getpid(), SIGSTOP);

	return TEST_OK;
}

static int readmem(int memfd, long addr, long *res)
{
	if (pread(memfd, res, sizeof(long), addr) != sizeof(long)) {
		perror("read mem");
		return TEST_ABORTED;
	}

	return TEST_OK;
}

static int get_memfd(pid_t pid)
{
	char fname[64];

	snprintf(fname, sizeof fname,
		 "/proc/%ld/mem", (long) pid);
	return open(fname, O_RDWR);
}

int run_debugger(pid_t pid)
{
	int memfd;
        long regaddr, regval;
	int status;
	int ret;

	if ((memfd = get_memfd(pid)) < 0) {
		perror("open mem");
		return TEST_ABORTED;
	}

	if (waitpid(pid, &status, WUNTRACED) == -1) {
		perror("waitpid SIGSTOP");
		return TEST_ABORTED;
	}

	if (ptrace(PTRACE_ATTACH, pid, 0, 0)) {
		perror("PTRACE_ATTACH");
		return TEST_ABORTED;
	}

	errno = 0;
	regaddr = ptrace(PTRACE_PEEKUSR, pid, PT_AR_BSP, 0);
	if (errno) {
		perror("PTRACE_PEEKUSR ar.bsp");
		return TEST_ABORTED;
	}
	regaddr -= 3 * 8;

	if (readmem(memfd, regaddr, &regval))
		return TEST_ABORTED;
	printf("Test variable via mem: 0x%lx\n", regval);
	if (regval == MAGIC_NEW) {
		ret = TEST_OK;
	} else if (regval == MAGIC_STALE) {
		printf("Found STALE value in RBS\n");
		ret = TEST_FAILED;
	} else {
		printf("Found unknown incorrect value in RBS\n");
		ret = TEST_FAILED;
	}

	if (ptrace(PTRACE_DETACH, pid, 0, SIGCONT)) {
		perror("PTRACE_DETACH");
		return TEST_ABORTED;
	}

	return ret;
}

int main(int argc, char **argv)
{
	int pid;

	if ((pid = fork()))
		return run_debugger(pid);
	else
		return run_child();
}

