/*
 * exec.c, Copyright (c) 2004 Aurema, Pty Ltd.  All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 *
 * This provides functionality to send exec notifications.
 */

#include <linux/module.h>
#include <linux/init.h>
#include <linux/relayfs_fs.h>

#include <asm/atomic.h>


/* Reader reference count. */
static atomic_t exec_client_cnt = ATOMIC_INIT(0);

/* Relayfs channel ID. */
static int exec_cid = -1;

/* Size of buffer to hold pid (32 bits). */
#define EXEC_BUFSIZE 16

/*
 * Called whenever an exec event happens.
 */
void exec_write_event(pid_t pid)
{
	char buf[EXEC_BUFSIZE]; 
	int len, result;

	if (exec_cid < 0) 
		return;
	if (atomic_read(&exec_client_cnt) < 1)
		return;

	len = snprintf(buf, sizeof(buf), "%d\n", pid);
	result = relay_write(exec_cid, buf, len, -1, NULL);
	if (unlikely(result <= 0))
		printk(KERN_ERR "%s - write failed\n", __FUNCTION__);
}

/*
 * File operation callback.
 */
static int 
exec_fileop_notify(int rchan_id, struct file *filp, enum relay_fileop op)
{
	if (unlikely(rchan_id != exec_cid)) {
		printk(KERN_ERR "%s - bad file number\n", __FUNCTION__);
		return -EBADF;
	}

	switch (op) {
	case RELAY_FILE_OPEN:
		atomic_inc(&exec_client_cnt);
		break;
	case RELAY_FILE_CLOSE:
		if (atomic_dec_and_test(&exec_client_cnt) == 0)
			relay_reset(exec_cid);
		break;
	default:
		/* do nothing */
		break;
	}

	return 0;
}


static struct rchan_callbacks exec_callbacks = {
	.buffer_start = NULL,
	.buffer_end = NULL,
	.deliver = NULL,
	.user_deliver = NULL,
	.needs_resize = NULL,
        .fileop_notify = exec_fileop_notify,
	.ioctl = NULL,
};

/* 
 * Creates the exec logger channel.
 */
int __init exec_init(void) 
{
	exec_cid = relay_open("execs",
			      RELAY_MIN_BUFSIZE, RELAY_MIN_BUFS,
			      RELAY_SCHEME_ANY | RELAY_USAGE_GLOBAL | 
			      RELAY_DELIVERY_PACKET | 
			      RELAY_TIMESTAMP_ANY, &exec_callbacks,
			      0, 0, 0, 0, 0, 0444, NULL, 0);
	if (exec_cid < 0) {
		printk(KERN_ERR "%s - open failed\n", __FUNCTION__);
		return exec_cid;
	}

	return 0;
}

/*
 * Closes exec logger channel.
 */
void __exit exec_deinit(void)
{
	if (relay_close(exec_cid) < 0)
		printk(KERN_ERR "%s - close failed\n", __FUNCTION__);

	exec_cid = -1;
}
