/*
 * pagg.c, Copyright (C) 2004, Aurema Pty. Ltd., support@aurema.com
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
 * License for more details.
 *
 *
 * This file has pagg callbacks required to track Linux task execs.
 */

#include <linux/module.h>
#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/pagg.h>
#include <linux/slab.h>
#include <linux/spinlock.h>

#include <asm/semaphore.h>


#define EXEC_PAGG "Exec Notifier" /* Maximum of 32. */

/*
 * Invoked during initialisation for every existing task.
 */
static int pagg_init_hook(task_t *task, struct pagg *pagg)
{
	return 0;
}

/*
 * Invoked when new tasks are forked.
 */
static int pagg_attach_hook(task_t *task, struct pagg *pagg, void *data)
{	
	return 0;
}

/*
 * Invoked when tasks exit the system.
 */
static void pagg_detach_hook(task_t *task, struct pagg *pagg)
{
}


extern void exec_write_event(pid_t pid);

/*
 * Invoked when tasks perform an exec() syscall.
 */
static void pagg_exec_hook(task_t *task, struct pagg *pagg)
{
       exec_write_event(task->pid);
}


static struct pagg_hook pagg_hooks = {
	.module = THIS_MODULE,
	.name = EXEC_PAGG,
	.data = NULL,
	.entry = LIST_HEAD_INIT(pagg_hooks.entry),
	.init = pagg_init_hook,
	.attach = pagg_attach_hook,
	.detach = pagg_detach_hook,
	.exec = pagg_exec_hook,
};


/*
 * Register pagg callbacks.  This must happen prior to any client
 * registration.
 */
int __init pagg_init(void)
{
	int result = pagg_hook_register(&pagg_hooks);

	if (result < 0)
		printk(KERN_ERR "%s - registration failed\n", __FUNCTION__);

	return result;
}

/*
 * Deregister pagg callbacks.  This should only ever occur after all
 * of our clients have deregistered.
 */
void __exit pagg_deinit(void)
{
	pagg_hook_unregister(&pagg_hooks);
}

