/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * Authors: Waiman Long <longman@redhat.com>
 *
 * This kernel module enables us to evaluate the contention behavior of locks.
 *
 * The following debugfs variables will be created:
 *  1) /sys/kernel/debug/locktest/c_count
 *  2) /sys/kernel/debug/locktest/i_count
 *  3) /sys/kernel/debug/locktest/l_count
 *  4) /sys/kernel/debug/locktest/p_count
 *  5) /sys/kernel/debug/locktest/locktype
 *  6) /sys/kernel/debug/locktest/r_percent
 *  7) /sys/kernel/debug/locktest/loadtype	# load type
 *  8) /sys/kernel/debug/locktest/etime	# elapsed time in us
 *  9) /sys/kernel/debug/locktest/status	# current test status
 * 10) /sys/kernel/debug/locktest/latency	# enable latency check
 * 11) /sys/kernel/debug/locktest/rwstat	# rwsem/rwlock stat
 * 12) /sys/kernel/debug/locktest/dumplock	# Dump out lock state
 *
 *  A negative r_percent means number of reader-only threads.
 *  A negative i_count means execution time in seconds.
 */
#include <linux/module.h>	// included for all kernel modules
#include <linux/kernel.h>	// included for KERN_INFO
#include <linux/init.h>		// included for __init and __exit macros
#include <linux/kobject.h>
#include <linux/debugfs.h>
#include <linux/string.h>
#include <linux/atomic.h>
#include <linux/delay.h>
#include <linux/spinlock.h>
#include <linux/sched.h>

#ifdef TASK_REPORT_IDLE
#include <linux/sched/clock.h>	/* Upstream kernel only */
#endif

#ifndef MAX_INT
#define MAX_INT		0x7fffffff
#endif

#ifndef ARRAY_SIZE
#define ARRAY_SIZE(a)	(sizeof(a) / sizeof((a)[0]))
#endif

static u32 c_count ;	/* Contention count	*/
static u32 i_count ;	/* Iteration count	*/
static u32 l_count ;	/* Load count		*/
static u32 p_count ;	/* Pause count		*/
static u32 locktype;	/* Lock type		*/
static u32 loadtype;	/* Load type		*/
static u32 r_percent;	/* Reader percentage	*/
static u32 rwstat;	/* RW stat		*/
static u32 latency;	/* Latency flag		*/
static atomic_t rcnt, wcnt;

int read_dummy;

/*
 * Lock type
 */
#define	LOCK_SPINLOCK	0
#define	LOCK_RWLOCK	1
#define LOCK_MUTEX	2
#define LOCK_RWSEM	3
#define LOCK_SPINIRQ	4
#define LOCK_MIXSPIN	5
#define LOCK_TRYRWSEM   6
#define LOCK_TERMSPIN	7
#define LOCK_OTHER	8

/*
 * Timing tests
 */
#define TIME_PAUSE	10	/* pause timing test  */
#define TIME_RMB	11	/* lfence timing test */
#define TIME_MB		12	/* mfence timing test */

/*
 * Load type
 * 1) Standalone - lock & protected data in separate cachelines
 * 2) Embedded   - lock & protected data in the same cacheline
 */
#define	LOAD_STANDALONE	0
#define	LOAD_EMBEDDED	1

/*
 * External functions
 */
extern void *__kmalloc_node(size_t size, gfp_t flags, int node);

/*
 * Locks
 */
struct load {
	atomic_t c1, c2;
};

struct s_lock {
	struct {
		spinlock_t  lock;
		struct load data;
	} ____cacheline_aligned_in_smp;
};

struct rw_lock {
	struct {
		rwlock_t    lock;
		struct load data;
	} ____cacheline_aligned_in_smp;
};

struct mutex_lock {
	struct {
		struct mutex lock;
		struct load  data;
	} ____cacheline_aligned_in_smp;
};

struct rwsem_lock {
	struct {
		struct rw_semaphore lock;
		struct load	    data;
		atomic_t	    readers;
		int		    readers_max, batches, total_reads;
		int		    rlock_ok, rlock_fail;
		int		    wlock_ok, wlock_fail;
	} ____cacheline_aligned_in_smp;
};

struct lstat {
	bool reader;
	long latency_max;
	long latency_total;
};

static struct s_lock     *slock, *tlock;
static struct rw_lock    *rwlock;
static struct mutex_lock *mutex;
static struct rwsem_lock *rwsem;

#ifndef lockdep_set_terminal_class
#define lockdep_set_terminal_class(lock)	
#endif

/*
 * Compute time difference in microsecond (us)
 */
static unsigned long compute_us(struct timespec *start, struct timespec *stop)
{
	return ((stop->tv_sec  - start->tv_sec ) * NSEC_PER_SEC +
		(stop->tv_nsec - start->tv_nsec) + 500)/1000;
}

static noinline int load(struct load *data, int ltype, int write, int lcnt)
{
	/*
	 * Negative lcnt means sleep for lcnt us
	 */
	if (lcnt < 0) {
		usleep_range(-lcnt, -lcnt+1);
	} else if (ltype == LOAD_STANDALONE) {
		for (; lcnt > 0; lcnt--)
			cpu_relax();
	} else if (write) {
		for (; lcnt > 0; lcnt--)
			(void)atomic_add(lcnt, &data->c1);
	} else {
		int sum;
		for (sum = 0; lcnt > 0; lcnt--) {
			sum += atomic_read(&data->c1) + atomic_read(&data->c2);
			rmb();
		}
		return sum;
	}
	return 0;
}

static noinline int test_spinlock(struct s_lock *lock)
{
	int i = i_count ;
	int p = p_count ;
	int c = l_count ;
	int l = loadtype;
	int j		;

	if (i < 0) {
		/*
		 * Use a fixed timeout.
		 */
		unsigned long timeout = jiffies + (-i) * HZ;

		for (i = 0; !time_after(jiffies, timeout); i++) {
			spin_lock(&lock->lock);
			load(&lock->data, l, 1, c);
			spin_unlock(&lock->lock);
			for (j = p ; j > 0 ; j--)
				cpu_relax();
			if (need_resched())
				cond_resched();
		}
		return i;
	}

	for ( ; i > 0 ; i--) {
		spin_lock(&lock->lock);
		load(&lock->data, l, 1, c);
		spin_unlock(&lock->lock);
		for (j = p ; j > 0 ; j--)
			cpu_relax();
		if (need_resched())
			cond_resched();
	}
	return 0;
}

static noinline void test_spinirq(struct s_lock *lock)
{
	int i = i_count ;
	int p = p_count ;
	int c = l_count ;
	int l = loadtype;
	int j		;

	for ( ; i > 0 ; i--) {
		spin_lock_irq(&lock->lock);
		load(&lock->data, l, 1, c);
		spin_unlock_irq(&lock->lock);
		for (j = p ; j > 0 ; j--)
			cpu_relax();
		if (need_resched())
			cond_resched();
	}
}

/*
 * A negative reader percentage is interpreted as the number of reader-only
 * threads required. The rests will be writer-only thread.
 */
static noinline int test_rwlock(int cnt, struct lstat *stat)
{
	int i = i_count ;
	int p = p_count ;
	int c = l_count ;
	int l = loadtype;
	int r = r_percent;
	unsigned long timeout = 0;
	int j;

	if (stat)
		stat->reader = 0;

	if (i < 0) {
		timeout = jiffies + (-i) * HZ;
		i = MAX_INT;
	}

	if (r < 0)
		goto xthread;

	/*
	 * Mixed reader/writer thread
	 */
	for (; i > 0 ; i--) {
		if (i%100 >= r) {
			/* Write lock */
			write_lock(&rwlock->lock);
			load(&rwlock->data, l, 1, c);
			write_unlock(&rwlock->lock);
		} else {
			/* Read lock */
			read_lock(&rwlock->lock);
			load(&rwlock->data, l, 0, c);
			read_unlock(&rwlock->lock);
		}
		for (j = p ; j > 0 ; j--)
			cpu_relax();
		if (timeout && time_after(jiffies, timeout))
			break;
		if (need_resched())
			cond_resched();
	}
	return timeout ? MAX_INT - i : 0;

xthread:
	/*
	 * Reader or writer only thread
	 */
	if (cnt < -r) {
		/*
		 * Reader only thread
		 */
		atomic_inc(&rcnt);
		if (stat)
			stat->reader = 1;
		while (--i >= 0) {
			read_lock(&rwlock->lock);
			load(&rwlock->data, l, 0, c);
			read_unlock(&rwlock->lock);

			if (timeout) {
				unsigned long now = jiffies;

				if (time_after(now, timeout))
					break;
				/*
				 * Exit if all the writers gone.
				 */
				if (time_after(now, timeout - 1) &&
				    !atomic_read(&wcnt))
					break;
			}

			for (j = p ; j > 0 ; j--)
				cpu_relax();
			if (need_resched())
				cond_resched();
		}
		atomic_dec(&rcnt);
	} else {
		/*
		 * Writer only thread
		 */
		atomic_inc(&wcnt);
		while (--i >= 0) {
			write_lock(&rwlock->lock);
			load(&rwlock->data, l, 1, c);
			write_unlock(&rwlock->lock);

			if (timeout) {
				unsigned long now = jiffies;
				
				if (time_after(jiffies, timeout))
					break;
				/*
				 * Exit if all the readers gone.
				 */
				if (time_after(now, timeout - 1) &&
				    !atomic_read(&rcnt))
				    	break;
			}

			for (j = p ; j > 0 ; j--)
				cpu_relax();
			if (need_resched())
				cond_resched();
		}
		atomic_dec(&wcnt);
	}
	return timeout ? MAX_INT - i : 0;
}

static noinline void test_mutex(void)
{
	int i = i_count ;
	int p = p_count ;
	int c = l_count ;
	int l = loadtype;
	int j		;

	for ( ; i > 0 ; i--) {
		mutex_lock(&mutex->lock);
		load(&mutex->data, l, 1, c);
		mutex_unlock(&mutex->lock);
		for (j = p ; j > 0 ; j--)
			cpu_relax();
		if (need_resched())
			cond_resched();
	}
}

static inline void update_stat(struct lstat *stat, u64 before)
{
	long delta = sched_clock() - before;

	if (delta > stat->latency_max)
		stat->latency_max = delta;
	stat->latency_total += delta;
}

/*
 * A negative reader percentage is interpreted as the number of reader-only
 * threads required. The rests will be writer-only thread.
 */
static noinline int test_rwsem(int cnt, struct lstat *stat)
{
	int i = i_count ;
	int p = p_count ;
	int c = l_count ;
	int l = loadtype;
	int r = r_percent;
	int s = rwstat;
	int lat = 0;
	unsigned long timeout = 0;
	u64 before;
	int max, j;

	if (stat) {
		stat->reader = 0;
		stat->latency_max = 0;
		stat->latency_total = 0;
		lat = latency;
	}

	if (i < 0) {
		timeout = jiffies + (-i) * HZ;
		i = MAX_INT;
	}

	if (r < 0)
		goto xthread;

	/*
	 * Mixed reader/writer thread
	 */
	while (--i >= 0) {
		if (i%100 >= r) {
			/* Write lock */
			if (lat)
				before = sched_clock();
			down_write(&rwsem->lock);
			if (lat)
				update_stat(stat, before);

			load(&rwsem->data, l, 1, c);
			if (s) {
				max = atomic_read(&rwsem->readers);
				if (max) {
					if (rwsem->readers_max < max)
						rwsem->readers_max = max;
					rwsem->batches++;
					rwsem->total_reads += max;
					atomic_set(&rwsem->readers, 0);
				}
			}
			up_write(&rwsem->lock);
		} else {
			/* Read lock */
			if (lat)
				before = sched_clock();
			down_read(&rwsem->lock);
			if (lat)
				update_stat(stat, before);
			load(&rwsem->data, l, 0, c);
			if (s)
				atomic_inc(&rwsem->readers);
			up_read(&rwsem->lock);
		}
		if (timeout && time_after(jiffies, timeout))
			break;

		for (j = p ; j > 0 ; j--)
			cpu_relax();
		if (need_resched())
			cond_resched();
	}
	return timeout ? MAX_INT - i : 0;

xthread:
	/*
	 * Reader or writer only thread
	 */
	if (cnt < -r) {
		/*
		 * Reader only thread
		 */
		atomic_inc(&rcnt);
		if (stat)
			stat->reader = 1;
		while (--i >= 0) {
			if (lat)
				before = sched_clock();
			down_read(&rwsem->lock);
			if (lat)
				update_stat(stat, before);
			load(&rwsem->data, l, 0, c);
			if (s)
				atomic_inc(&rwsem->readers);
			up_read(&rwsem->lock);

			if (timeout) {
				unsigned long now = jiffies;

				if (time_after(now, timeout))
					break;
				/*
				 * Exit if all the writers gone.
				 */
				if (time_after(now, timeout - 1) &&
				    !atomic_read(&wcnt))
					break;
			}

			for (j = p ; j > 0 ; j--)
				cpu_relax();
			if (need_resched())
				cond_resched();
		}
		atomic_dec(&rcnt);
	} else {
		/*
		 * Writer only thread
		 */
		atomic_inc(&wcnt);
		while (--i >= 0) {
			if (lat)
				before = sched_clock();
			down_write(&rwsem->lock);
			if (lat)
				update_stat(stat, before);
			load(&rwsem->data, l, 1, c);
			if (s) {
				max = atomic_read(&rwsem->readers);
				if (max) {
					if (rwsem->readers_max < max)
						rwsem->readers_max = max;
					rwsem->batches++;
					rwsem->total_reads += max;
					atomic_set(&rwsem->readers, 0);
				}
			}
			up_write(&rwsem->lock);

			if (timeout) {
				unsigned long now = jiffies;
				
				if (time_after(jiffies, timeout))
					break;
				/*
				 * Exit if all the readers gone.
				 */
				if (time_after(now, timeout - 1) &&
				    !atomic_read(&rcnt))
				    	break;
			}

			for (j = p ; j > 0 ; j--)
				cpu_relax();
			if (need_resched())
				cond_resched();
		}
		atomic_dec(&wcnt);
	}
	return timeout ? MAX_INT - i : 0;
}

static noinline int test_tryrwsem(int cnt, struct lstat *stat)
{
	int i = i_count ;
	int p = p_count ;
	int c = l_count ;
	int l = loadtype;
	int r = r_percent;
	unsigned long timeout = 0;
	int j;

	if (i < 0) {
		timeout = jiffies + (-i) * HZ;
		i = MAX_INT;
	}

	if (stat) {
		stat->reader = 0;
		stat->latency_max = 0;
		stat->latency_total = 0;
	}

	/*
	 * Mixed reader/writer thread
	 */
	while (--i >= 0) {
		if (i%100 >= r) {
			/* Write lock */
			if (!down_write_trylock(&rwsem->lock)) {
				rwsem->wlock_fail++;
				continue;
			}
			load(&rwsem->data, l, 1, c);
			rwsem->wlock_ok++;
			up_write(&rwsem->lock);
		} else {
			/* Read lock */
			if (!down_read_trylock(&rwsem->lock)) {
				rwsem->rlock_fail++;
				continue;
			}
			load(&rwsem->data, l, 0, c);
			rwsem->rlock_ok++;
			up_read(&rwsem->lock);
		}
		if (timeout && time_after(jiffies, timeout))
			break;

		for (j = p ; j > 0 ; j--)
			cpu_relax();
		if (need_resched())
			cond_resched();
	}
	return timeout ? MAX_INT - i : 0;
}

static noinline int test_pause(char *buf)
{
	struct timespec start, stop;
	unsigned long us;
	int i;

	/*
	 * Measure time with a 1-pause loop, then with a 11-pause loop.
	 * Get the time difference & divided by 10.
	 */
	getnstimeofday(&start);
	for (i = i_count; i && !READ_ONCE(read_dummy); i--) {
		cpu_relax();
		if (need_resched())
			cond_resched();
	}
	getnstimeofday(&stop);
	us = compute_us(&start, &stop);
	getnstimeofday(&start);
	for (i = i_count; i && !READ_ONCE(read_dummy); i--) {
		cpu_relax();	/* 01 */
		cpu_relax();	/* 02 */
		cpu_relax();	/* 03 */
		cpu_relax();	/* 04 */
		cpu_relax();	/* 05 */
		cpu_relax();	/* 06 */
		cpu_relax();	/* 07 */
		cpu_relax();	/* 08 */
		cpu_relax();	/* 09 */
		cpu_relax();	/* 10 */
		cpu_relax();	/* 11 */
		if (need_resched())
			cond_resched();
	}
	getnstimeofday(&stop);

	us = (compute_us(&start, &stop) - us + 5)/10;
	return sprintf(buf, "%ld\n", us);
}

static noinline int test_rmb(char *buf)
{
	struct timespec start, stop;
	unsigned long us;
	int i;

	/*
	 * Measure time with a 1-rmb loop, then with a 11-rmb loop.
	 * Get the time difference & divided by 10.
	 */
	getnstimeofday(&start);
	for (i = i_count; i && !READ_ONCE(read_dummy); i--) {
		rmb();
		if (need_resched())
			cond_resched();
	}
	getnstimeofday(&stop);
	us = compute_us(&start, &stop);
	getnstimeofday(&start);
	for (i = i_count; i && !READ_ONCE(read_dummy); i--) {
		rmb();	/* 01 */
		rmb();	/* 02 */
		rmb();	/* 03 */
		rmb();	/* 04 */
		rmb();	/* 05 */
		rmb();	/* 06 */
		rmb();	/* 07 */
		rmb();	/* 08 */
		rmb();	/* 09 */
		rmb();	/* 10 */
		rmb();	/* 11 */
		if (need_resched())
			cond_resched();
	}
	getnstimeofday(&stop);

	us = (compute_us(&start, &stop) - us + 5)/10;
	return sprintf(buf, "%ld\n", us);
}

static noinline int test_mb(char *buf)
{
	struct timespec start, stop;
	unsigned long us;
	int i;

	/*
	 * Measure time with a 1-mb loop, then with a 11-mb loop.
	 * Get the time difference & divided by 10.
	 */
	getnstimeofday(&start);
	for (i = i_count; i && !READ_ONCE(read_dummy); i--) {
		mb();
		if (need_resched())
			cond_resched();
	}
	getnstimeofday(&stop);
	us = compute_us(&start, &stop);
	getnstimeofday(&start);
	for (i = i_count; i && !READ_ONCE(read_dummy); i--) {
		mb();	/* 01 */
		mb();	/* 02 */
		mb();	/* 03 */
		mb();	/* 04 */
		mb();	/* 05 */
		mb();	/* 06 */
		mb();	/* 07 */
		mb();	/* 08 */
		mb();	/* 09 */
		mb();	/* 10 */
		mb();	/* 11 */
		if (need_resched())
			cond_resched();
	}
	getnstimeofday(&stop);

	us = (compute_us(&start, &stop) - us + 5)/10;
	return sprintf(buf, "%ld\n", us);
}

static noinline void test_other(void)
{
	printk(KERN_INFO "Info: run recursive read_lock test\n");
	/*
	 * Take recursive read_lock three times
	 */
	read_lock(&rwlock->lock);
	cpu_relax();
	read_lock(&rwlock->lock);
	cpu_relax();
	read_lock(&rwlock->lock);
	cpu_relax();
	read_unlock(&rwlock->lock);
	cpu_relax();
	read_unlock(&rwlock->lock);
	cpu_relax();
	read_unlock(&rwlock->lock);

	printk(KERN_INFO "Info: run rwsem lock/unlock mismatch test\n");
	up_write(&rwsem->lock);
	down_read(&rwsem->lock);
	init_rwsem(&rwsem->lock);
}

/*
 * Test mixed spinlock types
 * Even - spin_lock
 * Odd  - spin_lock_irq
 */
static noinline void test_mixspin(int count)
{
	int i = i_count ;
	int p = p_count ;
	int c = l_count ;
	int l = loadtype;
	int j		;

	if (count & 1) {
		for ( ; i > 0 ; i--) {
			spin_lock_irq(&slock->lock);
			load(&slock->data, l, 1, c);
			spin_unlock_irq(&slock->lock);
			for (j = p ; j > 0 ; j--)
				cpu_relax();
			if (need_resched())
				cond_resched();
		}
	} else {
		for ( ; i > 0 ; i--) {
			spin_lock(&slock->lock);
			load(&slock->data, l, 1, c);
			spin_unlock(&slock->lock);
			for (j = p ; j > 0 ; j--)
				cpu_relax();
			if (need_resched())
				cond_resched();
		}
	}
}

static unsigned int etime_show(char *buf)
{
	struct timespec start, stop;
	int cnt, n = 0;
	struct lstat stat;
	char *end = buf;

	stat.reader = 0;
	stat.latency_max = 0;
	if (locktype == TIME_PAUSE)
		return test_pause(buf);	/* pasue instruction timing test */
	else if (locktype == TIME_RMB)
		return test_rmb(buf);	/* lfence instruction timing test */
	else if (locktype == TIME_MB)
		return test_mb(buf);	/* mfence instruction timing test */

	cnt = atomic_dec_return((atomic_t *)&c_count);

	/* Wait until the count reaches 0 */
	while (READ_ONCE(c_count)) {
		cpu_relax();
		if (need_resched())
			cond_resched();
	}

	getnstimeofday(&start);
	if (locktype == LOCK_SPINLOCK)
		n = test_spinlock(slock);
	else if (locktype == LOCK_RWLOCK)
		n = test_rwlock(cnt, &stat);
	else if (locktype == LOCK_MUTEX)
		test_mutex();
	else if (locktype == LOCK_RWSEM)
		n = test_rwsem(cnt, &stat);
	else if (locktype == LOCK_TRYRWSEM)
		n = test_tryrwsem(cnt, &stat);
	else if (locktype == LOCK_OTHER)
		test_other();
	else if (locktype == LOCK_SPINIRQ)
		test_spinirq(slock);
	else if (locktype == LOCK_TERMSPIN)
		test_spinirq(tlock);
	else if (locktype == LOCK_MIXSPIN)
		test_mixspin(cnt);
	getnstimeofday(&stop);
	end += sprintf(buf, "%ld %d %c", compute_us(&start, &stop),
		       n, stat.reader ? 'r' : 'w');
	if (latency && stat.latency_max)
		end += sprintf(end, "%ld %ld", stat.latency_total/n,
			       stat.latency_max);
	return end - buf + 1;
}

static ssize_t etime_read(struct file *file, char __user *user_buf,
			  size_t count, loff_t *ppos)
{
	char buf[1024];
	unsigned int len = etime_show(buf);

	return simple_read_from_buffer(user_buf, count, ppos, buf, len);
}

static ssize_t etime_write(struct file *file, const char __user *user_buf,
			   size_t count, loff_t *ppos)
{
	atomic_set(&rcnt, 0);
	atomic_set(&wcnt, 0);
	return count;
}

static ssize_t dumplock_write(struct file *file, const char __user *user_buf,
			      size_t count, loff_t *ppos)
{
	if ((count > 5) && !memcmp(user_buf, "rwsem", 5) && rwsem) {
		printk("rwsem.count = 0x%lx\n",
			atomic_long_read(&rwsem->lock.count));
		printk("rwsem.owner = 0x%lx\n", (long)rwsem->lock.owner);
	}
	return count;
}

static unsigned int status_show(char *buf)
{
	char *end = buf;

	if (locktype == LOCK_RWSEM) {
		/*
		 * Print out status of the rwsem
		 */
		end += sprintf(end, "rwsem.count     = 0x%lx\n",
			atomic_long_read((atomic_long_t *)&rwsem->lock.count));
		end += sprintf(end, "rwsem.wait_list = %sempty\n",
			rwsem_is_contended(&rwsem->lock) ? "not" : "");
#ifdef raw_spin_is_locked
		end += sprintf(end, "rwsem.wait_lock = %slocked\n",
			raw_spin_is_locked(&rwsem->lock.wait_lock)
			? "" : "not ");
#endif
#ifdef CONFIG_RWSEM_SPIN_ON_OWNER
		end += sprintf(end, "rwsem.osq       = %d\n",
			atomic_read(&rwsem->lock.osq.tail));
		end += sprintf(end, "rwsem.owner     = 0x%lx\n",
			(unsigned long)rwsem->lock.owner);
#endif
		/*
		 * Print reader batches information
		 */
		end += sprintf(end, "readers_max     = %d\n",
				rwsem->readers_max);
		end += sprintf(end, "reader batches  = %d\n",
				rwsem->batches);
		end += sprintf(end, "total reads     = %d\n\n",
				rwsem->total_reads);

		atomic_set(&rwsem->readers, 0);
		rwsem->readers_max = rwsem->batches = rwsem->total_reads = 0;
	} else if (locktype == LOCK_TRYRWSEM) {
		/*
		 * Print out the # of successful trylocks
		 * N.B. The counts may not be accurate.
		 */
		end += sprintf(end, "rwsem.rlock_ok  = %d\n",
				rwsem->rlock_ok);
		end += sprintf(end, "rwsem.rlock_fail= %d\n",
				rwsem->rlock_fail);
		end += sprintf(end, "rwsem.wlock_ok  = %d\n",
				rwsem->wlock_ok);
		end += sprintf(end, "rwsem.wlock_fail= %d\n",
				rwsem->wlock_fail);
		rwsem->rlock_ok = rwsem->rlock_fail =
		rwsem->wlock_ok = rwsem->wlock_fail = 0;
	}
	return end - buf;
}

static ssize_t status_read(struct file *file, char __user *user_buf,
			   size_t count, loff_t *ppos)
{
	char buf[1024];
	unsigned int len = status_show(buf);

	return simple_read_from_buffer(user_buf, count, ppos, buf, len);
}

static struct {
	char *name;
	u32  *pvar;
} debugfs_u32_vars[] = {
	{ "c_count"  , &c_count   },
	{ "i_count"  , &i_count   },
	{ "l_count"  , &l_count   },
	{ "p_count"  , &p_count   },
	{ "locktype" , &locktype  },
	{ "loadtype" , &loadtype  },
	{ "r_percent", &r_percent },
	{ "latency"  , &latency   },
	{ "rwstat"   , &rwstat    },
};

static const struct file_operations fops_etime = {
	.read  = etime_read,
	.write = etime_write,
	.llseek = default_llseek,
};

static const struct file_operations fops_status = {
	.read  = status_read,
	.llseek = default_llseek,
};

static const struct file_operations fops_dumplock = {
	.write  = dumplock_write,
	.llseek = default_llseek,
};


static struct dentry *locktest_dir;

/*
 * Module init function
 */
static int __init locktest_init(void)
{
	int i;

	locktest_dir = debugfs_create_dir("locktest", NULL);
	if (!locktest_dir)
		return -ENOMEM;

	for (i = 0 ; i < ARRAY_SIZE(debugfs_u32_vars); i++) {
		if (!debugfs_create_u32(debugfs_u32_vars[i].name,
			0644, locktest_dir, debugfs_u32_vars[i].pvar))
			goto err_out;
	}

	if (!debugfs_create_file("etime", 0644, locktest_dir, NULL,
				 &fops_etime) ||
	    !debugfs_create_file("status", 0644, locktest_dir, NULL,
	    			 &fops_status) ||
	    !debugfs_create_file("dumplock", 0200, locktest_dir, NULL,
	    			 &fops_dumplock))
		goto err_out;

	/*
	 * Allocate node 0 memory for locks
	 */
	slock  = __kmalloc_node(sizeof(*slock ), GFP_KERNEL, 0);
	tlock  = __kmalloc_node(sizeof(*tlock ), GFP_KERNEL, 0);
	rwlock = __kmalloc_node(sizeof(*rwlock), GFP_KERNEL, 0);
	mutex  = __kmalloc_node(sizeof(*mutex ), GFP_KERNEL, 0);
	rwsem  = __kmalloc_node(sizeof(*rwsem ), GFP_KERNEL, 0);

	if (!slock || !rwlock || !mutex || !rwsem) {
		printk(KERN_WARNING "locktest: __kmalloc_node failed!\n");
		goto err_out;
	}
	spin_lock_init(&slock->lock);
	spin_lock_init(&tlock->lock);
	rwlock_init(&rwlock->lock);
	mutex_init(&mutex->lock);
	init_rwsem(&rwsem->lock);
	lockdep_set_terminal_class(&tlock->lock);

	printk(KERN_INFO "locktest module loaded!\n");

	// Non-zero return means that the module couldn't be loaded.
	return 0;

err_out:
	debugfs_remove_recursive(locktest_dir);
	return -ENOMEM;
}

static void __exit locktest_cleanup(void)
{
	/*
	kfree(slock );
	kfree(rwlock);
	kfree(mutex );
	kfree(rwsem );
	*/
	printk(KERN_INFO "locktest module unloaded.\n");
	if (locktest_dir)
		debugfs_remove_recursive(locktest_dir);
}

module_init(locktest_init);
module_exit(locktest_cleanup);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Waiman Long");
MODULE_DESCRIPTION("Lock testing module");
