#!/bin/sh
#

# Command line arguments.
SECS=$1
CLIENTS=$2
PORTS=$3
PERIOD=$4

# Parameter defaults.
[ -z "${SECS}"    ] && SECS=300
[ -z "${CLIENTS}" ] && CLIENTS=20
[ -z "${PORTS}"   ] && PORTS=4
[ -z "${PERIOD}"  ] && PERIOD=64

# Local tools must be on same directory...
BASEDIR=`dirname $0`

# Make sure our local tools are in place.
(cd ${BASEDIR}; make all > /dev/null) || exit 1

# nmeter configuration.
NMETER="${BASEDIR}/jack_test3_nmeter"
NMETER_ARGS="t c i x b m"

# jackd configuration.
JACKD="/usr/bin/jackd"
JACKD_DRIVER="alsa"
JACKD_DEVICE="hw:0"
JACKD_PRIO=60
JACKD_RATE=44100
JACKD_PORTS=$(((${CLIENTS} + 1) * ${PORTS} * 2))
JACKD_ARGS="-v -R -P${JACKD_PRIO} -p${JACKD_PORTS} -d${JACKD_DRIVER} -d${JACKD_DEVICE} -r${JACKD_RATE} -p${PERIOD} -n2 -S -P"

# client test configuration.
CLIENT="${BASEDIR}/jack_test3_client"
CLIENT_ARGS="${SECS} ${PORTS}"

# process/thread list configuration.
PIDOF="/sbin/pidof"
PLIST="ps -o pid,tid,class,rtprio,ni,pri,pcpu,stat,comm"

# Log filename.
LOG="jack_test3-`uname -r`-`date +'%Y%m%d%H%M'`.log"

# Command executive and logger.
exec_log ()
{
	CMD="$*"
	echo "-----------------------"		>> ${LOG} 2>&1
	echo "# ${CMD}"				>> ${LOG} 2>&1
	${CMD}					>> ${LOG} 2>&1
	echo					>> ${LOG} 2>&1
}

# Process/thread status loggers.
plist_log ()
{
	PIDS="$*"
	if [ -n "${PIDS}" ]; then
		echo "- - - - - - - - - - - -"	>> ${LOG} 2>&1
		${PLIST} -m ${PIDS}		>> ${LOG} 2>&1
		echo				>> ${LOG} 2>&1
	fi
}


# IRQ thread status loggers.
ilist_log ()
{
	echo "- - - - - - - - - - - -"	>> ${LOG} 2>&1
	${PLIST} --sort -rtprio -e \
	| egrep '(^[ ]+PID|IRQ|jack)'	>> ${LOG} 2>&1
	echo				>> ${LOG} 2>&1
}


#
# Log headings -- show some relevant info...
#
echo "*** Started `date` ***" >> ${LOG} 2>&1

echo >> ${LOG} 2>&1
echo "Seconds to run        (SECS) = ${SECS}"		>> ${LOG} 2>&1
echo "Number of clients  (CLIENTS) = ${CLIENTS}"	>> ${LOG} 2>&1
echo "Ports per client     (PORTS) = ${PORTS}"		>> ${LOG} 2>&1
echo "Frames per buffer   (PERIOD) = ${PERIOD}"		>> ${LOG} 2>&1
echo >> ${LOG} 2>&1

exec_log "uname -a"

exec_log "cat /proc/asound/version"
exec_log "cat /proc/asound/cards"

#exec_log "grep . /proc/sys/kernel/*_preemption"
#exec_log "grep . /proc/irq/*/*/threaded"
#exec_log "grep . /sys/block/hd*/queue/max_sectors_kb"

exec_log "cat /proc/interrupts"

# This is just about to be sure...
ilist_log 

#
# Lauch nmeter in the background...
#
echo -n "Launching `basename ${NMETER}`..."
(${NMETER} ${NMETER_ARGS} >> ${LOG} 2>&1) &
sleep 2
echo "done."
sleep 1

# Lauch the test client suite...
SLEEP=6
echo -n "Launching ${CLIENTS} ${CLIENT}(s) instance(s)..."
for NUM in `seq 1 ${CLIENTS}`; do
	CLIENT_CMDLINE="${CLIENT} ${CLIENT_ARGS}"
	SLEEP=$((${SLEEP} + 3))
	(sleep ${SLEEP}; echo -n "$NUM..."; exec_log ${CLIENT_CMDLINE}) &
done
echo "done."

# Let there be some evidence of process status...
SLEEP=$((${SLEEP} + 6))
(sleep ${SLEEP}; plist_log -C `basename ${JACKD}`; plist_log -C `basename ${CLIENT}`) &

# Let jackd live for extended but limited time...
SLEEP=$((${SLEEP} + ${SECS}))
(sleep ${SLEEP}; killall `basename ${JACKD}`) &

#
# Lauch jackd and wait for it...
#
echo -n "Running `basename ${JACKD}`..."
exec_log ${JACKD} ${JACKD_ARGS}
echo "done."
sleep 1

#echo -n "Killing `basename ${CLIENT}`..."
#killall `basename ${CLIENT}` && echo "OK." || echo "FAILED."
#sleep 1

echo -n "Killing `basename ${NMETER}`..."
killall `basename ${NMETER}` && echo "OK." || echo "FAILED."

echo "*** Terminated `date` ***" >> ${LOG} 2>&1

sync
sleep 1
sync

# Summary log analynis...
cat ${LOG} | awk -f ${BASEDIR}/jack_test3_summary.awk | tee -a ${LOG}

# Finally, plot log output...
${BASEDIR}/jack_test3_plot.sh ${LOG}
