/* jack_test4_client.c */
/****************************************************************************
   Copyright (C) 2004, Florian Schimdt.
   Copyright (C) 2005, rncbc aka Rui Nuno Capela.

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version 2
   of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

*****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include <jack/jack.h>

jack_client_t *client;
jack_port_t **iports;
jack_port_t **oports;

unsigned int  seconds_to_run = 60;
unsigned int  num_of_ports   = 4;
unsigned int  client_id      = 0;
unsigned int  client_count   = 1;

unsigned long signal_frame   = 0;
unsigned int  signal_index   = 0;
unsigned int  signal_period  = 0;

jack_default_audio_sample_t *signal;

float         signal_max_delta = 0.0;
unsigned long signal_max_frame = 0;

#define SIGNAL_GAIN			0.1
#define SIGNAL_FREQ			1000

#define SIGNAL_SYNC_PULSE	1.0
#define SIGNAL_SYNC_THRESH	(SIGNAL_SYNC_PULSE * (1.0 - SIGNAL_GAIN))

/* setup the test signal (wave)table. */
void init_signal (const char *cname, float gain, float freq, 
	jack_nframes_t srate)
{
	unsigned int index;
	
	signal_frame  = 0;
	signal_index  = 0;
	signal_period = (unsigned int) ((float) srate / freq);

	signal = (jack_default_audio_sample_t *)
		malloc(signal_period * sizeof(jack_default_audio_sample_t));

	signal[0] = SIGNAL_SYNC_PULSE;
	for (index = 1; index < signal_period; index++) {
		signal[index] = gain *
			sin(2.0 * M_PI * (float) index / (float) signal_period);
	}

	fprintf(stderr, "%s: init_signal(gain=%g,freq=%g,srate=%g,period=%u)\n",
		cname, gain, freq, (float) srate, signal_period);

	signal_max_delta = 0.0;
	signal_max_frame = 0;
}

/* stand-alone process. mix all the input ports to each output port. */
int process_0 (jack_nframes_t frames, void *arg)
{
	jack_default_audio_sample_t *ibuf;
	jack_default_audio_sample_t *obuf;
	jack_nframes_t k;
	int i, j;

	for (i = 0; i < num_of_ports; i++) {
		obuf = (jack_default_audio_sample_t*)
			jack_port_get_buffer(oports[i], frames);
		for (j = 0; j < num_of_ports; j++) {
			ibuf = (jack_default_audio_sample_t*)
				jack_port_get_buffer(iports[j], frames);
			for (k = 0; k < frames; k++) {
				if (j == 0)
					obuf[k] = 0.0;
				if (ibuf[k] < -1E-9 || ibuf[k] > +1E-9)
					obuf[k] += ibuf[k] / (float) num_of_ports;
			}
		}
	}
	return 1;
}

/* signal generator process. */
int process_1 (jack_nframes_t frames, void *arg)
{
	jack_default_audio_sample_t *obuf;
	unsigned long frame;
	unsigned int  index;
	jack_nframes_t k;
	int i;

	for (i = 0; i < num_of_ports; i++) {
		obuf = (jack_default_audio_sample_t*)
			jack_port_get_buffer(oports[i], frames);
		frame = signal_frame;
		index = signal_index;
		for (k = 0; k < frames; k++, frame++) {
			obuf[k] = signal[index];
			if (++index >= signal_period)
			    index = 0;
		}
	}
	signal_index = index;
	signal_frame = frame;
	return 1;
}

/* normal signal process. mix all the input ports to each output port. */
int process_N (jack_nframes_t frames, void *arg)
{
	jack_default_audio_sample_t *ibuf;
	jack_default_audio_sample_t *obuf;
	unsigned long frame;
	unsigned int  index;
	jack_default_audio_sample_t sample;
	jack_nframes_t k;
	int i, j;
	float delta;

	for (i = 0; i < num_of_ports; i++) {
		obuf = (jack_default_audio_sample_t*)
			jack_port_get_buffer(oports[i], frames);
		for (j = 0; j < num_of_ports; j++) {
			ibuf = (jack_default_audio_sample_t*)
				jack_port_get_buffer(iports[j], frames);
			frame = signal_frame;
			index = signal_index;
			for (k = 0; k < frames; k++) {
				sample = ibuf[k];
				if (frame == 0 && sample < SIGNAL_SYNC_THRESH)
					continue;	/* wait for sync pulse. */
				if (j == 0)
					obuf[k] = 0.0; 
				if (sample < -1E-9 || sample > +1E-9) {
					if (client_id < client_count ||
						sample < SIGNAL_SYNC_THRESH) {
						obuf[k] += sample / (float) num_of_ports;
					}
					delta = fabsf(signal[index] - sample);
					if (signal_max_delta < delta) {
						signal_max_delta = delta;
						signal_max_frame = frame;
						fprintf(stderr, "*** jack_test4-%u:"
							" delta=%g frame=%u\n",
							client_id, delta, frame);
					}
				}
				if (++index >= signal_period)
					index = 0;
				frame++;
			}
		}
	}
	/* take last one as next. */
	signal_index = index; 
	signal_frame = frame; 
	return 1;
}


int main(int argc, char *argv[])
{
	char client_name[33];
	char iport_name[33];
	char oport_name[33];
	char prev_client_name[33];
	unsigned int flags;
	const char **pports;
	int i;

	/* seconds_to_run: default = 60 seconds. */
	if (argc > 1)
		seconds_to_run = (unsigned int) atoi(argv[1]);
	fprintf(stderr, "seconds to run: %u\n", seconds_to_run);

	/* num_of_ports: default = 4 ports. */
	if (argc > 2)
		num_of_ports = (unsigned int) atoi(argv[2]);
	fprintf(stderr, "num.of ports: %u\n", num_of_ports);

	/* client_id: default = 0 (unchained). */
	if (argc > 3) {
		client_id = (unsigned int) atoi(argv[3]);
		fprintf(stderr, "client id: %u\n", client_id);
	}

	/* client_count: default = 0 (unchained). */
	if (argc > 4) {
		client_count = (unsigned int) atoi(argv[4]);
		fprintf(stderr, "client count: %u\n", client_count);
	}

	sprintf(client_name, "jack_test4-%u",
		(client_id > 0 ? client_id : getpid()));
	fprintf(stderr, "client_name: %s\n", client_name);

	fprintf(stderr, "%s: client_new\n", client_name);
	client = jack_client_new(client_name);
	if (!client) {
		fprintf(stderr, "%s: jackd not running?\n", client_name);
		return 1;
	}

	/* allocate a 1Khz sinusoid reference signal. */
	if (client_id > 0) {
		init_signal(client_name, SIGNAL_GAIN, SIGNAL_FREQ,
			jack_get_sample_rate(client));
	}
	
	iports = (jack_port_t **) malloc(num_of_ports * sizeof(jack_port_t *));
	oports = (jack_port_t **) malloc(num_of_ports * sizeof(jack_port_t *));

	fprintf(stderr, "%s: port_register\n", client_name);
	for (i = 0; i < num_of_ports; i++) {
		if (client_id != 1) {
			sprintf(iport_name, "in_%d", i);
			flags = 0;
			if (client_id >= client_count)
				flags |= JackPortIsTerminal;
			iports[i] = jack_port_register(client, iport_name,
				JACK_DEFAULT_AUDIO_TYPE, JackPortIsInput|flags, 0);
			if (iports[i] == NULL) {
				fprintf(stderr, "%s:%s port registration failed\n", 
					client_name, iport_name);
				goto exit1;
			}
		}
		sprintf(oport_name, "out_%d", i);
		flags = 0;
		if (client_id == 1)
			flags |= JackPortIsTerminal;
		oports[i] = jack_port_register(client, oport_name,
			JACK_DEFAULT_AUDIO_TYPE, JackPortIsOutput|flags, 0);
		if (oports[i] == NULL) {
			fprintf(stderr, "%s:%s port registration failed\n", 
				client_name, oport_name);
			goto exit1;
		}
	}

	fprintf(stderr, "%s: set_process_callback\n", client_name);
	switch (client_id) {
	case 0:
		jack_set_process_callback(client, process_0, 0);
		break;
	case 1:
		jack_set_process_callback(client, process_1, 0);
		break;
	default:
		jack_set_process_callback(client, process_N, 0);
		break;
	}

	fprintf(stderr, "%s: activate\n", client_name);
	jack_activate(client);

	/* try to connect to previous in chain... */
	if (client_id > 1) {
		sprintf(prev_client_name, "jack_test4-%d", (client_id - 1));
		for (i = 0; i < num_of_ports; i++) {
			sprintf(oport_name, "%s:out_%d", prev_client_name, i);
			sprintf(iport_name, "%s:in_%d", client_name, i);
			fprintf(stderr, "%s: connect: %s -> %s\n",
				client_name, oport_name, iport_name);
			if (jack_connect(client, oport_name, iport_name) != 0) {
				fprintf(stderr, "%s: connect failed\n");
				goto exit2;
			}
		}
	}

	/* try to connect to available physical outputs. */
	if (client_id >= client_count) {
		pports = jack_get_ports(client, 0, 0,
			JackPortIsInput | JackPortIsPhysical);
		if (pports) {
			for (i = 0; i < num_of_ports && pports[i]; i++) {
				sprintf(oport_name, "%s:out_%d", client_name, i);
				fprintf(stderr, "%s: connect: %s -> %s\n",
					client_name, oport_name, pports[i]);
				if (jack_connect(client, oport_name, pports[i]) != 0) {
					fprintf(stderr, "%s: connect failed\n");
					goto exit2;
				}
			}
			free(pports);
		}
	}

	/* ok, we're up and running... */
	fprintf(stderr, "%s: running...\n", client_name);
	sleep(seconds_to_run);

exit2:

	fprintf(stderr, "%s: deactivate\n", client_name);
	jack_deactivate(client);

exit1:

	fprintf(stderr, "%s: close\n", client_name);
	jack_client_close(client);

	free(iports);
	free(oports);

	/* final report. */
	if (client_id > 0) {
		fprintf(stderr, "%s: max_delta = %g", client_name, signal_max_delta);
		if (signal_max_delta > 0.0)
			fprintf(stderr, " (%g dB)",
				 20.0 * log10(SIGNAL_GAIN / signal_max_delta));
		fprintf(stderr, ", max_frame = %lu, frames = %lu\n",
			signal_max_frame, signal_frame);
		free(signal);
	}

	return 0;
}

/* end of jack_test4_client.c */
