#include <linux/init.h>
#include <linux/version.h>
#include <linux/module.h>
#include <linux/slab.h>
#include <linux/compiler.h>
#include <asm/cacheflush.h>
#include <asm/io.h>
#include <asm/msr.h>

#define NVDIMM_ADDR	0x240000000		// CHANGE THIS!
#define MAP_SIZE	(100*1024*1024)		// 100 MB

static void test_read(void *addr)
{
	unsigned char *wp = (unsigned char *) addr;
	unsigned char tmp;
	u64 tsc1, tsc2;
	int i;

	rdtscll(tsc1);
	for (i=0; i<(MAP_SIZE/sizeof(wp)); i++) {
		tmp = ACCESS_ONCE(*wp);
		wp++;
	}
	rdtscll(tsc2);
	pr_notice("test_read:  %10llu\n", (tsc2-tsc1));
}

static void test_write(void *addr)
{
	unsigned char *wp = (unsigned char *) addr;
	u64 tsc1, tsc2;
	int i;

	rdtscll(tsc1);
	for (i=0; i<(MAP_SIZE/sizeof(wp)); i++) {
		ACCESS_ONCE(*wp) = 1;
		wp++;
	}
	rdtscll(tsc2);
	pr_notice("test_write: %10llu\n", (tsc2-tsc1));
}

void test_set_memory_RAM(void)
{
	void *addr;
	int ret;

	pr_notice("=== test_set_memory_RAM ===\n");

	addr = kmalloc(sizeof(PAGE_SIZE), GFP_KERNEL);
	if (!addr)
		return;

	pr_notice("call set_memory_wt\n");
	ret = set_memory_wt((unsigned long)addr, 1);
	if (ret)
		pr_err("set_memory_wt failed %d\n", ret);

	pr_notice("call set_memory_uc\n");
	ret = set_memory_uc((unsigned long)addr, 1);
	if (ret)
		pr_err("set_memory_uc failed %d\n", ret);

	pr_notice("call set_memory_wb\n");
	ret = set_memory_wb((unsigned long)addr, 1);
	if (ret)
		pr_err("set_memory_wb failed %d\n", ret);

	kfree((void *)addr);
}

void test_ioremap_wt(void)
{
	void *addr;

	pr_notice("=== test_ioremap_wt ===\n");

	pr_notice("ioremap_wt\n");
	addr = ioremap_wt(NVDIMM_ADDR, MAP_SIZE);
	if (!addr) {
		pr_err("ioremap_wt failed\n");
		return;
	}

	test_write(addr);
	test_read(addr);

	iounmap(addr);
}

void test_ioremap_wc(void)
{
	void *addr;

	pr_notice("=== test_ioremap_wc ===\n");

	pr_notice("ioremap_wc\n");
	addr = ioremap_wc(NVDIMM_ADDR, MAP_SIZE);
	if (!addr) {
		pr_err("ioremap_wc failed\n");
		return;
	}

	test_write(addr);
	test_read(addr);

	iounmap(addr);
}

void test_ioremap(void)
{
	void *addr;

	pr_notice("=== test_ioremap ===\n");

	pr_notice("ioremap\n");
	addr = ioremap(NVDIMM_ADDR, MAP_SIZE);
	if (!addr) {
		pr_err("ioremap failed\n");
		return;
	}

	test_write(addr);
	test_read(addr);

	iounmap(addr);
}

void test_ioremap_prot(void)
{
	pgprot_t prot;
	void *addr;

	pr_notice("=== test_ioremap_prot ===\n");

	pgprot_val(prot) = 0;
	prot = pgprot_writethrough(prot);

	pr_notice("ioremap_prot\n");
	addr = ioremap_prot(NVDIMM_ADDR, MAP_SIZE, pgprot_val(prot));
	if (!addr) {
		pr_err("ioremap_prot failed\n");
		return;
	}

	test_write(addr);
	test_read(addr);

	iounmap(addr);
}

void test_set_memory_NVD(void)
{
	void *addr;
	int ret;

	pr_notice("=== test_set_memory_NVD ===\n");

	pr_notice("ioremap_cache\n");
	addr = ioremap_cache(NVDIMM_ADDR, MAP_SIZE);
	if (!addr) {
		pr_err("ioremap_cache failed\n");
		return;
	}

	test_write(addr);
	test_read(addr);

	pr_notice("call set_memory_wt\n");
	ret = set_memory_wt((unsigned long)addr, (MAP_SIZE >> PAGE_SHIFT));
	if (ret)
		pr_err("set_memory_wt failed %d\n", ret);

	test_write(addr);
	test_read(addr);

	pr_notice("call set_memory_wb\n");
	ret = set_memory_wb((unsigned long)addr, (MAP_SIZE >> PAGE_SHIFT));
	if (ret)
		pr_err("set_memory_wb failed %d\n", ret);

	test_write(addr);
	test_read(addr);

	iounmap(addr);
}

void test_set_memory_array_wt_NVD(void)
{
	void *addr;
	unsigned long *alist;
	int i, n, ret;

	pr_notice("=== test_set_memory_array_wt_NVD ===\n");

	pr_notice("ioremap_cache\n");
	addr = ioremap_cache(NVDIMM_ADDR, MAP_SIZE);
	if (!addr) {
		pr_err("ioremap_cache failed\n");
		return;
	}

	test_write(addr);
	test_read(addr);

	n = (MAP_SIZE >> PAGE_SHIFT);

	alist = (unsigned long *)kmalloc((sizeof(unsigned long) * n),
						GFP_KERNEL);
	if (!alist) {
		pr_err("kmalloc failed\n");
		return;
	}

	for (i = 0; i < n; i++)
		alist[i] = (unsigned long)addr + (PAGE_SIZE * i);

	pr_notice("call set_memory_array_wt\n");
	ret = set_memory_array_wt(alist, n);
	if (ret)
		pr_err("set_memory_array_wt failed %d\n", ret);

	test_write(addr);
	test_read(addr);

	pr_notice("call set_memory_array_wb\n");
	ret = set_memory_array_wb(alist, n);
	if (ret)
		pr_err("set_memory_array_wb failed %d\n", ret);

	test_write(addr);
	test_read(addr);

	kfree((void *)alist);
	iounmap(addr);
}

void test_set_memory_array_wc_NVD(void)
{
	void *addr;
	unsigned long *alist;
	int i, n, ret;

	pr_notice("=== test_set_memory_array_wc_NVD ===\n");

	pr_notice("ioremap_cache\n");
	addr = ioremap_cache(NVDIMM_ADDR, MAP_SIZE);
	if (!addr) {
		pr_err("ioremap_cache failed\n");
		return;
	}

	test_write(addr);
	test_read(addr);

	n = (MAP_SIZE >> PAGE_SHIFT);

	alist = (unsigned long *)kmalloc((sizeof(unsigned long) * n),
						GFP_KERNEL);
	if (!alist) {
		pr_err("kmalloc failed\n");
		return;
	}

	for (i = 0; i < n; i++)
		alist[i] = (unsigned long)addr + (PAGE_SIZE * i);

	pr_notice("call set_memory_array_wc\n");
	ret = set_memory_array_wc(alist, n);
	if (ret)
		pr_err("set_memory_array_wc failed %d\n", ret);

	test_write(addr);
	test_read(addr);

	pr_notice("call set_memory_array_wb\n");
	ret = set_memory_array_wb(alist, n);
	if (ret)
		pr_err("set_memory_array_wb failed %d\n", ret);

	test_write(addr);
	test_read(addr);

	kfree((void *)alist);
	iounmap(addr);
}

void test_set_memory_array_uc_NVD(void)
{
	void *addr;
	unsigned long *alist;
	int i, n, ret;

	pr_notice("=== test_set_memory_array_uc_NVD ===\n");

	pr_notice("ioremap_cache\n");
	addr = ioremap_cache(NVDIMM_ADDR, MAP_SIZE);
	if (!addr) {
		pr_err("ioremap_cache failed\n");
		return;
	}

	test_write(addr);
	test_read(addr);

	n = (MAP_SIZE >> PAGE_SHIFT);

	alist = (unsigned long *)kmalloc((sizeof(unsigned long) * n),
						GFP_KERNEL);
	if (!alist) {
		pr_err("kmalloc failed\n");
		return;
	}

	for (i = 0; i < n; i++)
		alist[i] = (unsigned long)addr + (PAGE_SIZE * i);

	pr_notice("call set_memory_array_uc\n");
	ret = set_memory_array_uc(alist, n);
	if (ret)
		pr_err("set_memory_array_uc failed %d\n", ret);

	test_write(addr);
	test_read(addr);

	pr_notice("call set_memory_array_wb\n");
	ret = set_memory_array_wb(alist, n);
	if (ret)
		pr_err("set_memory_array_wb failed %d\n", ret);

	test_write(addr);
	test_read(addr);

	kfree((void *)alist);
	iounmap(addr);
}

void test_set_pages_array_wt_NVD(void)
{
	void *addr;
	unsigned long pfn;
	struct page **plist;
	int i, n, ret;

	pr_notice("=== test_set_pages_array_wt_NVD ===\n");

	pr_notice("ioremap_cache\n");
	addr = ioremap_cache(NVDIMM_ADDR, MAP_SIZE);
	if (!addr) {
		pr_err("ioremap_cache failed\n");
		return;
	}

	test_write(addr);
	test_read(addr);

	pfn = (unsigned long)__pa(addr) >> PAGE_SHIFT;
	n = MAP_SIZE >> PAGE_SHIFT;

	plist = (struct page **)kmalloc((sizeof(struct page *)*n), GFP_KERNEL);
	if (!plist) {
		pr_err("kmalloc failed\n");
		return;
	}

	for (i = 0; i < n; i++)
		plist[i] = pfn_to_page(pfn + i);

	pr_notice("call set_pages_array_wt\n");
	ret = set_pages_array_wt(plist, n);
	if (ret)
		pr_err("set_pages_array_wt failed %d\n", ret);

	test_write(addr);
	test_read(addr);

	pr_notice("call set_pages_array_wb\n");
	ret = set_pages_array_wb(plist, n);
	if (ret)
		pr_err("set_pages_array_wb failed %d\n", ret);

	test_write(addr);
	test_read(addr);

	kfree((void *)plist);
	iounmap(addr);
}

void test_set_pages_array_wc_NVD(void)
{
	void *addr;
	unsigned long pfn;
	struct page **plist;
	int i, n, ret;

	pr_notice("=== test_set_pages_array_wc_NVD ===\n");

	pr_notice("ioremap_cache\n");
	addr = ioremap_cache(NVDIMM_ADDR, MAP_SIZE);
	if (!addr) {
		pr_err("ioremap_cache failed\n");
		return;
	}

	test_write(addr);
	test_read(addr);

	pfn = (unsigned long)__pa(addr) >> PAGE_SHIFT;
	n = MAP_SIZE >> PAGE_SHIFT;

	plist = (struct page **)kmalloc((sizeof(struct page *)*n), GFP_KERNEL);
	if (!plist) {
		pr_err("kmalloc failed\n");
		return;
	}

	for (i = 0; i < n; i++)
		plist[i] = pfn_to_page(pfn + i);

	pr_notice("call set_pages_array_wc\n");
	ret = set_pages_array_wc(plist, n);
	if (ret)
		pr_err("set_pages_array_wc failed %d\n", ret);

	test_write(addr);
	test_read(addr);

	pr_notice("call set_pages_array_wb\n");
	ret = set_pages_array_wb(plist, n);
	if (ret)
		pr_err("set_pages_array_wb failed %d\n", ret);

	test_write(addr);
	test_read(addr);

	kfree((void *)plist);
	iounmap(addr);
}

void test_set_pages_array_uc_NVD(void)
{
	void *addr;
	unsigned long pfn;
	struct page **plist;
	int i, n, ret;

	pr_notice("=== test_set_pages_array_uc_NVD ===\n");

	pr_notice("ioremap_cache\n");
	addr = ioremap_cache(NVDIMM_ADDR, MAP_SIZE);
	if (!addr) {
		pr_err("ioremap_cache failed\n");
		return;
	}

	test_write(addr);
	test_read(addr);

	pfn = (unsigned long)__pa(addr) >> PAGE_SHIFT;
	n = MAP_SIZE >> PAGE_SHIFT;

	plist = (struct page **)kmalloc((sizeof(struct page *)*n), GFP_KERNEL);
	if (!plist) {
		pr_err("kmalloc failed\n");
		return;
	}

	for (i = 0; i < n; i++)
		plist[i] = pfn_to_page(pfn + i);

	pr_notice("call set_pages_array_uc\n");
	ret = set_pages_array_uc(plist, n);
	if (ret)
		pr_err("set_pages_array_uc failed %d\n", ret);

	test_write(addr);
	test_read(addr);

	pr_notice("call set_pages_array_wb\n");
	ret = set_pages_array_wb(plist, n);
	if (ret)
		pr_err("set_pages_array_wb failed %d\n", ret);

	test_write(addr);
	test_read(addr);

	kfree((void *)plist);
	iounmap(addr);
}

int test_init(void)
{
	pr_notice("test_init\n");
	test_set_memory_RAM();
	test_set_memory_NVD();
	test_set_memory_array_wt_NVD();
	test_set_memory_array_wc_NVD();
	test_set_memory_array_uc_NVD();
	test_set_pages_array_wt_NVD();
	test_set_pages_array_wc_NVD();
	test_set_pages_array_uc_NVD();
	test_ioremap();
	test_ioremap_wt();
	test_ioremap_wc();
	test_ioremap_prot();
	return 0;
}
module_init(test_init);

void test_exit(void)
{
	pr_notice("test_exit\n");
}
module_exit(test_exit);

MODULE_LICENSE("GPL v2");
