/*
 * Copyright (c) 2010 Philip Frey, Systems Group ETH Zurich.
 *
 * This software is available to you under a choice of one of two
 * licenses.  You may choose to be licensed under the terms of the GNU
 * General Public License (GPL) Version 2, available from the file
 * COPYING in the main directory of this source tree, or the
 * OpenIB.org BSD license below:
 *
 *     Redistribution and use in source and binary forms, with or
 *     without modification, are permitted provided that the following
 *     conditions are met:
 *
 *      - Redistributions of source code must retain the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer.
 *
 *      - Redistributions in binary form must reproduce the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer in the documentation and/or other materials
 *        provided with the distribution.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

/*
 * iWARP functions.
 */

#ifndef BASIC_RDMA_H_
#define BASIC_RDMA_H_

#include <stdio.h>

#include "iwarp_objs.h"

/* shared rdma library (OFED) */
#ifdef __x86_64__
	#define RDMA_LIB_PATH	"/usr/lib64/librdmacm.so"	//TODO: set at Makefile
//	#define RDMA_LIB_PATH	"/usr/local/lib/librdmacm.so"
#else
	#define RDMA_LIB_PATH	"/usr/lib/librdmacm.so"
#endif

/* syntactic sugar */
#define IN
#define OUT

/*
 * default qp init attrs (for Chelsio T3)
 *
 * empirical max values that do not result in a failure upon QP creation:
 * max_send_wr		16384
 * max_recv_wr		1023
 * max_send_sge		4
 * max_recv_sge		4
 * max_inline_data	64
 */
#define MAX_INLINE_DATA	64
#define MAX_SEND_WR		16384
#define MAX_RECV_WR		1023
#define MAX_SEND_SGE	4
#define	MAX_RECV_SGE	4

/* default connection parameters */
#define RESPONDER_RESOURCES		8	/* outbound read queue depth */
#define INITIATOR_DEPTH			8	/* inbound read queue depth */
#define RETRY_COUNT				3

struct iw_lib *lib;

#define sem_wait_safe(sem)									\
	if (lib && lib->cm_thread_running && !stop_cm_thread) {	\
		sem_wait(sem);										\
	}														\

////////////////////////////////////////////////////////////////////////////////
// Helper Functions
////////////////////////////////////////////////////////////////////////////////

#define max(a, b)	\
	(a > b ? a : b)

#define min(a, b)	\
	(a < b ? a : b)

/*
 * Make sure that the iwarp connection context is not a NULL pointer.
 *
 * @ctx_conn:	the iwarp connection context which should be checked
 *
 * Return 1 if the context is NULL; 0 otherwise.
 */
#define ctx_null(ctx_conn)			\
	ctx_null0(ctx_conn, __func__)
static inline int ctx_null0(
		IN const struct iw_ctx_conn *ctx_conn,
		IN const char*				 func)
{
	if (!ctx_conn) {
		printf("ERROR: [%s] iwarp connection context missing\n", func);
		return 1;
	} else {
		return 0;
	}
}


/*
 * Convert the iw_state enum into its string representation.
 *
 * @iw_state:	the iwarp state in its enum representation
 *
 * Return the string representing the iwarp state or "UNKNOWN STATE" if the
 * state is not known.
 */
static inline const char* iw_state_str(
		IN const enum iw_state state)
{
	switch (state) {
	case IWARP_INVALID:
		return "IWARP_INVALID";
	case IWARP_ADDR_RES:
		return "IWARP_ADDR_RES";
	case IWARP_ROUTE_RES:
		return "IWARP_ROUTE_RES";
	case IWARP_CONN_REQ:
		return "IWARP_CONN_REQ";
	case IWARP_VALID:
		return "IWARP_VALID";
	case IWARP_REJECTED:
		return "IWARP_REJECTED";
	case IWARP_CONNECTED:
		return "IWARP_CONNECTED";
	case IWARP_DISCONNECTED:
		return "IWARP_DISCONNECTED";
	case IWARP_ERROR:
		return "IWAPR_ERROR";
	default:
		return "UNKNOWN STATE";
	}
}


/*
 * Convert the wc status into its string representation.
 *
 * @wc_status:	the wc status in its enum representation
 *
 * Return the string representing the wc status or "UNKNOWN STATUS" if the
 * status is not known.
 */
static inline const char* wc_status_str(
		IN const enum ibv_wc_status wc_status)
{
	switch (wc_status) {
	case IBV_WC_SUCCESS:
		return "IBV_WC_SUCCESS";
	case IBV_WC_LOC_LEN_ERR:
		return "IBV_WC_SUCCESS";
	case IBV_WC_LOC_QP_OP_ERR:
		return "IBV_WC_LOC_QP_OP_ERR";
	case IBV_WC_LOC_EEC_OP_ERR:
		return "IBV_WC_LOC_EEC_OP_ERR";
	case IBV_WC_LOC_PROT_ERR:
		return "IBV_WC_LOC_PROT_ERR";
	case IBV_WC_WR_FLUSH_ERR:
		return "IBV_WC_WR_FLUSH_ERR";
	case IBV_WC_MW_BIND_ERR:
		return "IBV_WC_MW_BIND_ERR";
	case IBV_WC_BAD_RESP_ERR:
		return "IBV_WC_BAD_RESP_ERR";
	case IBV_WC_LOC_ACCESS_ERR:
		return "IBV_WC_LOC_ACCESS_ERR";
	case IBV_WC_REM_INV_REQ_ERR:
		return "IBV_WC_REM_INV_REQ_ERR";
	case IBV_WC_REM_ACCESS_ERR:
		return "IBV_WC_REM_ACCESS_ERR";
	case IBV_WC_REM_OP_ERR:
		return "IBV_WC_REM_OP_ERR";
	case IBV_WC_RETRY_EXC_ERR:
		return "IBV_WC_RETRY_EXC_ERR";
	case IBV_WC_RNR_RETRY_EXC_ERR:
		return "IBV_WC_RNR_RETRY_EXC_ERR";
	case IBV_WC_LOC_RDD_VIOL_ERR:
		return "IBV_WC_LOC_RDD_VIOL_ERR";
	case IBV_WC_REM_INV_RD_REQ_ERR:
		return "IBV_WC_REM_INV_RD_REQ_ERR";
	case IBV_WC_REM_ABORT_ERR:
		return "IBV_WC_REM_ABORT_ERR";
	case IBV_WC_INV_EECN_ERR:
		return "IBV_WC_INV_EECN_ERR";
	case IBV_WC_INV_EEC_STATE_ERR:
		return "IBV_WC_INV_EEC_STATE_ERR";
	case IBV_WC_FATAL_ERR:
		return "IBV_WC_FATAL_ERR";
	case IBV_WC_RESP_TIMEOUT_ERR:
		return "IBV_WC_RESP_TIMEOUT_ERR";
	case IBV_WC_GENERAL_ERR:
		return "IBV_WC_GENERAL_ERR";
	default:
		return "UNKNOWN STATUS";
	}
}


/*
 * Convert the rdma connection management event type into its
 * string representation.
 *
 * @rdma_event:	the rdma event type in its enum representation
 *
 * Return the string representing the rdma event type or "UNKNOWN STATUS" if it
 * is not known.
 */
static inline const char* rdma_evt_str(
		IN enum rdma_cm_event_type rdma_event)
{
	switch(rdma_event) {
	case RDMA_CM_EVENT_ADDR_RESOLVED:
		return "RDMA_CM_EVENT_ADDR_RESOLVED";
	case RDMA_CM_EVENT_ADDR_ERROR:
		return "RDMA_CM_EVENT_ADDR_ERROR";
	case RDMA_CM_EVENT_ROUTE_RESOLVED:
		return "RDMA_CM_EVENT_ROUTE_RESOLVED";
	case RDMA_CM_EVENT_ROUTE_ERROR:
		return "RDMA_CM_EVENT_ROUTE_ERROR";
	case RDMA_CM_EVENT_CONNECT_REQUEST:
		return "RDMA_CM_EVENT_CONNECT_REQUEST";
	case RDMA_CM_EVENT_CONNECT_RESPONSE:
		return "RDMA_CM_EVENT_CONNECT_RESPONSE";
	case RDMA_CM_EVENT_CONNECT_ERROR:
		return "RDMA_CM_EVENT_CONNECT_ERROR";
	case RDMA_CM_EVENT_UNREACHABLE:
		return "RDMA_CM_EVENT_UNREACHABLE";
	case RDMA_CM_EVENT_REJECTED:
		return "RDMA_CM_EVENT_REJECTED";
	case RDMA_CM_EVENT_ESTABLISHED:
		return "RDMA_CM_EVENT_ESTABLISHED";
	case RDMA_CM_EVENT_DISCONNECTED:
		return "RDMA_CM_EVENT_DISCONNECTED";
	case RDMA_CM_EVENT_DEVICE_REMOVAL:
		return "RDMA_CM_EVENT_DEVICE_REMOVAL";
	case RDMA_CM_EVENT_MULTICAST_JOIN:
		return "RDMA_CM_EVENT_MULTICAST_JOIN";
	case RDMA_CM_EVENT_MULTICAST_ERROR:
		return "RDMA_CM_EVENT_MULTICAST_ERROR";
	default:
		return "UNKNOWN EVENT";
	}
}


/*
 * Check if the iwarp connection context is in the expected state.
 *
 * Return 1 (true) if the state of @ctx_conn matches @expected; 0 (false)
 * otherwise.
 */
#define ctx_state(expected, ctx_conn)			\
	ctx_state0(expected, ctx_conn, __func__)
static inline int ctx_state0(
		IN	const enum iw_state			 expected,
		IN	const struct iw_ctx_conn	*ctx_conn,
		IN	const char*					 func)
{
	if (ctx_null(ctx_conn)) {
		return 0;
	}
	if (ctx_conn->state != expected) {
		printf("ERROR: [%s] connection context is in state %s instead of %s\n",
				func, iw_state_str(ctx_conn->state),
				iw_state_str(expected));
		return 0;
	}

	return 1;

}


////////////////////////////////////////////////////////////////////////////////
// Connection Management
////////////////////////////////////////////////////////////////////////////////

/*
 * Allocate per-connection iWARP context:
 * - allocate protection domain
 * - create send completion channel
 * - create receive completion channel
 * - create send completion queue
 * - create receive completion queue
 * - create queue pair
 *
 * IMPORTANT: - the type must be set to CTX_CONNECT (active side) or
 * 				CTX_ACCEPT (passive side) before calling this function to
 *              specify what type of context should be created
 *            - function pointers MUST NOT be set BEFORE calling this function
 *              as they will be cleared here
 *
 * Client context state transition:
 * IWARP_INVALID => IWARP_VALID
 *
 * Server context state transition:
 * IWARP_CONN_REQ => IWARP_VALID
 *
 * @remote_host:	IP address or name of the remote host
 * 					(ignored for CTX_ACCEPT)
 * @remote_port:	port on which the remote host is listening
 * 					(ignored for CTX_ACCEPT)
 * @qp_cap:			initial qp capacity settings (NULL for default)
 * 					it contains the values applied when this function returns
 * @shared_pd:		if not NULL, specifies a shared protection domain in which
 * 					the context should be created
 * @create_srq		whether or not to create a shared receive queue
 * @existing_srq	attach this srq to the qp (NULL for none)
 * @ctx_conn:		iwarp connection context
 * 					- if passive context: state must be IWARP_CONN_REQ and the
 * 					  cm_id must be valid
 * 					- if active context: state must be IWARP_INVALID
 * 					  (context will be 0-filled)
 *
 * This function will have no effect in case of a failure.
 *
 * Return 0 on success; -1 otherwise.
 */
int iw_ctx_alloc(
		IN		const char*			 remote_host,
		IN		const char*			 remote_port,
		IN OUT	struct ibv_qp_cap	*qp_cap,
		IN OUT	struct iw_pd		*shared_pd,
		IN		int					 create_srq,
		IN		struct iw_srq		*existing_srq,
		IN OUT	struct iw_ctx_conn 	*ctx_conn);


/*
 * Destory per-connection iWARP context:
 * - destroy qp
 * - destroy receive cq
 * - destroy send cq
 * - destroy receive completion channel
 * - destroy send completion channel
 * - deallocate pd (if it is not used anymore)
 * - free private data
 *
 * @ctx_conn:	the context to be destroyed
 *
 * Return 0 on success; -1 otherwise.
 */
int iw_ctx_free(
		IN OUT 	struct iw_ctx_conn	*ctx_conn);


/*
 * Connect to a host waiting for iwarp connection requests.
 *
 * @priv_data:		private data to append to connection request
 * @priv_data_len:	length of the private data in bytes
 * @ctx_conn:		allocated iwarp context (must be in IWARP_VALID state)
 * 					(use iw_ctx_alloc() for allocation)
 *
 * Send a connection request to the passive side and wait for the connection
 * response. If after RETRY_COUNT retries, there is still no
 * RDMA_CM_EVENT_ESTABLISHED the connection attempt has failed and the function
 * returns -1.
 *
 * In case of success, the private data field of the iwarp connection context
 * was calloc'ed and contains the private data sent by the remote host. It must
 * be freed by using either iw_disconnect(), iw_await_disconnect() or manually
 * using free() when it is no longer needed.
 *
 * Return 0 on success; -1 otherwise.
 */
int iw_connect(
		IN		const void				*priv_data,
		IN		uint8_t					 priv_data_len,
		IN OUT	struct iw_ctx_conn		*ctx_conn);


/*
 * Create a listen context which is listening for inbound connection requests.
 *
 * @local_ip:	the IP address of the device to listen on (0 = any)
 * @local_port:	the port on which to listen
 * @backlog:	the number of connections that the backlog can hold
 * 				(waiting for calls to listen)
 * @listen_ctx:	the new listen context
 *
 * Return 0 on success; -1 otherwise.
 */
int iw_open(
		IN		const char				*local_ip,
		IN		const char				*local_port,
		IN		int						 backlog,
		IN OUT	struct iw_ctx_listen	*ctx_listen);


/*
 * Close an existing listen context.
 *
 * @listen_ctx:	the context to close
 *
 * Return 0 on success; -1 otherwise.
 */
int iw_close(
		IN OUT	struct iw_ctx_listen	*listen_ctx);


/*
 * Listen for an inbound connection request. This call will block until a
 * connection request has arrived.
 * The @ctx_conn returned is temporary and its state is set to IWARP_CONN_REQ.
 * It does not have a valid and connected qp yet. The only valid members are
 *  - state	(set to IWARP_CONN_REQ)
 *  - cm_id
 *  - priv_data_in_len
 * 	- priv_data_in		(sent by the remote host in the connection request)
 *
 * The @ctx_conn has to be handed over to iw_accept() or iw_reject() to
 * complete the connection establishment.
 *
 * In case of success, the private data field of the iwarp connection context
 * was calloc'ed and contains the private data sent by the remote host. It must
 * be freed by using either iw_disconnect(), iw_await_disconnect() or manually
 * using free() when it is no longer needed.
 *
 * @listen_ctx:	the context (previously created with iw_open) which specifies
 * 				the listen interface and cm id
 * @ctx_conn:	iwarp connection context in IWARP_INVALID state
 *
 * Return 0 on success; -1 otherwise.
 */
int iw_listen(
		IN OUT	struct iw_ctx_listen	*listen_ctx,
		IN OUT	struct iw_ctx_conn		*ctx_conn);


/*
 * Accept the inbound connection request referenced by @ctx_conn. The iwarp
 * connection context will be in state IWARP_CONNECTED if the accept call was
 * successful. Otherwise it will be in state IWARP_ERROR.
 *
 * @priv_data:		private data to append to connection response
 * @priv_data_len:	length of the private data in bytes
 * @ctx_conn:		allocated iwarp context (must be in IWARP_VALID state)
 * 					(use iw_ctx_alloc() on the context returend by
 * 					iw_listen() for allocation)
 *
 * Return 0 on success; -1 otherwise.
 */
int iw_accept(
		IN		const void				*priv_data,
		IN		uint8_t					 priv_data_len,
		IN OUT 	struct iw_ctx_conn		*ctx_conn);


/*
 * Reject the inbound connection request referenced by @ctx_conn. The iwarp
 * connection context @ctx_conn will be set to state IWARP_DISCONNECTED.
 *
 * @priv_data:		private data to append to connection response
 * @priv_data_len:	length of the private data in bytes
 * @ctx_conn:		allocated iwarp context (must be in IWARP_VALID state)
 * 					(use iw_ctx_alloc() on the context returend by
 * 					iw_listen() for allocation)
 *
 * Return 0 on success; -1 otherwise.
 */
int iw_reject(
		IN		const void			*priv_data,
		IN		uint8_t				 priv_data_len,
		IN OUT	struct iw_ctx_conn	*ctx_conn);


/*
 * Wait for a disconnect request from the remote peer.
 * This method blocks until a RDMA_CM_EVENT_DISCONNECT is received.
 *
 * The @ctx_conn will be set to state IWARP_DISCONNECTED when the disconnect
 * has completed.
 *
 * @ctx_conn:	the iwarp connection context which should be disconnected
 *
 * Return 0 on success; -1 otherwise.
 */
int iw_await_disconnect(
		IN OUT	struct iw_ctx_conn *ctx_conn);


/*
 * Disconnect from the remote host by closing the iwarp connection.
 *
 * The @ctx_conn will be set to state IWARP_DISCONNECTED when the disconnect
 * has completed.
 *
 * @ctx_conn:	the iwarp connection context which should be disconnected
 *
 * Return 0 on success; -1 otherwise.
 */
int iw_disconnect(
		IN OUT	struct iw_ctx_conn	*ctx_conn);


/*
 * Get local port number of the connection context.
 * Supports IPv4 and IPv6 port information.
 *
 * @ctx_conn:	iwarp connection context
 *
 * Return the local port on success; 0 otherwise.
 */
in_port_t iw_ctx_get_local_port(
		IN	const struct iw_ctx_conn	*ctx_conn);


/*
 * Get remote port number of the connection context.
 * Supports IPv4 and IPv6 port information.
 *
 * @ctx_conn:	iwarp connection context
 *
 * Return the remote port on success; 0 otherwise.
 */
in_port_t iw_ctx_get_remote_port(
		IN	const struct iw_ctx_conn	*ctx_conn);


/*
 * Get local IP address of the connection context.
 * Supports IPv4 and IPv6.
 *
 * @ctx_conn:	iwarp connection context
 *
 * Return a string representing the local IP in dotted notation on success;
 * an empty string otherwise.
 */
char* iw_ctx_get_local_ip(
		IN	const struct iw_ctx_conn	*ctx_conn);


/*
 * Get remote IP address of the connection context.
 * Supports IPv4 and IPv6.
 *
 * @ctx_conn:	iwarp connection context
 *
 * Return a string representing the remote IP in dotted notation on success;
 * an empty string otherwise.
 */
char* iw_ctx_get_remote_ip(
		IN	const struct iw_ctx_conn	*ctx_conn);


////////////////////////////////////////////////////////////////////////////////
// Memory Management
////////////////////////////////////////////////////////////////////////////////

/*
 * Allocate a new buffer and register it as a (local) memory region.
 *
 * @size:		the size in bytes of the new memory region
 * @access:		access permissions for this mr
 * @lmr:		the pre-allocated lmr struct through which the new memory region
 * 				will be returned
 * @ctx_conn:	the iwarp connection context
 *
 * The buffer for the memory region will be calloc'ed in this function - make
 * sure to free it again when it is no longer used (by using iw_lmr_destroy())!
 *
 * This method can be called before the iwarp connection is established. The
 * @ctx_conn has be either in state IWARP_VALID or IWARP_CONNECTED.
 *
 * Return 0 on success; -1 otherwise.
 */
int iw_lmr_create(
		IN		uint32_t				 size,
		IN		enum ibv_access_flags	 access,
		OUT		struct iw_lmr			*lmr,
		IN OUT	struct iw_ctx_conn		*ctx_conn);


/*
 * Register an existing buffera as a (local) memory region.
 *
 * @buf:		pointer to the already existing buffer of size > 0 which should
 * 				be made	available for RDMA transport
 * @size:		the length of @buf in bytes (must be > 0)
 * @access:		access permissions for this memory region
 * @lmr:		the pre-allocated lmr struct through which the new memory region
 * 				will be returned
 * @ctx_conn:	the iwarp connection context
 *
 * This method can be called before the iwarp connection is established. The
 * @ctx_conn has be either in state IWARP_VALID or IWARP_CONNECTED.
 *
 * Return 0 on success; -1 otherwise.
 */
int iw_lmr_register(
		IN		void					*buf,
		IN		uint32_t				 size,
		IN		enum ibv_access_flags	 access,
		OUT		struct iw_lmr			*lmr,
		IN OUT	struct iw_ctx_conn		*ctx_conn);


/*
 * Destroy a (local) memory region.
 *
 * The memory region is deregistered and if @free_buf is set to 1, its
 * underlying buffer is freed.
 *
 * @lmr:		the memory region to deregister
 * @free_buf:	iff set to 1, lmr->buf will be freed
 *
 * Return 0 on success; -1 otherwise.
 */
int iw_lmr_destroy(
		IN OUT	struct iw_lmr	*lmr,
		IN		int				 free_buf);


/*
 * Advertise the local buffer referred to by @lmr to the remote side.
 * The triple which is transmitted consists of:
 * - address (absolute)
 * - length of the buffer
 * - rkey
 *
 * @lmr:		the memory region which should be advertised
 * @ctx_conn:	the iwarp connection context
 *
 * Return 0 on success; -1 otherwise.
 */
int iw_post_send_adv(
		IN 		const struct iw_lmr	*lmr,
		IN OUT	struct iw_ctx_conn	*ctx_conn);


/*
 * Post a receive work request to prepare for the next inbound remote buffer
 * advertisement. This call does not block.
 *
 * @ctx_conn:	the iwarp connection context on which to wait for the remote
 * 				buffer advertisement
 *
 * Return the work request id of the receive wr on success; -1 otherwise.
 */
int iw_post_recv_adv(
		IN OUT	struct iw_ctx_conn	*ctx_conn);


/*
 * Wait for an inbound buffer advertisement.
 *
 * @rmr:			the rmr through which the received data will be returned
 * @expected_wr_id:	the work request id returned by the preceeding
 * 					iw_post_recv_adv() call
 * @ctx_conn:		the iwarp connection context
 *
 * Return 0 on success; -1 otherwise.
 */
int iw_wait_recv_adv(
		IN OUT	struct iw_rmr		*rmr,
		IN		unsigned int		 expected_wr_id,
		IN OUT	struct iw_ctx_conn	*ctx_conn);


////////////////////////////////////////////////////////////////////////////////
// RDMA Operations
////////////////////////////////////////////////////////////////////////////////

/*
 * Wait for the next @num_wc work completions on the target cq.
 *
 * This function may block and will only return once @num_wcs have been reaped
 * from the target cq.
 * If 0 is returned, the @wc array contains the @nm_wcs work completions
 * reaped from the cq.
 *
 * @cq:			the target cq on which to wait for the completions
 * @num_wcs:	number of wcs that are expected
 * @wc:			a pre-allocated array to return the work completoins
 * 				(size must be at least num_wcs)
 * @ctx_conn:	the connection context
 *
 * Return 0 on success; -1 otherwise.
 */
int await_completions(
		IN		enum iw_cq_type				 cq,
		IN		int							 num_wcs,
		IN OUT	struct ibv_wc				*wc,
		IN OUT	struct iw_ctx_conn			*ctx_conn);


/*
 * Post a receive work request onto the receive queue.
 *
 * This call returns immediately after having posted the work request.
 *
 * @dst:		the scatter-gather list into which inbound data will be placed
 * @ctx_conn:	the iwarp connection context
 *
 * Return the work request id on success; -1 otherwise.
 */
int post_recv_sgl(
		IN	const struct iw_sgl			*dst,
		IN	const struct iw_ctx_conn	*ctx_conn);


/*
 * Post a receive work request onto the receive queue.
 *
 * This call returns immediately after having posted the work request.
 *
 * @dst:		the memory region into which inbound data will be placed
 * @dst_off:	the offset within the mr where the data placement will start
 * @len:		the length of the data to be received
 * @ctx_conn:	the iwarp connection context
 *
 * Return the work request id on success; -1 otherwise.
 */
int post_recv_lmr(
		IN	const struct iw_lmr			*dst,
		IN	uint32_t					 dst_off,
		IN	uint32_t					 len,
		IN	const struct iw_ctx_conn	*ctx_conn);


/*
 * Post a send work request onto the send queue.
 *
 * This call returns immediately after having posted the work request.
 *
 * @src:		the scatter-gather list from which the data will be sent
 * @flags:		flags determining the behavior of the work request
 * 				(e.g. signaled or not)
 * @ctx_conn:	the iwarp connection context
 *
 * Return the work request id on success; -1 otherwise.
 */
int post_send_sgl(
		IN	const struct iw_sgl			*src,
		IN	enum ibv_send_flags			 flags,
		IN	const struct iw_ctx_conn	*ctx_conn);


/*
 * Post a send work request onto the send queue.
 *
 * This call returns immediately after having posted the work request.
 *
 * @src:		the memory region from which the data will be sent
 * @src_off:	the offset within the mr where the data transfer will start
 * @len:		the length of the data to be sent (in bytes)
 * @flags:		flags determining the behavior of the work request
 * 				(e.g. signaled or not)
 * @ctx_conn:	the iwarp connection context
 *
 * Return the work request id on success; -1 otherwise.
 */
int post_send_lmr(
		IN	const struct iw_lmr			*src,
		IN	uint32_t					 src_off,
		IN	uint32_t					 len,
		IN	enum ibv_send_flags			 flags,
		IN	const struct iw_ctx_conn	*ctx_conn);


/*
 * Post a write work request onto the send queue.
 *
 * This call returns immediately after having posted the work request.
 *
 * @src:		the scatter-gather list from which the data will be sent
 * @dst:		the remote memory region into which data will be written
 * @dst_off:	the offset within the remote mr
 * @flags:		flags determining the behavior of the work request
 * 				(e.g. signaled or not)
 * @ctx_conn:	the iwarp connection context
 *
 * Return the work request id on success; -1 otherwise.
 */
int post_write_sgl(
		IN	const struct iw_sgl			*src,
		IN	const struct iw_rmr			*dst,
		IN	uint32_t					 dst_off,
		IN	enum ibv_send_flags			 flags,
		IN	const struct iw_ctx_conn	*ctx_conn);


/*
 * Post a write work request onto the send queue.
 *
 * This call returns immediately after having posted the work request.
 *
 * @src:		the memory region from which the data will be sent
 * @src_off:	the offset within the mr where the data transfer will start
 * @dst:		the remote memory region into which data will be written
 * @dst_off:	the offset within the remote mr
 * @len:		the length of the data to be written (in bytes)
 * @flags:		flags determining the behavior of the work request
 * 				(e.g. signaled or not)
 * @ctx_conn:	the iwarp connection context
 *
 * Return the work request id on success; -1 otherwise.
 */
int post_write_lmr(
		IN	const struct iw_lmr			*src,
		IN	uint32_t					 src_off,
		IN	const struct iw_rmr			*dst,
		IN	uint32_t					 dst_off,
		IN	uint32_t					 len,
		IN	enum ibv_send_flags			 flags,
		IN	const struct iw_ctx_conn	*ctx_conn);


/*
 * Post a read work request onto the send queue.
 *
 * This call returns immediately after having posted the work request.
 *
 * @dst:		the scatter-gather list into which the data will be read
 * @src:		the remote memory region from which data will be read
 * @src_off:	the offset within the remote mr
 * @flags:		flags determining the behavior of the work request
 * 				(unsignaled read work requests are not supported by the Chelsio
 * 				 RNIC - a work completion is therefore always generated!)
 * @ctx_conn:	the iwarp connection context
 *
 * Return the work request id on success; -1 otherwise.
 */
int post_read_sgl(
		IN	const struct iw_sgl			*dst,
		IN	const struct iw_rmr			*src,
		IN	uint32_t					 src_off,
		IN	enum ibv_send_flags			 flags,
		IN	const struct iw_ctx_conn	*ctx_conn);


/*
 * Post a read work request onto the send queue.
 *
 * This call returns immediately after having posted the work request.
 *
 * @dst:		the memory region into which the data will be read
 * @dst_off:	the offset within the local mr where data placement will start
 * @src:		the remote memory region from which data will be read
 * @src_off:	the offset within the remote mr
 * @len:		the length of the data to be read (in bytes)
 * @flags:		flags determining the behavior of the work request
 * 				(unsignaled read work requests are not supported by the Chelsio
 * 				 RNIC - a work completion is therefore always generated!)
 * @ctx_conn:	the iwarp connection context
 *
 * Return the work request id on success; -1 otherwise.
 */
int post_read_lmr(
		IN	const struct iw_lmr			*dst,
		IN	uint32_t					 dst_off,
		IN	const struct iw_rmr			*src,
		IN	uint32_t					 src_off,
		IN	uint32_t					 len,
		IN	enum ibv_send_flags			 flags,
		IN	const struct iw_ctx_conn	*ctx_conn);

#endif /*BASIC_RDMA_H_*/
