/*
 * Copyright (c) 2010 Philip Frey, Systems Group ETH Zurich.
 *
 * This software is available to you under a choice of one of two
 * licenses.  You may choose to be licensed under the terms of the GNU
 * General Public License (GPL) Version 2, available from the file
 * COPYING in the main directory of this source tree, or the
 * OpenIB.org BSD license below:
 *
 *     Redistribution and use in source and binary forms, with or
 *     without modification, are permitted provided that the following
 *     conditions are met:
 *
 *      - Redistributions of source code must retain the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer.
 *
 *      - Redistributions in binary form must reproduce the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer in the documentation and/or other materials
 *        provided with the distribution.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

/*
 * Asynchronous event handling.
 */

#include <stdlib.h>

#include "iwarp.h"
#include "iwarp_debug.h"


/*
 * Print the error signaled by the event type as described in the manpage
 * of ibv_get_async_event().
 *
 * @ae:	the async event
 */
static inline void print_ae(struct ibv_async_event *ae)
{
	switch(ae->event_type) {
	/* QP events */
	case IBV_EVENT_QP_FATAL:
		dprint(DBG_AE, LOG_EVENT, "[QP] Error occurred on a QP and it "
				"transitioned to error state");
		break;
	case IBV_EVENT_QP_REQ_ERR:
		dprint(DBG_AE, LOG_EVENT, "[QP] Invalid Request Local Work Queue "
				"Error");
		break;
	case IBV_EVENT_QP_ACCESS_ERR:
		dprint(DBG_AE, LOG_EVENT, "[QP] Local access violation error");
		break;
	case IBV_EVENT_COMM_EST:
		dprint(DBG_AE, LOG_EVENT, "[QP] Communication was established on a QP");
		break;
	case IBV_EVENT_SQ_DRAINED:
		dprint(DBG_AE, LOG_EVENT, "[QP] Send Queue was drained of outstanding "
				"messages in progress");
		break;
	case IBV_EVENT_PATH_MIG:
		dprint(DBG_AE, LOG_EVENT, "[QP] A connection has migrated to the "
				"alternate path");
		break;
	case IBV_EVENT_PATH_MIG_ERR:
		dprint(DBG_AE, LOG_EVENT, "[QP] A connection failed to migrate to the "
				"alternate path");
		break;
	case IBV_EVENT_QP_LAST_WQE_REACHED:
		dprint(DBG_AE, LOG_EVENT, "[QP] Last WQE Reached on a QP associated "
				"with an SRQ");
		break;
	/* CQ events */
	case IBV_EVENT_CQ_ERR:
		dprint(DBG_AE, LOG_EVENT, "[CQ] CQ is in error (CQ overrun)");
		break;
	/* SRQ events */
	case IBV_EVENT_SRQ_ERR:
		dprint(DBG_AE, LOG_EVENT, "[SRQ] Error occurred on an SRQ");
		break;
	case IBV_EVENT_SRQ_LIMIT_REACHED:
		dprint(DBG_AE, LOG_EVENT, "[SRQ] SRQ limit was reached");
		break;
	/* Port events */
	case IBV_EVENT_PORT_ACTIVE:
		dprint(DBG_AE, LOG_EVENT, "[Port] Link became active on a port");
		break;
	case IBV_EVENT_PORT_ERR:
		dprint(DBG_AE, LOG_EVENT, "[Port] Link became unavailable on a port");
		break;
	case IBV_EVENT_LID_CHANGE:
		dprint(DBG_AE, LOG_EVENT, "[Port] LID was changed on a port");
		break;
	case IBV_EVENT_PKEY_CHANGE:
		dprint(DBG_AE, LOG_EVENT, "[Port] P_Key table was changed on a port");
		break;
	case IBV_EVENT_SM_CHANGE:
		dprint(DBG_AE, LOG_EVENT, "[Port] SM was changed on a port");
		break;
	case IBV_EVENT_CLIENT_REREGISTER:
		dprint(DBG_AE, LOG_EVENT, "[Port] SM sent a CLIENT_REREGISTER request "
				"to a port");
		break;
	/* CA events */
	case IBV_EVENT_DEVICE_FATAL:
		dprint(DBG_AE, LOG_EVENT, "[CA] CA is in FATAL state");
		break;

	default:
		dprint(DBG_AE, LOG_EVENT, "Unknown async event: %d", ae->event_type);
	}
}

/*
 * Asynchronous event handler.
 *
 * @arg:	pointer to the malloc'ed iw_ctx_aeq containing the *ibv_context
 */
static void *aeq_thread(void *arg)
{
	struct iw_ctx_aeq		*aeq_ctx = (struct iw_ctx_aeq*)arg;
	struct ibv_async_event	 ae;

	dprint(DBG_AE, LOG_INFO, "AEQ thread started for context %p",
			aeq_ctx->ibv_ctx);

	while (aeq_ctx->ref) {
		(*lib->ops.s_ibv_get_async_event)(aeq_ctx->ibv_ctx, &ae);
		print_ae(&ae);
		(*lib->ops.s_ibv_ack_async_event)(&ae);
	}

	dprint(DBG_AE, LOG_EVENT, "AEQ thread stopped for context %p",
			aeq_ctx->ibv_ctx);

	free(aeq_ctx);

	return NULL;

}

/*
 * List operation:
 * Find an iw_ctx_aeq struct based on the ibv_context.
 *
 * @needle:	the element to look for
 */
static inline struct iw_ctx_aeq* iw_ctx_aeq_find(struct ibv_context *needle)
{
	struct iw_ctx_aeq *cur = lib->aeq_ctx_list;

	while (cur && cur->ibv_ctx != needle) {
		cur = cur->next;
	}

	return cur;
}

/*
 * List operation:
 * Add an iw_ctx_aeq struct at the head.
 *
 * @new:	the new element to be inserted
 */
static inline void iw_ctx_aeq_add(struct iw_ctx_aeq *new)
{
	new->next = lib->aeq_ctx_list;
	lib->aeq_ctx_list = new;
}

/*
 * List operation:
 * Delete an iw_ctx_aeq struct based on its ibv_context.
 *
 * @old:	the element to be deleted
 */
static inline void iw_ctx_aeq_del(struct ibv_context *old)
{
	struct iw_ctx_aeq	*cur, *prev;
	cur = prev = lib->aeq_ctx_list;

	while (cur && cur->ibv_ctx != old) {
		prev = cur;
		cur = cur->next;
	}

	if (cur) {
		if (cur == lib->aeq_ctx_list) {
			lib->aeq_ctx_list = cur->next;
			cur->next = NULL;
		} else {
			prev->next = cur->next;
			cur->next = NULL;
		}
	} else {
		dprint(DBG_ON, LOG_WARNING, "failed to delete aeq context from list");
	}
}

/*
 * Make sure there is a thread that handles async events on @ibv_ctx.
 * If there is no thread yet, a new one will be spawned and started. Otherwise
 * the reference of the existing one will be increased.
 *
 * @ibv_ctx:	the infiniband verbs context to listen for async events on
 */
void create_aeq_thread(struct ibv_context *ibv_ctx)
{
	struct iw_ctx_aeq	*aeq_ctx;

	pthread_mutex_lock(&lib->aeq_ctx_list_mutex);

	aeq_ctx = iw_ctx_aeq_find(ibv_ctx);
	if (aeq_ctx) {
		aeq_ctx->ref++;
		goto out;
	} else {
		aeq_ctx = malloc(sizeof(struct iw_ctx_aeq));
		aeq_ctx->ibv_ctx = ibv_ctx;
		aeq_ctx->ref = 1;
		pthread_create(&aeq_ctx->thread, NULL, aeq_thread, aeq_ctx);
		iw_ctx_aeq_add(aeq_ctx);
	}

out:
	pthread_mutex_unlock(&lib->aeq_ctx_list_mutex);
}

/*
 * Stop listening for async events on the given @ibv_ctx.
 * The reference counter of the corresponding thread will be decreased and it
 * will stop automatically once the reference counter reaches 0.
 *
 * @ibv_ctx:	the infiniband verbs context on which listening for async events
 * 				is no longer required
 */
void destroy_aeq_thread(struct ibv_context *ibv_ctx)
{
	struct iw_ctx_aeq	*aeq_ctx;

	pthread_mutex_lock(&lib->aeq_ctx_list_mutex);
	aeq_ctx = iw_ctx_aeq_find(ibv_ctx);
	if (aeq_ctx) {
		aeq_ctx->ref--;
		if (aeq_ctx->ref == 0) {
			iw_ctx_aeq_del(ibv_ctx);
		}
	}
	pthread_mutex_unlock(&lib->aeq_ctx_list_mutex);
}
