/*
 * Copyright (c) 2010 Philip Frey, Systems Group ETH Zurich.
 *
 * This software is available to you under a choice of one of two
 * licenses.  You may choose to be licensed under the terms of the GNU
 * General Public License (GPL) Version 2, available from the file
 * COPYING in the main directory of this source tree, or the
 * OpenIB.org BSD license below:
 *
 *     Redistribution and use in source and binary forms, with or
 *     without modification, are permitted provided that the following
 *     conditions are met:
 *
 *      - Redistributions of source code must retain the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer.
 *
 *      - Redistributions in binary form must reproduce the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer in the documentation and/or other materials
 *        provided with the distribution.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

/*
 * Debug output.
 */

#ifndef BASIC_DEBUG_
#define BASIC_DEBUG_

#include <stdio.h>

/*
 * Debug categories:
 *
 * DBG_OFF	No output whatsoever
 * DBG_ON	Always on, for really important events or error conditions
 * DBG_TMP	Temporarily on for fine-grained debugging
 * DBG_CM	Connection manager
 * DBG_MEM	Memory management
 * DBG_OPS	RDMA operations
 * DBG_AE	Asynchronous events
 * DBG_ALL	All of the above
 */
#define DBG_OFF		0
#define DBG_ON		1 << 0
#define DBG_TMP		1 << 1
#define DBG_CM		1 << 2
#define DBG_MEM		1 << 3
#define DBG_OPS		1 << 4
#define DBG_AE		1 << 5
#define DBG_ALL		(DBG_ON | DBG_TMP | DBG_CM | DBG_MEM | DBG_OPS | DBG_AE)


/*
 * Set DPRINT_MASK to tailor your debugging needs:
 * Use OR-ed combination of the above DBG_* options.
 */
//#define DPRINT_MASK	DBG_OFF
#define DPRINT_MASK	(DBG_ON | DBG_AE)
//#define DPRINT_MASK	(DBG_ON | DBG_MEM | DBG_OPS)
//#define DPRINT_MASK	DBG_ALL


/*
 * Loglevels
 */
#define LOG_ERROR	"ERROR: "		/* error conditions			*/
#define LOG_WARNING	"WARN: "		/* warning conditions		*/
#define LOG_INFO	"INFO: "		/* informational			*/
#define LOG_DEBUG	"DEBUG: "		/* debug-level messages		*/
#define LOG_EVENT	"ASYNC_EVENT:"	/* async events				*/

/**
 * dprint - Selective debug print
 *
 * Debug print with selectable debug categories and loglevel,
 * starting with header "__func__".
 *
 * @dbgcat	:	Set of debug categories (OR-ed combination of DBG_* above),
 *				to which this debug message is assigned.
 * @loglevel:	One of USER_ERROR, USER_WARNING, USER_NOTICE, USER_INFO,
 *				or USER_DEBUG, in the order of decreasing importance.
 * @fmt		:	printf compliant format string
 * @args	:	printf compliant argument list
 */
#define dprint(dbgcat, loglevel, fmt, args...)							\
	do {																\
		if ((dbgcat) & DPRINT_MASK)	{									\
				printf(loglevel "[%s] " fmt "\n", __func__, ## args);	\
		}																\
	} while (0)


#endif /*BASIC_DEBUG_*/
