/*
 * Copyright (c) 2010 Philip Frey, Systems Group ETH Zurich.
 *
 * This software is available to you under a choice of one of two
 * licenses.  You may choose to be licensed under the terms of the GNU
 * General Public License (GPL) Version 2, available from the file
 * COPYING in the main directory of this source tree, or the
 * OpenIB.org BSD license below:
 *
 *     Redistribution and use in source and binary forms, with or
 *     without modification, are permitted provided that the following
 *     conditions are met:
 *
 *      - Redistributions of source code must retain the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer.
 *
 *      - Redistributions in binary form must reproduce the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer in the documentation and/or other materials
 *        provided with the distribution.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

/*
 * Memory management.
 */

#include <stdlib.h>
#include <string.h>

#include "iwarp.h"
#include "iwarp_debug.h"


////////////////////////////////////////////////////////////////////////////////
// PRIVATE HELPER METHODS
////////////////////////////////////////////////////////////////////////////////

/*
 * Make sure that @mr is not a NULL pointer.
 */
static inline int mr_null(
		IN	const void *mr)
{
	if (!mr) {
		dprint(DBG_ON, LOG_ERROR, "memory region is NULL");
		return 1;
	} else {
		return 0;
	}
}


////////////////////////////////////////////////////////////////////////////////
// PRIVATE METHODS
////////////////////////////////////////////////////////////////////////////////

/*
 * Initialize the buffers and memory regions required for buffer advertisement.
 *
 * Return 0 on success; -1 otherwise.
 */
static int bufadv_alloc(
		IN OUT struct iw_ctx_conn *ctx_conn)
{
	int						 ret;

	if (ctx_null(ctx_conn)) {
		return -1;
	}

	if (ctx_conn->bufadv_valid) {
		dprint(DBG_MEM, LOG_WARNING, "buffer advertisment objs already exist");
		return 0;
	}

	/* alloc buffer advertisement objects (lmr) */
	ctx_conn->bufadv_rx = calloc(1, sizeof(struct iw_lmr));
	if (!ctx_conn->bufadv_rx) {
		dprint(DBG_ON, LOG_ERROR, "out of memory");
		return -1;
	}
	ctx_conn->bufadv_tx = calloc(1, sizeof(struct iw_lmr));
	if (!ctx_conn->bufadv_tx) {
		dprint(DBG_ON, LOG_ERROR, "out of memory");
		goto free_bufadv_rx;
	}

	/* alloc buffer advertisement memory regions */
	ret = iw_lmr_create(sizeof(struct iw_rmr), IBV_ACCESS_LOCAL_WRITE,
			ctx_conn->bufadv_rx, ctx_conn);
	if (ret) {
		dprint(DBG_ON, LOG_ERROR, "out of memory");
		goto free_bufadv_tx;
	}
	ret = iw_lmr_create(sizeof(struct iw_rmr), IBV_ACCESS_LOCAL_WRITE,
			ctx_conn->bufadv_tx, ctx_conn);
	if (ret) {
		dprint(DBG_ON, LOG_ERROR, "out of memory");
		goto free_bufadv_rx_mr;
	}

	ctx_conn->bufadv_valid = 1;

	dprint(DBG_MEM, LOG_INFO, "buffer advertisement infrastructure created");

	return 0;

free_bufadv_rx_mr:
	iw_lmr_destroy(ctx_conn->bufadv_rx, 1);
free_bufadv_tx:
	free(ctx_conn->bufadv_tx);
	ctx_conn->bufadv_tx = NULL;
free_bufadv_rx:
	free(ctx_conn->bufadv_rx);
	ctx_conn->bufadv_rx = NULL;

	return -1;

}


/*
 * Free the buffers and memory regions required for buffer advertisement.
 */
void bufadv_free(
		IN OUT struct iw_ctx_conn *ctx_conn)
{
	int ret;

	if (ctx_null(ctx_conn) || !ctx_conn->bufadv_valid) {
		return;
	}

	/* memory regions */
	if (ctx_conn->bufadv_tx->mr) {
		ret = iw_lmr_destroy(ctx_conn->bufadv_tx, 1);
		if (ret) {
			dprint(DBG_ON, LOG_ERROR, "failed to deregister bufadv_tx mr (%m)");
		}
	}
	if (ctx_conn->bufadv_rx->mr) {
		ret = iw_lmr_destroy(ctx_conn->bufadv_rx, 1);
		if (ret) {
			dprint(DBG_ON, LOG_ERROR, "failed to deregister bufadv_rx mr (%m)");
		}
	}

	/* memory region objects (lmr) */
	free(ctx_conn->bufadv_tx);
	ctx_conn->bufadv_tx = NULL;
	free(ctx_conn->bufadv_rx);
	ctx_conn->bufadv_rx = NULL;

	dprint(DBG_MEM, LOG_INFO, "buffer advertisement infrastructure destroyed");

	ctx_conn->bufadv_valid = 0;

	return;

}


////////////////////////////////////////////////////////////////////////////////
// INTERFACE METHODS
////////////////////////////////////////////////////////////////////////////////

int iw_lmr_create(
		IN		uint32_t				 size,
		IN		enum ibv_access_flags	 access,
		OUT		struct iw_lmr			*lmr,
		IN OUT	struct iw_ctx_conn		*ctx_conn)
{
	/* make sure the lmr pointer is valid */
	if (mr_null(lmr)) {
		return -1;
	}

	if (size == 0) {
		dprint(DBG_ON, LOG_ERROR, "cannot create zero-length mr");
		return -1;
	}

	if (ctx_null(ctx_conn)) {
		return -1;
	}

	if (ctx_conn->state != IWARP_VALID &&
		ctx_conn->state != IWARP_CONNECTED) {
		dprint(DBG_ON, LOG_ERROR, "connection context must be in IWARP_VALID or"
				" IWARP_CONNECTED state (allocate it first!)");
		return -1;
	}

	/* callocate memory */
	lmr->len = size;
	lmr->buf = calloc(1, lmr->len);
	if (!lmr->buf) {
		dprint(DBG_ON, LOG_ERROR, "out of memory");
		lmr->len = 0;
		return -1;
	}

	/* create mr on buffer */
	lmr->mr = (*lib->ops.s_ibv_reg_mr)(ctx_conn->iw_pd->pd, lmr->buf,
			lmr->len, access);
	if (!lmr->mr) {
		dprint(DBG_ON, LOG_ERROR, "failed to create mr on buffer (%m)");
		goto free_buf;
	}

	/* permissions */
	if (!access) {
		dprint(DBG_MEM, LOG_WARNING, "creating local-read-only memory region!");
	}
	lmr->access = access;

	dprint(DBG_MEM, LOG_INFO, "buffer created and registered as lmr");

	return 0;

free_buf:
	free(lmr->buf);
	lmr->buf = NULL;

	return -1;

}


int iw_lmr_register(
		IN		void					*buffer,
		IN		uint32_t				 buffer_len,
		IN		enum ibv_access_flags	 access,
		OUT		struct iw_lmr			*lmr,
		IN OUT	struct iw_ctx_conn		*ctx_conn)
{
	/* make sure the lmr pointer is valid */
	if (mr_null(lmr)) {
		return -1;
	}

	if (buffer_len == 0) {
		dprint(DBG_ON, LOG_ERROR, "cannot register zero-length mr");
		return -1;
	}

	if (!buffer) {
		dprint(DBG_ON, LOG_ERROR, "buffer is NULL");
		return -1;
	}

	if (ctx_null(ctx_conn)) {
		return -1;
	}

	/* assign buffer */
	lmr->buf = buffer;
	lmr->len = buffer_len;

	/* create mr on buffer */
	lmr->mr = (*lib->ops.s_ibv_reg_mr)(ctx_conn->iw_pd->pd, lmr->buf,
			lmr->len, access);
	if (!lmr->mr) {
		dprint(DBG_ON, LOG_ERROR, "failed to create mr on buffer (%m)");
		return -1;
	}

	/* permissions */
	if (!access) {
		dprint(DBG_MEM, LOG_WARNING, "creating local-read-only memory region!");
	}
	lmr->access = access;

	dprint(DBG_MEM, LOG_INFO, "buffer registered as lmr");

	return 0;

}


int iw_lmr_destroy(
		IN OUT	struct iw_lmr	*lmr,
		IN		int				 free_buffer)
{
	int ret, error = 0;

	if (mr_null(lmr)) {
		return -1;
	}

	/* deregister the mr */
	ret = (*lib->ops.s_ibv_dereg_mr)(lmr->mr);
	if (ret) {
		dprint(DBG_ON, LOG_ERROR, "failed to deregister mr (%m)");
		error = -1;
	}
	lmr->mr = NULL;

	/* free the buffer */
	if (free_buffer) {
		free(lmr->buf);
		lmr->buf = NULL;
		dprint(DBG_MEM, LOG_INFO, "lmr deregistered and buffer freed");
	} else {
		dprint(DBG_MEM, LOG_INFO, "lmr deregistered");
	}

	lmr->len = 0;

	return error;

}


int iw_post_send_adv(
		IN 		const struct iw_lmr	*lmr,
		IN OUT	struct iw_ctx_conn	*ctx_conn)
{
	int				 ret, bufadv_created = 0;
	struct iw_rmr	*bufadv;

	/* check the iwarp connection context */
	if (ctx_null(ctx_conn)) {
		return -1;
	}

	/* make sure we were given a valid @lmr */
	if (mr_null(lmr)) {
		return -1;
	}

	/* create buffer advertisement infrastructure if necessary */
	if (!ctx_conn->bufadv_valid) {
		ret = bufadv_alloc(ctx_conn);
		if (ret) {
			return -1;
		}
		bufadv_created = 1;
	}

	bufadv = (struct iw_rmr*)ctx_conn->bufadv_tx->buf;

	if (!lmr->access) {
		dprint(DBG_ON, LOG_WARNING, "advertising local-read-only memory region "
				"- this is probably not what you want!");
	}

	bufadv->addr = (uint64_t)lmr->mr->addr;
	bufadv->len  = lmr->mr->length;
	bufadv->rkey = lmr->mr->rkey;

	ret = post_send_lmr(ctx_conn->bufadv_tx, 0, sizeof(struct iw_rmr), 0,
			ctx_conn);
	if (ret < 0) {
		dprint(DBG_ON, LOG_ERROR, "failed to send local memory region");
		goto destroy_tag_info;
	}

	dprint(DBG_MEM, LOG_INFO, "sent buffer advertisement: "
			"[addr=%p, len=%d, rkey=%d]",
			(void *)bufadv->addr, bufadv->len, bufadv->rkey);

	return 0;

destroy_tag_info:
	if (bufadv_created)
		bufadv_free(ctx_conn);

	return -1;

}


int iw_post_recv_adv(
		IN OUT	struct iw_ctx_conn	*ctx_conn)
{
	int ret, bufadv_created = 0;

	/* check the iwarp connection context */
	if (ctx_null(ctx_conn)) {
		//TODO: doesn't it have to be valid already?
		return -1;
	}

	/* create tag exchange infrastructure if necessary */
	if (!ctx_conn->bufadv_valid) {
		ret = bufadv_alloc(ctx_conn);
		if (ret) {
			return -1;
		}
		bufadv_created = 1;
	}

	/* post the receive wr */
	ret = post_recv_lmr(ctx_conn->bufadv_rx, 0, sizeof(struct iw_rmr),
			ctx_conn);
	if (ret < 0) {
		dprint(DBG_ON, LOG_ERROR, "failed to post receive");
		goto destroy_tag_info;
	}

	return ret;

destroy_tag_info:
	if (bufadv_created)
		bufadv_free(ctx_conn);

	return -1;

}


int iw_wait_recv_adv(
		IN OUT	struct iw_rmr		*rmr,
		IN		unsigned int		 expected_wr_id,
		IN OUT	struct iw_ctx_conn	*ctx_conn)
{
	int				 ret;
	struct iw_rmr	*bufadv;
	struct ibv_wc	 wc;

	/* check the iwarp connection context */
	if (ctx_null(ctx_conn)) {
		//TODO: ctx_conn should probably be valid...
		return -1;
	}

	/* make sure we were given a valid @rmr */
	if (mr_null(rmr)) {
		return -1;
	}

	/* make sure the buffer advertisement infrastructure is valid */
	if (!ctx_conn->bufadv_valid) {
		dprint(DBG_ON, LOG_ERROR, "tag exchange infrastructure invalid - did "
				"you mean to call post_bufadv_receive()?");
		return -1;
	}

	/* wait for the inbound buffer advertisement */
	ret = await_completions(IW_RCQ, 1, &wc, ctx_conn);
	if (ret < 0 || wc.status != IBV_WC_SUCCESS || wc.wr_id != expected_wr_id) {
		dprint(DBG_ON, LOG_ERROR, "failed to wait for remote buffer "
				"advertisement");
		return -1;
	}

	/* get the data back to the user */
	bufadv = (struct iw_rmr *)ctx_conn->bufadv_rx->buf;
	rmr->addr = bufadv->addr;
	rmr->len  = bufadv->len;
	rmr->rkey = bufadv->rkey;

	dprint(DBG_MEM, LOG_INFO, "received remote buffer advertisement: "
			"[addr=%p, len=%d, rkey=%d]",
			(void *)bufadv->addr, bufadv->len, bufadv->rkey);

	return 0;

}
