/*
 * Copyright (c) 2010 Philip Frey, Systems Group ETH Zurich.
 *
 * This software is available to you under a choice of one of two
 * licenses.  You may choose to be licensed under the terms of the GNU
 * General Public License (GPL) Version 2, available from the file
 * COPYING in the main directory of this source tree, or the
 * OpenIB.org BSD license below:
 *
 *     Redistribution and use in source and binary forms, with or
 *     without modification, are permitted provided that the following
 *     conditions are met:
 *
 *      - Redistributions of source code must retain the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer.
 *
 *      - Redistributions in binary form must reproduce the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer in the documentation and/or other materials
 *        provided with the distribution.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

/*
 * Simple client/server application for demonstrating libiwarp usage.
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <getopt.h>

#include <netinet/in.h>
#include <arpa/inet.h>

#include <iwarp.h>

#define CTRL_BUF_SIZE		64
#define RDMA_BUF_SIZE		512

/* Error and Debug Messages */
#define DEBUG

#define error(msg, args...) \
	fprintf(stderr, "ERROR: "msg"\n", ## args)
#ifdef DEBUG
#define debug(msg, args...) \
	printf("DEBUG: "msg"\n", ## args)
#else
#define debug(msg, args...)
#endif

/*
 * Control Block
 */
struct {
	int			 is_server;

	char		*srv_addr;
	char		*srv_port;

	void		*priv_data;
	uint8_t	 	 priv_data_len;
} cb;


int run_client()
{
	int							 ret, rwr_id, swr_id;
	struct ibv_wc				 wc;

	/* for convenience */
	struct iw_ctx_conn			iw_ctx;

	/* local memory regions */
	struct iw_lmr				 mem_ctrl, mem_write_src, mem_read_dst;

	/* remote memory regions */
	struct iw_rmr				 mem_write_dst;


	/* allocate the iwarp connection context */
	iw_ctx.state = IWARP_INVALID;
	iw_ctx.type = CTX_CONNECT;
	ret = iw_ctx_alloc(cb.srv_addr, cb.srv_port, NULL, NULL, 0,
			NULL, &iw_ctx);

	/* control buffer */
	ret = iw_lmr_create(CTRL_BUF_SIZE, IBV_ACCESS_LOCAL_WRITE, &mem_ctrl,
			&iw_ctx);
	if (ret) {
		error("failed to allocate mem_ctrl");
		goto free_iw_ctx;
	}

	/* rdma write source */
	ret = iw_lmr_create(RDMA_BUF_SIZE, 0, &mem_write_src, &iw_ctx);
	if (ret) {
		error("failed to allocate mem_write_src");
		goto free_ctrl;
	}

	/* rdma read destination */
	ret = iw_lmr_create(RDMA_BUF_SIZE,
			IBV_ACCESS_LOCAL_WRITE|IBV_ACCESS_REMOTE_WRITE,
			&mem_read_dst, &iw_ctx);
	if (ret) {
		error("failed to allocate mem_read_dst");
		goto free_write_src;
	}

	rwr_id = iw_post_recv_adv(&iw_ctx);
	if (rwr_id == -1) {
		error("failed to post receive for rmr advertisement");
		goto free_read_dst;
	}

	/* IWARP CONNECT **********************************************************/

	cb.priv_data = "please";
	cb.priv_data_len = strlen(cb.priv_data);
	ret = iw_connect(cb.priv_data, cb.priv_data_len, &iw_ctx);
	if (ret) {
		error("connection failed");
		goto free_read_dst;
	} else {
		debug("connection successful");
		if (iw_ctx.priv_data_in && iw_ctx.priv_data_in_len) {
			debug("private data received (%d):", iw_ctx.priv_data_in_len);
			debug(" \"%s\"", (char*) iw_ctx.priv_data_in);
		}
	}

	/* RDMA SEND **************************************************************/

	/* rdma send */
	strcpy(mem_ctrl.buf, "hello lmr world!");
	swr_id = post_send_lmr(&mem_ctrl, 0, CTRL_BUF_SIZE, IBV_SEND_SIGNALED,
			&iw_ctx);
	if (swr_id < 0) {
		goto free_read_dst;
	}
	/* wait for completion */
	ret = await_completions(IW_SCQ, 1, &wc, &iw_ctx);
	if (ret) {
		goto free_read_dst;
	}
	if (wc.status != IBV_WC_SUCCESS || wc.wr_id != swr_id) {
		error("sending the control message failed");
	}

	/* RDMA WRITE *************************************************************/

	/* receive write destination buffer */
	ret = iw_wait_recv_adv(&mem_write_dst, rwr_id, &iw_ctx);
	if (ret) {
		error("failed to wait for rmr advertisement");
		goto free_read_dst;
	}

	/* rdma write something to the server */
	strcpy(mem_write_src.buf, "hello rdma world!");
	swr_id = post_write_lmr(&mem_write_src, 0, &mem_write_dst, 0, RDMA_BUF_SIZE,
			IBV_SEND_SIGNALED, &iw_ctx);
	if (swr_id < 0) {
		error("failed to rdma write to the server");
		goto free_read_dst;
	}
	ret = await_completions(IW_SCQ, 1, &wc, &iw_ctx);
	if (ret) {
		goto free_read_dst;
	}
	if (wc.status != IBV_WC_SUCCESS || wc.wr_id != swr_id) {
		error("waiting for the completion of the rdma write failed");
	}

	/* RDMA READ **************************************************************/

	/* rdma read back what was written before with the rdma write call */
	swr_id = post_read_lmr(&mem_read_dst, 0, &mem_write_dst, 0, RDMA_BUF_SIZE,
			IBV_SEND_SIGNALED, &iw_ctx);
	if (swr_id < 0) {
		error("failed to rdma read from the server");
		goto free_read_dst;
	}
	ret = await_completions(IW_SCQ, 1, &wc, &iw_ctx);
	if (ret) {
		goto free_read_dst;
	}
	if (wc.status != IBV_WC_SUCCESS || wc.wr_id != swr_id) {
		error("waiting for the completion of the rdma read failed");
	}

	/* compare what was read back and make sure it matches what was written */
	if (memcmp(mem_write_src.buf, mem_read_dst.buf, RDMA_BUF_SIZE) != 0) {
		error("we have not read back what was written before :-(");
//		goto free_read_dst;
	} else {
		debug("rdma read matches rdma write :-)");
	}


	/* RDMA SEND **************************************************************/

	strcpy(mem_ctrl.buf, "done");
	swr_id = post_send_lmr(&mem_ctrl, 0, CTRL_BUF_SIZE, IBV_SEND_SIGNALED,
			&iw_ctx);
	if (swr_id < 0) {
		goto free_read_dst;
	}
	ret = await_completions(IW_SCQ, 1, &wc, &iw_ctx);
	if (ret) {
		goto free_read_dst;
	}
	if (wc.status != IBV_WC_SUCCESS || wc.wr_id != swr_id) {
		error("sending the control message failed");
	}

	/* free buffers */
	ret = iw_lmr_destroy(&mem_ctrl, 1);
	if (ret) {
		error("failed to destroy mem_done");
	}
	ret = iw_lmr_destroy(&mem_write_src, 1);
	if (ret) {
		error("failed to destroy mem_write_src");
	}
	ret = iw_lmr_destroy(&mem_read_dst, 1);
	if (ret) {
		error("failed to destroy mem_read_dst");
	}

	/* AWAIT IWARP DISCONNET **************************************************/

	ret = iw_await_disconnect(&iw_ctx);
	if (ret) {
		error("awaiting disconnect failed");
		return -1;
	}

	ret = iw_ctx_free(&iw_ctx);
	if (ret) {
		error("failed to free iwarp connection context");
		return -1;
	}

	return 0;

free_read_dst:
	ret = iw_lmr_destroy(&mem_read_dst, 1);
	if (ret) {
		error("failed to destroy mem_read_dst");
	}
free_write_src:
	ret = iw_lmr_destroy(&mem_write_src, 1);
	if (ret) {
		error("failed to destroy mem_write_src");
	}
free_ctrl:
	ret = iw_lmr_destroy(&mem_ctrl, 1);
	if (ret) {
		error("failed to destroy mem_ctrl");
	}
free_iw_ctx:
	ret = iw_ctx_free(&iw_ctx);
	if (ret) {
		error("failed to free iwarp connection context");
	}

	return -1;

}


int run_server()
{
	int						 ret, rwr_id = 0;
	struct ibv_wc			 wc;

	/* for convenience */
	struct iw_ctx_conn		 iw_ctx;
	struct iw_ctx_listen	 listen_ctx;

	/* local memory regions */
	struct iw_lmr			 mem_ctrl, mem_write_dst;

	/* private data for the connection reply */
	void					*priv_data_out;
	uint8_t					 priv_data_out_len;


	/* specify that we are a server in INVALID iwarp connection state */
	iw_ctx.type = CTX_ACCEPT;
	iw_ctx.state = IWARP_INVALID;

	/* IWARP LISTEN ***********************************************************/

	ret = iw_open("0", cb.srv_port, 3, &listen_ctx);
	if (ret) {
		error("failed to open listen context");
		return -1;
	}

	ret = iw_listen(&listen_ctx, &iw_ctx);
	if (ret) {
		error("failed to listen for incoming connections");
		goto close_listen;
	}

	ret = iw_ctx_alloc(0, 0, NULL, NULL, 0, NULL, &iw_ctx);
	if (ret) {
		error("failed to allocate iwarp connection context");
		goto free_iw_ctx;
	}

	ret = iw_lmr_create(CTRL_BUF_SIZE, IBV_ACCESS_LOCAL_WRITE,
			&mem_ctrl, &iw_ctx);
	if (ret) {
		error("failed to create mem_ctrl");
		goto free_iw_ctx;
	}

	ret = iw_lmr_create(RDMA_BUF_SIZE,
			IBV_ACCESS_LOCAL_WRITE|
			IBV_ACCESS_REMOTE_WRITE|
			IBV_ACCESS_REMOTE_READ,
			&mem_write_dst, &iw_ctx);
	if (ret) {
		error("failed to allocate mem_write_dst");
		goto free_ctrl;
	}

	/* inspect the private data from the connection request */
	if (iw_ctx.priv_data_in && iw_ctx.priv_data_in_len > 0) {
		debug("private data in connection request (%d bytes)",
				iw_ctx.priv_data_in_len);
		debug(" \"%s\"", (char*) iw_ctx.priv_data_in);
	} else {
		debug("no private data in request");
	}

	/* reject the connection if the inbound private data are not "please" */
	if (iw_ctx.priv_data_in &&
			strncmp(iw_ctx.priv_data_in, "please", 6) == 0) {

		/* IWARP ACCEPT *******************************************************/

		/* pre-post receive */
		rwr_id = post_recv_lmr(&mem_ctrl, 0, CTRL_BUF_SIZE, &iw_ctx);
		if (rwr_id == -1) {
			error("failed to pre-post receive");
			goto free_write_dst;
		}

		priv_data_out = "you are a nice guy, pleased to meet you :-)";
		priv_data_out_len = strlen(priv_data_out);
		ret = iw_accept(priv_data_out, priv_data_out_len, &iw_ctx);
		if (ret) {
			error("failed to accept incoming connection");
			goto free_write_dst;
		}

	} else {

		/* IWARP REJECT *******************************************************/

		priv_data_out = "you are rude - I do not want to talk to you :-(";
		priv_data_out_len = strlen(priv_data_out);
		ret = iw_reject(priv_data_out, priv_data_out_len, &iw_ctx);
		if (ret) {
			error("failed to reject the connection");
			goto free_write_dst;
		}

	}

	/* RDMA RECEIVE ***********************************************************/

	/* await receive */
	ret = await_completions(IW_RCQ, 1, &wc, &iw_ctx);
	if (ret < 0 || wc.status != IBV_WC_SUCCESS || wc.wr_id != rwr_id) {
		error("awaiting receive failed");
		goto free_write_dst;
	} else {
		debug("received %d bytes", wc.byte_len);
	}
	debug("received: \"%s\"", (char *)mem_ctrl.buf);

	/* ADVERTISE LOCAL BUFFER *************************************************/

	/* rdma receive */
	rwr_id = post_recv_lmr(&mem_ctrl, 0, CTRL_BUF_SIZE, &iw_ctx);
	if (rwr_id < 0) {
		error("failed to post rdma receive");
		goto free_write_dst;
	}

	/* advertise the local rdma buffer */
	ret = iw_post_send_adv(&mem_write_dst, &iw_ctx);
	if (ret) {
		error("advertising mem_write_dst failed");
		goto free_write_dst;
	}


	/***************************************************************************
	****************************************************************************
	* CLIENT DOES RDMA OPERATIONS HERE *****************************************
	****************************************************************************
	***************************************************************************/


	/* RDMA RECEIVE ***********************************************************/

	/* wait for done from client */
	ret = await_completions(IW_RCQ, 1, &wc, &iw_ctx);
	if (ret < 0 || wc.status != IBV_WC_SUCCESS || wc.wr_id != rwr_id) {
		error("awaiting receive failed");
		goto free_write_dst;
	} else {
		debug("received %d bytes", wc.byte_len);
	}


	/* display what the client has written */
	debug("client wrote: \"%s\"", (char *)mem_write_dst.buf);


	/* display what the client has sent */
	debug("client sent: \"%s\"", (char *)mem_ctrl.buf);


	/* cleanup buffers */
	ret = iw_lmr_destroy(&mem_write_dst, 1);
	if (ret) {
		error("failed to destroy mem_write_dst");
	}
	ret = iw_lmr_destroy(&mem_ctrl, 1);
	if (ret) {
		error("failed to destroy mem_ctrl");
	}

	/* IWARP DISCONNECT *******************************************************/

	ret = iw_disconnect(&iw_ctx);
	if (ret) {
		error("disconnect failed");
		return -1;
	}

	ret = iw_ctx_free(&iw_ctx);
	if (ret) {
		error("failed to free iwarp connection context");
	}

	ret = iw_close(&listen_ctx);
	if (ret) {
		error("failed to close listen context");
	}

	return 0;


free_write_dst:
	ret = iw_lmr_destroy(&mem_write_dst, 1);
	if (ret) {
		error("failed to destroy mem_write_dst");
	}
free_ctrl:
	ret = iw_lmr_destroy(&mem_ctrl, 1);
	if (ret) {
		error("failed to destroy mem_ctrl");
	}
free_iw_ctx:
	ret = iw_ctx_free(&iw_ctx);
	if (ret) {
		error("failed to free iwarp connection context");
	}
close_listen:
	ret = iw_close(&listen_ctx);
	if (ret) {
		error("failed to close listen context");
	}

	return -1;

}


void usage(char *app_name)
{
	printf("Usage:\n");
	printf("Server: %s -s [-p <srv_port>]\n", app_name);
	printf("Client: %s -c -a <srv_addr> [-p <srv_port>]\n", app_name);
	exit(-1);
}


int main(int argc, char **argv)
{
	int option, ret;

	/* application control block */
	cb.is_server = -1;
	cb.srv_addr = "0";
	cb.srv_port = "8712";
	cb.priv_data_len = 0;

	/* parse command line arguments */
	while ((option = getopt(argc, argv, "csa:p:")) != -1) {
		switch (option) {
		case 's':	// run as server
			cb.is_server = 1;
			break;
		case 'c':
			cb.is_server = 0;
			break;
		case 'a':	// server address
			cb.srv_addr = strdup(optarg);
			debug("server addr: %s", optarg);
			break;
		case 'p':	// server port
			cb.srv_port = strdup(optarg);
			debug("server port: %s", optarg);
			break;
		default:
			usage(argv[0]);
			break;
		}
	}

	if (cb.is_server == -1) {
		usage(argv[0]);
	}

	if (!cb.is_server && !cb.srv_addr) {
		error("client must supply a server ip address!");
		usage(argv[0]);
	}

	if (cb.is_server) {
		ret = run_server();
	} else {
		ret = run_client();
	}

	if (ret) {
		error("%s failed", argv[0]);
	} else {
		debug("%s successful", argv[0]);
	}
	return 0;
}
