/*
 * DRIVER for testing the usbdmx
 */
#include <linux/miscdevice.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/poll.h>
#include <linux/usb.h>
#include <linux/slab.h>

#include "usbdmx.h"

#define DMX_MINOR 99                /* dummy */

/* wait timeout for usbdmx [msec] */
#define TIMEOUT (100)

struct dmx_usb_data {
  struct usb_device *dmx_dev;	/* init: probe_dmx */
  int isopen;			/* nz if open */
  int present;			/* Device is present on the bus */
  int memory;			/* selected memory map of read/write */
  int offset;			/* memory offset */
  int blocking; 		/* blocking read/write? */
};

static struct dmx_usb_data dmx_instance;

static int open_dmx(struct inode *inode, struct file *file) {
  struct dmx_usb_data *dmx = &dmx_instance;
  
  if (dmx->isopen || !dmx->present) {
    return -EBUSY;
  }
  dmx->isopen = 1;
  
  MOD_INC_USE_COUNT;
  
  dbg("USBDMX opened.");
  
  return 0;
}

static int close_dmx(struct inode *inode, struct file *file) {
  struct dmx_usb_data *dmx = &dmx_instance;
  
  dmx->isopen = 0;
  
  MOD_DEC_USE_COUNT;
  
  dbg("USBDMX closed.");
  return 0;
}

static int
ioctl_dmx(struct inode *inode, struct file *file, unsigned int cmd,
	  unsigned long arg) {
  struct dmx_usb_data *dmx = &dmx_instance;
  int result = 0;
  int d;
  
  /* Sanity check to make sure usbdmx is connected, powered, etc */
  if ( dmx == NULL ||
       dmx->present == 0 ||
       dmx->dmx_dev == NULL )
    return -1;

  dbg("USBDMX ioctl");
  
  switch(cmd) {
  case CONFIG_SET:
    d = (int)arg;
    if(d<1 || d>3) {
      err("confiuration out of range (1..3)");
      return -EINVAL;
    }
      
    if(!usb_set_configuration(dmx->dmx_dev, d))
      info("configuration changed to %i", d);
    break;

  case DMX_BLOCKING_SET:
    d = (int)arg;
    
    dmx->blocking = d?VALUE_BLOCKING:0;
    break;

  case DMX_BLOCKING_GET:
    return dmx->blocking?1:0;

  case DMX_SERIAL_GET:
    d = 0;
    result = usb_control_msg(dmx->dmx_dev, usb_rcvctrlpipe(dmx->dmx_dev, 0),
			     DMX_SERIAL,
			     USB_DIR_IN | USB_TYPE_VENDOR, 0, 0, &d, 4, TIMEOUT);
    if(result == 4)
      return d;
    else
      dbg("ioctl: DMX_SERIAL_GET: got %i/%i bytes back", result, 4);
    break;

  case DMX_MEM_MAP_SET:
    d = (int)arg;
    
    if(d == DMX_MRG_MEM || d == DMX_RC_MEM || d == DMX_TX_MEM)
      dmx->memory = d;
    else
      dmx->memory = 0;

    dmx->offset = 0;
    break;

  case DMX_MEM_MAP_GET:
    return dmx->memory;

  case DMX_TX_FRAMES_GET:
    d = 0;
    result = usb_control_msg(dmx->dmx_dev, usb_rcvctrlpipe(dmx->dmx_dev, 0),
			     DMX_TX_FRAMES,
			     USB_DIR_IN | USB_TYPE_VENDOR, 0, 0, &d, 4, TIMEOUT);
    if(result == 4)
      return d;
    else
      dbg("ioctl: DMX_TX_FRAMES_GET: got %i/%i bytes back", result, 4);
    break;
    
  case DMX_RC_FRAMES_GET:
    d = 0;
    result = usb_control_msg(dmx->dmx_dev, usb_rcvctrlpipe(dmx->dmx_dev, 0),
			     DMX_RC_FRAMES,
			     USB_DIR_IN | USB_TYPE_VENDOR, 0, 0, &d, 4, TIMEOUT);
    if(result == 4)
      return d;
    else
      dbg("ioctl: DMX_RC_FRAMES_GET: got %i/%i bytes back", result, 4);
    break;
    
  case DMX_TX_STARTCODE_GET:
    d = 0;
    result = usb_control_msg(dmx->dmx_dev, usb_rcvctrlpipe(dmx->dmx_dev, 0),
			     DMX_TX_STARTCODE,
			     USB_DIR_IN | USB_TYPE_VENDOR, 0, 0, &d, 1, TIMEOUT);
    if(result == 1)
      return d;
    else
      dbg("ioctl: DMX_TX_STARTCODE_GET: got %i/%i bytes back", result, 1);
    break;
   
  case DMX_TX_STARTCODE_SET:
    /* get value from arg-list */
    d = (int)arg;
    if(d > 0xff || d < 0)
      break;

    result = usb_control_msg(dmx->dmx_dev, usb_sndctrlpipe(dmx->dmx_dev, 0),
			     DMX_TX_STARTCODE,
			     USB_DIR_OUT | USB_TYPE_VENDOR, d, 0, 0, 0, TIMEOUT);
    break;

  case DMX_RC_STARTCODE_GET:
    d = 0;
    result = usb_control_msg(dmx->dmx_dev, usb_rcvctrlpipe(dmx->dmx_dev, 0),
			     DMX_RC_STARTCODE,
			     USB_DIR_IN | USB_TYPE_VENDOR, 0, 0, &d, 1, TIMEOUT);
    if(result == 1)
      return d;
    else
      dbg("ioctl: DMX_RC_STARTCODE_GET: got %i/%i bytes back", result, 1);
    break;
   
  case DMX_RC_STARTCODE_SET:
    /* get value from arg-list */
    d = (int)arg;
    if(d > 0xff || d < 0)
      break;

    result = usb_control_msg(dmx->dmx_dev, usb_sndctrlpipe(dmx->dmx_dev, 0),
			     DMX_RC_STARTCODE,
			     USB_DIR_OUT | USB_TYPE_VENDOR, d, 0, 0, 0, TIMEOUT);
    break;

  case DMX_RC_SLOTS_GET:
    d = 0;
    result = usb_control_msg(dmx->dmx_dev, usb_rcvctrlpipe(dmx->dmx_dev, 0),
			     DMX_RC_SLOTS,
			     USB_DIR_IN | USB_TYPE_VENDOR, 0, 0, &d, 2, TIMEOUT);
    if(result == 2)
      return d;
    else
      dbg("ioctl: DMX_RC_SLOTS_GET: got %i/%i bytes back", result, 2);
    break;
   
  case DMX_TX_SLOTS_GET:
    d = 0;
    result = usb_control_msg(dmx->dmx_dev, usb_rcvctrlpipe(dmx->dmx_dev, 0),
			     DMX_TX_SLOTS,
			     USB_DIR_IN | USB_TYPE_VENDOR, 0, 0, &d, 2, TIMEOUT);
    if(result == 2)
      return d;
    else
      dbg("ioctl: DMX_TX_SLOTS_GET: got %i/%i bytes back", result, 2);
    break;
   
  case DMX_TX_SLOTS_SET:
    /* get value from arg-list */
    d = (int)arg;
    if(d > 0x200 || d < 24)
      break;

    result = usb_control_msg(dmx->dmx_dev, usb_sndctrlpipe(dmx->dmx_dev, 0),
			     DMX_TX_SLOTS,
			     USB_DIR_OUT | USB_TYPE_VENDOR, d, 0, 0, 0, TIMEOUT);
    break;

  default:
    return -ENOIOCTLCMD;
    break;
  }
  return result;
}

static loff_t
seek_dmx(struct file *file, loff_t offset, int whence) {
  struct dmx_usb_data *dmx = &dmx_instance;
  
  if(dmx == NULL)
    return -1;

  switch(whence) {
  case 0:
    dmx->offset = offset;
    break;

  case 1:
    dmx->offset += offset;
    break;

  default:
    return -EINVAL;
  }
  return dmx->offset;
}

static ssize_t
write_dmx(struct file *file, const char *buffer,
	  size_t count, loff_t * ppos) {
  struct dmx_usb_data *dmx = &dmx_instance;
  int result;
  char *buf;
  
  /* Sanity check to make sure usbdmx is connected, powered, etc */
  if( dmx == NULL ||
      dmx->present == 0 ||
      dmx->dmx_dev == NULL ||
      dmx->memory == 0 ) {
    warn("write_dmx: sanity check failed");

    return -1;
  }

  if(!(buf = (char *) kmalloc(count, GFP_KERNEL))) {
    err("write_dmx: Not enough memory for buffer");
    return -ENOMEM;
  }
  
  if (copy_from_user(buf, buffer, count)) {
    warn("write_dmx: copy_from_user failed");

    kfree(buf);
    return -EFAULT;
  }

  result = usb_control_msg(dmx->dmx_dev, usb_sndctrlpipe(dmx->dmx_dev, 0),
			   dmx->memory,
			   USB_DIR_OUT | USB_TYPE_VENDOR, dmx->blocking, 
			   dmx->offset, buf, count, TIMEOUT);
  
  if(result != count && result != -ETIMEDOUT) {
    unsigned int e;
    
    e = ioctl_dmx(0, 0, DMX_ERROR_GET, 0);
    err("write_dmx: finished with result: %i, error code: "
	"0x%02x 0x%02x 0x%02x 0x%02x", 
	result, e&0xff, (e>>8)&0xff, (e>>16)&0xff, (e>>24)&0xff);

    ioctl_dmx(0, 0, DMX_ERROR_SET, 0);
  }
  kfree(buf);
  
  return result;
}

static ssize_t
read_dmx(struct file *file, char *buffer, size_t count, loff_t * ppos) {
  struct dmx_usb_data *dmx = &dmx_instance;
  int result;
  char *buf;
  
  /* Sanity check to make sure usbdmx is connected, powered, etc */
  if( dmx == NULL ||
      dmx->present == 0 ||
      dmx->dmx_dev == NULL ||
      dmx->memory == 0 ) {
    warn("read_dmx: sanity check failed");

    return -1;
  }
  
  if(!(buf = (char *) kmalloc(count, GFP_KERNEL))) {
    err("read_dmx: Not enough memory for buffer");
    return -ENOMEM;
  }
  
  result = usb_control_msg(dmx->dmx_dev, usb_rcvctrlpipe(dmx->dmx_dev, 0),
			   dmx->memory,
			   USB_DIR_IN | USB_TYPE_VENDOR, dmx->blocking, 
			   dmx->offset, buf, count, TIMEOUT);

  if(result != count && result != -ETIMEDOUT) {
    unsigned int e;
    
    e = ioctl_dmx(0, 0, DMX_ERROR_GET, 0);
    err("read_dmx: finished with result: %i, error code: "
	"0x%02x 0x%02x 0x%02x 0x%02x", 
	result, e&0xff, (e>>8)&0xff, (e>>16)&0xff, (e>>24)&0xff);
    ioctl_dmx(0, 0, DMX_ERROR_SET, 0);
  }
  if (result >= 0 && copy_to_user(buffer, buf, result)) {
    warn("read_dmx: copy_to_user failed");
    result = -EFAULT;
  }
  kfree(buf);
  return result;
}

static void *probe_dmx(struct usb_device *dev, unsigned int ifnum,
                      const struct usb_device_id *id) {
  struct dmx_usb_data *dmx = &dmx_instance;
  char str[128];
  int res;

  if (dev->descriptor.idVendor != 0x0ce1) {
    return NULL;
  }
  
  if (dev->descriptor.idProduct != 0x1 ) {
    warn(KERN_INFO "USBDMX model not supported/tested.");
    return NULL;
  }
  
  info("USBDMX found at address %d", dev->devnum);
  
  dmx->present = 1;
  dmx->dmx_dev = dev;
  dmx->memory = 0;
  dmx->offset = 0;
  dmx->blocking = 0;

  if(usb_set_configuration(dev, 2)) /* set configuration to best */
    err ("setting configuration failed");

  if (dev->actconfig->iConfiguration) {
    if((res = usb_string(dev, dev->actconfig->iConfiguration, str, sizeof(str))) < 0 )
      err ("reading string for current config failed (error=%i)", res);
    else
      info ("configuration changed to \"%s\"", str);
  }

  ioctl_dmx(0, 0, DMX_ERROR_SET, 0);
    
  return dmx;
}

static void disconnect_dmx(struct usb_device *dev, void *ptr) {
  struct dmx_usb_data *dmx = (struct dmx_usb_data *) ptr;

#if 0  
  if (dmx->isopen) {
    dmx->isopen = 0;
    
    /* better let it finish - the release will do whats needed */
    dmx->dmx_dev = NULL;
    return;
  }
#endif
  dmx->present = 0;
  dmx->dmx_dev = NULL;

  info("USBDMX disconnected.");
}

static struct
file_operations usb_dmx_fops = {
  llseek:  seek_dmx,
  read:    read_dmx,
  write:   write_dmx,
  ioctl:   ioctl_dmx,
  open:	   open_dmx,
  release: close_dmx,
};

static struct
usb_driver dmx_driver = {
  name:       "usbdmx",
  probe:      probe_dmx,
  disconnect: disconnect_dmx,
  fops:	      &usb_dmx_fops,
  minor:      DMX_MINOR
};

int usb_dmx_init(void) {
  struct dmx_usb_data *dmx = &dmx_instance;

  if (usb_register(&dmx_driver) < 0)
    return -1;

  dmx->present = 0;
  dmx->dmx_dev = NULL;

  info("USBDMX support registered.");
  return 0;
}


void usb_dmx_cleanup(void) {
  struct dmx_usb_data *dmx = &dmx_instance;
  
  dmx->present = 0;
  usb_deregister(&dmx_driver);
}

module_init(usb_dmx_init);
module_exit(usb_dmx_cleanup);

