/*
 * This file is part of the LinuxBIOS project.
 *
 * Copyright (C) 2007 Philipp Degler <pdegler@rumms.uni-mannheim.de>
 * (Written by Philipp Degler)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

#include <stdio.h>
#include <sys/io.h>
#include <unistd.h>

#define IT8712F_CONFIG_REG_LDN       0x07	/* Logical Device Number. */

/* The content of IT8712F_CONFIG_REG_LDN (index 0x07) must be set to the
   LDN the register belongs to, before you can access the register. */
void it8712f_sio_write(int port, int ldn, int index, int value)
{
	outb(IT8712F_CONFIG_REG_LDN, port);	//base
	outb(ldn, port + 1);	//data (logical device)
	outb(index, port);	//base (index)
	outb(value, port + 1);	//data (config value)
}

int it8712f_sio_read(int port, int ldn, int index)
{
	outb(IT8712F_CONFIG_REG_LDN, port);	//base
	outb(ldn, port + 1);	//data (logical device)
	outb(index, port);	//base (index)
	return inb(port + 1);	//data (config value)
}

void probe_superio(unsigned short port)
{
	/* Perform MB PnP setup to put the SIO chip at 0x2e. */
	/* Base address 0x2e: 0x87 0x01 0x55 0x55. */
	/* Base address 0x4e: 0x87 0x01 0x55 0xaa. */
	outb(0x87, port);
	outb(0x01, port);
	outb(0x55, port);
	if (port == 0x2e)
		outb(0x55, port);
	else
		outb(0xaa, port);

	/* Probe Global Register values */
	printf("\t-- Global Configuration Registers (LDN: All) --\n");
	printf("\tConfigure Control 02h = %02x \n",
	       it8712f_sio_read(port, 0x00, 0x02));
	printf("\tLogical Device Number (LDN) 07h = %02x \n",
	       it8712f_sio_read(port, 0x00, 0x07));
	printf("\tChipID Byte1 20h = %02x \n",
	       it8712f_sio_read(port, 0x00, 0x20));
	printf("\tChipID Byte2 21h = %02x \n",
	       it8712f_sio_read(port, 0x00, 0x21));
	printf("\tConfig Select and Chip Version 22h = %02x \n",
	       it8712f_sio_read(port, 0x00, 0x22));
	printf("\tClock Selection Register 23h = %02x \n",
	       it8712f_sio_read(port, 0x00, 0x23));
	printf("\tSoftware Suspend 24h = %02x \n",
	       it8712f_sio_read(port, 0x00, 0x24));

	printf("\n\t-- Extended Global Config (GPIO) --\n");
	printf("\tGPIO 1 25h = %02x (Default=01h)\n",
	       it8712f_sio_read(port, 0x07, 0x25));
	printf("\tGPIO 2 26h = %02x (Default=00h)\n",
	       it8712f_sio_read(port, 0x07, 0x26));
	printf("\tGPIO 3 27h = %02x (Default=00h)\n",
	       it8712f_sio_read(port, 0x07, 0x27));
	printf("\tGPIO 4 28h = %02x (Default=40h)\n",
	       it8712f_sio_read(port, 0x07, 0x28));
	printf("\tGPIO 5 29h = %02x (Default=00h)\n",
	       it8712f_sio_read(port, 0x07, 0x29));
	printf
	    ("\tExtended 1 Multi-Function Pin Selection Register 2ah = %02x (Default=00h)\n",
	     it8712f_sio_read(port, 0x07, 0x2a));
	printf("\tLogical Block Lock Register 2bh = %02x (Default=00h)\n",
	       it8712f_sio_read(port, 0x00, 0x2b));
	printf
	    ("\tExtended 2 Multi-Function Pin Selection Register 2ch = %02x (Default=00h)\n",
	     it8712f_sio_read(port, 0x07, 0x2c));
	printf("\tTest Register1 2eh = %02x (Default=00h)\n",
	       it8712f_sio_read(port, 0xf4, 0x2e));
	printf("\tTest Register2 2fh = %02x (Default=00h)\n",
	       it8712f_sio_read(port, 0xf4, 0x2f));

	printf("\n\t-- Serial Port 1 ldn=01h--\n");
	printf("\tActivate 30h = %02x \n", it8712f_sio_read(port, 0x01, 0x30));
	printf("\tBase Address MSB Register 60h = %02x \n",
	       it8712f_sio_read(port, 0x01, 0x60));
	printf("\tBase Address LSB Register 61h = %02x \n",
	       it8712f_sio_read(port, 0x01, 0x61));
	printf("\tInterrupt Level Select 70h = %02x \n",
	       it8712f_sio_read(port, 0x01, 0x70));
	printf("\tSpecial Configuration Register(1) f0h = %02x \n",
	       it8712f_sio_read(port, 0x01, 0xf0));
	printf("\tSpecial Configuration Register(2) f1h = %02x \n",
	       it8712f_sio_read(port, 0x01, 0xf1));
	printf("\tSpecial Configuration Register(3) f2h = %02x \n",
	       it8712f_sio_read(port, 0x01, 0xf2));
	printf("\tSpecial Configuration Register(4) f3h = %02x \n",
	       it8712f_sio_read(port, 0x01, 0xf3));

	printf("\n\t-- Environment Controller ldn=04h --\n");
	printf("\tEnvironment Controller Activate 30h = %02x (Default=00h)\n",
	       it8712f_sio_read(port, 0x04, 0x30));
	printf
	    ("\tEnvironment Controller Base Address MSB Register 60h = %02x (Default=02h)\n",
	     it8712f_sio_read(port, 0x04, 0x60));
	printf
	    ("\tEnvironment Controller Base Address LSB Register 61h = %02x (Default=90h)\n",
	     it8712f_sio_read(port, 0x04, 0x61));
	printf
	    ("\tPME Direct Access Base Address MSB Register 62h = %02x (Default=02h)\n",
	     it8712f_sio_read(port, 0x04, 0x62));
	printf
	    ("\tPME Direct Access Base Address LSB Register 63h = %02x (Default=30h)\n",
	     it8712f_sio_read(port, 0x04, 0x63));
	printf
	    ("\tEnvironment Controller Interrupt Level Select 70h = %02x (Default=09h)\n",
	     it8712f_sio_read(port, 0x04, 0x70));
	printf("\tAPC/PME Event Enable Register f0h = %02x (Default=00h)\n",
	       it8712f_sio_read(port, 0x04, 0xf0));
	printf("\tAPC/PME Status Register f1h = %02x (Default=00h)\n",
	       it8712f_sio_read(port, 0x04, 0xf1));
	printf("\tAPC/PME Control Register1 f2h = %02x (Default=00h)\n",
	       it8712f_sio_read(port, 0x04, 0xf2));
	printf
	    ("\tEnvironment Controller Special Configuration Register f3h = %02x (Default=00h)\n",
	     it8712f_sio_read(port, 0x04, 0xf3));
	printf("\tAPC/PME Control Register2 f4h = %02x (Default=00h)\n",
	       it8712f_sio_read(port, 0x04, 0xf4));
	printf
	    ("\tAPC/PME Special Code Index Register f5h = %02x (Default=--)\n",
	     it8712f_sio_read(port, 0x04, 0xf5));
	printf("\tAPC/PME Special Code Data Register f6h = %02x (Default=--)\n",
	       it8712f_sio_read(port, 0x04, 0xf6));

	printf("\n\t-- keyboard ldn=05h --\n");
	printf("\tActivate 30h = %02x (Default=01h or 00h)\n",
	       it8712f_sio_read(port, 0x05, 0x30));
	printf("\tData Base Address MSB Register 60h = %02x (Default=00h)\n",
	       it8712f_sio_read(port, 0x05, 0x60));
	printf("\tData Base Address LSB Register 61h = %02x (Default=60h)\n",
	       it8712f_sio_read(port, 0x05, 0x61));
	printf("\tCommand Base Address MSB Register 62h = %02x (Default=00h)\n",
	       it8712f_sio_read(port, 0x05, 0x62));
	printf("\tCommand Base Address LSB Register 63h = %02x (Default=64h)\n",
	       it8712f_sio_read(port, 0x05, 0x63));
	printf("\tInterrupt Level Select 70h = %02x (Default=01h)\n",
	       it8712f_sio_read(port, 0x05, 0x70));
	printf("\tInterrupt Type 71h = %02x (Default=02h)\n",
	       it8712f_sio_read(port, 0x05, 0x71));
	printf("\tSpecial Configuration Register f0h = %02x (Default=00h)\n",
	       it8712f_sio_read(port, 0x05, 0xf0));

	printf("\n\t-- mouse ldn=06h --\n");
	printf("\tActivate 30h = %02x (Default=00h)\n",
	       it8712f_sio_read(port, 0x06, 0x30));
	printf("\tInterrupt Level Select 70h = %02x (Default=01h)\n",
	       it8712f_sio_read(port, 0x06, 0x70));
	printf("\tInterrupt Type 71h = %02x (Default=02h)\n",
	       it8712f_sio_read(port, 0x06, 0x71));
	printf("\tSpecial Configuration Register f0h = %02x (Default=00h)\n",
	       it8712f_sio_read(port, 0x06, 0xf0));

	printf("\n\t-- GPIO Configuration Registers --\n");
	printf
	    ("\tIRQ3 Ext. Routing Input Pin Mapping Reg. e3h = %02x (Default=00h)\n",
	     it8712f_sio_read(port, 0x07, 0xe3));
	printf
	    ("\tIRQ4 Ext. Routing Input Pin Mapping Reg. e4h = %02x (Default=00h)\n",
	     it8712f_sio_read(port, 0x07, 0xe4));
	printf
	    ("\tIRQ5 Ext. Routing Input Pin Mapping Reg. e5h = %02x (Default=00h)\n",
	     it8712f_sio_read(port, 0x07, 0xe5));
	printf
	    ("\tIRQ6 Ext. Routing Input Pin Mapping Reg. e6h = %02x (Default=00h)\n",
	     it8712f_sio_read(port, 0x07, 0xe6));
	printf
	    ("\tIRQ7 Ext. Routing Input Pin Mapping Reg. e7h = %02x (Default=00h)\n",
	     it8712f_sio_read(port, 0x07, 0xe7));
	printf
	    ("\tIRQ9 Ext. Routing Input Pin Mapping Reg. e9h = %02x (Default=00h)\n",
	     it8712f_sio_read(port, 0x07, 0xe9));
	printf
	    ("\tIRQ10 Ext. Routing Input Pin Mapping Reg. eah = %02x (Default=00h)\n",
	     it8712f_sio_read(port, 0x07, 0xea));
	printf
	    ("\tIRQ11 Ext. Routing Input Pin Mapping Reg. ebh = %02x (Default=00h)\n",
	     it8712f_sio_read(port, 0x07, 0xeb));
	printf
	    ("\tIRQ12 Ext. Routing Input Pin Mapping Reg. ech = %02x (Default=00h)\n",
	     it8712f_sio_read(port, 0x07, 0xec));
	printf
	    ("\tIRQ14 Ext. Routing Input Pin Mapping Reg. eeh = %02x (Default=00h)\n",
	     it8712f_sio_read(port, 0x07, 0xee));
	printf
	    ("\tIRQ15 Ext. Routing Input Pin Mapping Reg. efh = %02x (Default=00h)\n",
	     it8712f_sio_read(port, 0x07, 0xef));
	printf("\tKeyboard Lock Pin Mapping Registe f7h = %02x (Default=00h)\n",
	       it8712f_sio_read(port, 0x07, 0xf7));
}

int main(int argc, char *argv[])
{

	if (iopl(3) < 0) {
		perror("iopl");
		_exit(1);
	}

	/* try the 2e */
	printf("Probing port 0x2e ...\n");
	probe_superio(0x2e);

	printf("\n");

	/* now try the 4e */
	printf("Probing port 0x4e ...\n");
	probe_superio(0x4e);

	return 0;

}
