/*
 * Copyright (C) The Apache Software Foundation. All rights reserved.
 *
 * This software is published under the terms of the Apache Software License
 * version 1.1, a copy of which has been included  with this distribution in
 * the LICENSE file.
 */

package org.apache.log4j;

import java.io.*;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.GregorianCalendar;

import org.apache.log4j.*;
import org.apache.log4j.FileAppender;
import org.apache.log4j.DailyRollingFileAppender;
import org.apache.log4j.HTMLLayout;
import org.apache.log4j.spi.*;
import org.apache.log4j.helpers.*;
import org.apache.log4j.helpers.LogLog;


/**
 * DailyRollingFileAppenderExt extends {@link org.apache.log4j.FileAppender}
 * so that the underlying file is rolled over at a user chosen frequency.
 *
 * <p>The rolling schedule is specified by the <b>DatePattern</b>
 * option. This pattern should follow the {@link java.text.SimpleDateFormat}
 * conventions. In particular, you <em>must</em> escape literal text
 * within a pair of single quotes. A formatted version of the date
 * pattern is used before the extension, if any, of the rolled file name.
 *
 * <p>For example, if the <b>File</b> option is set to
 * <code>/foo/bar.log</code> and the <b>DatePattern</b> set to
 * <code>'.'yyyy-MM-dd</code>, if today is 2001-02-16 then the
 * logging file <code>/foo/bar.2001-02-16.log</code> will be used.
 * At midnight logging will continue in the logging file
 * <code>/foo/bar.2001-02-17.log</code> until it is rolled over
 * itself the next day.
 *
 * <p>In the above example, if the <b>File</b> option had been
 * set to <code>/foo/bar</code> instead, then the logging files
 * would have been <code>/foo/bar.2001-02-16</code> and
 * <code>/foo/bar.2001-02-17</code> respectively.
 *
 * <p>Is is possible to specify monthly, weekly, half-daily, daily,
 * hourly, or minutely rollover schedules.
 *
 * <p><table border="1">
 * <tr>
 * <th>DatePattern</th>
 * <th>Rollover schedule</th>
 *
 * <tr>
 * <td><code>'.'yyyy-MM</code>
 * <td>Rollover at the beginning of each month</td>
 *
 *
 * <tr>
 * <td><code>'.'yyyy-ww</code>
 *
 * <td>Rollover at the first day of each week. The first day of the
 * week depends on the locale.</td>
 *
 *
 * <tr>
 * <td><code>'.'yyyy-MM-dd</code>
 *
 * <td>Rollover at midnight each day.</td>
 *
 *
 * <tr>
 * <td><code>'.'yyyy-MM-dd-a</code>
 *
 * <td>Rollover at midnight and midday of each day.</td>
 *
 *
 * <tr>
 * <td><code>'.'yyyy-MM-dd-HH</code>
 *
 * <td>Rollover at the top of every hour.</td>
 *
 *
 * <tr>
 * <td><code>'.'yyyy-MM-dd-HH-mm</code>
 *
 * <td>Rollover at the beginning of every minutue.</td>
 *
 * </table>
 *
 * <p>Do not use the colon ":" character in anywhere in the
 * <b>DatePattern</b> option. The text before the colon is interpeted
 * as the protocol specificaion of a URL which is probably not what
 * you want.</p>
 *
 * <p>Ripped largely from {@link org.apache.log4j.DailyRollingFileAppender}
 * but that wasn't written in an extensible way.</p>
 *
 * @author <a href="mailto:donald_l_taylor_jr@yahoo.com">Don Taylor</a>
 */

public class DailyRollingFileAppenderExt extends FileAppender {
    // The code assumes that the following constants are in a increasing
    // sequence.
    public static final int TOP_OF_TROUBLE = -1;
    public static final int TOP_OF_MINUTE = 0;
    public static final int TOP_OF_HOUR = 1;
    public static final int HALF_DAY = 2;
    public static final int TOP_OF_DAY = 3;
    public static final int TOP_OF_WEEK = 4;
    public static final int TOP_OF_MONTH = 5;

    /**
     * A string constant used in naming the option for setting the
     * filename pattern. Current value of this string constant is
     * <strong>DatePattern</strong>.
     */
    public static final String DATE_PATTERN_OPTION = "DatePattern";

    /**
     * <p>Maximum number of logs to maintain.</p>
     *
     * <p>If more than this number
     * of logs exist, then the extra logs will be deleted, starting
     * with the oldest logs. If this value is not greater than 0, then
     * no logs will be deleted. Defaults to -1.</p>
     */
    protected static final int MAX_LOGS = -1;

    /**
     * The date pattern. By default, the pattern is set to
     * "'.'yyyy-MM-dd" meaning daily rollover.
     */
    private String datePattern = "'.'yyyy-MM-dd";

    /**
     * The timestamp when we shall next recompute the filename.
     */
    private long nextCheck = System.currentTimeMillis() - 1;

    private Date now = new Date();
    private SimpleDateFormat sdf;
    private RollingCalendar rc = new RollingCalendar();
    private int checkPeriod = TOP_OF_TROUBLE;
    private String baseFilename;
    private int max_logs = MAX_LOGS;
    private boolean writeHeaderDuringAppend = true;
    private boolean appendingLog;

    /**
     * The default constructor does nothing.
     */
    public DailyRollingFileAppenderExt() {
    }

    /**
     * Instantiate a <code>DailyRollingFileAppenderExt</code> and open the
     * file designated by <code>filename</code>. The opened filename will
     * become the ouput destination for this appender.
     */
    public DailyRollingFileAppenderExt(Layout layout, String filename,
            String datePattern) throws IOException {
        setLayout(layout);
        baseFilename = filename;
        this.datePattern = datePattern;
        activateOptions();
    }

    /**
     * The <b>DatePattern</b> takes a string in the same format as
     * expected by {@link java.text.SimpleDateFormat}. This options
     * determines the rollover schedule.
     */
    public void setDatePattern(String pattern) {
        datePattern = pattern;
    }

    /**
     * Returns the value of the <b>DatePattern</b> option.
     */
    public String getDatePattern() {
        return datePattern;
    }

    public void activateOptions() {
        if (datePattern != null && baseFilename != null) {
            sdf = new SimpleDateFormat(datePattern);
            int type = computeCheckPeriod(datePattern);

            printPeriodicity(type);
            rc.setType(type);

            String datedFileName = insertDate(baseFilename, sdf.format(new Date()));
            File target = new File(datedFileName);

            try {
                // If the log file doesn't exist or has a zero size then
                // we should append. File.length() is supposed to return 0L
                // if the file doesn't exist.
                appendingLog = (0L >= target.length()) ? false : true;
                super.setFile(datedFileName, appendingLog);
                deleteOldLogs();
            } catch (IOException ioe) {
                errorHandler.error("super.setFile(" + datedFileName + ", " +
                        target.exists() + ") call failed.");
            }

            long currentTimeMillis = System.currentTimeMillis();
            now.setTime(currentTimeMillis);
            nextCheck = rc.getNextCheckMillis(now);
            LogLog.debug("nextCheck = "+nextCheck);
            super.activateOptions();
        } else {
            LogLog.error("Either Filename or DatePattern options are not set " +
                    "for [" + name + "].");
        }
    }

    /**
     * <p>Output to the debug log the rollover schedule.</p>
     * @see org.apache.log4j.helpers.LogLog
     */
    protected void printPeriodicity(int type) {
        switch (type) {
        case TOP_OF_MINUTE:
            LogLog.debug("Appender [" + name + "] to be rolled every minute.");
            break;
        case TOP_OF_HOUR:
            LogLog.debug("Appender [" + name +
                    "] to be rolled on top of every hour.");
            break;
        case HALF_DAY:
            LogLog.debug("Appender [" + name +
                    "] to be rolled at midday and midnight.");
            break;
        case TOP_OF_DAY:
            LogLog.debug("Appender [" + name + "] to be rolled at midnight.");
            break;
        case TOP_OF_WEEK:
            LogLog.debug("Appender [" + name + "] to be rolled at start " +
                    "of week.");
            break;
        case TOP_OF_MONTH:
            LogLog.debug("Appender [" + name +
                    "] to be rolled at start of every month.");
            break;
        default:
            LogLog.warn("Unknown periodicity for appender [" + name + "].");
            break;
        }
    }

    /**
     * <p>Determines the type of check period for the specified
     * <code>datePattern</code>.</p>
     *
     * @return One of <ul>
     * <li><code>TOP_OF_MINUTE</code></li>
     * <li><code>TOP_OF_HOUR</code></li>
     * <li><code>HALF_DAY</code></li>
     * <li><code>TOP_OF_DAY</code></li>
     * <li><code>TOP_OF_WEEK</code></li>
     * <li><code>TOP_OF_MONTH</code></li>
     * </ul>
     */
     protected int computeCheckPeriod(String datePattern) {
        RollingCalendar c = new RollingCalendar();

        // set sate to 1970-01-01 00:00:00 GMT
        final Date epoch = new Date(0);

        if (datePattern != null) {
            for (int i = TOP_OF_MINUTE; i <= TOP_OF_MONTH; i++) {
                String r0 = sdf.format(epoch);             
                c.setType(i);
                Date next = new Date(c.getNextCheckMillis(epoch));
                String r1 = sdf.format(next);
                LogLog.debug("Type = "+i+", r0 = "+r0+", r1 = "+r1);
                if ( r0 != null && r1 != null && !r0.equals(r1) ) {
                    return i;
                }
            }
        }

        return TOP_OF_TROUBLE;  // Deliberately head for trouble...
    }

    /**
     * Rollover to a new file.
     */
    protected void rollOver() throws IOException {

        // Compute filename, but only if datePattern is specified
        if (datePattern == null) {
            errorHandler.error("Missing DatePattern option in rollOver().");
            return;
        }

        String datedFileName = insertDate(baseFilename, sdf.format(new Date()));
        File target = new File(datedFileName);

        if (target.exists()) {
            target.delete();
        }

        LogLog.debug(super.getFile() + " -> " + datedFileName);

        try {
            writeFooter();
            super.setFile(datedFileName, false);
            deleteOldLogs();
        } catch (IOException e) {
            errorHandler.error("super.setFile("+datedFileName+", false) " + 
                    "call failed.");
        }
    }

    /**
     * This method differentiates DailyRollingFileAppenderExt from its
     *   super class.
     */
    protected void subAppend(LoggingEvent event) {
        long n = System.currentTimeMillis();

        if (n >= nextCheck) {
            now.setTime(n);
            nextCheck = rc.getNextCheckMillis(now);
            try {
                rollOver();
            } catch (IOException ioe) {
                LogLog.error("rollOver() failed.", ioe);
            }
        }

        super.subAppend(event);
    }

    /**
     * <p>Inserts the specified date string into the filename before
     * the filename's extension yielding a result such as:<br>
     * filename_base.date.filename_ext.</p>
     */
    private String insertDate(String filename, String date) {
        String fullname;
        int dot = filename.lastIndexOf(".");

        if (-1 < dot) {
            fullname = filename.substring(0,dot) + date +
                    filename.substring(dot);
        } else {
            fullname = filename + date;
        }

        return fullname;
    }

    /**
     * <p>Sets the <b>base</b> filename we're to log to. This is not
     * the actual file that will be logged to, as the actual file
     * will include a date/time stamp.</p>
     */
    public void setFile(String filename) {
        baseFilename = filename;
    }

    /**
     * <p>Returns the <b>base</b> filename we're logging to.</p>
     *
     * @see #setFile
     */
    public String getFile() {
        return baseFilename;
    }

    /**
     * <p>Set the maximum number of logs to retain. Excess logs will
     * be deleted, oldest logs first. If the value <= 0; then no logs
     * will be deleted.</p>
     *
     * @see #deleteOldLogs
     */
    public void setMaxLogs(int maxlogs) {
        max_logs = maxlogs;
    }

    /**
     * <p>The maximum number of logs being retained.</p>
     *
     * @see #setMaxLogs
     * @see #deleteOldLogs
     */
    public int getMaxLogs() {
        return max_logs;
    }

    /**
     * <p>If the <b>maxLogs</b> property has been set, then this will
     * delete the excess logs, oldest logs first.</p>
     *
     * @see #setMaxLogs
     */
    protected void deleteOldLogs() {
        if (0 < max_logs && null != baseFilename) {
            // Get list of log files
            File file = 
                    new File(baseFilename).getAbsoluteFile().getParentFile();
            File[] logs = file.listFiles(new LogFilter());

            if ( (null != logs) && (max_logs < logs.length) ) {
                // Too many logs, we have to delete the oldest.

                // First sort the logs by their last modified time.
                // The oldest logs will thus be first in the list.
                TreeMap tm = new TreeMap();
                for (int log=0; log < logs.length; log++) {
                    tm.put(new Long(logs[log].lastModified()), logs[log]);
                }

                // Now delete the oldest.
                for (int oldest=0; oldest < (logs.length - max_logs);
                        oldest++) {
                    try {
                        ((File) tm.get(tm.firstKey())).delete();
                        tm.remove(tm.firstKey());
                    } catch (ClassCastException cce) {
                        // this should never happen
                        LogLog.error("TreeMap object wasn't a File!", cce);
                    }
                }
            }
        } else {
            LogLog.debug("Either max_logs <= 0 or baseFilename is not set.");
        }
    }

    /**
     * <p>Sets the <b>writeHeaderDuringAppend</b> property. If <code>true</code>,
     * then when appending to an already existing log, the contents of the
     * associated {@link org.apache.log4j.Layout#getHeader} will be written to
     * the log, otherwise it will not.</p>
     *
     * <p>This property exists to solve a problem when using
     * {@link org.apache.log4j.HTMLLayout}-based layouts, it is an error to
     * write an HTML header multiple times to an HTML file.</p>
     */
    public void setWriteHeaderDuringAppend(boolean flag) {
        writeHeaderDuringAppend = flag;
    }

    /**
     * <p>Retrieve the <b>writeHeaderDuringAppend</b> property.</p>
     *
     * @see #setWriteHeaderDuringAppend
     */
    public boolean getWriteHeaderDuringAppend() {
        return writeHeaderDuringAppend;
    }

    /**
     * <p>Write the contents of this class'
     * {@link org.apache.log4j.Layout#getHeader} to the log, unless
     * we're appending to this log and the <b>writeHeaderDuringAppend</b>
     * property is <code>false</code>.</p>
     *
     * @see #setWriteHeaderDuringAppend
     */
    protected void writeHeader() {
        if (appendingLog && !writeHeaderDuringAppend) {
            return;
        } else {
            super.writeHeader();
        }
    }

    
    /**
     * RollingCalendar is a helper class to
     * DailyRollingFileAppenderExt. Using this class, it is easy to compute
     * and access the next Millis().
     *
     * It subclasses the standard {@link GregorianCalendar}-object, to
     * allow access to the protected function getTimeInMillis(), which it
     * then exports.
     *
     * @author <a HREF="mailto:eirik.lygre@evita.no">Eirik Lygre</a>
     */
    protected static class RollingCalendar extends GregorianCalendar {
        int type = DailyRollingFileAppenderExt.TOP_OF_TROUBLE;

        /**
         * Method declaration
         */
        void setType(int type) {
            this.type = type;
        }

        /**
         * Method declaration
         */
        public long getNextCheckMillis(Date now) {
            return getNextCheckDate(now).getTime();
        }

        /**
         * Method declaration
         */
        public Date getNextCheckDate(Date now) {
            this.setTime(now);

            switch (type) {
            case DailyRollingFileAppenderExt.TOP_OF_MINUTE:
                this.set(Calendar.SECOND, 0);
                this.set(Calendar.MILLISECOND, 0);
                this.add(Calendar.MINUTE, 1);
                break;
            case DailyRollingFileAppenderExt.TOP_OF_HOUR:
                this.set(Calendar.MINUTE, 0);
                this.set(Calendar.SECOND, 0);
                this.set(Calendar.MILLISECOND, 0);
                this.add(Calendar.HOUR_OF_DAY, 1);
                break;
            case DailyRollingFileAppenderExt.HALF_DAY:
                this.set(Calendar.MINUTE, 0);
                this.set(Calendar.SECOND, 0);
                this.set(Calendar.MILLISECOND, 0);

                int hour = get(Calendar.HOUR_OF_DAY);

                if (hour < 12) {
                    this.set(Calendar.HOUR_OF_DAY, 0);
                } else {
                    this.set(Calendar.HOUR_OF_DAY, 12);
                }

                break;
            case DailyRollingFileAppenderExt.TOP_OF_DAY:
                this.set(Calendar.HOUR_OF_DAY, 0);
                this.set(Calendar.MINUTE, 0);
                this.set(Calendar.SECOND, 0);
                this.set(Calendar.MILLISECOND, 0);
                this.add(Calendar.DATE, 1);
                break;
            case DailyRollingFileAppenderExt.TOP_OF_WEEK:
                this.set(Calendar.DAY_OF_WEEK, getFirstDayOfWeek());
                this.set(Calendar.HOUR_OF_DAY, 0);
                this.set(Calendar.SECOND, 0);
                this.set(Calendar.MILLISECOND, 0);
                this.add(Calendar.WEEK_OF_YEAR, 1);
                break;
            case DailyRollingFileAppenderExt.TOP_OF_MONTH:
                this.set(Calendar.DATE, 1);
                this.set(Calendar.HOUR_OF_DAY, 0);
                this.set(Calendar.SECOND, 0);
                this.set(Calendar.MILLISECOND, 0);
                this.add(Calendar.MONTH, 1);
                break;
            default:
                throw new IllegalStateException("Unknown periodicity type.");
            }

            return getTime();
        }

    }  // class RollingCalendar

    
    /**
     * <p>Used by {@link DailyRollingFileAppenderExt#deleteOldLogs} to
     * determine the current set of log files that are in use.</p>
     */
    protected class LogFilter implements FilenameFilter {

        public boolean accept(File dir, String name) {
            int dot = baseFilename.lastIndexOf('.');
            String date = null;

            if (-1 < dot) {
                String base = baseFilename.substring(0, dot);
                String ext = baseFilename.substring(dot);
                if (name.startsWith(base) && name.endsWith(ext)) {
                    date = name.substring(base.length(), name.lastIndexOf('.'));
		}
            } else if (name.startsWith(baseFilename)) {
                date = name.substring(baseFilename.length());
            }

            if (null != date) {
                sdf.setLenient(false);
                try {
                    Date d = sdf.parse(date);
                    if (null != d) {
                        return true;
                    } else {
                        return false;
                    }
                } catch (java.text.ParseException pe) {
                    return false;
                }
            } else {
                return false;
            }
        }
    } // class LogFilter
}
