/**
 * Logback: the reliable, generic, fast and flexible logging framework.
 * Copyright (C) 2011-2012, Anthony Trinh. All rights reserved.
 * Copyright (C) 1999-2011, QOS.ch. All rights reserved.
 *
 * This program and the accompanying materials are dual-licensed under
 * either the terms of the Eclipse Public License v1.0 as published by
 * the Eclipse Foundation
 *
 *   or (per the licensee's choosing)
 *
 * under the terms of the GNU Lesser General Public License version 2.1
 * as published by the Free Software Foundation.
 */

package ch.qos.logback.classic;

import org.eclipse.core.runtime.Status;

import com.example.e4.helloworld.Activator;

import ch.qos.logback.classic.Level;
import ch.qos.logback.classic.encoder.PatternLayoutEncoder;
import ch.qos.logback.classic.spi.ILoggingEvent;
import ch.qos.logback.core.AppenderBase;

/**
 * An appender that wraps the Eclipse platform logging facility
 * and redirects messages to Eclipse's Error Log. 
 * 
 * @author Anthony Trinh
 */
public class EclipseLogAppender extends AppenderBase<ILoggingEvent> {

	private PatternLayoutEncoder encoder = null;
	
	/**
	 * Checks that required parameters are set, and if everything is in order,
	 * activates this appender.
	 */
	@Override
	public void start() {
		if ((this.encoder == null) || (this.encoder.getLayout() == null)) {
			addError("No layout set for the appender named [" + name + "].");
			return;
		}

		super.start();
	}

	/**
	 * Writes an event to Eclipse's Error Log
	 * 
	 * @param event
	 *            the event to be logged
	 */
	@Override
	protected void append(ILoggingEvent event) {
		
		if (!isStarted()) {
			return;
		}
				
		// Map log level to appropriate Eclipse status (one of: CANCEL, ERROR, INFO, OK, WARNING)
		int status = Status.OK;
		int lev = event.getLevel().levelInt;
		switch (lev) {
		case Level.WARN_INT:
			status = Status.WARNING;
			break;
			
		case Level.ERROR_INT:
			status = Status.ERROR;
			break;
			
		default:
			status = Status.INFO;
			break;
		}
		
		// Log the message if level allows (not off)
		if (lev != Level.OFF_INT) {
			String msg = this.encoder.getLayout().doLayout(event);
			
			Activator
				.getDefault()
				.getLog()
				.log(new Status(Status.INFO, Activator.PLUGIN_ID, status, msg, null));
		}
	}

	/**
	 * Gets the pattern-layout encoder for this appender's <i>logcat</i> message
	 * 
	 * @return the pattern-layout encoder
	 */
	public PatternLayoutEncoder getEncoder() {
		return this.encoder;
	}

	/**
	 * Sets the pattern-layout encoder for this appender's <i>logcat</i> message
	 * 
	 * @param encoder the pattern-layout encoder
	 */
	public void setEncoder(PatternLayoutEncoder encoder) {
		this.encoder = encoder;
	}

}
