package net.charabia.normalizer;

/* ====================================================================
 * Copyright (c) 2002 Rodrigo Reyes.  All rights reserved.
 * ====================================================================
 */

import java.util.*;

/**
 * A rule object is a single processing rule, which defines a string
 * to match, but also a right and a left context, along with the
 * replacement string.
 *
 * <p> When a rule is applied on a transducer buffer, it checks for
 * the matching string, then verifies first the right context, then
 * the left context. If the rule happens to match, it injects in the
 * TransducerBuffer output the replacement string.
 *
 */

public class Rule
{
    private Vector m_match;
    private Vector m_leftcontext;
    private Vector m_rightcontext;
    private String m_replacement;

    /**
     * Public constructor.
     *
     * @param match the string to be matched (and replaced, if the rule applies)
     * @param left the left context, to be matched for the rule to be applied
     * @param right the right context
     * @param repl the replacement string, that is injected in the TransducerBuffer output.
     */

    public Rule(Vector match, Vector left, Vector next, String repl)
    {
	m_match = match;
	m_leftcontext = left;
	m_rightcontext = next;
	m_replacement = repl;
    }

    public Vector getMatched()
    {
	return m_match;
    }

    /**
     * Applies the rule on TransducerBuffer. If the rule is applied
     * (which is the case when the main string, the left context, and
     * the right one are true at the current position of the
     * TransducerBuffer), then the TransducerBuffer position is
     * updated, the replacement string is injected, and the method
     * returns true.
     *
     * @return true if the rule is applied, false otherwise.  */

    public boolean matches(TransducerBuffer buffer)
    {
	int index;

	// Check the central rules
	index = buffer.getIndex();
	int size = 0;
	for (Iterator iter = m_match.iterator(); iter.hasNext(); index++, size++)
	    {
		TokenMatcher matcher = (TokenMatcher)iter.next();
		if (matcher.matches(buffer, index) == false)
		    return false;
	    }

	//	System.out.println("passed rule " + this + ", central " + m_match);

	// check the right context rules
	for (Iterator iter = m_rightcontext.iterator(); iter.hasNext(); index++)
	    {
		TokenMatcher matcher = (TokenMatcher)iter.next();
		if (matcher.matches(buffer, index) == false)
		    return false;
	    }
	// check the left context rules
	index = buffer.getIndex() - 1;
	for (Iterator iter = m_leftcontext.iterator(); iter.hasNext(); index--)
	    {
		TokenMatcher matcher = (TokenMatcher)iter.next();
		if (matcher.matches(buffer, index) == false)
		    return false;
	    }

	//	System.out.println("Applying rule " + this);

	buffer.inject(m_replacement);
	buffer.setIndex(buffer.getIndex() + size);

	return true;
    }

}
