/*
 *  Copyright (C) 2002 , Torsten Mohr, tmohr@s.netic.de
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  as published by the Free Software Foundation; either version 2
 *  of the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#include "slImg.h"
#include "slPng.h"
#include "slLAV.h"


#define QWE fprintf(stderr, "file '%s', func '%s', line %i\n", __FILE__, __func__, __LINE__)

static SL_LAV_IN left_avi;
static SL_LAV_IN right_avi;
static SL_LAV_OUT out_avi;

static SL_IMG left_pic;
static SL_IMG right_pic;
static SL_IMG blend_pic;

static int offset = 0;
static char* inleft = NULL;
static char* inright = NULL;
static char* outstream = NULL;
static char* picname = NULL;
static int seq = -1;
static int x1 = 374;
static int x2 = 394;

static void usage(void) {
	fprintf(stderr,
		"pngblend [options]\n"
		"           -a offset in right input stream\n"
		"           -l left input stream name\n"
		"           -r right input stream name\n"
		"           -o output stream name\n"
		"           -p picture name take alpha from picture\n"
		"           -n left pic starts to fade out\n"
		"           -m right pic is faded in here\n"
	);
}

static int min(int a, int b) {
	return a < b ? a : b;
}


int main(int argc, char** argv) {
	char* namepattern = NULL;
	char* ep;
	int i, ret, len;
	int w, h;
	unsigned long int tmp;

	while(i = getopt(argc,argv, "l:r:o:a:p:n:m:"), i != -1) {
		switch(i) {
			case 'o':
				outstream = optarg;
				break;

			case 'l':
				inleft = optarg;
				break;

			case 'r':
				inright = optarg;
				break;

			case 'p':
				picname = optarg;
				break;

			case 'a':
				tmp = strtoul(optarg, &ep, 0);
				if(*ep != 0) {
					fprintf(stderr, "didn't recognise '%s'\n", optarg);
					exit(1);
				}
				if(tmp < 0) {
					fprintf(stderr, "offset not valid\n");
					exit(1);
				}
				offset = tmp;
				break;

			case 'n':
				tmp = strtoul(optarg, &ep, 0);
				if(*ep != 0) {
					fprintf(stderr, "didn't recognise '%s'\n", optarg);
					exit(1);
				}
				if(tmp < 0) {
					fprintf(stderr, "'n' not valid\n");
					exit(1);
				}
				x1 = tmp;
				break;

			case 'm':
				tmp = strtoul(optarg, &ep, 0);
				if(*ep != 0) {
					fprintf(stderr, "didn't recognise '%s'\n", optarg);
					exit(1);
				}
				if(tmp < 0) {
					fprintf(stderr, "'m' not valid\n");
					exit(1);
				}
				x2 = tmp;
				break;

			case '?':
			case ':':
			default:
				fprintf(stderr, "got %i\n", i);
				exit(1);
				break;
		}
	}

	if(inleft == NULL) {
		printf("left input stream missing\n");
		usage();
		exit(1);
	}

	if(inright == NULL) {
		printf("right input stream missing\n");
		usage();
		exit(1);
	}

	if(outstream == NULL) {
		printf("output stream missing\n");
		usage();
		exit(1);
	}

	if(x1 > x2) {
		printf("impossible: n > m\n");
		usage();
		exit(1);
	}

	if(picname != NULL) {
		printf("impossible: n > m\n");
		usage();
		exit(1);
	}

	ret = slLAVReadInit(&left_avi, inleft);
	if(ret != 0) {
		printf("couldn't read-open '%s'\n", inleft);
		exit(1);
	}
	w = left_avi.w;
	h = left_avi.h;

	ret = slLAVReadInit(&right_avi, inright);
	if(ret != 0) {
		printf("couldn't read-open '%s'\n", inright);
		exit(1);
	}

	if(w != right_avi.w || h != right_avi.h) {
		printf("differing width or height in left and right input streams\n");
		exit(1);
	}

	ret = slImgMalloc(&left_pic, w, h);
	if(ret != 0) {
		printf("couldn't get memory for left temp picture\n");
		exit(1);
	}

	ret = slImgMalloc(&right_pic, w, h);
	if(ret != 0) {
		printf("couldn't get memory for right temp picture\n");
		exit(1);
	}

	if(picname == NULL) {
		ret = slImgMalloc(&blend_pic, w, h);
		if(ret != 0) {
			printf("couldn't get memory for right temp picture\n");
			exit(1);
		}
		slImgBlend(&blend_pic, x1, x2);
//		ret = slPngWrite("blendfile.png", &blend_pic);
	}
	else {
		ret = slPngRead(picname, &blend_pic);
		if(ret != 0) {
			printf("couldn't read blend_pic %s\n", picname);
			exit(1);
		}
		if(blend_pic.w != w || blend_pic.h != h) {
			printf("differing sizes, blend_pic %s doesn't fit\n", picname);
			exit(1);
		}
		// convert pic to grayscale
		slImgGrayscale(&blend_pic);
	}

	ret = slLAVWriteInit(&out_avi, outstream, w, h, 70, 0);
	if(ret != 0) {
		printf("couldn't write-open '%s'\n", outstream);
		exit(1);
	}

	len = min(left_avi.num_frames, right_avi.num_frames);
	for(i = 0; i < len; i++) {
		printf("processing frame %i\n", i);

		ret = slLAVReadFrame(&left_avi, &left_pic, i);
		if(ret != 0) {
			printf("couldn't read left frame %i\n", i);
			exit(1);
		}

		ret = slLAVReadFrame(&right_avi, &right_pic, i);
		if(ret != 0) {
			printf("couldn't read right frame %i\n", i);
			exit(1);
		}

		slImgMix(&left_pic, &right_pic, &blend_pic);

		ret = slLAVWriteFrame(&out_avi, &left_pic);
		if(ret != 0) {
			printf("couldn't write frame %i\n", i);
			exit(1);
		}
		
	}

	slLAVInClose(&left_avi);
	slLAVInClose(&right_avi);
	slLAVOutClose(&out_avi);


	return EXIT_SUCCESS;
}

