#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>

#include "slLAV.h"
#include "slImg.h"


static int initialized = 0;

static unsigned char* tmp;

static unsigned char* Y;
static unsigned char* Cb;
static unsigned char* Cr;


static void check_init(void) {
	if(!initialized) {
		initialized = 1;

		Y = (unsigned char*)malloc(SL_LAV_BS_Y);
		Cb = (unsigned char*)malloc(SL_LAV_BS_C);
		Cr = (unsigned char*)malloc(SL_LAV_BS_C);
		tmp = (unsigned char*)malloc(SL_LAV_BS_RGB);

		if(Y == NULL || Cb == NULL || Cr == NULL || tmp == NULL) {
			fprintf(stderr, "couldn't get memory");
			exit(1);
		}
	}
}


int slLAVReadInit(SL_LAV_IN* src, char* fname) {
	check_init();

	src->in = lav_open_input_file(fname);
	if(src->in == NULL) {
		fprintf(stderr, "couldn't load '%s'\n", fname);
		exit(1);
	}

	src->w = lav_video_width(src->in);
	src->h = lav_video_height(src->in);
	src->num_frames = lav_video_frames(src->in);
	src->act_frame = -1;

	return 0;
}


extern int slLAVReadFrame(SL_LAV_IN* src, SL_IMG* img, int num) {
	int rr, gg, bb, jy, jx, j;
	size_t size;
	unsigned char* ptr;
	char buf[1024];

	if(!initialized) {
		fprintf(stderr, "can't load Frame when not initialized");
		exit(1);
	}

	lav_set_video_position(src->in, num);
	size = lav_frame_size(src->in, num);

	// read the frame
	lav_read_frame(src->in, tmp);

	// convert it to YCbCr
	decode_jpeg_raw(tmp, size, lav_video_interlacing(src->in),
			0, src->w, src->h, Y, Cb, Cr);

	ptr = img->data;
	for(jy = src->h-1; jy >= 0; jy--) {
		for(jx = 0; jx < src->w; jx++) {
			j = jy/SL_LAV_SUBY*src->w/SL_LAV_SUBX + jx/SL_LAV_SUBX;
			YCbCr2RGB(Y[jy*src->w+jx], Cb[j], Cr[j], &rr, &gg, &bb);
			*ptr++ = rr;
			*ptr++ = gg;
			*ptr++ = bb;
			*ptr++ = 255;
		}
	}

	slImgFlipY(img);

	return 0;
}


int slLAVInClose(SL_LAV_IN* in) {
	return lav_close(in->in);
}


int slLAVWriteInit(SL_LAV_OUT* out, char* fname,
		int w, int h, int qual, int inter) {

	check_init();

	out->out = lav_open_output_file(fname, 'a',
                    w, h, inter, 25.0,
                    0, 0, 0);
	if(out == NULL) {
		fprintf(stderr, "couldn't write '%s'", fname);
		exit(-1);
	}

	out->w = w;
	out->h = h;
	out->quality = qual;
	out->inter = inter;

	return 0;
}


int slLAVWriteFrame(SL_LAV_OUT* out, SL_IMG* img) {
	int jy, jx, jxp, jyp, ji, jo, jj;
	int yy, cb, cr, sumcb, sumcr;
	int size;
	unsigned char* rgb;

	rgb = img->data;

	slImgFlipY(img);

	for(jy = 0; jy < img->h; jy += SL_LAV_SUBY) {
		for(jx = 0; jx < img->w; jx += SL_LAV_SUBX) {
			sumcb = 0;
			sumcr = 0;
			for(jyp = 0; jyp < SL_LAV_SUBY; jyp++) {
				for(jxp = 0; jxp < SL_LAV_SUBX; jxp++) {
					ji = (img->h - (jy + jyp)) * img->w + (jx + jxp);
					jo = (jy + jyp) * out->w + (jx + jxp);

					RGB2YCbCr(rgb[4*ji], rgb[4*ji+1], rgb[4*ji+2], &yy, &cb, &cr);

					Y[jo] = yy;
					sumcb += cb;
					sumcr += cr;
				}
			}
			for(jyp = 0; jyp < SL_LAV_SUBY; jyp++) {
				for(jxp = 0; jxp < SL_LAV_SUBX; jxp++) {
					jj = (jy+jyp)/SL_LAV_SUBY
						* out->w/SL_LAV_SUBX
						+ (jx/SL_LAV_SUBX + jxp/SL_LAV_SUBX);

					Cb[jj] = sumcb / SL_LAV_SUBX / SL_LAV_SUBY;
					Cr[jj] = sumcr / SL_LAV_SUBX / SL_LAV_SUBY;
				}
			}
		}
	}

	size = encode_jpeg_raw(tmp, 100000L, out->quality, out->inter, 0,
		out->w, out->h, Y, Cb, Cr);
	lav_write_frame(out->out, tmp, size, 1);

	return 0;
}


int slLAVOutClose(SL_LAV_OUT* out) {
	return lav_close(out->out);
}

