#include "y_cb_cr.h"

#define USE_OTHER 1

#if USE_OTHER

typedef unsigned char uint8_t;

#define FP_BITS 18

/* precomputed tables */

static int Y_R[256];
static int Y_G[256];
static int Y_B[256];
static int Cb_R[256];
static int Cb_G[256];
static int Cb_B[256];
static int Cr_R[256];
static int Cr_G[256];
static int Cr_B[256];
static int conv_RY_inited = 0;

static int RGB_Y[256];
static int R_Cr[256];
static int G_Cb[256];
static int G_Cr[256];
static int B_Cb[256];
static int conv_YR_inited = 0;


static int myround(double n)
{
  if (n >= 0) 
    return (int)(n + 0.5);
  else
    return (int)(n - 0.5);
}



static void init_RGB_to_YCbCr_tables(void)
{
  int i;

  /*
   * Q_Z[i] =   (coefficient * i
   *             * (Q-excursion) / (Z-excursion) * fixed-point-factor)
   *
   * to one of each, add the following:
   *             + (fixed-point-factor / 2)         --- for rounding later
   *             + (Q-offset * fixed-point-factor)  --- to add the offset
   *             
   */
  for (i = 0; i < 256; i++) {
    Y_R[i] = myround(0.299 * (double)i 
		     * 219.0 / 255.0 * (double)(1<<FP_BITS));
    Y_G[i] = myround(0.587 * (double)i 
		     * 219.0 / 255.0 * (double)(1<<FP_BITS));
    Y_B[i] = myround((0.114 * (double)i 
		      * 219.0 / 255.0 * (double)(1<<FP_BITS))
		     + (double)(1<<(FP_BITS-1))
		     + (16.0 * (double)(1<<FP_BITS)));

    Cb_R[i] = myround(-0.168736 * (double)i 
		      * 224.0 / 255.0 * (double)(1<<FP_BITS));
    Cb_G[i] = myround(-0.331264 * (double)i 
		      * 224.0 / 255.0 * (double)(1<<FP_BITS));
    Cb_B[i] = myround((0.500 * (double)i 
		       * 224.0 / 255.0 * (double)(1<<FP_BITS))
		      + (double)(1<<(FP_BITS-1))
		      + (128.0 * (double)(1<<FP_BITS)));

    Cr_R[i] = myround(0.500 * (double)i 
		      * 224.0 / 255.0 * (double)(1<<FP_BITS));
    Cr_G[i] = myround(-0.418688 * (double)i 
		      * 224.0 / 255.0 * (double)(1<<FP_BITS));
    Cr_B[i] = myround((-0.081312 * (double)i 
		       * 224.0 / 255.0 * (double)(1<<FP_BITS))
		      + (double)(1<<(FP_BITS-1))
		      + (128.0 * (double)(1<<FP_BITS)));
  }
  conv_RY_inited = 1;
}




static void init_YCbCr_to_RGB_tables(void)
{
  int i;

  /*
   * Q_Z[i] =   (coefficient * i
   *             * (Q-excursion) / (Z-excursion) * fixed-point-factor)
   *
   * to one of each, add the following:
   *             + (fixed-point-factor / 2)         --- for rounding later
   *             + (Q-offset * fixed-point-factor)  --- to add the offset
   *             
   */

  /* clip Y values under 16 */
  for (i = 0; i < 16; i++) {
    RGB_Y[i] = myround((1.0 * (double)(16) 
		     * 255.0 / 219.0 * (double)(1<<FP_BITS))
		    + (double)(1<<(FP_BITS-1)));
  }
  for (i = 16; i < 236; i++) {
    RGB_Y[i] = myround((1.0 * (double)(i - 16) 
		     * 255.0 / 219.0 * (double)(1<<FP_BITS))
		    + (double)(1<<(FP_BITS-1)));
  }
  /* clip Y values above 235 */
  for (i = 236; i < 256; i++) {
    RGB_Y[i] = myround((1.0 * (double)(235) 
		     * 255.0 / 219.0 * (double)(1<<FP_BITS))
		    + (double)(1<<(FP_BITS-1)));
  }
    
  /* clip Cb/Cr values below 16 */	 
  for (i = 0; i < 16; i++) {
    R_Cr[i] = myround(1.402 * (double)(-112)
		   * 255.0 / 224.0 * (double)(1<<FP_BITS));
    G_Cr[i] = myround(-0.714136 * (double)(-112)
		   * 255.0 / 224.0 * (double)(1<<FP_BITS));
    G_Cb[i] = myround(-0.344136 * (double)(-112)
		   * 255.0 / 224.0 * (double)(1<<FP_BITS));
    B_Cb[i] = myround(1.772 * (double)(-112)
		   * 255.0 / 224.0 * (double)(1<<FP_BITS));
  }
  for (i = 16; i < 241; i++) {
    R_Cr[i] = myround(1.402 * (double)(i - 128)
		   * 255.0 / 224.0 * (double)(1<<FP_BITS));
    G_Cr[i] = myround(-0.714136 * (double)(i - 128)
		   * 255.0 / 224.0 * (double)(1<<FP_BITS));
    G_Cb[i] = myround(-0.344136 * (double)(i - 128)
		   * 255.0 / 224.0 * (double)(1<<FP_BITS));
    B_Cb[i] = myround(1.772 * (double)(i - 128)
		   * 255.0 / 224.0 * (double)(1<<FP_BITS));
  }
  /* clip Cb/Cr values above 240 */	 
  for (i = 241; i < 256; i++) {
    R_Cr[i] = myround(1.402 * (double)(112)
		   * 255.0 / 224.0 * (double)(1<<FP_BITS));
    G_Cr[i] = myround(-0.714136 * (double)(112)
		   * 255.0 / 224.0 * (double)(1<<FP_BITS));
    G_Cb[i] = myround(-0.344136 * (double)(i - 128)
		   * 255.0 / 224.0 * (double)(1<<FP_BITS));
    B_Cb[i] = myround(1.772 * (double)(112)
		   * 255.0 / 224.0 * (double)(1<<FP_BITS));
  }
  conv_YR_inited = 1;
}





/* 
 * in-place conversion [R', G', B'] --> [Y', Cb, Cr]
 *
 */

void convert_RGB_to_YCbCr(uint8_t *planes[], int length)
{
  uint8_t *Y, *Cb, *Cr;
  int i;

  if (!conv_RY_inited) init_RGB_to_YCbCr_tables();

  for ( i = 0, Y = planes[0], Cb = planes[1], Cr = planes[2];
	i < length;
	i++, Y++, Cb++, Cr++ ) {
    int r = *Y;
    int g = *Cb;
    int b = *Cr;

    *Y = (Y_R[r] + Y_G[g]+ Y_B[b]) >> FP_BITS;
    *Cb = (Cb_R[r] + Cb_G[g]+ Cb_B[b]) >> FP_BITS;
    *Cr = (Cr_R[r] + Cr_G[g]+ Cr_B[b]) >> FP_BITS;
  }
}



/* 
 * in-place conversion [Y', Cb, Cr] --> [R', G', B']
 *
 */

void convert_YCbCr_to_RGB(uint8_t *planes[], int length)
{
  uint8_t *R, *G, *B;
  int i;

  if (!conv_YR_inited) init_YCbCr_to_RGB_tables();

  for ( i = 0, R = planes[0], G = planes[1], B = planes[2];
	i < length;
	i++, R++, G++, B++ ) {
    int y = *R;
    int cb = *G;
    int cr = *B;

    int r = (RGB_Y[y] + R_Cr[cr]) >> FP_BITS;
    int g = (RGB_Y[y] + G_Cb[cb]+ G_Cr[cr]) >> FP_BITS;
    int b = (RGB_Y[y] + B_Cb[cb]) >> FP_BITS;

    *R = (r < 0) ? 0 : (r > 255) ? 255 : r ;
    *G = (g < 0) ? 0 : (g > 255) ? 255 : g ;
    *B = (b < 0) ? 0 : (b > 255) ? 255 : b ;
  }
  
}

void RGB2YCbCr(int r, int g, int b, int* y, int* cb, int* cr) {
  if (!conv_RY_inited) init_RGB_to_YCbCr_tables();

    *y = (Y_R[r] + Y_G[g]+ Y_B[b]) >> FP_BITS;
    *cb = (Cb_R[r] + Cb_G[g]+ Cb_B[b]) >> FP_BITS;
    *cr = (Cr_R[r] + Cr_G[g]+ Cr_B[b]) >> FP_BITS;
	
}

void YCbCr2RGB(int y, int cb, int cr, int* r, int* g, int* b) {
	int rr, gg, bb;

  if (!conv_YR_inited) init_YCbCr_to_RGB_tables();

    rr = (RGB_Y[y] + R_Cr[cr]) >> FP_BITS;
    gg = (RGB_Y[y] + G_Cb[cb]+ G_Cr[cr]) >> FP_BITS;
    bb = (RGB_Y[y] + B_Cb[cb]) >> FP_BITS;

    *r = (rr < 0) ? 0 : (rr > 255) ? 255 : rr ;
    *g = (gg < 0) ? 0 : (gg > 255) ? 255 : gg ;
    *b = (bb < 0) ? 0 : (bb > 255) ? 255 : bb ;
}


#else



void RGB2YCbCr(int rin, int gin, int bin, int* y, int* cb, int* cr) {
	double R = (double)rin/256.0;
	double G = (double)gin/256.0;
	double B = (double)bin/256.0;

	double Y, Cb, Cr;

	double c11 =  0.299;
	double c12 =  0.587;
	double c13 =  0.114;
	double c21 = -0.168736;
	double c22 = -0.331264;
	double c23 =  0.5;
	double c31 =  0.5;
	double c32 = -0.418688;
	double c33 = -0.081312;

	Y  = c11 * R + c12 * G + c13 * B;
	Cb = c21 * R + c22 * G + c23 * B;
	Cr = c31 * R + c32 * G + c33 * B;

	Y  = Y * 219.0 + 16;
	Cb = Cb * 224.0 + 128;
	Cr = Cr * 224.0 + 128;

	if(Y  < 0.0) Y  = 0.0;
	if(Cr < 0.0) Cr = 0.0;
	if(Cb < 0.0) Cb = 0.0;
	if(Y  > 255.0) Y  = 255.0;
	if(Cb > 255.0) Cb = 255.0;
	if(Cr > 255.0) Cr = 255.0;

	*y = Y;
	*cb = Cb;
	*cr = Cr;
}

void YCbCr2RGB(int Yin, int Cbin, int Crin, int* r, int* g, int* b) {
	double Y = ((double)Yin-16.0)/219.0;
	double Cb = ((double)Cbin-128.0)/224.0;
	double Cr = ((double)Crin-128.0)/224.0;

	double R, G, B;

	double c11 = 1.0;
	double c12 = 0.0000000;
	double c13 = 1.4020000;
	double c21 = 1.0000000;
	double c22 =-0.3441360;
	double c23 =-0.7141360;
	double c31 = 1.0000000;
	double c32 = 1.7720000;
	double c33 = 0.0000000;

	R = c11 * Y + c12 * Cb + c13 * Cr;
	G = c21 * Y + c22 * Cb + c23 * Cr;
	B = c31 * Y + c32 * Cb + c33 * Cr;

	R *= 256.0;
	G *= 256.0;
	B *= 256.0;

	if(R < 0.0) R = 0.0;
	if(G < 0.0) G = 0.0;
	if(B < 0.0) B = 0.0;
	if(R > 255.0) R = 255.0;
	if(G > 255.0) G = 255.0;
	if(B > 255.0) B = 255.0;

	*r = R;
	*g = G;
	*b = B;
}

#endif

