#######################################################################
#
# sessiond - server process to store values identified by a session id.
#
# Used together with mod_dtcl - the tcl module to Apache web server.
#
# Copyright 2000 by Claus Bo Nielsen - cl@us-bo.dk
# ---------------------------------------------------------------------
#
# Changelog
# ---------
#
# 050300/cbn     Initial version
#
#######################################################################

# Globals - can be change
# -----------------------

# the tcp/ip port number this service uses
set service_port 8880		
set session_timeout [expr 60 * 15]

# Globals - NOT to be changed
# ---------------------------
set server_live 1
array set sessions [list]
set start_session [pid]

# Background error handler
proc bgerror { str } {
    logevent $str
}

# Log function
proc logevent { str } {
    puts "[clock format [clock seconds] -format {%d/%m/%y %H:%M:%S}] $str"
}

# Insert new session
proc new { {str "" } } {
    global sessions
    global start_session

    set sessionid [incr start_session]
    array set sessions [list $sessionid [list [clock seconds] $str]]

    return $sessionid
}

# Update session (with no 'str' just update the date stamp)
proc update { sessionid { str "" }} {
    global sessions

    if { $str == "" } {
	set val $sessions($sessionid)
	set val [lreplace $val 0 0 [clock seconds]]
	array set sessions [list $sessionid $val]
    } else {
	array set sessions [list $sessionid [list [clock seconds] $str]]
    }

    logevent $sessions($sessionid)
}

# get a session
proc get { sessionid } {
    global sessions

    return [lindex [lindex [array get sessions $sessionid] 1] 1]
}

# Remove sessions
proc remove { sessionid } {
    global sessions
    
    array unset sessions $sessionid
    logevent "Session $sessionid  - deleted"
}

# cleanup sessions
proc cleanup { } {
    global sessions
    global session_timeout

    set dellist [list]

    set sid [array startsearch sessions]

    while 1 {
	set elem [array get sessions [array nextelement sessions $sid]]

	if { $elem == "" } {
	    break;
	}
	if { [lindex [lindex $elem 1] 0] > [expr [clock seconds] - $session_timeout] } {
	    continue
	}

	lappend dellist [lindex $elem 0]	
    }

    foreach e $dellist {
	logevent "auto deleting $e"
	array unset sessions $e
    }
}

# Client input
proc clientinput { sock {sessionid 0} } {
    global server_live
    global sessions
    global start_session
    global session_timeout
    
    if { [gets $sock input] == -1 } {
	set input "quit"
    }
    
    switch [lindex $input 0] {
	"create" {
	    puts $sock [new [lindex $input 1]]
	    logevent "$sock create new session"
    	}
	
	"exist" -
	"get" {
	    puts $sock [get [lindex $input 1]]
	}
	
	"delete" {
	    logevent "$sock delete session [lindex $input 1]"
	    
	    remove [lindex $input 1]
	    puts $sock "0"
	}
	
	"set" {
	    logevent "$sock set [lindex $input 2] in session [lindex $input 1]"

	    update [lindex $input 1] [lindex $input 2]
	    puts $sock "0"
	}
	
	"quit"  { 
	    logevent "$sock disconnect"
	    close $sock 
	}
	
	"shutdown" {
	    logevent "$sock sending server shutdown"
	    puts $sock "0"
	    set server_live 0
	}
	
	"dump" {
	    logevent "[array get sessions]"
	    puts $sock "0"
	}
	
	default {
	    logevent "$sock unknown command: $input"
	    puts $sock "-1"
	}
    }

    catch {flush $sock}
    
    cleanup
} 

# Called when at client connects
proc clientconnect { sock address port } {
    fconfigure $sock -blocking 0
    set cliname [lindex [fconfigure $sock -sockname] 1]
    
    logevent "$sock connected ($cliname\[$address\]:$port)"
    
    fileevent $sock readable "clientinput $sock"
}


#######################################################################
# Main()
#######################################################################

logevent "sessiond starting on [info hostname] on port $service_port"

# Start the server
set srvsock [socket -server {clientconnect} $service_port]

logevent "sessiond ready"

# Process events (exit when "server_live" will be modified)
vwait server_live

logevent "sessiond shuts down..."

close $srvsock

#######################################################################
# End of script
#######################################################################
