package Apache::NavGraphics;

##Derived from work by Lincoln Stein in his article
##"A Dynamic Navigation Bar with mod_perl" in
##The Perl Journal, Issue #12, Vol.3, No.4

use strict;
use Apache::Constants qw(:common);
use Apache::File ();
use Apache::URI  ();
use Apache::OutputChain ();
use File::Basename;
use GD ();
use Text::Template ();

use vars qw( @ISA $VERSION );
@ISA = qw(Apache::OutputChain);

my %BARS = ();
my %TB = ();

$VERSION = '0.08';

sub handler {
  my $r = shift;
  Apache::OutputChain::handler($r, __PACKAGE__);
}

sub PRINT {
  my $self = shift;
  my $r = $self->[0];
  my $bar   = load_nav($r);
  my $plate = load_plate($r);

  local $_ = join '', @_;
  my ($header, $content);
  ($header,$content) = m#<HEAD>(.+)</HEAD>.*<BODY.*?>(.+)</BODY>#si;
  my $thash = {
   'left_nav_java'    => $bar->java, 
   'header'           => $header, 
   'navigation_table' => $bar->table, 
   'content'          => $content 
  };
  if (ref($plate) ne 'Text::Template') {
    warn join '', "NavGraphics($$) BOOGER: ", $r->filename, "\n";
  }
  my $result = $plate->fill_in(HASH => $thash);
  if (defined $result) {
    $self->Apache::OutputChain::PRINT($result);
  }
  else {
    warn "ERROR: $Text::Template::ERROR\n";
  }
  return OK;
}


sub load_nav {
  my $r = shift;
  my $uri = $r->parsed_uri();
  my $mod_time;
  my $root = $r->document_root();
  my $localdir = join '' , $root, dirname($uri->path);
  chop $localdir if (substr $localdir, -1, 1 eq '/');
  my $conf_file = join '/', $localdir, 'navConf.txt';
  my $dirname = $r->dir_config('NavGraphicsDir');
  ##We always return an object, even if there's no configuration file
  ##so that the page is wrapped by the header and footer.
  if (-e $conf_file) {
    $mod_time = (stat _)[9];
  }
  else {
    $conf_file = '';
    $mod_time = 0;
  }

  return $BARS{$conf_file}
    if exists($BARS{$conf_file}) && $BARS{$conf_file}->modified >= $mod_time;
  warn "NavBar: nav not cached\n";
  $BARS{$conf_file} = Apache::NavGraphics::NavBar->new($conf_file, $dirname, $root);
  warn "NavBar nav: No object\n" unless exists $BARS{$conf_file};
  warn "NavBar nav: ",$BARS{$conf_file}->modified," >= $mod_time\n";
  return $BARS{$conf_file};

}

sub load_plate {
  my ($r) = @_;
  my $conf_file = $r->server_root_relative($r->dir_config('MyTemplate'));
  my $mod_time;
  if (-e $conf_file) {
    $mod_time = (stat _)[9];
  }
  else {
    $conf_file = '';
    $mod_time = 0;
  }

  warn "NavGraphics($$): conf_file = >$conf_file<\n";
  warn "NavGraphics($$) time=$mod_time\n";
  warn join '', "NavGraphics($$): exists ", exists $TB{$conf_file}, "\n";

  if (exists($TB{$conf_file})) {
    warn join ' ', "Nav Obj ($$):", $TB{$conf_file}->modified(), $mod_time, "\n";
    warn join ' ', "Nav keys ($$):", keys %{$TB{$conf_file}}, "\n";
  }
  return $TB{$conf_file}->template()
    if exists($TB{$conf_file})
    && ($TB{$conf_file}->modified() >= $mod_time)
    && (ref($TB{$conf_file}->template()) eq 'Text::Template');
  warn "NavGraphics($$): plate not cached\n";
  warn "NavGraphics($$): plate $TB{$conf_file}\n";
  warn "NavGraphics($$): before new $TB{$conf_file}\n";
  $TB{$conf_file} = Apache::NavGraphics::NavTemplate->new($conf_file);
  warn "NavGraphics($$): after new $TB{$conf_file}\n";
  warn join '', "NavGraphics plate($$): ", ref($TB{$conf_file}->template()), "\n";
  warn join '', "NavGraphics plate key($$): ", ref($TB{$conf_file}->{template}), "\n";
  return $TB{$conf_file}->template();

}

package Apache::NavGraphics::NavBar;

my @size = (179,19);
my @size2 = (161,19);
my @left_nav = map { hex $_ } qw(CC CC 99);
my @text_on  = map { hex $_ } qw(FF FF CC);
my @mouse_on = map { hex $_ } qw(99 99 66);
my @text_off = map { hex $_ } qw(66 66 33);

sub new {                        # create a new NavBar object
  my ($class, $conf_file, $dirname, $root) = @_;
  my (@c, %c, %d);
  unless ($conf_file) {
    return bless { 'labels'   => [],
		   'url'      => {},
		   'table'    => '',
		   'java'     => '',
		   'modified' => 0 }, $class;
  }
  my $fh = Apache::File->new($conf_file) || return;
  local $/ = "\n";
  while (<$fh>) {
    tr/\r//d;
    chomp;
    my (undef, $label, $url) = split /\t/;
    push @c, $label;
    $c{$label} = $url;
  }
  my $self =  bless { 'labels'   => \@c,
		      'url'      => \%c,
		      'table'    => '',
		      'java'     => '',
		      'modified' => (stat $conf_file)[9] }, $class;
  $self->_make_nav($dirname,$root);
  return $self;
}

#return ordered list of all the URLs in the navigation bar
sub url { return $_[0]->{'url'}->{$_[1]} }

#return the label for a particular URL in the bar
sub labels { return @{shift->{'labels'}}; }

#return the modification date of the configuration file
sub modified { return $_[0]->{'modified'}; }

sub table {
  my $self = shift;
  if (@_) {
    $self->{'table'} = shift; 
  }
  else {
    return $self->{'table'};
  }
}

sub java {
  my $self = shift;
  if (@_) {
    $self->{'java'} = shift; 
  }
  else {
    return $self->{'java'};
  }
}

sub _make_nav {
  my ($bar,$dirname,$root) = @_;            #Create the navigation bar
  my @java;
  my @table;
  my @bounds;
  my ($java,$table) = ('','');
  my $num = 1;
  my ($fileOn, $fileOff);
  my $myLabel;
  foreach my $label ($bar->labels) {
    ($fileOff = $label) =~ tr/a-zA-Z0-9_/_/cs;
    $fileOn  = join '_', 'lnav', $fileOff, 'On.png';
    $fileOff = join '_', 'lnav', $fileOff, 'Off.png';
    $fileOff = join '/', $dirname, $fileOff;
    $fileOn  = join '/', $dirname, $fileOn;
    push @java, <<EOJAVA;
  nav${num}On = new Image();
  nav${num}On.src = "$fileOn";

  nav${num}Off = new Image();
  nav${num}Off.src = "$fileOff";

EOJAVA
    my $url   = $bar->url($label);
    push @table, <<EOTABLE;
<tr>
<td background="/images/left_nav_bar_background.png" width="179" height="27" valign="center"><a href="$url" onMouseOver="changeImages('nav$num', 'nav${num}On')"
onMouseOut="changeImages('nav$num', 'nav${num}Off')"><img name="nav$num" src="$fileOff" alt="$label" border="0" width=179 height=19></img></a></td>
</tr>

EOTABLE

    my $imageOff = new GD::Image(@size);

    my $white     = $imageOff->colorAllocate(255,255,255);
    my $left_nav  = $imageOff->colorAllocate(@left_nav);
    my $text_off  = $imageOff->colorAllocate(@text_off);

    $imageOff->filledRectangle(0,0,@size,$white);
    $imageOff->filledRectangle(0,0,@size2,$left_nav);
    ##Allowed width 153
    @bounds = GD::Image->stringTTF($text_off,
                         '/usr/X11R6/lib/X11/fonts/webfonts/arialb.ttf',
		         8,0,8,13,$label);
    if ($bounds[2] - $bounds[0] > 153) {
      $myLabel = "Too long"; 
    }
    else {
      $myLabel = $label;
    }
    $imageOff->stringTTF($text_off,
                         '/usr/X11R6/lib/X11/fonts/webfonts/arialb.ttf',
		         8,0,8,13,$myLabel);

    my $imageOn = new GD::Image(@size);

    $white        = $imageOn->colorAllocate(255,255,255);
    my $text_on   = $imageOn->colorAllocate(@text_on);
    my $mouse_on  = $imageOn->colorAllocate(@mouse_on);

    $imageOn->filledRectangle(0,0,@size,$white);
    $imageOn->filledRectangle(0,0,@size2,$mouse_on);
    $imageOn->stringTTF($text_on,
                        '/usr/X11R6/lib/X11/fonts/webfonts/arialb.ttf',
		        8,0,8,13,$myLabel);

    $fileOff = join '/', $root, $fileOff;
    $fileOn  = join '/', $root, $fileOn;
    my $onFH = Apache::File->new(">$fileOn") or
      warn "Problem with opening $fileOn: $!\n";
    print $onFH $imageOn->png;
    $onFH->close;
    my $offFH = Apache::File->new(">$fileOff") or
      warn "Problem with opening $fileOff: $!\n";
    print $offFH $imageOff->png;
    $offFH->close;

    ++$num;
  }
  $bar->table(join "", @table);
  $bar->java(join "", @java);

}

package Apache::NavGraphics::NavTemplate;

sub new {
  my ($class, $templateFile) = @_;
  warn "NavPlate($$): new <$templateFile>\n";
  unless ($templateFile) {
    warn "NavPlate($$) Empty templateFile sent\n";
    return bless { 'template' => '',
		   'modified' => 0 }, $class;
  }
  my $fh = Apache::File->new($templateFile) || return;
  warn "NavPlate($$): after fh\n";
  my $template = Text::Template->new(
				   TYPE   => 'FILEHANDLE',
				   SOURCE => $fh,
				   DELIMITERS => ['{{', '}}']
                                 )
    or warn "($$)Problem with template: $Text::Template::ERROR\n";
  warn join '', "NavPlate($$) ref: ", ref($template), "\n";
  return bless({ 'template' => $template,
		 'modified' => (stat $templateFile)[9] }, $class);

}

sub template { return $_[0]->{'template'}; }
sub modified { return $_[0]->{'modified'}; }

1;
