/* vim:set ts=2 sw=2 et cindent: */
/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is Mozilla.
 *
 * The Initial Developer of the Original Code is IBM Corporation.
 * Portions created by IBM Corporation are Copyright (C) 2004
 * IBM Corporation.  All Rights Reserved.
 *
 * Contributor(s):
 *   Darin Fisher <darin@meer.net>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

#include <stdlib.h>
#include <stdio.h>
#include <iconv.h>
#include "nsEmbedString.h"

#ifdef IS_LITTLE_ENDIAN
#define ASTRING_CHARSET "UTF-16LE"
#else
#define ASTRING_CHARSET "UTF-16BE"
#endif

/**
 * This function uses iconv to convert the input wchar_t array to the charset
 * used by nsAString.  It writes the resulting string to the output nsAString.
 */
PRBool WCSToAString(const wchar_t *input, size_t inputLen, nsAString &output)
{
  iconv_t cd = iconv_open(ASTRING_CHARSET, "WCHAR_T");
  if (cd == (iconv_t) -1)
  {
    perror("iconv_open");
    return PR_FALSE;
  }

  // Start with an empty buffer
  NS_StringSetData(output, NULL, 0);
   
  char *inbuf = (char *) input;
  size_t inbytesleft = inputLen * sizeof(wchar_t);

  char outbuf[512], *outp;

  PRBool rv = PR_TRUE;
  do
  {
    size_t outbytesleft = sizeof(outbuf);
    outp = outbuf;

    if (iconv(cd, &inbuf, &inbytesleft, &outp, &outbytesleft) == (size_t) -1)
    {
      rv = PR_FALSE;
      break;
    }
    // Append converted data to the output buffer
    PRUint32 len = (outp - outbuf) / sizeof(PRUnichar);
    NS_StringAppendData(output, (const PRUnichar *) outbuf, len);
  }
  while (inbytesleft != 0);

  iconv_close(cd);
  return rv;
}

/**
 * This function uses iconv to convert the input nsAString to an array of
 * wchar_t.  It returns a malloc'd result in output, with length given by
 * outputLen.
 */
PRBool AStringToWCS(const nsAString &input, wchar_t **output, size_t *outputLen)
{
  iconv_t cd = iconv_open("WCHAR_T", ASTRING_CHARSET);
  if (cd == (iconv_t) -1)
  {
    perror("iconv_open");
    return PR_FALSE;
  }

  const PRUnichar *inData;
  PRUint32 inLen = NS_StringGetData(input, &inData);

  PRBool rv = PR_FALSE;

  // Allocate result buffer (assume worse case)
  size_t outbytesleft = sizeof(wchar_t) * inLen;
  *output = (wchar_t *) malloc(outbytesleft + sizeof(wchar_t));
  char *outbuf = (char *) *output;
  if (outbuf)
  {
    char *inbuf = (char *) inData;
    size_t inbytesleft = inLen * sizeof(PRUnichar);

    if (iconv(cd, &inbuf, &inbytesleft, &outbuf, &outbytesleft) != (size_t) -1)
    {
      // null-terminate result
      ((wchar_t *) outbuf)[0] = wchar_t(0);

      rv = PR_TRUE;
    }
  }

  iconv_close(cd);
  return rv;
}

int main(int argc, char **argv)
{
  printf("\n");

  
  // start by converting a multibyte string to a wchar_t array...

  wchar_t buf[512];
  size_t len;

  len = mbstowcs(buf, argv[0], sizeof(buf)/sizeof(buf[0]));

  printf("as wchar_t: %S\n", buf);


  // now convert from wchar_t array to nsAString

  nsEmbedString s;
  if (!WCSToAString(buf, len, s))
    return -1;

  const PRUnichar *data = s.get();
  while (*data)
    printf("%hx ", *data++);
  printf("\n");


  // now convert from nsAString to wchar_t array

  wchar_t *result;
  size_t resultLen;
  if (!AStringToWCS(s, &result, &resultLen))
    return -1;

  printf("as wchar_t: %S\n", result);
  free(result);


  printf("\n");
  return 0;
}
