package samples.businessnew;

/*
 * Copyright  1998 - 2000 IBM Corporation.
 *
 * Redistribution and use in source (source code) and binary (object code)
 * forms, with or without modification, are permitted provided that the
 * following conditions are met:
 * 1. Redistributed source code must retain the above copyright notice, this
 * list of conditions and the disclaimer below.
 * 2. Redistributed object code must reproduce the above copyright notice,
 * this list of conditions and the disclaimer below in the documentation
 * and/or other materials provided with the distribution.
 * 3. The name of IBM may not be used to endorse or promote products derived
 * from this software or in any other form without specific prior written
 * permission from IBM.
 * 4. Redistribution of any modified code must be labeled "Code derived from
 * the original OpenCard Framework".
 *
 * THIS SOFTWARE IS PROVIDED BY IBM "AS IS" FREE OF CHARGE. IBM SHALL NOT BE
 * LIABLE FOR INFRINGEMENTS OF THIRD PARTIES RIGHTS BASED ON THIS SOFTWARE.  ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IBM DOES NOT WARRANT THAT THE FUNCTIONS CONTAINED IN THIS
 * SOFTWARE WILL MEET THE USER'S REQUIREMENTS OR THAT THE OPERATION OF IT WILL
 * BE UNINTERRUPTED OR ERROR-FREE.  IN NO EVENT, UNLESS REQUIRED BY APPLICABLE
 * LAW, SHALL IBM BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.  ALSO, IBM IS UNDER NO OBLIGATION
 * TO MAINTAIN, CORRECT, UPDATE, CHANGE, MODIFY, OR OTHERWISE SUPPORT THIS
 * SOFTWARE.
 */

import javacard.framework.APDU;
import javacard.framework.ISO;
import javacard.framework.ISOException;
import javacard.framework.Applet;
import javacard.framework.OwnerPIN;
import javacard.framework.Util;

/**
 * <tt>BizCardApplet</tt> is a demo Java Card applet for
 * the IBM Java Card.
 *
 * It can store a certain number of business card
 * information entries containing the fields title,
 * name, E-mail address, phone and address.
 * The information can be accessed only after a successful
 * card holder verification. As this is a Demo-Applet,
 * the pin is "guess".
 *
 * Note: This applet works with the mask 2 of the IBM Java
 * Card, as available in April 1999.  There are slight
 * differences to the Java Card 2.1 Spec released in
 * in April 1999.  For example is the name of the class
 * containing ISO constants "ISO" versus "ISO7816" (spec).
 *
 * @author  Thomas Schaeck (schaeck@de.ibm.com)
 * @author  Frank Seliger  (seliger@de.ibm.com)
 */
public class BizCardApplet
					   extends javacard.framework.Applet {

  final static byte PIN_TRY_LIMIT = (byte) 10;
  final static byte MAX_PIN_SIZE  = (byte)  8;

  /** A BizCard has two business info entries. */
  private BizCardRecord bizCardRecord0_;
  private BizCardRecord bizCardRecord1_;

  /** The OwnerPin must be matched before this applet
	  cooperates */
  private OwnerPIN pin_
		   = new OwnerPIN(PIN_TRY_LIMIT, MAX_PIN_SIZE);

/**
 * Create a new BizCardApplet.
 */
protected BizCardApplet() {
  super();
  bizCardRecord0_ = new BizCardRecord();
  bizCardRecord1_ = new BizCardRecord();

  /** Set the pin to "guess" */
  byte[] pin = {(byte)'g', (byte)'u', (byte)'e',
	 (byte)'s', (byte)'s', (byte) 0, (byte) 0, (byte) 0 };
  pin_.update(pin, (short) 0, (byte)pin.length);	
}
/**
 * Actions to be performed it this applet is deselected.
 */
public void deselect() {  // Switching the applet
  pin_.reset();           // .. resets PIN-validated flag
}
/**
 * Gets a field as specified in the given APDU.
 *
 * @param apdu The APDU to be processed.
 */
private void getField(APDU apdu) {
  byte[] buffer = apdu.getBuffer();

  // Check class byte. Only 0x80 is allowed here.
  if (buffer[ISO.OFFSET_CLA] != (byte) 0x80)
	ISOException.throwIt(ISO.SW_CONDITIONS_NOT_SATISFIED);

  // Check P2. This is the field ID of the field
  // to be read. It must be between 0 and 4.
  if ((buffer[ISO.OFFSET_P2]) > 4)
	ISOException.throwIt(ISO.SW_INCORRECT_P1P2);

  // Check if the CHV has already been verified.
  if (pin_.isValidated() != true)
	ISOException.throwIt(ISO.SW_SECURITY_STATUS_NOT_SATISFIED);

  // Retrieve the information to send outgoing
  BizCardRecord bcRecord = null;

  switch (buffer[ISO.OFFSET_P1]) {
	case 0x00 : bcRecord = bizCardRecord0_; break;
	case 0x01 : bcRecord = bizCardRecord1_; break;
	default : // Index is out of bounds
	  ISOException.throwIt(ISO.SW_INCORRECT_P1P2);
  }
  BizCardField bcField =
	bcRecord.getField(buffer[ISO.OFFSET_P2]);

  // Le currently 	
  short le = apdu.setOutgoing();
  apdu.setOutgoingLength(bcField.length);

  // Send the response.
  apdu.sendBytesLong(
	bcField.data, (short) 0, bcField.length);
}
/**
 * Install the applet on the Java Card.
 *
 * @param apdu Currently ignored.
 *
 * Note: In Java Card 2.1 the function signature has
 * three parameters instead of this single parameter.
 */
public static void install(APDU apdu)
						   throws ISOException {
  BizCardApplet me = new BizCardApplet();
  me.register();
}
/**
 * Perform card holder verification as specified
 * in the given APDU.
 *
 * @param apdu The APDU to be processed.
 */
private void performCHV(APDU apdu) {
  byte[] buffer = apdu.getBuffer();
  byte lc = buffer[4];

  // Check class byte. Only 0x80 is allowed here.
  if (buffer[ISO.OFFSET_CLA] != (byte) 0x80)
	ISOException.throwIt(ISO.SW_CONDITIONS_NOT_SATISFIED);

  // Check P1: Only 0x01 for CHV 1 is allowed here.
  if (buffer[ISO.OFFSET_P1] != (byte) 0x01)
	ISOException.throwIt(ISO.SW_INCORRECT_P1P2);

  // Check P2. Only 0x00 is allowed here.
  if (buffer[ISO.OFFSET_P2] != (byte) 0x00)
	ISOException.throwIt(ISO.SW_INCORRECT_P1P2);

  // Check the pin.	
  if (pin_.check(buffer, ISO.OFFSET_CDATA, lc) == false)
	ISOException.throwIt(
	   ISO.SW_SECURITY_STATUS_NOT_SATISFIED);

  // Send the response: No data, return code 0x9000
  apdu.setOutgoingAndSend((short) 0, (short) 0);
}
/**
 * Processes incoming APDUs.
 * When a Java Card Applet receives an APDU,
 * it calls this method to process it.
 *
 * @param apdu The APDU to be processed.
 */
public void process(APDU apdu) throws ISOException {
  byte[] apduBuffer = apdu.getBuffer();

  // Class byte must be 0x80 or 0x00 here.
  if (   (apduBuffer[ISO.OFFSET_CLA] != (byte) 0x00)
	  && (apduBuffer[ISO.OFFSET_CLA] != (byte) 0x80) )
  	ISOException.throwIt(ISO.SW_CLA_NOT_SUPPORTED);

  // Dispatch commands depending on
  // .. class and instruction bytes.
  switch (Util.makeShort(
	(byte)(apduBuffer[ISO.OFFSET_CLA] & (byte) 0xF0),
	apduBuffer[ISO.OFFSET_INS])) {

	case (short) 0x8001 :
	  performCHV(apdu);
	  break;
	case (short) 0x8002 :
	  getField(apdu);
	  break;
	case (short) 0x8003 :
	  setField(apdu);
	  break;
	case (short) 0x00A4 :
	  selectFile(apdu);
	  break;
	default :
	  ISOException.throwIt(ISO.SW_INS_NOT_SUPPORTED);
	  break;
  }
}
/**
 * Return the select response when the applet is selected.
 *
 * @param apdu The select APDU that was sent to the card.
 */
private void selectFile(APDU apdu) {
  byte[] buffer = apdu.getBuffer();
  short lc = (short) ((short) 0x00FF & buffer[4]);
  short offset = ISO.OFFSET_CDATA;

  // Check class byte. Only 0x00 is allowed here.
  if (buffer[ISO.OFFSET_CLA] != (byte) 0x00)
	ISOException.throwIt(ISO.SW_CONDITIONS_NOT_SATISFIED);

  // Check P1. Only SELECT by name is supported.
  if (buffer[ISO.OFFSET_P1] != (byte) 0x04)
	ISOException.throwIt(ISO.SW_INCORRECT_P1P2);

  // Check P2. Only 0x00 (return FCI template) is allowed here.
  if (buffer[ISO.OFFSET_P2] != (byte) 0x00)
	ISOException.throwIt(ISO.SW_INCORRECT_P1P2);

  // Check AID length.
  if ((lc < ISO.OFFSET_CDATA) || (lc > (short) 0x10))
	ISOException.throwIt(ISO.SW_WRONG_LENGTH);

  // Compare AIDs.	
  if (!(javacard.framework.System.getAID()
		.equals(buffer, offset, buffer[ISO.OFFSET_LC])))
	ISOException.throwIt(ISO.SW_WRONG_DATA);

  // Calculate length to make SELECT processing general
  short proLen
	= (short) 0x00; // 13 bytes proprietary data
  short start
	= (offset -= 4); // to use aid in APDU buffer

  // -> command ok, now prepare response
  buffer[offset++] = (byte) 0x6F;
  buffer[offset++] = (byte) ((byte) 0xFF
							& (4 + proLen + lc));
  buffer[offset++] = (byte) 0x84;
  buffer[offset++] = (byte) ((byte) 0xFF & (lc));
  offset += lc;
  buffer[offset++] = (byte) 0xA5;
  buffer[offset++] = (byte) ((byte) 0xFF & (proLen));

  // Send the response.
  apdu.setOutgoingAndSend(start, (short) (start + 1
				  + (short) ((byte) 0xFF & buffer[2])));
}
/**
 * Sets a field as specified in the given APDU.
 *
 * @param apdu The SetField Command APDU to be processed.
 */
private void setField(APDU apdu) {
  byte[] buffer = apdu.getBuffer();
  short lc = (short) ((short) 0x00FF & buffer[4]);

  // Check class byte. Only 0x80 is allowed here.
  if (buffer[ISO.OFFSET_CLA] != (byte) 0x80)
	ISOException.throwIt(ISO.SW_CONDITIONS_NOT_SATISFIED);

  // Check P2. This is the field to be read:
  //  1 <= field <= 5
  if ((buffer[ISO.OFFSET_P2]) > 4)
	ISOException.throwIt(ISO.SW_INCORRECT_P1P2);

  // Check if the CHV has already been verified.
  if (pin_.isValidated() != true)
	ISOException.throwIt(ISO.SW_SECURITY_STATUS_NOT_SATISFIED);

  // Determine which information to update
  BizCardRecord bcRecord = null;

  switch (buffer[ISO.OFFSET_P1]) {
	case 0x00 : bcRecord = bizCardRecord0_; break;
	case 0x01 : bcRecord = bizCardRecord1_; break;
	default : // Index is out of bounds
	  ISOException.throwIt(ISO.SW_INCORRECT_P1P2);
  }

  // Set the field to the value transmitted in the APDU.
  bcRecord.setField(buffer[ISO.OFFSET_P2],
					buffer, ISO.OFFSET_CDATA, lc);
  	
  // Send the response: No data, status word 0x9000.
  apdu.setOutgoingAndSend((short) 0, (short) 0);
}
}
