package samples.businessnew;

/*
 * Copyright  1998 - 2000 IBM Corporation.
 *
 * Redistribution and use in source (source code) and binary (object code)
 * forms, with or without modification, are permitted provided that the
 * following conditions are met:
 * 1. Redistributed source code must retain the above copyright notice, this
 * list of conditions and the disclaimer below.
 * 2. Redistributed object code must reproduce the above copyright notice,
 * this list of conditions and the disclaimer below in the documentation
 * and/or other materials provided with the distribution.
 * 3. The name of IBM may not be used to endorse or promote products derived
 * from this software or in any other form without specific prior written
 * permission from IBM.
 * 4. Redistribution of any modified code must be labeled "Code derived from
 * the original OpenCard Framework".
 *
 * THIS SOFTWARE IS PROVIDED BY IBM "AS IS" FREE OF CHARGE. IBM SHALL NOT BE
 * LIABLE FOR INFRINGEMENTS OF THIRD PARTIES RIGHTS BASED ON THIS SOFTWARE.  ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IBM DOES NOT WARRANT THAT THE FUNCTIONS CONTAINED IN THIS
 * SOFTWARE WILL MEET THE USER'S REQUIREMENTS OR THAT THE OPERATION OF IT WILL
 * BE UNINTERRUPTED OR ERROR-FREE.  IN NO EVENT, UNLESS REQUIRED BY APPLICABLE
 * LAW, SHALL IBM BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.  ALSO, IBM IS UNDER NO OBLIGATION
 * TO MAINTAIN, CORRECT, UPDATE, CHANGE, MODIFY, OR OTHERWISE SUPPORT THIS
 * SOFTWARE.
 */

import opencard.core.event.CTListener;
import opencard.core.event.CardTerminalEvent;
import opencard.core.service.SmartCard;
import opencard.core.service.CardService;
import opencard.core.terminal.CardTerminalRegistry;
import opencard.core.terminal.CardTerminalException;

/**
 * BusinessCard represents a JavaCard with a BusinessCardApplet to applications. 
 * A BusinessCard object offers methods to get and set BusinessInfo entries on the card.
 *
 * @author  Thomas Schaeck (schaeck@de.ibm.com)
 * @version $Id: BusinessCard.java,v 1.1 1998/09/02 16:12:08 cvsusers Exp $
 */
public class BusinessCard implements CTListener
{
  /** The object representing the Smart Card and OCF. */
  SmartCard card_ = null;

  /** The Applet Proxy for the BusinessCardApplet. */
  BusinessCardProxy businessCardProxy_ = null;
/**
 * Creates a new BusinessCard object.
 */
BusinessCard() throws BusinessCardException {
	try {
		// Start the OpenCard Framework.
		SmartCard.start();

		// Register the new SignatureCard as a Card Terminal Event Listener
		CardTerminalRegistry.getRegistry().addCTListener(this);

		// Let the registry create events for cards which are already present
		CardTerminalRegistry.getRegistry().createEventsForPresentCards(this);
	} catch (Exception e) {
		// Do exception handling
		e.printStackTrace();
	}
}
/**
 * React on card insertion events sent by OCF: Get new card and card service
 *
 * @param ctEvent The card insertion event.
 */
public void cardInserted(CardTerminalEvent ctEvent) {
	try {
		// Get a SmartCard object
		card_ = SmartCard.getSmartCard(ctEvent);

		// Get a Proxy instance for the BusinessCardProxy.
		businessCardProxy_ = (BusinessCardProxy) card_.getCardService(BusinessCardProxy.class, true);
	} catch (Exception e) {
		// Exception handling to be inserted here.
		e.printStackTrace();
	}
}
/**
 * React on card removal events sent by OCF: Invalidate card and card service.
 *
 * @param ctEvent The card removal event.
 */
public synchronized void cardRemoved(CardTerminalEvent ctEvent) {
	card_ = null;
	businessCardProxy_ = null;
}
/**
 * Close a <tt>BusinessCard</tt> object.
 */
public void close() throws CardTerminalException {
	// Shut down the OpenCard Framework.
	SmartCard.shutdown();
}
/**
 * Gets a <tt>BusinessInfo</tt> object for the entry with the given index.
 *
 * @param index The index of the entry for which a <tt>BusinessInfo</tt>
 *              object shall be obtained.
 *
 * @exception samples.business.BusinessCardException
 *            Thrown when problems occurr while reading the info from card.
 */
public BusinessInfo getBusinessInfo(int index) throws BusinessCardException {
	if (card_ == null)
		throw new BusinessCardException("No card found.");
	try {
		// We need mutual exclusion here to prevent other Card Services from
		// modifying data on the card while we read the information fields.
		card_.beginMutex();

		// Read the fields using an Card Applet Proxy.
		return new BusinessInfo(businessCardProxy_.getName(index), 
								businessCardProxy_.getTitle(index), 
								businessCardProxy_.getEmail(index), 
								businessCardProxy_.getPhone(index), 
								businessCardProxy_.getAddress(index));
	} catch (Exception e) {
		e.printStackTrace();
		throw new BusinessCardException(e.getMessage());
	} finally {
		// End mutual exclusion. From now on other Card Services may
		// access the card again.
		card_.endMutex();
	}
}
/**
 * Sets the entry with the given index to the given <tt>BusinessInfo</tt> object. 
 *
 * @param index        The index of the entry for which a
 *                     <tt>BusinessInfo</tt> object shall be written.
 * @param businessInfo The <tt>BusinessInfo</tt> object to be written.
 *
 * @see samples.business.BusinessInfo
 *
 * @exception samples.business.BusinessCardException
 *            Thrown when problems occurr while writing the info to  card.
 */
public void setBusinessInfo(int index, BusinessInfo businessInfo) 
throws BusinessCardException {
	// Check for card presence.
	if (card_ == null)
		throw new BusinessCardException("No card found.");
	try {
		// We need mutual exclusion here to prevent other CardServices from
		// modifying data on the card while we write the information fields.
		card_.beginMutex();

		// Write the fields to the card using an AppletProxy.
		businessCardProxy_.setName(index, businessInfo.getName());
		businessCardProxy_.setTitle(index, businessInfo.getTitle());
		businessCardProxy_.setEmail(index, businessInfo.getEmail());
		businessCardProxy_.setPhone(index, businessInfo.getPhone());
		businessCardProxy_.setAddress(index, businessInfo.getAddress());
	} catch (Exception e) {
		e.printStackTrace();
		throw new BusinessCardException(e.getMessage());
	} finally {
		// End mutual exclusion. From now on other CardServices may access
		// the card again. We make this call in the finally statement to ensure
		// that it is performed under all circumstances.
		card_.endMutex();
	}
}
}
