# Copyright (c) 2005 Cisco Systems.  All rights reserved.
# 
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License as
# published by the Free Software Foundation; either version 2 of the
# License, or (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

import os, array, struct, fcntl, subnmgt, subnadm, devmgt
from os import O_RDWR
from umadioctl import REGISTER_AGENT, UNREGISTER_AGENT

ABI_VERSION = 5

QP1_QKEY = 0x80010000L

class MgmtClass:
    SUBN_LID_ROUTED		= 0x01
    SUBN_DIRECT_ROUTED		= 0x81
    SUBN_ADM			= 0x03
    PERF			= 0x04
    BM				= 0x05
    DEV_MGT			= 0x06
    COM_MGT			= 0x07

class Method:
    GET				= 0x01
    SET				= 0x02
    GET_RESP			= 0x81
    SEND			= 0x03
    TRAP			= 0x05
    REPORT			= 0x06
    REPORT_RESP			= 0x86
    TRAP_REPRESS		= 0x07

class Mad:
    "A management datagram (MAD)"
    def header(mgmt_class, class_version, method, status, class_specific,
               transaction_id, attribute_id, attribute_modifier):
        return struct.pack('!BBBBHHQHHI', 1, mgmt_class, class_version, method,
                           status, class_specific, transaction_id, attribute_id,
                           0, attribute_modifier)
    header = staticmethod(header)

    def dr_header(mgmt_class, class_version, method, status, hop_pointer,
                  hop_count, transaction_id, attribute_id, attribute_modifier,
                  mkey, dr_slid, dr_dlid):
        return (Mad.header(mgmt_class, class_version, method, status,
                           hop_pointer << 8 | hop_count,
                           transaction_id, attribute_id, attribute_modifier) +
                struct.pack('!QHH', mkey, dr_slid, dr_dlid))
    dr_header = staticmethod(dr_header)

    def __init__(self, data, status = 0, timeout_ms = 0, retries = 0,
                 length = 0, qpn = 0, qkey = 0, lid = 0, sl = 0, path_bits = 0,
                 grh_present = 0, gid_index = 0, hop_limit = 0,
                 traffic_class = 0, gid = '\x00' * 16, flow_label = 0):
        self._data          = data
        self._status        = status
        self._timeout_ms    = timeout_ms
        self._retries       = retries
        self._qpn           = qpn
        self._qkey          = qkey
        self._lid           = lid
        self._sl            = sl
        self._path_bits     = path_bits
        self._grh_present   = grh_present
        self._gid_index     = gid_index
        self._hop_limit     = hop_limit
        self._traffic_class = traffic_class
        self._gid           = gid
        self._flow_label    = flow_label

class UmadAgent:
    "A userspace MAD agent"
    def __init__(self, file, id):
        self._file = file
        self._id   = id

    def __del__(self):
        del self._file._agents[self._id]
        buf = struct.pack('@I', self._id)
        fcntl.ioctl(self._file._fd, UNREGISTER_AGENT, buf)

    def send(self, mad):
        os.write(self._file._fd, struct.pack('@I', self._id) +
                 struct.pack('!IIIIIIHBBBBBB16sI', 0, mad._timeout_ms, mad._retries, 0,
                             mad._qpn, mad._qkey, mad._lid, mad._sl, mad._path_bits,
                             mad._grh_present, mad._gid_index, mad._hop_limit,
                             mad._traffic_class, mad._gid, mad._flow_label) +
                 mad._data)

class RecvMad:
    "A MAD received by userspace"
    def __init__(self, agent, mad):
        self._agent = agent
        self._mad   = mad

class UmadFile:
    "A userspace MAD file descriptor"
    def __init__(self, dev = '/dev/infiniband/umad0'):
        self._fd     = os.open(dev, O_RDWR)
        self._agents = { }

    def __del__(self):
        os.close(self._fd)
        
    def reg_agent(self, qpn, method_mask = '\x00' * 16, mgmt_class = 0,
                  mgmt_class_version = 0, oui = '\x00' * 3, rmpp_version = 0):
        buf = array.array('B',
                          struct.pack('@I16sBBB3sB', 0, method_mask, qpn, mgmt_class,
                                      mgmt_class_version, oui, rmpp_version))
        if fcntl.ioctl(self._fd, REGISTER_AGENT, buf, 1):
            raise IOError
        id, = struct.unpack('@I23x', buf)

        agt = UmadAgent(self, id)
        self._agents[id] = agt
        return agt

    def recv(self, max_data = 256):
        buf = os.read(self._fd, max_data + 56)
        if len(buf) < 56:
            raise IOError
        agt_id, = struct.unpack('@I', buf[:4])
        hdr = struct.unpack('!IIIIIIHBBBBBB16sI', buf[4:56])
        return RecvMad(self._agents[agt_id],
                       Mad(buf[56:], status = hdr[0], timeout_ms = hdr[1], retries = hdr[2],
                           length = hdr[3], qpn = hdr[4], qkey = hdr[5], lid = hdr[6],
                           sl = hdr[7], path_bits = hdr[8], grh_present = hdr[9],
                           gid_index = hdr[10], hop_limit = hdr[11], traffic_class = hdr[12],
                           gid = hdr[13], flow_label = hdr[14]))
