
OSG_BEGIN_NAMESPACE

class OSG_SYSTEM_DLLMAPPING ContentResolver : public MemoryObject
{
  public:
    /*---------------------------------------------------------------------*/
    /*! \name Types                                                        */
    /*! \{                                                                 */
    
    typedef MemoryObject    Inherited;
    typedef ContentResolver Self;
    
    typedef RefCountPtr<Self, MemObjRefCountPolicy> ObjRefPtr;
    typedef TransitPtr <Self                      > ObjTransitPtr;
    
    typedef std::map<std::string, ContentProviderRefPtr> MountMap;
    typedef std::map<std::string, FiletypeHandlerRefPtr> TypeMap;
    
    /*! \}                                                                 */
    /*---------------------------------------------------------------------*/
    /*! \name Create                                                       */
    /*! \{                                                                 */
    
    static  ObjTransitPtr create(void);
    
    virtual ObjTransitPtr clone (void);
    
    /*! \}                                                                 */
    /*---------------------------------------------------------------------*/
    /*! \name Mounts                                                       */
    /*! \{                                                                 */
    
    static bool addGlobalMount(      ContentProvider *provider,
                               const std::string     &mountPoint);
    static bool subGlobalMount(const std::string     &mountPoint);
    
           bool addMount      (      ContentProvider *provider,
                               const std::string     &mountPoint);
           bool subMount      (const std::string     &mountPoint);
    
    /*! \}                                                                 */
    /*---------------------------------------------------------------------*/
    /*! \name Filetypes                                                    */
    /*! \{                                                                 */
    
    static bool addGlobalFiletype(      FiletypeHandler *handler,
                                  const std::string     &fileType);
    static bool subGlobalFiletype(const std::string     &fileType);
    
           bool addFiletype      (      FiletypeHandler *handler,
                                  const std::string     &fileType);
           bool subFiletype      (const std::string     &fileType);
    
    /*! \}                                                                 */
    /*---------------------------------------------------------------------*/
    /*! \name Filetypes                                                    */
    /*! \{                                                                 */
    
    FieldContainerTransitPtr read (const std::string    &file,
                                   const std::string    &fileType = std::string());
    bool                     write(      FieldContainer *obj,
                                   const std::string    &file,
                                   const std::string    &fileType = std::string());
    
    FileHandleTransitPtr      createHandle     (
        const std::string &file                     );
    FiletypeHandlerTransitPtr createTypeHandler(
              FileHandle  *fh,
        const std::string &fileTyp = std::string()  );

    /*! \}                                                                 */
    /*---------------------------------------------------------------------*/
  protected:
    static  ContentResolver *_prototype;
    
    static  MountMap         _globalMounts;
    static  TypeMap          _globalTypes;
    
            ContentCache    *_cache;
    
            MountMap         _mounts;
            TypeMap          _types;
};

typedef ContentResolver::ObjRefPtr      ContentResolverRefPtr;
typedef ContentResolver::ObjTransitPtr  ContentResolverTransitPtr;


FieldContainerTransitPtr ContentResolver::read(
    const std::string &file, const std::string &fileType)
{
    FieldContainerTransitPtr returnValue(NULL    );
    std::string              typeStr    (fileType);
    
    // check cache
    if(_cache != NULL && _cache->hasResource(file) == true)
        returnValue = _cache->getResource(file);

    
    if(returnValue == NULL)
    {
        if(typeStr.empty() == true)
            typeStr = getExtension(file);
        
        FileHandleRefPtr        fh = createHandle     (file   );
        FiletypeHandlerRefPtr   ft = createTypeHandler(typeStr);
        
        returnValue = ft->read(fh);
        
        // cache resource - we know it is not cached already
        if(_cache != NULL && returnValue != NULL)
            _cache->addResource(file, returnValue);
    }

    return returnValue;
}

bool ContentResolver::write(
    FieldContainer *obj, const std::string &file, const std::string &fileType)
{
    bool        returnValue(false   );
    std::string typeStr    (fileType);
    
    if(typeStr.empty() == true)
        typeStr = getExtension(file);
    
    FileHandleRefPtr      fh = createHandle     (file   );
    FiletypeHandlerRefPtr ft = createTypeHandler(typeStr);
    
    returnValue = ft->write(obj, fh);
    
    return returnValue;
}



FileHandleTransitPtr ContentResolver::createHandle(const std::string &file)
{
    FileHandleTransitPtr returnValue(NULL);
    
    MountMap::const_iterator mmIt  = _mounts.begin();
    MountMap::const_iterator mmEnd = _mounts.end  ();
    
    // check instance mounts
    for(; mmIt != mmEnd; ++mmIt)
    {
        if(ba::starts_with(file, mmIt->first) == false)
            continue;
        
        if(mmIt->second->exists(file) == true)
        {
            returnValue = mmIt->second->createHandle(file);
            break;
        }
    }
    
    // check global mounts
    if(returnValue == NULL)
    {
        mmIt  = _globalMounts.begin();
        mmEnd = _globalMounts.end  ();
        
        for(; mmIt != mmEnd; ++mmIt)
        {
            if(ba::starts_with(file, mmIt->first) == false)
                continue;
            
            if(mmIt->second->exists(file) == true)
            {
                returnValue = mmIt->second->createHandle(file);
                break;
            }
        }
    }
    
    return returnValue;
}

FiletypeHandlerTransitPtr ContentResolver::createTypeHandler(
    FileHandle *fh, const std::string &fileType)
{
    FiletypeHandlerTransitPtr returnValue(NULL);
    
    TypeMap::const_iterator tmIt  = _types.begin();
    TypeMap::const_iterator tmEnd = _types.end  ();
    
    // check instance type handlers
    for(; tmIt != tmEnd; ++tmIt)
    {
        if(tmIt->second->handlesType(fileType) == true)
        {
            returnValue = tmIt->second->createHandler(fh);
            break;
        }
    }
    
    // check global type handlers
    if(returnValue == NULL)
    {
        tmIt  = _globalTypes.begin();
        tmEnd = _globalTypes.end  ();
        
        for(; tmIt != tmEnd; ++tmIt)
        {
            if(tmIt->second->handlesType(fileType) == true)
            {
                returnValue = tmIt->second->createHandler(fh);
                break;
            }
        }
    }
    
    return returnValue;
}





bool ContentResolver::doAddMount(
    MountMap &mounts, ContentProvider *provider, const std::string &mountPoint)
{
    bool                returnValue = false;
    MountMap::iterator  mmIt        = mounts.find(mountPoint);
    
    if(mmIt == mounts.end())
    {
        mounts.insert(MountMap::value_type(mountPoint, provider));
        returnValue = true;
    }
    else
    {
        SWARNING << "Can not mount here [" << mountPoint
                 << "], mount point already in use."
                 << std::endl;
    }
    
    return returnValue;
}

bool ContentResolver::doSubMount(
    MountMap &mounts, const std::string &mountPoint)
{
    bool               returnValue = false;
    MountMap::iterator mmIt        = mounts.find(mountPoint);
    
    if(mmIt != mounts.end())
    {
        mounts.erase(mmIt);
        returnValue = true;
    }
    else
    {
        SWARNING << "Can not unmount [" << mountPoint
                 << "], not a mount point."
                 << std::endl;
    }
    
    return returnValue;
}


bool ContentResolver::doAddFiletype(
    TypeMap &types, FiletypeHandler *handler, const std::string &fileType)
{
    bool              returnValue = false;
    TypeMap::iterator tmIt        = types.find(fileType);
    
    if(tmIt == types.end())
    {
        types.insert(TypesMap::value_type(fileType, handler));
        returnValue = true;
    }
    else
    {
        // do override handling ??
    }

    return returnValue;
}

OSG_END_NAMESPACE
