/*!
 * \class OSGNSPlugin
 * \brief
 * \author Andreas Zieringer
 */

// Qt stuff
#include <qapplication.h>
#include <qpainter.h>
#include <qmessagebox.h>
#include <qlabel.h>
#include <qlayout.h>
#include <qvbox.h>
#include <qpushbutton.h>
#include <qstringlist.h>
#include <qdragobject.h>
#include <qdir.h>
#include <qcursor.h>

#include <OSGConfig.h>
#include <OSGSceneFileHandler.h>
#include <OSGSimpleGeometry.h>
#include <OSGGradientBackground.h>

#include "OSGNSPlugin.h"

OSG_USING_NAMESPACE

// *************************
// Qt OpenSG render widget.
// *************************

OpenSGWidget::OpenSGWidget( QGLFormat f, QWidget *parent, const char *name )
     : QGLWidget( f, parent, name ),
    _mgr(NULL),
    _pwin(NullFC)
{
    setAcceptDrops(true);
    setAutoBufferSwap(false);
    
    _mgr = new SimpleSceneManager;
    _pwin = PassiveWindow::create();
    
    _mgr->setWindow(_pwin);
}

SimpleSceneManager *OpenSGWidget::getManager(void)
{
    return _mgr;
}

void OpenSGWidget::initializeGL()
{
    _pwin->init();
}

void OpenSGWidget::resizeGL( int width, int height )
{
    _mgr->resize(width, height);
}

void OpenSGWidget::paintGL()
{
    _mgr->redraw();
    swapBuffers();
}

void OpenSGWidget::mousePressEvent( QMouseEvent *ev )
{
    UInt32 button;
    
    switch ( ev->button() ) 
    {
        case LeftButton:  button = SimpleSceneManager::MouseLeft;   break;
        case MidButton:   button = SimpleSceneManager::MouseMiddle; break;
        case RightButton: button = SimpleSceneManager::MouseRight;  break;
        default:          return;
    }
    _mgr->mouseButtonPress(button, ev->x(), ev->y());
    update();
}

void OpenSGWidget::mouseReleaseEvent( QMouseEvent *ev )
{
    UInt32 button;
    
    switch ( ev->button() ) 
    {
        case LeftButton:  button = SimpleSceneManager::MouseLeft;   break;
        case MidButton:   button = SimpleSceneManager::MouseMiddle; break;
        case RightButton: button = SimpleSceneManager::MouseRight;  break;
        default:          return;
    }

    _mgr->mouseButtonRelease(button, ev->x(), ev->y());
    update();
}

void OpenSGWidget::mouseMoveEvent( QMouseEvent *ev )
{
    _mgr->mouseMove(ev->x(), ev->y());
    update();
}

void OpenSGWidget::wheelEvent( QWheelEvent *ev )
{
    _mgr->mouseButtonPress(ev->delta() > 0 ? SimpleSceneManager::MouseUp
                                          : SimpleSceneManager::MouseDown, 
                              ev->x(), ev->y());

    ev->accept();
    update();
}

/*!
 * \brief Drag enter event.
 * \param event
 */
void OpenSGWidget::dragEnterEvent(QDragEnterEvent *e)
{
    if(!strcmp(e->format(), "text/uri-list"))
    {
        e->accept(true);
    }
    else
    {
        e->accept(false);
    }
}

/*!
 * \brief Drop event, accepts urls and tries to load them.
 * \param event
 */
void OpenSGWidget::dropEvent(QDropEvent *e)
{
    if(!strcmp(e->format(), "text/uri-list"))
    {
        QStringList list;

        if(QUriDrag::decodeLocalFiles(e, list))
        {
            emit droppedFiles(list); 
        }
    }
}


// ****************************
// NS Plugin
// ****************************

OSGNSPlugin::OSGNSPlugin() :
        QNPWidget(),
        _main(NULL),
        _tools(NULL),
        _gl(NULL),
        _root(NullFC),
        _show_tools(false)
{
    setAcceptDrops(true);
    
    _main = new QVBox(this);
    _main->setSpacing(2);

    QVBoxLayout *vl = new QVBoxLayout(this);
    vl->addWidget(_main);
    
    // create render widget.
    _gl = new OpenSGWidget(QGLFormat(QGL::DoubleBuffer | QGL::DepthBuffer |
                                     QGL::Rgba | QGL::DirectRendering), _main);

    connect(_gl, SIGNAL(droppedFiles(const QStringList &)), this,
            SLOT(droppedFiles(const QStringList &)));

    // tool bar.
    _tools = new QHBox(_main);
    _tools->setSizePolicy(QSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed));
    
    // add show all button
    QPushButton *show_all = new QPushButton("ShowAll", _tools);
    connect(show_all, SIGNAL(clicked()), this, SLOT(showAll()));
    
    QPushButton *headlight = new QPushButton("Headlight", _tools);
    headlight->setToggleButton(true);
    connect(headlight, SIGNAL(toggled(bool)), this, SLOT(toggledHeadlight(bool)));
    headlight->blockSignals(true);
    headlight->setOn(true);
    headlight->blockSignals(false);
    
    // create a gradient background.
    GradientBackgroundPtr gback = GradientBackground::create();
    beginEditCP(gback, GradientBackground::LineFieldMask);
        gback->clearLines();
        gback->addLine(Color3f(0.7, 0.7, 0.8), 0);
        gback->addLine(Color3f(0.0, 0.1, 0.3), 1);
    endEditCP(gback, GradientBackground::LineFieldMask);

    WindowPtr win = _gl->getManager()->getWindow();
    beginEditCP(win);
        for(int i=0;i<win->getPort().size();++i)
        {
            ViewportPtr vp = win->getPort()[i];
            beginEditCP(vp);
                vp->setBackground(gback);
            endEditCP(vp);
        }
    endEditCP(win);

    // create root node.
    _root = Node::create();
    GroupPtr group = Group::create();
    beginEditCP(_root);
        _root->setCore(group);
    endEditCP(_root);

    // create a scene
    NodePtr scene = makeTorus(.5, 3, 16, 16);

    beginEditCP(_root);
        _root->addChild(scene);
    endEditCP(_root);
        
    _gl->getManager()->setRoot(_root);
    _gl->getManager()->showAll();
    _gl->update();
}

NodePtr OSGNSPlugin::load(const QString &filename)
{
    QString fname = QDir::convertSeparators(filename);
    QApplication::setOverrideCursor(QCursor(Qt::waitCursor));
    NodePtr scene = SceneFileHandler::the().read(fname.latin1());
    QApplication::restoreOverrideCursor();
    
    if(scene != NullFC)
    {
        beginEditCP(_root);
            _root->addChild(scene);
        endEditCP(_root);
    }
    else
    {
        QString text;
        text.sprintf("Couldn't load '%s'!", fname.latin1());
        QMessageBox::warning(this, "OSGNSPlugin", text);
    }
    
    _gl->getManager()->showAll();
    _gl->update();

    return scene;
}

void OSGNSPlugin::setShowTools(bool s)
{
    _show_tools = s;
    if(_show_tools)
        _tools->show();
    else
        _tools->hide();
}

void OSGNSPlugin::paintEvent(QPaintEvent* event)
{
    _main->repaint();
    _gl->update();
}

void OSGNSPlugin::resizeEvent(QResizeEvent*)
{
    //_main->resize(size());
    _gl->update();
}

void OSGNSPlugin::showAll(void)
{
    _gl->getManager()->showAll();
    _gl->update();
}

void OSGNSPlugin::toggledHeadlight(bool s)
{
    if(s)
        _gl->getManager()->turnHeadlightOn();
    else
        _gl->getManager()->turnHeadlightOff();
    _gl->update();
}

void OSGNSPlugin::droppedFiles(const QStringList &files)
{
    // Destroy old scene
    beginEditCP(_root);
    while(_root->getNChildren() > 0)
        _root->subChild(_root->getChild(0));
    endEditCP(_root);
    
    for(QStringList::ConstIterator it = files.begin(); it != files.end(); ++it)
        NodePtr scene = load((*it));

    _gl->getManager()->showAll();
    _gl->update();
}

// ***************************
// Plugin instance
// ***************************

OSGNSPluginInstance::OSGNSPluginInstance() :
    QNPInstance(),
    _plugin(NULL)
{
}

QNPWidget* OSGNSPluginInstance::newWindow()
{
    _plugin = new OSGNSPlugin;
    return _plugin;
}

bool OSGNSPluginInstance::newStreamCreated(QNPStream *, StreamMode &smode)
{
    smode = AsFileOnly;
    return true;
}

void OSGNSPluginInstance::streamAsFile(QNPStream *, const char *fname)
{
    // load file given with src=""
    if(_plugin != NULL)
        _plugin->load(fname);
}

// -------------------------------------------

QNPInstance *OSGNSPluginPlugin::newInstance()
{
    return new OSGNSPluginInstance;
}

const char *OSGNSPluginPlugin::getMIMEDescription() const
{
    return "OSGNSPlugin::A OpenSG render plugin";
}

const char *OSGNSPluginPlugin::getPluginNameString() const
{
    return "OSGNSPlugin Qt-based Plugin";
}

const char *OSGNSPluginPlugin::getPluginDescriptionString() const
{
    return "A Qt-based LiveConnected plug-in with a OpenSG render widget.";
}

QNPlugin* QNPlugin::create()
{
    // init OpenSG
    osgInit(0, NULL);
    osg::FieldContainerFactory::the()->initializePendingTypes();
    
    if(!QGLFormat::hasOpenGL())
        qWarning( "This system has no OpenGL support!" );

    return new OSGNSPluginPlugin;
}
