/*----------------------------------------------------------------------------
// Licensed materials - Property of IBM                                      
//
// (C) Copyright IBM Corp.  2007
// This code was donated to the OpenSSL project under the terms of the 
// OpenSSL license.
//
//---------------------------------------------------------------------------*/


/* Note !
   AES-CCM is defined as a one-shot encrypt/mac operation
   hence there's no AES_CCM_CTX, no Init/Update/Final
   It'll also eat a LOT of RAM for long messages.
*/
#ifndef AES_DEBUG
# ifndef NDEBUG
#  define NDEBUG
# endif
#endif
#include <assert.h>

#include "openssl/evp.h"
#include "openssl/aes.h"
#include "aes_locl.h"


/*
\debug Define: Uncomment to enable debugging of the AES-CCM code
#define AES_CCM_DEBUG 1
*/

#if defined(AES_CCM_DEBUG)


/*! @brief print a tagged binary buffer
    @param tag the tag
    @param buf the buffer to convert to hex and print
    @param blen the length of the buffer
    \debug Code: printbin: controlled by AES_CCM_DEBUG
*/
static void printbin(char *tag,unsigned char *buf, int blen)
{
  int i;
  printf("\t%-12s: ",tag);
  for(i = 0; i < blen; i++) {
    printf("%02x",(unsigned)buf[i]);
    if( (i % 4) == 3) printf(" ");
    
  }
  printf("\n");
}
/*! @brief print a tagged binary buffer with an incrementing counter on the tag
    @param tag the tag
    @param ctr a pointer to the counter
    @param buf the buffer to convert to hex and print
    @param blen the length of the buffer
    \debug Code: printbinCTR: controlled by AES_CCM_DEBUG
*/
static void printbinCTR(char *tag,int *ctr,unsigned char *buf, int blen)
{

  char buffer[21];
  sprintf(buffer,"%s%1d",tag,*ctr);
  (*ctr)++;
  printbin(buffer,buf,blen);
}
#else
#define printbin(x,y,z)
#define printbinCTR(x,y,z,q)
#endif


/*! @brief
  Increment the CCM CTR, which is variable length, big endian
  @param counter the counter to increment
  @param q the number of bytes in the counter
*/
static void AES_CCM_inc(unsigned char *counter,unsigned q) {
  int i;
  for(i = 15; q > 0 ; i--,q--) {
    counter[i]++;
    if(0 != counter[i] ) break;
  }            
}


/** @brief xor two buffers into a destination. 
    (Which may be one of the source buffers )
    @param dest the destination buffer
    @param s1 first source buffer
    @param s2 second source buffer
    @param blen buffer length
*/

static void xor(unsigned char *dest, unsigned char *s1, unsigned char *s2, unsigned blen)
{
    unsigned int i;
    for(i = 0; i < blen; i++) {
	dest[i] = s1[i] ^ s2[i];
    }
}
/** @brief perform the CCM incremental hash function 
    @param pcb An ICC library context 
    @param B The retained hash buffer 
    @param A the new data to hash in
    @param ctx and initialzied Cipher CTX   
*/
static void CCM_HASH(unsigned char *B,unsigned char *A,EVP_CIPHER_CTX *ctx) 
{
  int outl = 0;
  /* XOR the encrypted last block (now in B) 
     with the newly prepared block 
  */
  xor(A,B,A,AES_BLOCK_SIZE);
  /* Encrypt the result, and save it as the "last block" */
  /* AES_encrypt(A,B,akey); */
  EVP_EncryptUpdate(ctx,B,&outl,A,AES_BLOCK_SIZE);  
}

/** @brief Initialize the CCM counter
    @param CTR a pointer to the counter buffer
    @param iv the nonce
    @param ivlen the length of the nonce
    @param q the size of the counter fields
*/
static void CCM_FORMAT_CTR(unsigned char *CTR,
			   unsigned char *iv,
			   unsigned long ivlen,
			   unsigned int q) 
{
  unsigned int i;
  /* Format the CTR */
  memset(CTR,0,AES_BLOCK_SIZE);
  /* Copy in the IV fields */
  for(i = 1; i <= ivlen; i ++) {
    CTR[i] = iv[i-1];
  }
  CTR[0] |= ((q-1) & 7);
}

/** @brief
    Perform the code common to AES_CCM encrypt and decrypt 
    provide the iv (Nonce),aad, data and key, output buffer and taglength
    @param pcb an ICC library context
    @param iv the Nonce, can be 32-128 bits long, < 64 is not recommended
    @param ivlen the length of the IV
    @param key an aes
    @param keylen the length of the AES key in bytes
    @param aad Additional Authentication data, hashed, but not encrypted
    @param aadlen the length of the aad
    @param data the data buffer to encrypt
    @param datalen the length of the data buffer
    @param out the output buffer
    @param outlen a place to store the returned output length
    - which WILL be rounded up to a 16 byte boundary +16 bytes
    @param taglen the desired length of the auth tag
    @param enc 0 = decrypt, else encrypt
    @return 1 if O.K., 0 otherwise
*/ 
static int AES_CCM_Common(unsigned char *iv,unsigned int ivlen,
			  unsigned char *key,unsigned int keylen,
			  unsigned char *aad, unsigned long aadlen,
			  unsigned char *data,unsigned long datalen,
			  unsigned char *out, unsigned long *outlen,
			  unsigned int taglen,int enc
			  )
{
    int rv = 1;
    EVP_CIPHER_CTX *ctx = NULL;
    const EVP_CIPHER *cipher = NULL;
    unsigned char A0[AES_BLOCK_SIZE];
    unsigned char B0[AES_BLOCK_SIZE];
    unsigned char CTR[AES_BLOCK_SIZE];
    unsigned char S0[AES_BLOCK_SIZE];
    unsigned t = 0;
    unsigned q = 0;
    unsigned long p = 0;
    unsigned char *oldout = out;
    unsigned int aadbytes = 0;
    unsigned int offset = 0;
    int outl = 0;
    unsigned int i,j,k;
    unsigned int aadenc = 2;
#if defined(AES_CCM_DEBUG)
    int b = 0; /* Index counters to aid formatting during debug */
    int s = 0;
    int c = 0;
#endif
    /* Check for a valid aad length */
    if(aadlen > 0xFFFFFFFFL) {
      rv = 0;
    }
    /* Check for a valid nonce length 7-13 */
    if( (ivlen < 7) || (ivlen > 13) ) {
      rv = 0;
    } else {
      /* Calculate q, the bytes required to hold the possible representation 
	 of the LENGTH of the plaintext from the nonce 
      */
      q = 15 - ivlen;

    }
    t = (taglen-2)/2;
    /* check for a valid tag length 4,6,8,10,12,14,16 */
    
    if( (0 != (taglen & 1)) || (taglen < 4) || (taglen > 16)) {
      rv = 0;      
    }
    if(NULL == outlen) {
      rv = 0;
    }
    switch(keylen) {
    case 16:
      cipher = EVP_get_cipherbyname("AES-128-ECB");
      break;
    case 24:
      cipher = EVP_get_cipherbyname("AES-192-ECB");
      break;
    case 32:
      cipher = EVP_get_cipherbyname("AES-256-ECB");
      break;
    default:
      rv = 0;
      break;
    }
    ctx = EVP_CIPHER_CTX_new();
    if( 1 == rv ) {
      *outlen = 0; /* Clear the output data length */
      /* Now - to prevent the user from knowing why we failed 
	 always flush the output buffers 
      */
      if(enc) {
	/* In encrypt mode, clear the output buffer */
	memset(out,0,(((datalen+15)/16)*16)+taglen);
      } else {
	/* Same in decrypt mode */
	memset(out,0,datalen-taglen);
      }
      EVP_CIPHER_CTX_set_padding(ctx,0);
      EVP_EncryptInit(ctx,cipher,key,NULL);      
    } 
    if (1 == rv ) {
      /* If we are decrypting, we don't decrypt the tag ... */
      if( ! enc ) {
	datalen -= taglen;
      }

      /* Format the first block */
      memset(B0,0,AES_BLOCK_SIZE);
      /* Do we have adata ? */
      if(aad != NULL && aadlen > 0) {
	B0[0] |= 0x40;
      }
      B0[0] |= (((taglen-2)/2) & 7) << 3;
      B0[0] |= (q - 1) & 7;
      /* Copy in the IV fields */
      for(i = 1; i <= ivlen; i ++) {
	B0[i] = iv[i-1];
      }

      /* 
	 Encode the number of bytes of plaintext 
	 in the remaining space 
      */
      p = datalen;;
      for( j = 15 ; j > i ; j--) {
	B0[j] = (p & 0xff);
	p >>= 8;
      }
      
      printbinCTR("B",&b,B0,16);
      memset(A0,0,AES_BLOCK_SIZE); /* Start with original data == 0 */
      CCM_HASH(B0,A0,ctx);

      CCM_FORMAT_CTR(CTR,iv,ivlen,q);

      /*
	 now any aadata 
      */
    
      memset(A0,0,AES_BLOCK_SIZE);
      
      if(aad != NULL && aadlen > 0) {
	if(aadlen < (0x10000L - 0x100L)) {
	  aadbytes = 2;	    
	  aadenc = 2;
	} else if(aadlen <= 0xFFFFFFFF) {
	  aadbytes = 6;
	  aadenc = 4;
	  A0[0] = 0xff;
	  A0[1] = 0xfe;
	} else {
	  aadbytes = 10;
	  aadenc = 8;
	  A0[0] = 0xff;
	  A0[1] = 0xff;
	}
	j = aadlen;
	for(i = 0, k = aadbytes-1; i < aadenc; i++,k--) {
	  A0[k] = j & 0xff;
	  j = j / 256;
	}
	/* Now roll through the aad ? */
      }
      /* Free space in A0.16 - (1 byte flags + aadbytes) */
      offset = aadbytes;
      j = 16 - offset; 
      /* j is the number of bytes to copy,
	 offset is the offset into the buffer to start 
	 the copy
      */

      for( ;aadlen > 0 ; aadlen -= j,j = AES_BLOCK_SIZE) {

	/* If the remaining free space is less than the aadlength
	   copy only the aadlength 
	*/
	if( j > aadlen) {
	  j = aadlen;
	}
	
	/* start at buffer + flags + stored length
	   copy the requistite ammount of aad
	*/
	memcpy(A0+offset,aad,j);
	if(offset+j < 16) {
	  memset(A0+offset+j,0,(16-(offset+j)));
	}

	printbinCTR("B",&b,A0,16);
	CCM_HASH(B0,A0,ctx);
 	aad += j;
	/* Now encrypt the result, and store it in B0 */

	offset = 0; /* offset can only be zero the first time through */
	if(aadlen <= j) break; /* aadlen is unsigned, this ensures we exit */
      }


      /* Process the first counter block */
      printbinCTR("CTR",&c,CTR,AES_BLOCK_SIZE);
      /* AES_encrypt(CTR,A0,akey); */
      EVP_EncryptUpdate(ctx,A0,&outl,CTR,AES_BLOCK_SIZE);
      printbinCTR("S",&s,A0,AES_BLOCK_SIZE);
      AES_CCM_inc(CTR,q); 
      memcpy(S0,A0,AES_BLOCK_SIZE);


      /* Now do the encrypt/decrypt phase, note that we are still adding data to the
	 hash function at this phase as well.
      */

      for( ; datalen > 0; datalen -= AES_BLOCK_SIZE) {
	int l = AES_BLOCK_SIZE;

	if(datalen < AES_BLOCK_SIZE) {
	  l = datalen;
	  /* In this case we need to zero A0 */
	  memset(A0,0,AES_BLOCK_SIZE);
	}

	if(enc) {
	  memcpy(A0,data,l);
	  printbinCTR("B",&b,A0,AES_BLOCK_SIZE);
	  CCM_HASH(B0,A0,ctx);
	} 
	/* Prep the counter - encrypt the current value
	   
	 */
	printbinCTR("CTR",&c,CTR,AES_BLOCK_SIZE);
	/* AES_encrypt(CTR,A0,akey); */
	EVP_EncryptUpdate(ctx,A0,&outl,CTR,AES_BLOCK_SIZE);
	printbinCTR("S",&s,A0,AES_BLOCK_SIZE);
	/* Increment the counter */
	AES_CCM_inc(CTR,q);

	/* XOR the encrypted counter with the incoming data */
	xor(A0,data,A0,l);

	/* Copy this to the output */
	memcpy(out,A0,l);
	printbin("C",A0,l);

	/* In decrypt mode, that gave us plaintext which
	   we hash 
	*/
	if( !enc ) {
	  if(l != AES_BLOCK_SIZE) {
            memset(A0+l,0,AES_BLOCK_SIZE-l);
          }
	  printbinCTR("B",&b,A0,AES_BLOCK_SIZE);
	  CCM_HASH(B0,A0,ctx);
	}
	data += l;
	out += l;
	*outlen += l;
	if(datalen < AES_BLOCK_SIZE) {
	  break;
	}
      }
    }
    if(1 == rv ) {
      /* last encrypted hash block is our tag */
      printbin("T",B0,taglen);
      /* XOR this with the saved first counter block */
      xor(B0,S0,B0,taglen);
      /* append this to the encrypted data in encrypt mode */
      if( enc ) {
	memcpy(out,B0,taglen);
	*outlen += taglen;
      } else {	
	/* If it's decrypt mode, we compare it with the end of the data */
	if(memcmp(B0,data,taglen) != 0 ) {
	  rv = 0;
	  memset(oldout,0,*outlen);
	}
      }
    }
    if(NULL != ctx) {
      EVP_CIPHER_CTX_cleanup(ctx);
      EVP_CIPHER_CTX_free(ctx);
   }
    return rv;
}

/** @brief
    Perform an AES CCM Encrypt operation, 
    provide the iv (Nonce),aad, data and key, output buffer and taglength
    @param pcb an ICC library context
    @param iv The Nonce, can be 32-128 bits long, < 64 is not recommended
    @param ivlen the length of the IV
    @param key an aes key
    @param keylen the length of the AES key in bytes
    @param aad Additional Authentication data, hashed, but not encrypted
    @param aadlen the length of the aad
    @param data the data buffer to encrypt
    @param datalen the length of the data buffer
    @param out the output buffer
    @param outlen a place to store the returned output length
    - which WILL be rounded up to a 16 byte boundary +16 bytes
    @param taglen the desired length of the auth tag
    @return 1 if O.K., 0 otherwise
*/ 
int AES_CCM_Encrypt(unsigned char *iv,unsigned int ivlen,
		    unsigned char *key,unsigned int keylen,
		    unsigned char *aad, unsigned long aadlen,
		    unsigned char *data,unsigned long datalen,
		    unsigned char *out, unsigned long *outlen,
		    unsigned int taglen
		    )
{
  int rv = 1;
  rv = AES_CCM_Common(iv,ivlen,
		      key,keylen,
		      aad,aadlen,
		      data,datalen,
		      out,outlen,
		      taglen,1);

 
    return rv;
}


/** @brief
    Perform an AES CCM Decrypt operation, 
    provide the iv (Nonce),aad, data and key, output buffer and taglength
    @param pcb an ICC library context
    @param iv The Nonce, can be 32-128 bits long, < 64 is not recommended
    @param ivlen the length of the IV
    @param key an aes key
    @param keylen the length of the AES key
    @param aad Additional Authentication data, hashed, but not encrypted
    @param aadlen the length of the aad
    @param data the data buffer to encrypt
    @param datalen the length of the data buffer
    @param out the output buffer
    @param outlen a place to store the returned output length
    @param taglen the length of the auth tag
    @return 1 if O.K., 0 otherwise
    @note This (by spec) returns no data on failure. However as
    we'd have to allocate an internal buffer, which would still be
    accessable to the caller within the same process, we simple
    erase any partial data in the output buffer on failure instead.
    - So , be aware that the output buffer WILL be overwritten,
    no matter what.
    @note the fact that this isn't allowed to return any data if the
    tag match fails is the reason why there can't be the usual 
    Init/Update/Update/Final API.
*/ 
int AES_CCM_Decrypt(unsigned char *iv,unsigned int ivlen,
		    unsigned char *key,unsigned int keylen,
		    unsigned char *aad, unsigned long aadlen,
		    unsigned char *data,unsigned long datalen,
		    unsigned char *out, unsigned long *outlen,
		    unsigned int taglen
		    )
{
  int rv = 1;
  rv = AES_CCM_Common(iv,ivlen,
		      key,keylen,
		      aad,aadlen,
		      data,datalen,
		      out,outlen,
		      taglen,0);
  
 
  if(0 == rv) { /* If anything failed, kill the output buffer */
    memset(out,0,*outlen);
    *outlen = 0;    
  }

  return rv;
}



