/*----------------------------------------------------------------------------
// Licensed materials - Property of IBM                                      
//
// (C) Copyright IBM Corp.  2007,2009
// This code was donated to the OpenSSL project under the terms of the 
// OpenSSL license.
//
// 
-----------------------------------------------------------------------------*/

#include <string.h>
#include "openssl/evp.h"
#include "cmac.h"

/* 
\debug Define: Uncomment this to debug the CMAC code
#define DEBUG_CMAC 
*/


/* 
   We reuse one buffer, which gets set to zero on the first
   CMAC_Update() operation. This is used as a flag to say the next
   update will be the first. It's possible to perform a CMAC
   with no data passed in.
*/
#define STUPID_VALUE 0xdeadbeef
/*
  These are what the constants should be, but we always XOR them, so just
  xoring the final value is enough.
static const unsigned char R128[16] = {0,0,0,0, 0,0,0,0, 0,0,0,0, 0,0,0,0x87};
static const unsigned char R64[8] =   {0,0,0,0, 0,0,0,0x1b};
*/
#define R128 0x87
#define R64 0x1b

static const char iv[16]; /*!< zero filled buffer NIST CMAC always uses a 0 IV and it's reused for the 0 data block input to generate subkeys */

/* DES odd parity */
static const unsigned char parity[256]={
    1,  1,  2,  2,  4,  4,  7,  7,  8,  8, 11, 11, 13, 13, 14, 14,
   16, 16, 19, 19, 21, 21, 22, 22, 25, 25, 26, 26, 28, 28, 31, 31,
   32, 32, 35, 35, 37, 37, 38, 38, 41, 41, 42, 42, 44, 44, 47, 47,
   49, 49, 50, 50, 52, 52, 55, 55, 56, 56, 59, 59, 61, 61, 62, 62,
   64, 64, 67, 67, 69, 69, 70, 70, 73, 73, 74, 74, 76, 76, 79, 79,
   81, 81, 82, 82, 84, 84, 87, 87, 88, 88, 91, 91, 93, 93, 94, 94,
   97, 97, 98, 98,100,100,103,103,104,104,107,107,109,109,110,110,
  112,112,115,115,117,117,118,118,121,121,122,122,124,124,127,127,
  128,128,131,131,133,133,134,134,137,137,138,138,140,140,143,143,
  145,145,146,146,148,148,151,151,152,152,155,155,157,157,158,158,
  161,161,162,162,164,164,167,167,168,168,171,171,173,173,174,174,
  176,176,179,179,181,181,182,182,185,185,186,186,188,188,191,191,
  193,193,194,194,196,196,199,199,200,200,203,203,205,205,206,206,
  208,208,211,211,213,213,214,214,217,217,218,218,220,220,223,223,
  224,224,227,227,229,229,230,230,233,233,234,234,236,236,239,239,
  241,241,242,242,244,244,247,247,248,248,251,251,253,253,254,254
};

/*!
  #brief fix up the key parity.
  This does duplicate function inside the DES code
  the DES_CBLOCK data structure is really ugly - this
  is simpler
  @param key a des/3des key
  @param leylen the length in bytes of the key
*/
static void fix_des(unsigned char *key, int keylen)
{
  int i;
  for(i = 0; i < keylen; i++) {
    key[i] = parity[key[i]];
  }
}




#if defined(DEBUG_CMAC)
/*! @brief print a binary buffer
    @param s the buffer to print
    @param l the length of the buffer
    \debug Code printbin: controlled by DEBUG_CMAC
*/
static void printbin( char *s,int l)
{
  int i;
  printf("len = %d :",l);
  for(i = 0; i < l ; i++) {
	printf("%02x",((unsigned)s[i] & 0xff));
  }
  printf("\n");
}
#endif


/** @brief xor two buffers into a destination. 
    (Which may be one of the source buffers )
    @param dest the destination buffer
    @param s1 first source buffer
    @param s2 second source buffer
    @param blen buffer length
*/

static void xor(unsigned char *dest, unsigned char *s1, unsigned char *s2, unsigned blen)
{
  unsigned int i;
  for(i = 0; i < blen; i++) {
    dest[i] = s1[i] ^ s2[i];
  }
}

/** @brief left shift a buffer (assumed to be a big endian byte stream)
    @param buf  source buffer
    @param blen buffer length
*/
static void shl(unsigned char *buf,unsigned blen)
{
  unsigned cry = 0;
  unsigned crynxt = 0;
  if(blen != 0) {
    do {
      blen --;
      if(buf[blen] & 0x80) {
	crynxt = 1;
      }
      buf[blen] = (buf[blen] << 1) | cry;
      cry = crynxt;
      crynxt = 0;
    } while(blen != 0);
  }
}
/**
   @brief Call EVP_EncryptInit() for the CMAC context
          this is used in more than one place, and the logic is a bit messy
	  with the need to handle variable length keys	  
    @param cmac_ctx a pointer to an CMAC_CTX_t
    @return 1 if O.K. 0 otherwise
*/
static int CMAC_EVP_Init(CMAC_CTX_t *cmac_ctx)
{
  int rv = 1;
  unsigned int keybytes = 0;
  int nid = -1;

  nid =  EVP_CIPHER_type(cmac_ctx->cip);
  EVP_CIPHER_CTX_cleanup(cmac_ctx->cctx);
  if(cmac_ctx->keylen == 0) { /* Fixed length keys, AES, DES etc */
    /* Check for DES keys here as we need to fixup the key parity */
    switch(nid) {
    case NID_des_cbc:
      fix_des(cmac_ctx->key,8);
      break;
    case NID_des_ede_cbc:
    case NID_des_ede3_cbc:
      fix_des(cmac_ctx->key,24);
      break;
    default:
      break;
    }    
    rv = EVP_EncryptInit(cmac_ctx->cctx,cmac_ctx->cip,cmac_ctx->key,(unsigned char *)iv);
    EVP_CIPHER_CTX_set_padding(cmac_ctx->cctx,0);
  } else { /* Alg's with variable length keys, RC2, RC4 etc */
    keybytes = (cmac_ctx->keylen + 7)/8;
    EVP_EncryptInit_ex(cmac_ctx->cctx,cmac_ctx->cip,NULL,NULL,NULL);
    EVP_CIPHER_CTX_set_key_length(cmac_ctx->cctx,keybytes);
    /* RC2 non-byte length keys */
    if(NID_rc2_cbc == nid ) {
      EVP_CIPHER_CTX_ctrl(cmac_ctx->cctx,EVP_CTRL_SET_RC2_KEY_BITS,cmac_ctx->keylen,NULL);
    }
    EVP_CIPHER_CTX_set_padding(cmac_ctx->cctx,0);
    rv = EVP_EncryptInit_ex(cmac_ctx->cctx,NULL,NULL,cmac_ctx->key,(unsigned char *)iv);
  }
  return rv;
}

/** @brief generate the CMAC subkeys
    @param pcb a pointer to an ICC library context
    @param cmac_ctx a pointer to an CMAC_CTX_t 
    @return 1 if O.K., 0 on failure (unsupported cipher block size etc).    
*/
    
static int SubKeys(CMAC_CTX_t *cmac_ctx)
{
  int rv = 1;
  int olen = 0;
  unsigned char R;
  switch(cmac_ctx->bs) {
  case 8:
    R = R64;
    break;
  case 16:
    R = R128;
    break;
  default:
    rv = 0;
    break;
  }
  if(1 == rv) {
    rv = CMAC_EVP_Init(cmac_ctx);
    if( 1 == rv) {
      /* We need to encrypt with a block of 0's here */
      EVP_EncryptUpdate(cmac_ctx->cctx,
			     cmac_ctx->k1,&olen,(unsigned char *)iv, cmac_ctx->bs);
      
      /* k1 now contains one block of 0 encrypted with our key */
      
      /* We only need this for testing, save a copy of the encrypted key 
	 so we can verify it against NIST known answers.
	 This buffer is reused as a scratch area during CMAC_Update
      */
      memcpy(cmac_ctx->cip_k,cmac_ctx->k1, cmac_ctx->bs);
      
      if(0 == (cmac_ctx->k1[0] & 0x80)) { /* MSBit of K1 is zero */
	shl(cmac_ctx->k1, cmac_ctx->bs);
      } else {
	shl(cmac_ctx->k1, cmac_ctx->bs);
	cmac_ctx->k1[cmac_ctx->bs-1] ^= R;
	/* xor(cmac_ctx->k1,cmac_ctx->k1,R,cmac_ctx->bs); */
      }
      memcpy(cmac_ctx->k2,cmac_ctx->k1, cmac_ctx->bs);
      if(0 == (cmac_ctx->k2[0] & 0x80)) { /* MSBit of K2 is zero */
	shl(cmac_ctx->k2, cmac_ctx->bs);
      } else {
	shl(cmac_ctx->k2, cmac_ctx->bs);      
	cmac_ctx->k2[cmac_ctx->bs-1] ^= R;
	/* xor(cmac_ctx->k2,cmac_ctx->k2,R,cmac_ctx->bs);*/
      }
    }
  }
  if(rv != 1) printf("CMAC_EVP_Init() fails %d\n",rv);
  return rv;
}
/** @brief Allocate a new CMAC_CTX (CMAC context)
    @param pcb a pointer to an ICC library context
    @return NULL on failure, or a pointer to an uninitialized CMAC_CTX
*/
CMAC_CTX * CMAC_CTX_new()
{
  CMAC_CTX *ctx;
  ctx = (CMAC_CTX *)CRYPTO_malloc(sizeof(CMAC_CTX),__FILE__,__LINE__);
  if(NULL != ctx) memset(ctx,0,sizeof(CMAC_CTX));
  return (ctx);
}
/** @brief Clean up an CMAC_CTX_t (erase sensitive data)
    @param pcb a pointer to an ICC library context
    @param cmac_ctx a pointer to a CMAC_CTX_t
*/
static void CMAC_CTX_cleanup(CMAC_CTX_t *cmac_ctx)
{
  int clen = 0;
  memset(cmac_ctx->k1,0,cmac_ctx->bs);
  memset(cmac_ctx->k2,0,cmac_ctx->bs);
  memset(cmac_ctx->buf,0,cmac_ctx->bs);
  memset(cmac_ctx->cip_k,0,cmac_ctx->bs);
  if(cmac_ctx->keylen) {
    clen = (cmac_ctx->keylen + 7)/8;
  } else {
    clen = EVP_CIPHER_key_length(cmac_ctx->cip);
  }
  memset(cmac_ctx->key,0,clen);
}
/** @brief free an CMAC_CTX_t 
    @param pcb a pointer to an ICC library context 
    @param cmac a pointer to the CMAC_CTX to free
*/
void CMAC_CTX_free(CMAC_CTX *cmac) 
{
  CMAC_CTX_t *cmac_ctx = (CMAC_CTX_t *)cmac;

  if(NULL != cmac_ctx) {
    CMAC_CTX_cleanup(cmac_ctx);
    if(NULL != cmac_ctx->k1) CRYPTO_free(cmac_ctx->k1);
    if(NULL != cmac_ctx->k2) CRYPTO_free(cmac_ctx->k2);
    if(NULL != cmac_ctx->buf) CRYPTO_free(cmac_ctx->buf);
    if(NULL != cmac_ctx->cip_k) CRYPTO_free(cmac_ctx->cip_k);
    if(NULL != cmac_ctx->cctx) {
        EVP_CIPHER_CTX_cleanup(cmac_ctx->cctx);
        EVP_CIPHER_CTX_free(cmac_ctx->cctx);
    }
    if(NULL != cmac_ctx->key) CRYPTO_free(cmac_ctx->key);
    CRYPTO_free(cmac_ctx);
  }
}
/** @brief Initialize a CMAC operation
    @param pcb a pointer to an ICC library context 
    @param cmac a pointer to an CMAC_CTX
    @param cip a pointer to a cipher structure
    @param key a key buffer, the size must match that
           needed by the CMAC cipher
    @param keylen the length of the key in bits for variable key length algs,
           Set this to zero for fixed length key algs
    @return 1 on success, 0 otherwise
*/
    
int CMAC_Init(CMAC_CTX *cmac,
	      const EVP_CIPHER *cip,
	      unsigned char *key,
	      unsigned int keylen) 
{
  
  CMAC_CTX_t *cmac_ctx = (CMAC_CTX_t *)cmac;
  int rv = 1;
  unsigned int nid = 0;

  cmac_ctx->bs = EVP_CIPHER_block_size(cip);
  if( 8 == cmac_ctx->bs || 16 == cmac_ctx->bs) {

    cmac_ctx->cip = cip;
    if(NULL == cmac_ctx->k1) {
      cmac_ctx->k1 = (unsigned char *)CRYPTO_malloc(cmac_ctx->bs,__FILE__,__LINE__);
    }
    memset(cmac_ctx->k1,0,cmac_ctx->bs);

    if(NULL == cmac_ctx->k2) {
      cmac_ctx->k2 = (unsigned char *)CRYPTO_malloc(cmac_ctx->bs,__FILE__,__LINE__);
    }
    memset(cmac_ctx->k2,0,cmac_ctx->bs);
 
    if(NULL == cmac_ctx->buf ) {
      cmac_ctx->buf = (unsigned char *)CRYPTO_malloc(cmac_ctx->bs,__FILE__,__LINE__);
    } 
    memset(cmac_ctx->buf,0,cmac_ctx->bs);
    if( 0 == keylen) {
      keylen = EVP_CIPHER_key_length(cip)*8;
    }
    if(NULL == cmac_ctx->key) {
      /* Allows for RC2 keys */
      cmac_ctx->key = (unsigned char *)CRYPTO_malloc(keylen+7/8,__FILE__,__LINE__);
    }
    memcpy(cmac_ctx->key,key,EVP_CIPHER_key_length(cip));
 
    
    /* Used in Init() to hold the encrypted key,
       after that it's used as scratch storage for Update()
    */
    if(NULL == cmac_ctx->cip_k) {
      cmac_ctx->cip_k = (unsigned char *)CRYPTO_malloc(cmac_ctx->bs,__FILE__,__LINE__);
    }
    memset(cmac_ctx->cip_k,0,cmac_ctx->bs);

    if(NULL == cmac_ctx->cctx) {
      cmac_ctx->cctx = EVP_CIPHER_CTX_new();
    } else {
      EVP_CIPHER_CTX_cleanup(cmac_ctx->cctx);
    }
    cmac_ctx->pbs = STUPID_VALUE;
    nid =  EVP_CIPHER_type(cip);
    switch(nid) { /* This is just in case the caller messes up ! */
    case NID_aes_128_cbc:
    case NID_aes_192_cbc:
    case NID_aes_256_cbc:
    case NID_camellia_128_cbc:
    case NID_camellia_192_cbc:
    case NID_camellia_256_cbc:
    case NID_des_cbc:
    case NID_des_ede3_cbc:
    case NID_cast5_cbc:
      cmac_ctx->keylen = 0; 
      break;
    default: /* RC2, RC4 */
      cmac_ctx->keylen = keylen;
      break;
    }
    
    if( 1 != SubKeys(cmac_ctx) ) {
      rv = 0;
    }
  } else {
    rv = 0;
  }
  return rv;
}
/* 
   This needs to be just a bit clever to cope with partial blocks
   and the special handling of the final block
   Basically CMAC_Final needs either a partial or a full block 
   left over. 
   Since we'll be kind and allow partial blocks to be input here
   we need to be sure we'll have at least a partial block left,
   and possibly a full one.

   Note that this code does assume certain behaviour from OpenSSL's
   EncryptUpdate - i.e. that with padding off, and complete blocks being fed
   in, complete blocks are output immediately. That's not unreasonable,
   but I'm not sure it's guaranteed.

*/
/** @brief Update phase of a CMAC operation
    @param pcb a pointer to an ICC library context
    @param cmac a pointer to a CMAC_CTX
    @param in a pointer to data to MAC
    @param inlen the length of the input data
    @note blocked/aligned data will be more efficient, but this will
    survive incorrectly blocked/misaligned data
*/
int CMAC_Update(CMAC_CTX *cmac,
		unsigned char *in, 
		unsigned int inlen)
{
  int rv = 1;
  unsigned int i = 0;
  unsigned int outl = 0;
  CMAC_CTX_t *cmac_ctx = (CMAC_CTX_t *)cmac;

  /* We need to erase this buffer anyway - and we avoid allocating
     one extra buffer this way - is it worth it ? - Probably since
     we needed somewhere to save cip_k to pass the known answer tests 
     and it still needs to be set to zero for the first block.
  */
  if(STUPID_VALUE == cmac_ctx->pbs) {
    memset(cmac_ctx->cip_k,0,cmac_ctx->bs);
    cmac_ctx->pbs = 0;
  } 
  /* This is a bit ugly, redundant memcpy's in what's hopefully 
     the common case, but the cost of memcpy is minor
     compared with the encrypt and we have to deal with the possibility
     of misaligned input data at any time
  */
  if(inlen > 0) {
    if( (inlen + cmac_ctx->pbs) > cmac_ctx->bs) {
      for( i = inlen+cmac_ctx->pbs; i > cmac_ctx->bs; i -= ((cmac_ctx->bs > i) ? i : cmac_ctx->bs) ) {
	/* Top up the retained buffer with incoming data */
	memcpy(cmac_ctx->buf + cmac_ctx->pbs,in,(cmac_ctx->bs - cmac_ctx->pbs));
	/* We stole this many bytes of input */
	in += (cmac_ctx->bs - cmac_ctx->pbs);
	/* And there's no partial buffer for next time */
	cmac_ctx->pbs = 0;
	/* Ci-1 ^ Mi */
	xor(cmac_ctx->buf,cmac_ctx->cip_k,cmac_ctx->buf,cmac_ctx->bs);
	/* Ci = CIPH(Ci-1 ^ Mi) */
	
	rv = CMAC_EVP_Init(cmac_ctx);
	if( 1 == rv) {
#if defined(DEBUG_CMAC)
	  printf("UPD:");
	  printbin(cmac_ctx->buf,cmac_ctx->bs);
#endif
	  rv = EVP_EncryptUpdate(cmac_ctx->cctx,
				 cmac_ctx->cip_k,
				 (int *)&outl,cmac_ctx->buf,cmac_ctx->bs);      
	}
      }
      if(i > 0) {
	/* We have at most one block left over, 
	   copy that to our retain buffer and set the length of the left over segment       
	*/
	memcpy(cmac_ctx->buf,in,i);
	cmac_ctx->pbs = i;
      }
    } else { 
      /* O.K. someone is being a bit silly, we don't have enough data to do an update yet,
	 maybe doing the CMAC 3 bytes at a time perhaps ...
	 Just top up the retain buffer ... *sigh*
      */
      memcpy(cmac_ctx->buf + cmac_ctx->pbs,in,inlen);
      cmac_ctx->pbs += inlen;
    }
  }
  return rv;
}
/** @brief Finish a CMAC operation and return the CMAC value
    @param pcb a pointer to an ICC library context
    @param cmac a pointer to an CMAC_CTX
    @param md a pointer to a buffer in which to store the CMAC,
           must be at least maclen bytes longs
    @param maclen the length of the desired mac value in bytes. (<= block length)
    @return 1 on success, 0 on failure
    @note a request for an overly long CMAC return will result in an error and
          no data being produced

*/
int CMAC_Final(CMAC_CTX *cmac,
	       unsigned char *md, 
	       unsigned int maclen)
{
  int rv = 1;
  int outl = 0;
  CMAC_CTX_t *cmac_ctx = (CMAC_CTX_t *)cmac;
  if(maclen <= cmac_ctx->bs) {
    /* A CMAC is valid with no input data, so depending on the user CMAC
       logic we COULD hit this case. i.e. 
       CMAC_Init()
       for(i = 0; i < len ; i+= blocksize) {
          CMAC_Update()
       }
       CMAC_Final()
    */
    if(STUPID_VALUE == cmac_ctx->pbs) {
      memset(cmac_ctx->cip_k,0,cmac_ctx->bs);
      cmac_ctx->pbs = 0;
    } 

    if(cmac_ctx->pbs == cmac_ctx->bs) { /* Complete block */
      /* Mn = K1 ^ Mn* */
      xor(cmac_ctx->buf,cmac_ctx->k1,cmac_ctx->buf,cmac_ctx->bs);
    } else { /* Partial block */
      /* Mn' = (Mn* || 10... ) - we are byte blocked so ... */
      memset(cmac_ctx->buf + cmac_ctx->pbs,0,(cmac_ctx->bs-cmac_ctx->pbs));
      /* MSBit of the unused section set to 1 */
      cmac_ctx->buf[cmac_ctx->pbs] = 0x80;
      /* XOR with k2 => Mn = (K2 ^ (Mn* || 10....) */
      xor(cmac_ctx->buf,cmac_ctx->k2,cmac_ctx->buf,cmac_ctx->bs);
    }
    /* Now Xor with the retained data (cip_k)*/
    xor(cmac_ctx->buf,cmac_ctx->cip_k,cmac_ctx->buf,cmac_ctx->bs);
    /* Now do the final run through the encryption */
#if defined(DEBUG_CMAC)
    printbin(cmac_ctx->buf,cmac_ctx->bs);
    printf("FIN:");
#endif
    CMAC_EVP_Init(cmac_ctx);

    rv = EVP_EncryptUpdate(cmac_ctx->cctx,
				cmac_ctx->cip_k,
				&outl,
				cmac_ctx->buf,cmac_ctx->bs);
    /* Return the requested amount of mac */
    memcpy(md,cmac_ctx->cip_k,maclen);    
  } else {
    rv = 0; /* Invalid mac length */
  }
  /* Scrub the CMAC ctx */
  CMAC_CTX_cleanup(cmac_ctx);
  EVP_CIPHER_CTX_cleanup(cmac_ctx->cctx);
  return rv;
}


