/* -*-c++-*- OpenSceneGraph - Copyright (C) 1998-2006 Robert Osfield 
 *
 * This library is open source and may be redistributed and/or modified under  
 * the terms of the OpenSceneGraph Public License (OSGPL) version 0.0 or 
 * (at your option) any later version.  The full license is in LICENSE file
 * included with this distribution, and on the openscenegraph.org website.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 * OpenSceneGraph Public License for more details.
*/

#include <osgDB/PluginImageWriter>
#include <osgDB/FileNameUtils>
#include <osgDB/FileUtils>
#include <osgDB/WriteFile>
#include <osg/Image>

namespace osgDB
{

/// Counts the number of direcories the given relative path goes "up" the current dir, or 0 if not.
/// This returns 0 for absolute paths.
/// Examples:
///    - "../a" goes 1 level up
///    - "../../a/b/c/d/e" goes 2
///    - "../a/../b/../.." goes 2
///    - "a/b/../c" goes 0
unsigned int countNbDirsUp(const std::string & path)
{
    // Algorithm:
    //    - For each path component, count +1 for "..", 0 for ".", and -1 for anything else
    //    - Ignore everything after  the last ".." of the path.
    if (osgDB::isAbsolutePath(path)) return 0;
    int result(0), tempResult(0);
    for(osgDB::PathIterator it(path); it.valid(); ++it)
    {
        if (*it == "..")
        {
            // Count +1, and "validates" temporary result
            ++tempResult;
            result = tempResult;
        }
        else if (*it != ".") --tempResult;
    }

    return result<=0 ? 0 : static_cast<unsigned int>(result);
}


/// Local hash function for a path.
/// Does not canonize the given path, but is not confused with mixed separators.
unsigned int pathHash(const std::string & s)
{
    // This is based on the DJB hash algorithm
    // Note: SDBM Hash initializes at 0 and is
    //        hash = c + (hash << 6) + (hash << 16) - hash;
    unsigned int hash = 5381;
    for(std::string::const_iterator it=s.begin(), itEnd=s.end(); it!=itEnd; ++it)
    {
        std::string::value_type c = *it;
        if (c == '\\') c = '/';        // We're processing a path and don't want to be affected by differences in separators
        hash = ((hash << 5) + hash) + c;
    }
    return hash;
}





PluginImageWriter::PluginImageWriter(const std::string & srcDirectory, const std::string & destDirectory, bool keepRelativePaths, unsigned int allowUpDirs)
    : _lastGeneratedImageIndex(0), _srcDirectory(srcDirectory), _destDirectory(destDirectory), _keepRelativePaths(keepRelativePaths), _allowUpDirs(allowUpDirs), cur(_images.end())
{}

PluginImageWriter::PluginImageWriter(const std::string & destDirectory)
    : _lastGeneratedImageIndex(0), _destDirectory(destDirectory), _keepRelativePaths(false), _allowUpDirs(0), cur(_images.end())
{}


void PluginImageWriter::setCurImage(const osg::Image * img)
{
    if (!img) return;
    ImagesSet::iterator it( _images.find(img) );
    if (it != _images.end())
    {
        cur = it;
        return;
    }

    // New entry

    // Get absolute source path
    std::string absoluteSourcePath;
    if (_keepRelativePaths && !img->getFileName().empty())        // if keepRelativePaths is false, absoluteSourcePath is not used, so we can skip this part
    {
        if (osgDB::isAbsolutePath(img->getFileName())) absoluteSourcePath = img->getFileName();
        else absoluteSourcePath = osgDB::concatPaths(_srcDirectory, img->getFileName());
        absoluteSourcePath = osgDB::getRealPath(osgDB::convertFileNameToNativeStyle(absoluteSourcePath));      // getRealPath() here is only used to canonize the path, not to add current directory in front of relative paths, hence the "concatPaths(_srcDirectory, ...)" just above
    }

    // Compute destination paths from the source path
    std::string relativeDestinationPath;
    std::string absoluteDestinationPath;
    if (absoluteSourcePath.empty())
    {
        // We have no name. Generate one.
        generateImageName(relativeDestinationPath, absoluteDestinationPath);
    }
    else
    {
        // We have a name.
        if (_keepRelativePaths)
        {
            // We'll try to keep images relative path.
            relativeDestinationPath = osgDB::getPathRelative(_srcDirectory, absoluteSourcePath);
            unsigned int nbDirsUp = countNbDirsUp(relativeDestinationPath);
            // TODO if nbDirsUp>nb dirs in _destDirectory, then issue a warning, and use simple file name
            if (nbDirsUp > _allowUpDirs) relativeDestinationPath = osgDB::getSimpleFileName(absoluteSourcePath);
        }
        else
        {
            // We keep only the simple file name.
            relativeDestinationPath = osgDB::getSimpleFileName(absoluteSourcePath);
        }
        absoluteDestinationPath = osgDB::getRealPath(osgDB::convertFileNameToNativeStyle( osgDB::concatPaths(_destDirectory, relativeDestinationPath) ));
        // TODO Check for absolute paths collisions between multiple images
    }

    // Add entry
    cur = _images.insert(ImagesSet::value_type(img, ImageData(relativeDestinationPath, absoluteDestinationPath))).first;
    _searchMap.insert(SearchMap::value_type(pathHash(absoluteDestinationPath), img));
}


void PluginImageWriter::write(const osgDB::Options * options) const
{
    if (!valid()) return;
    if (cur->second.written) return;        // Already written
    const std::string & absoluteDestinationPath = cur->second.absolutePath;
    if (!osgDB::makeDirectoryForFile(absoluteDestinationPath))
    {
        OSG_NOTICE << "Can't create directory for file '" << absoluteDestinationPath << "'. May fail creating the image file." << std::endl;
    }
    if (!osgDB::writeImageFile(*cur->first, absoluteDestinationPath, options))
    {
        OSG_WARN << "Can't write file '" << absoluteDestinationPath << "'." << std::endl;
    }
    cur->second.written = true;        // Set 'written' even if writing failed, to avoid having the same warning message in a loop
}

bool PluginImageWriter::absoluteImagePathExists(const std::string & path)
{
    // For all paths in the search map having the same hash as 'path', check if paths correspond
    std::pair<SearchMap::iterator, SearchMap::iterator> bounds( _searchMap.equal_range(pathHash(path)) );
    for(SearchMap::iterator it=bounds.first; it!=bounds.second; ++it)
    {
        const osg::Image * img( it->second );
        if (_images[img].absolutePath == path) return true;
    }
    return false;
}

void PluginImageWriter::generateImageName(std::string & out_relativePath, std::string & out_absolutePath)
{
    static const ImageIndex MAX_IMAGE_NUMBER = UINT_MAX-1;        // -1 to allow doing +1 without an overflow
    static const char * const IMAGE_PREFIX = "Image_";
    static const char * const IMAGE_EXT    = ".tga";
    for (ImageIndex imageNumber=_lastGeneratedImageIndex+1; imageNumber<MAX_IMAGE_NUMBER; ++imageNumber)
    {
        std::ostringstream oss;
        oss << IMAGE_PREFIX << imageNumber << IMAGE_EXT;
        out_relativePath = oss.str();
        out_absolutePath = osgDB::concatPaths(_destDirectory, out_relativePath);
        
        if (!absoluteImagePathExists(out_absolutePath))
        {
            _lastGeneratedImageIndex = imageNumber;
            return;
        }
    }
    throw std::runtime_error("Could not get a free index to write image.");
}

}
