/*
  The shader properties:
_____________________________
-----------------------------
-----------------------------

  input vertex attributes
  -----------------------
  vec4 position_mass
  vec3 velocity
  ivec4 connection


  uniform variables
  -----------------
  samplerBuffer tex_position_mass
  float t -- time step
  float k -- spring constant
  float c -- damping constant
  mat4 mvp -- model view projection matrix
  

  output vertex attributes
  ------------------------
  interface block Vertex with:
    vec4 position_mass
    vec3 velocity

 */

// minimum glsl shader version that this shader requires
# version 330

// only for OpenGL ES 2.0 compatibility
precision highp float;

// position and mass of the point
// vector contains the vertex position in xyz, and the mass in w
// (vertex attribute)
in vec4 position_mass;

// current velocity in xyz direction
// (vertex attribute)
in vec3 velocity;

// the connection vector define the connectivity between the points
// (vertex attribute)
in ivec4 connection;

// this is a texture buffer object that will be bound to the same buffer as the
// position_mass input attribute
// (uniform variable)
uniform samplerBuffer tex_position_mass;

// the outputs of the vertex shader are the same as the inputs, just
// wrapped in an interface block
//out Vertex
//{
out  vec4 vs_position_mass;
out  vec3 vs_velocity;
//} vertex;

// uniform for the time step
// the application update this
uniform float t;

// global spring constant
uniform float k;

// global damping constant
uniform float c;

// gravity
const vec3 gravity = vec3(0.0, 0.0, 0.0);

// reseting length of the spring
uniform float rest_length;

// model view projection matrix
uniform mat4 mvp;

void main()
{
  vec3 p = position_mass.xyz; // p can be our position
  float m = position_mass.w;  // m is the mass of our vertex
  vec3 u = velocity;          // u is the initial velocity
  vec3 F;                     // F is the force on the mass
  vec3 v = u;                 // v is the final velocity
  vec3 s = vec3(0.0);         // s is the displacement in this step

  // first we check if this is a 'fixed' vertex
  // -1 means 'fixed'
  if(connection[0] != -1)
  {
    // Initialize F using gravity and damping
    // Fg = gravity
    // Fd = -c * u
    //  F = Fg + Fd
    F = gravity - c * u;

    // Go through the neighbor points
    for(int i = 0; i < 4; i++)
    {
      // if the id the same like the current vertex id,
      // there is no neighbor
      if(connection[i] != gl_VertexID)
      {
        // q is the position of the neighbor current vertex
        // we won't care about the other vertex's mass
        vec3 q = texelFetch(tex_position_mass, connection[i]).xyz;
        vec3 d = q - p; // direction
        float x = length(d); // distance between p and q
        // Fs = -kxd
        // F_all = Fg + Fd + Fs
        F += -k * (rest_length - x) * normalize(d);
      }
    } // end of neighbor loop

    // acceleration due to force
    // Newton's second law F = m * a
    vec3 a = F / m;

    // final velocity
    // equation of motion
    v = u + a * t;

    // displacement
    // derivative of v with respect to time
    s = u * t + 0.5 * a * t * t;

  } // end of 'if fixed'


  // write the outputs
  //vertex.position_mass = vec4(p + s, m);
  //vertex.velocity = v;
  vs_position_mass = vec4(p + s, m);
  vs_velocity = v;


  // for the rendering
  // we have to update the gl_Position
  gl_Position = mvp * vec4(p + s, 1.0);

} // end of main

