#include "include/springmasssystem.h"

#include <stdio.h>
#include <stdlib.h>
#include <iostream>

#include <GL/gl.h>
#include <GL/glut.h>

SpringMassSystem::SpringMassSystem()
{
  _initialised = false;       // spring mass system isn't initialised
  setUseDisplayList(false);   // disable displaylists
  createMesh(2,2);            // create the spring mass mesh
}

SpringMassSystem::~SpringMassSystem()
{
  delete[] _pm_data;
  delete[] _vel_data;
  delete[] _connection_data;
  delete[] _idx_data;
  delete[] _ext;
}

void SpringMassSystem::createMesh(unsigned int m,unsigned int n)
{
  _numVerts = m*n;  // number of vertices

  // init vertices
  //////////////////////////////
  std::cout << "Position/Mass:\n-----------------------" << std::endl;
  _pm_data = new float[_numVerts*4]; // allocate memory
  float x = 0.0;
  float y = 0.0;
  float dx = 1.0 / (m-1);
  float dy = 1.0 / (n-1);
  for(unsigned int i=0; i < _numVerts; i++)
  {
    _pm_data[i*4] = x; // x
    _pm_data[i*4+1] = y; // y
    _pm_data[i*4+2] = 0.0; // z
    _pm_data[i*4+3] = 1.0; // mass

    std::cout << i << ": " << x << ", " << y << ", " << 0.0 << ", " << 1.0 << std::endl;

    x += dx;

    if(!((i+1)%m))
    {
      y += dy;
      x = 0.0;
    }
  } // vertices are initialised (CPU)
  //////////////////////////////


  // init vertex connections
  //////////////////////////////
  std::cout << "Connections:\n-----------------------" << std::endl;
  int* _connection_data = new int[_numVerts*4];
  for(unsigned int i=0; i<_numVerts; i++)
  {
    int tmp = 0;
    // i+n (top)
    tmp = i+m;
    if(tmp >= _numVerts)
      tmp = -1; // if vertex on the top, then pin it (value = -1)
    _connection_data[i*4] = tmp;
    std::cout << i << ": " << tmp;

    // i+1 (right)
    tmp = i+1;
    if(!(tmp % m))
      tmp = i; // there is no right neighbor (value = i)
    _connection_data[i*4+1] = tmp;
    std::cout << ", " << tmp;

    // i-n (bottom)
    tmp = i-m;
    if(tmp < 0)
      tmp = i; // there is no bottom vertex (value = i)
    _connection_data[i*4+2] = tmp;
    std::cout << ", " << tmp;

    // i-1 (left)
    tmp = i-1;
    if(!((tmp+1)%m))
      tmp = i;
    _connection_data[i*4+3] = tmp;
    std::cout << ", " << tmp << std::endl;

  } // connections are initialised (CPU)
  //////////////////////////////

  // TEST
  for(unsigned int i=0; i < _numVerts*4; i++)
    _connection_data[i] = -1;


  // init vertex velocity
  //////////////////////////////
  float* _vel_data = new float[_numVerts*3];
  for (unsigned int i = 0; i < _numVerts*3; i++)
    _vel_data[i] = 0.0;  // start value is zero
  // velocities are initialised (CPU)
  //////////////////////////////


  // init quad indices
  //////////////////////////////
  std::cout << "Indices:\n-----------------------" << std::endl;
  _numPrim = (m-1)*(n-1);
  _idx_data = new ushort[_numPrim*4];
  unsigned int row = 0;
  unsigned int a = -1;
  for(unsigned int i=0; i<_numPrim; i++)
  {
    a++;

    if(((a+1) % m) == 0)
      a++;

    _idx_data[i*4] = a;
    _idx_data[i*4+1] = a+1;
    _idx_data[i*4+2] = a+1+m;
    _idx_data[i*4+3] = a+m;

    std::cout << i << ": " << a << ", " << a+1 << ", " << a+1+m << ", " << a+m << std::endl;

  } // indices are initialised
  //////////////////////////////
}

char* SpringMassSystem::shaderSourceFromFile(char* filename) const
{
  FILE *fp; // file pointer
  char* buffer; // char memory/buffer
  unsigned long len; // len of file
  fp = fopen(filename,"r"); // open file
  fseek(fp,0,SEEK_END); // go to end
  len=ftell(fp); // get position at end (length)
  fseek(fp,0,SEEK_SET); // go to the beginning
  buffer=(char*)malloc(len); // allocate memory
  fread(buffer,len,1,fp); // read into buffer
  fclose(fp);
  return buffer;
}

void SpringMassSystem::shaderLog(GLuint Shader) const
{
  GLint isCompiled;
  _ext->glGetShaderiv(Shader,GL_COMPILE_STATUS,&isCompiled);
  if(isCompiled)
    std::cout << "Shader is compiled" << std::endl;
  else
  {
    std::cout << "Shader is NOT compiled" << std::endl;

    GLsizei infologLen = 0;
    GLsizei charsWritten = 0;
    GLchar* infoLog;

    _ext->glGetShaderiv(Shader, GL_INFO_LOG_LENGTH, &infologLen);

    if (infologLen > 0)
    {
      infoLog = (GLchar*) malloc(infologLen);
      if (infoLog == NULL)
      {
        printf("ERROR: Could not allocate InfoLog buffer\n");
        exit(1);
      }
    }

    _ext->glGetShaderInfoLog(Shader,infologLen,&charsWritten,infoLog);
    printf("InfoLog:\n%s\n\n",infoLog);
    free(infoLog);
  }
}

void SpringMassSystem::shaderLinkLog(GLuint Program) const
{
  bool IsLinked;
  char *shaderProgramInfoLog;
  int maxLength;

  // check status
  _ext->glGetProgramiv(Program, GL_LINK_STATUS, (int *)&IsLinked);

  if(IsLinked == false)
  {
    /* Noticed that glGetProgramiv is used to get the length for a shader program, not glGetShaderiv. */
    _ext->glGetProgramiv(Program, GL_INFO_LOG_LENGTH, &maxLength);

    /* The maxLength includes the NULL character */
    shaderProgramInfoLog = new char[maxLength];

    /* Notice that glGetProgramInfoLog, not glGetShaderInfoLog. */
    _ext->glGetProgramInfoLog(Program, maxLength, &maxLength, shaderProgramInfoLog);

    printf("LINKER: %s",shaderProgramInfoLog);
    /* Handle the error in an appropriate way such as displaying a message or writing to a log file. */
    /* In this simple program, we'll just leave */
    exit(2);
  }
}

// Error Print Function
void SpringMassSystem::openglLog(char* prefix) const
{
  GLenum errCode;
  const GLubyte* errString;
  if((errCode = glGetError()) != GL_NO_ERROR)
  {
    errString = gluErrorString(errCode);
    fprintf(stderr, "OpenGL Error %s: %s\n", prefix, errString);
  }
}

void SpringMassSystem::drawImplementation (osg::RenderInfo& renderInfo) const
{
  // initialise step
  if(!_initialised)
  {
    _initialised = true;
    _writeID = 1;
    _readID = 0;

    // opengl extension interface
    //////////////////////////////
    _ext = new MyExtensions(renderInfo.getContextID());
    std::cout << "OpenGL Version: " <<  _ext->getGlVersion() << std::endl;

    // update pass initialisation
    //////////////////////////////
    _ext->glGenVertexArrays(2, _vaoUpdate); // generate vaos update
    _ext->glGenBuffers(2, _vboPos); // generate vbos position mass data
    _ext->glGenBuffers(2, _vboVel); // generate vbos velocity data
    _ext->glGenBuffers(1, &_vboCon); // generate vbo connection data
    openglLog("glGenBuffers");

    for(int i=0; i < 2; i++)
    {
      _ext->glBindVertexArray(_vaoUpdate[i]); // bind vao update

      // First we define the position vbo for the update step
      //////////////////////////////
      _ext->glBindBuffer( GL_ARRAY_BUFFER, _vboPos[i] ); // bind vbo for position mass data
      _ext->glBufferData( GL_ARRAY_BUFFER, _numVerts*sizeof(float)*4, // allocate gpu memory
                          &(_pm_data[0]), // copy data to gpu memory
                          GL_DYNAMIC_COPY ); // dynamic copy mode (Driver-Memmory-Management)
      _ext->glEnableVertexAttribArray(0); // enable attribute array 0 for position mass data
      _ext->glVertexAttribPointer(0, 4, GL_FLOAT, GL_FALSE, 0, 0); // pointer on position data


      // then we define the velocity vbo
      //////////////////////////////
      _ext->glBindBuffer( GL_ARRAY_BUFFER, _vboVel[i] ); // bind vbo for velocity data
      _ext->glBufferData( GL_ARRAY_BUFFER, _numVerts*sizeof(float)*3, // allocate gpu memory
                          &(_vel_data[0]), // copy data to gpu memory
                          GL_DYNAMIC_COPY ); // use dynamic copy mode
      _ext->glEnableVertexAttribArray(1); // enable attribute array 1 for velocity data
      _ext->glVertexAttribPointer( 1, 3, GL_FLOAT, GL_FALSE, 0, 0 ); // pointer on velocity data

      // last but not least we set up the connection data vbo
      //////////////////////////////
      _ext->glBindBuffer( GL_ARRAY_BUFFER, _vboCon ); // bind vbo for connection data
      openglLog("glBindBuffer");
      if(i==0)
      {
        _ext->glBufferData( GL_ARRAY_BUFFER, _numVerts*sizeof(int)*4, // allocate gpu memory
                            &(_connection_data[0]), // copy data to gpu memory
                            GL_STATIC_DRAW ); // use static draw mode
        openglLog("glBufferData");
      }
      _ext->glEnableVertexAttribArray(2); // enable attribute array 2 for connection data
      openglLog("glEnableVertexAttribArray");
      _ext->glVertexAttribIPointer(2, 4, GL_INT, 0, 0); // pointer on index data
      openglLog("glVertexAttribIPointer");
    } // update vaos are ready
    //////////////////////////////


    // render pass initialisation
    //////////////////////////////
    _ext->glGenVertexArrays(2, _vaoRender); // generate vaos render
    _ext->glGenBuffers(1, &_vboIdx); // generate vbo indices

    for(int i=0; i < 2; i++)
    {
      _ext->glBindVertexArray(_vaoRender[i]); // bind vao render

      // first we define the position vbo for the render step
      _ext->glBindBuffer( GL_ARRAY_BUFFER, _vboPos[i] ); // bind the position mass vbo
      _ext->glEnableVertexAttribArray(0); // enable the attribute array 0 for position mass data
      _ext->glVertexAttribPointer(0, 4, GL_FLOAT, GL_FALSE, 0, 0); // pointer on the position mass data

      // then we define the indices
      _ext->glBindBuffer( GL_ELEMENT_ARRAY_BUFFER, _vboIdx ); // bind index buffer
      if(i==0)
        _ext->glBufferData( GL_ELEMENT_ARRAY_BUFFER, _numPrim*sizeof(ushort)*4,
                            &(_idx_data[0]), GL_STATIC_DRAW );
    } // render vaos are ready
    //////////////////////////////

    _ext->glBindVertexArray(0); // unbind vao


    // initialise the texture buffer with the _vboPos values
    //////////////////////////////
    glGenTextures(2, _texPos); // generate texture buffers for position mass data
    openglLog("glGenTextures");

    for(int i=0; i < 2; i++)
    {
      glBindTexture( GL_TEXTURE_BUFFER_ARB, _texPos[i] ); // bind textures with position mass data
      openglLog("glBindTexture");
      _ext->glTexBuffer( GL_TEXTURE_BUFFER_ARB, GL_RGBA32F, // use RGBA 32 bit float
                         _vboPos[i] ); // interpret position vbo as texture
      openglLog("glTexBuffer");
    } // position mass textures are ready


    // initialise the shader (update step)
    //////////////////////////////
    const GLchar* source = shaderSourceFromFile("spring_mass.vert"); // shader source
    GLuint vsHandle = _ext->glCreateShader(GL_VERTEX_SHADER); // get vertex shader identifier
    _ext->glShaderSource(vsHandle, 1, &source, NULL); // bind shader source and shader object
    _ext->glCompileShader(vsHandle); // compile shader
    shaderLog(vsHandle); // error check

    _program = _ext->glCreateProgram(); // create shader program
    _ext->glAttachShader(_program, vsHandle); // bind vertex shader to the program

    _ext->glBindAttribLocation(_program,0,"position_mass");
    _ext->glBindAttribLocation(_program,1,"velocity");
    _ext->glBindAttribLocation(_program,2,"connection");

    const char* varyings[2] = {"vs_position_mass","vs_velocity"}; // transform feedback target variables
    _ext->glTransformFeedbackVaryings(_program, 2, varyings, GL_SEPARATE_ATTRIBS); // define tf-varyings
    _ext->glLinkProgram(_program); // link the komplete shader program
    shaderLinkLog(_program);       // check the linking process
  } // end of the initialisation
  //////////////////////////////

  // if all necessary objects initialised,
  // we do the transform feedback and the rendering

  // first, the transform feedback
  //////////////////////////////
  _ext->glUseProgram(_program); // activate shader program

  const GLfloat mvp[16] = {1,0,0,0,0,1,0,0,0,0,1,0,0,0,0,1}; // isn't necessary!!!!!!!!!!!!
  _ext->glUniformMatrix4fv(_ext->glGetUniformLocation(_program,"mvp"),1,GL_FALSE,&mvp[0]); // model view matrix
  _ext->glUniform1f(_ext->glGetUniformLocation(_program,"t"), 0.00001f); // time step (constant)
  _ext->glUniform1f(_ext->glGetUniformLocation(_program,"k"), 15.0f); // spring constant
  _ext->glUniform1f(_ext->glGetUniformLocation(_program,"c"), 8.0f); // damping constant
  _ext->glUniform1f(_ext->glGetUniformLocation(_program,"rest_length"), 0.1); // spring rest length
  _ext->glUniform1i(_ext->glGetUniformLocation(_program,"tex_position_mass"),0);

  _ext->glActiveTexture(GL_TEXTURE0); // activate texture unit 0
  glBindTexture(GL_TEXTURE_BUFFER_ARB, _texPos[_writeID]); // bind texture with position data to tex unit 0

  _ext->glBindVertexArray(_vaoUpdate[_writeID]); // attach vao update
  _ext->glBindBufferBase(GL_TRANSFORM_FEEDBACK_BUFFER, 0, _vboPos[_readID]); // define read buffer (position)
  _ext->glBindBufferBase(GL_TRANSFORM_FEEDBACK_BUFFER, 1, _vboVel[_readID]); // define read buffer (velocity)

  glEnable(GL_RASTERIZER_DISCARD); // disable rasterization process
  _ext->glBeginTransformFeedback(GL_POINTS); // begin of the transform feedback process
  //glBeginQuery(GL_TRANSFORM_FEEDBACK_PRIMITVES_WRITTEN, query); // begin query
  glDrawArrays(GL_POINTS, 0, _numVerts); // draw array (vertex stage)
  //glEndQuery(GL_TRANSFORM_FEEDBACK_PRIMITIVES_WRITTEN); // end of the query
  _ext->glEndTransformFeedback(); // transform feedback is finished
  glDisable(GL_RASTERIZER_DISCARD); // enable rasterization process
  //glGetQueryObjectuiv(query, GL_QUERY_RESULT, &primitives_written); // get query result
  _ext->glUseProgram(0); // deactivate shader program

  // RENDER STEP
  //////////////////////////////
  glPointSize(6); // define points size
  glDisable(GL_LIGHTING);
  _ext->glBindVertexArray(_vaoRender[_writeID]); // bind vertex array object
  glColor3f(0.0f,0.0f,0.7f);
  glDrawElements(GL_QUADS,_numPrim*4,GL_UNSIGNED_SHORT,0); // draw quads
  glColor3f(0.7f,0.0f,0.0f);
  glDrawElements(GL_LINES,_numPrim*4,GL_UNSIGNED_SHORT,0); // draw lines
  glDrawArrays(GL_POINTS,0,_numVerts); // draw points
  glEnable(GL_LIGHTING);
  _ext->glBindVertexArray(0); // unbind vertex array object

  // FLIP BUFFER STEP
  //////////////////////////////
  int tmp = _readID;
  _readID =_writeID;
  _writeID=tmp;
}
