/*
 * This file contains the PPC32 PMU register description tables
 * and pmc checker used by perfmon.c.
 *
 * Philip Mucci, mucci@cs.utk.edu
 * 
 * Based on code from:
 * Copyright (c) 2005 David Gibson, IBM Corporation.
 *
 * Based on perfmon_p6.c:
 * Copyright (c) 2005-2006 Hewlett-Packard Development Company, L.P.
 * Contributed by Stephane Eranian <eranian@hpl.hp.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2 of the GNU General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 * 02111-1307 USA
  */
#include <linux/module.h>
#include <linux/perfmon.h>

MODULE_AUTHOR("Philip Mucci <mucci@cs.utk.edu>");
MODULE_DESCRIPTION("PPC32 PMU description table");
MODULE_LICENSE("GPL");

struct pfm_arch_pmu_info pfm_ppc32_pmu_info={
  .pmu_style = PM_NONE,
};

static struct pfm_reg_desc pfm_ppc32_pmc_desc[]={
/* mmcr0 */ PMC_D(PFM_REG_I, "MMCR0", 0x0, 0, 0, 0),
/* mmcr1 */ PMC_D(PFM_REG_I, "MMCR1", 0x0, 0, 0, 0),
/* mmcra */ PMC_D(PFM_REG_I, "MMCR2", 0x0, 0, 0, 0)
};
#define PFM_PM_NUM_PMCS	(sizeof(pfm_ppc32_pmc_desc)/sizeof(struct pfm_reg_desc))

static struct pfm_reg_desc pfm_ppc32_pmd_desc[]={
/* pmd1  */ PMD_D(PFM_REG_C, "PMD1", 0),
/* pmd2  */ PMD_D(PFM_REG_C, "PMD2", 0),
/* pmd3  */ PMD_D(PFM_REG_C, "PMD3", 0),
/* pmd4  */ PMD_D(PFM_REG_C, "PMD4", 0),
/* pmd5  */ PMD_D(PFM_REG_C, "PMD5", 0),
/* pmd6  */ PMD_D(PFM_REG_C, "PMD6", 0)
};
#define PFM_PM_NUM_PMDS	(sizeof(pfm_ppc32_pmd_desc)/sizeof(struct pfm_reg_desc))

static enum ppc32_pmu_type pm_type = PM_NONE;
static int nmmcr = 0;
static int npmds = 0;
static int intsok = 0;

static unsigned int get_nr_pmds(void)
{
	switch (pm_type) {
	case PM_7450:
		return 6;
	case PM_7400:
	case PM_750:
	case PM_604e:
		return 4;
	case PM_604:
		return 2;
	default: /* PM_NONE, but silences gcc warning */
		return 0;
	}
}

static char pmu_name[64] = "";

static int pfm_ppc32_probe_pmu(void)
{
  unsigned int pvr = mfspr(SPRN_PVR);
  
  switch (PVR_VER(pvr))
    {
    case 0x0004: /* 604 */
      strcpy(pmu_name,"PPC604");
      pm_type = PM_604;
      nmmcr = 1;
      break;
    case 0x0009: /* 604e;  */
    case 0x000A: /* 604ev */
      strcpy(pmu_name,"PPC604e");
      pm_type = PM_604e;
      nmmcr = 2;
      break;
    case 0x0008: /* 750/740 */
      strcpy(pmu_name,"PPC750");
      pm_type = PM_750;
      nmmcr = 2;
      break;
    case 0x7000: /* 750FX */
    case 0x7001:
      strcpy(pmu_name,"PPC750");
      pm_type = PM_750;
      nmmcr = 2;
      if ((pvr & 0xFF0F) >= 0x0203)
	intsok = 1;
      break;
    case 0x7002: /* 750GX */
      strcpy(pmu_name,"PPC750");
      pm_type = PM_750;
      nmmcr = 2;
      intsok = 1;
    case 0x000C: /* 7400 */
      strcpy(pmu_name,"PPC7400");
      pm_type = PM_7400;
      nmmcr = 3;
      break;
    case 0x800C: /* 7410 */
      strcpy(pmu_name,"PPC7400");
      pm_type = PM_7400;
      nmmcr = 3;
      if ((pvr & 0xFFFF) >= 0x01103)
	intsok = 1;
      break;
    case 0x8000: /* 7451/7441 */
    case 0x8001: /* 7455/7445 */
    case 0x8002: /* 7457/7447 */
    case 0x8003: /* 7447A */
    case 0x8004: /* 7448 */
      strcpy(pmu_name,"PPC7450");
      pm_type = PM_7450;
      nmmcr = 3;
      intsok = 1;
      break;
    default:
      PFM_INFO("Unknown PVR_VER(0x%x)\n",PVR_VER(pvr));
      return -1;
    }

  npmds = get_nr_pmds();
  return 0;
}

static void perfmon_perf_irq(struct pt_regs *regs) 
{
  /* BLATANTLY STOLEN FROM OPROFILE, then modified */

  /* set the PMM bit (see comment below) */
  mtmsr(mfmsr() | MSR_PMM);
  
  pfm_interrupt_handler(instruction_pointer(regs),regs);

  /* The freeze bit was set by the interrupt. */
  /* Clear the freeze bit, and reenable the interrupt.
   * The counters won't actually start until the rfi clears
   * the PMM bit */
  

  /* Unfreezes the counters on this CPU, enables the interrupt,
   * enables the counters to trigger the interrupt, and sets the
   * counters to only count when the mark bit is not set.
   */
  {
    u32 mmcr0 = mfspr(SPRN_MMCR0);
    
    mmcr0 &= ~(MMCR0_FC | MMCR0_FCM0);
    mmcr0 |= (MMCR0_FCECE | MMCR0_PMC1CE | MMCR0_PMCnCE | MMCR0_PMXE);
    
    mtspr(SPRN_MMCR0, mmcr0);
  }
}

/*
 * impl_pmcs, impl_pmds are computed at runtime to minimize errors!
 */
static struct pfm_pmu_config pfm_ppc32_pmu_conf = {
  .pmu_name = pmu_name,
  .counter_width = 31,
  .pmd_desc = pfm_ppc32_pmd_desc,
  .pmc_desc = pfm_ppc32_pmc_desc,
  .num_pmc_entries = PFM_PM_NUM_PMCS,
  .num_pmd_entries = PFM_PM_NUM_PMDS,
  .probe_pmu  = pfm_ppc32_probe_pmu,
  .flags = PFM_PMU_BUILTIN_FLAG,
  .owner = THIS_MODULE,
  .arch_info = &pfm_ppc32_pmu_info
};
	
static int __init pfm_ppc32_pmu_init_module(void)
{
  int retval;

  retval = pfm_ppc32_probe_pmu();
  if (retval)
    return(retval);

  if (intsok == 0)
    PFM_INFO("Interrupts unlikely to work\n");

  retval = reserve_pmc_hardware(perfmon_perf_irq);
  if (retval)
    return retval;

  pfm_ppc32_pmd_desc[nmmcr].type = PFM_REG_NA;
  pfm_ppc32_pmc_desc[npmds].type = PFM_REG_NA;

  pfm_ppc32_pmu_conf.num_pmc_entries = nmmcr;
  pfm_ppc32_pmu_conf.num_pmd_entries = npmds;

  pfm_ppc32_pmu_info.pmu_style = pm_type;
  retval = pfm_pmu_register(&pfm_ppc32_pmu_conf);
  if (retval)
      release_pmc_hardware();

  return retval;
}

static void __exit pfm_ppc32_pmu_cleanup_module(void)
{
  pfm_pmu_unregister(&pfm_ppc32_pmu_conf);
  release_pmc_hardware();
}

module_init(pfm_ppc32_pmu_init_module);
module_exit(pfm_ppc32_pmu_cleanup_module);
