
#include "postgres.h"
#include "catalog/pg_type.h" /* For xxxOID constants */
#include "postgres_fe.h"
#include "libpq-fe.h"
#include "libpq-int.h"

/*
 * Each param value requires an oid, ptr and 2 ints (oid, value, length, format).
 * This makes the maximum size of a param 20 bytes.  A stack size of 4k
 * would allow for 204 query params.  If more params are needed,
 * heap memory is used.  Needing more than 204 param values in a query
 * is very rare, although possible.
 */
#define PARAM_STACKSIZE 4096

/*
 * Declares some variables for the param value arrays:
 * oid, values, lengths, formats.  These values are assigned by
 * paramArrayAssign().
 */
#define PARAM_ARRAY_DECL \
  char _stackbuffer[PARAM_STACKSIZE]; \
  char *buf   = _stackbuffer; \
  Oid *oids   = NULL; \
  char **vals = NULL; \
  int *lens   = NULL; \
  int *fmts   = NULL

/*
 * On the very rare occasion that the number of params is too large for
 * the _stackbuffer, we have to free our allocated buffer.
 */
#define PARAM_ARRAY_FREE do{ \
  if(buf != _stackbuffer) \
    free(buf); \
}while(0)

#define getparam(_conn) &(_conn)->param

/* puts points from either a PGpath or PGpolygon.  If its a path,
 * closed indicates whether the path is open or closed.  Closed is
 * ignored when a polygon.
 */
static int putptarray(PGconn *conn, int npts, PGpoint *pts, Oid oid, int closed);

/*
 * Assigns param values to param arrays, for use with postgres
 * parameter API.  When needed, malloc/free is used for these arrays.
 * Returns 1 on success and 0 on error.
 *
 * This function used to be a macro, but it grow over time ... long macros
 * tend to be annoying.
 */
static int paramArrayAssign(
  PGconn *conn,
  char **buf, /* in=stack, out=stack|heap */
  Oid **oids,
  char ***vals,
  int **lens,
  int **fmts);

void pqParamReset(PGconn *conn)
{
  conn->param.vcnt = 0;
}

void PQparamClear(PGconn *conn)
{
  int i;
  PGparam *param = getparam(conn);

  if(!param->vals)
    return;

  for(i=0; i < param->vmax; i++)
    if(param->vals[i].ptr)
      free(param->vals[i].ptr);

  if(param->vals)
    free(param->vals);

  param->vcnt = 0;
  param->vmax = 0;
  param->vals = NULL;
}

int PQparamExec(
  PGconn *conn,
  const char *command,
  int resultFormat,
  PGresult **resultp)
{
  PGresult *res;
  PGparam *param = getparam(conn);
  PARAM_ARRAY_DECL;

  if(resultp)
    *resultp = NULL;

  if(!paramArrayAssign(conn, &buf, &oids, &vals, &lens, &fmts))
  {
    pqParamReset(conn);
    return 0;
  }

  res = PQexecParams(
    conn,
    command,
    param->vcnt,
    (const Oid *)oids,
    (const char * const *)vals,
    (const int *)lens,
    (const int *)fmts,
    resultFormat);

  PARAM_ARRAY_FREE;
  pqParamReset(conn);

  if(!res)
    return 0;

  /* If caller doesn't want result, clear it. */
  if(!resultp)
    PQclear(res);
  else
    *resultp = res;

  return 1;
}

int PQparamExecPrepared(
  PGconn *conn,
  const char *stmtName,
  int resultFormat,
  PGresult **resultp)
{
  PGresult *res;
  PGparam *param = getparam(conn);
  PARAM_ARRAY_DECL;

  if(resultp)
    *resultp = NULL;

  if(!paramArrayAssign(conn, &buf, &oids, &vals, &lens, &fmts))
  {
    pqParamReset(conn);
    return 0;
  }

  res = PQexecPrepared(
    conn,
    stmtName,
    param->vcnt,
    (const char * const *)vals,
    (const int *)lens,
    (const int *)fmts,
    resultFormat);

  PARAM_ARRAY_FREE;
  pqParamReset(conn);

  if(!res)
    return 0;

  /* If caller doesn't want result, clear it. */
  if(!resultp)
    PQclear(res);
  else
    *resultp = res;

  return 1;
}

int PQparamSend(
  PGconn *conn,
  const char *command,
  int resultFormat)
{
  int r;
  PGparam *param = getparam(conn);
  PARAM_ARRAY_DECL;

  if(!paramArrayAssign(conn, &buf, &oids, &vals, &lens, &fmts))
  {
    pqParamReset(conn);
    return 0;
  }

  r = PQsendQueryParams(
    conn,
    command,
    param->vcnt,
    (const Oid *)oids,
    (const char * const *)vals,
    (const int *)lens,
    (const int *)fmts,
    resultFormat);

  PARAM_ARRAY_FREE;
  pqParamReset(conn);
  return r;
}

int PQparamSendPrepared(
  PGconn *conn,
  const char *stmtName,
  int resultFormat)
{
  int r;
  PGparam *param = getparam(conn);
  PARAM_ARRAY_DECL;

  if(!paramArrayAssign(conn, &buf, &oids, &vals, &lens, &fmts))
  {
    pqParamReset(conn);
    return 0;
  }

  r = PQsendQueryPrepared(
    conn,
    stmtName,
    param->vcnt,
    (const char * const *)vals,
    (const int *)lens,
    (const int *)fmts,
    resultFormat);

  PARAM_ARRAY_FREE;
  pqParamReset(conn);
  return r;
}

int PQputchar(PGconn *conn, char ch)
{
  return PQput(conn, (const void *)&ch, 1, CHAROID, 1);
}

int PQputbool(PGconn *conn, int b)
{
	char c = b ? 't' : 'f';
  return PQput(conn, (const void *)&c, 1, BOOLOID, 1);
}

#if 0
int PQputmoney(PGconn *conn, double money)
{
  /* money is changing to 64-bit.  What length do we specify? */
}
#endif

int PQputint2(PGconn *conn, short i2)
{
  i2 = (short)htons(i2);
  return PQput(conn, (const void *)&i2, 2, INT2OID, 1);
}

int PQputint4(PGconn *conn, int i4)
{
  i4 = (int)htonl(i4);
  return PQput(conn, (const void *)&i4, 4, INT4OID, 1);
}

int PQputfloat4(PGconn *conn, float f4)
{
  void *voidp = &f4;
	int i = htonl(*(unsigned int *)voidp);
  return PQput(conn, (const void *)&i, 4, FLOAT4OID, 1);
}

int PQputint8(PGconn *conn, void *i8p)
{
  unsigned int buf[2];

  if(!i8p)
    return PQputnull(conn);

  return PQput(conn, pqSwapi8(buf, i8p, 1), 8, INT8OID, 1);
}

int PQputfloat8(PGconn *conn, double f8)
{
	unsigned int buf[2];
	return PQput(conn, pqSwapi8(buf, &f8, 1), 8, FLOAT8OID, 1);
}

int PQputtext(PGconn *conn, const char *text)
{
  if(!text)
    return PQputnull(conn);

  return PQput(conn, text, (int)strlen(text) + 1, TEXTOID, 0);
}

int PQputpoint(PGconn *conn, PGpoint *pt)
{
  unsigned int buf[4];

  if(!pt)
    return PQputnull(conn);

  pqSwapi8(buf,     &pt->x, 1);
  pqSwapi8(buf + 2, &pt->y, 1);
  return PQput(conn, buf, 16, POINTOID, 1);
}

int PQputlseg(PGconn *conn, PGlseg *lseg)
{
  unsigned int buf[8];

  if(!lseg)
    return PQputnull(conn);

  pqSwapi8(buf,     &lseg->pts[0].x, 1);
  pqSwapi8(buf + 2, &lseg->pts[0].y, 1);
  pqSwapi8(buf + 4, &lseg->pts[1].x, 1);
  pqSwapi8(buf + 6, &lseg->pts[1].y, 1);
  return PQput(conn, buf, 32, LSEGOID, 1);
}

int PQputbox(PGconn *conn, PGbox *box)
{
  unsigned int buf[8];

  if(!box)
    return PQputnull(conn);

  pqSwapi8(buf,     &box->high.x, 1);
  pqSwapi8(buf + 2, &box->high.y, 1);
  pqSwapi8(buf + 4, &box->low.x, 1);
  pqSwapi8(buf + 6, &box->low.y, 1);
  return PQput(conn, buf, 32, BOXOID, 1);
}

int PQputcircle(PGconn *conn, PGcircle *circle)
{
  unsigned int buf[6];

  if(!circle)
    return PQputnull(conn);

  pqSwapi8(buf,     &circle->center.x, 1);
  pqSwapi8(buf + 2, &circle->center.y, 1);
  pqSwapi8(buf + 4, &circle->radius, 1);
  return PQput(conn, buf, 24, CIRCLEOID, 1);
}

int PQputpath(PGconn *conn, PGpath *path)
{
  if(!path)
    return PQputnull(conn);
  return putptarray(conn, path->npts, path->pts, PATHOID, path->closed ? 1 : 0);
}

int PQputpolygon(PGconn *conn, PGpolygon *polygon)
{
  if(!polygon)
    return PQputnull(conn);
  return putptarray(conn, polygon->npts, polygon->pts, POLYGONOID, -1);
}

int PQputmacaddr(PGconn *conn, PGmacaddr *mac)
{
  unsigned char buf[6];

  if(!mac)
    return PQputnull(conn);

  buf[0] = (unsigned char)mac->a;
  buf[1] = (unsigned char)mac->b;
  buf[2] = (unsigned char)mac->c;
  buf[3] = (unsigned char)mac->d;
  buf[4] = (unsigned char)mac->e;
  buf[5] = (unsigned char)mac->f;
  return PQput(conn, buf, 6, MACADDROID, 1);
}

int PQput(
  PGconn *conn,
  const void *data,
  int datal,
  Oid oid,
  int format)
{
  PGvalue *v;
  PGparam *param = getparam(conn);

	if(format == 1)
	{
		int l = 0;

		switch(oid)
		{
			case BOOLOID:
			case CHAROID:
				l = 1;
				break;

			case INT2OID:
				l = 2;
				break;

			case FLOAT4OID:
			case INT4OID:
				l = 4;
				break;

			case FLOAT8OID:
			case INT8OID:
				l = 8;
				break;

			/*case MONEYOID:*/
		}

		if(l && l != datal)
		{
      printfPQExpBuffer(&conn->errorMessage,
        libpq_gettext("Invalid type length: typlen=%d != inlen=%d"), l, datal);
      pqParamReset(conn);
			return 0;
		}
	}

  /* need to grow param vals array */
  if(param->vcnt == param->vmax)
  {
    PGvalue *vals;
    int vmax = param->vmax ? (param->vmax * 3) / 2 : 16;

    if(!(vals = (PGvalue *)realloc(param->vals, sizeof(PGvalue) * vmax)))
    {
      printfPQExpBuffer(&conn->errorMessage,
        libpq_gettext("cannot allocate param value array"));
      pqParamReset(conn);
      return 0;
    }

    /* zero out the new array elements */
    memset(vals + param->vcnt, 0, (vmax - param->vcnt) * sizeof(PGvalue));
    param->vmax = vmax;
    param->vals = vals;
  }

  /* grab next param value */
  v = &param->vals[param->vcnt];

  if(!data)
    datal = 0;

  /* need more mem for param value ptr */
  if(v->ptrl < datal)
  {
    char *ptr = (char *)realloc(v->ptr, datal);

    if(!ptr)
    {
      printfPQExpBuffer(&conn->errorMessage,
        libpq_gettext("cannot allocate param value data pointer"));
      pqParamReset(conn);
      return 0;
    }

    v->ptrl = datal;
    v->ptr = ptr;
  }

  /* assign the value, length and format to the param value */
  v->datal  = datal;
	v->oid    = oid;
  v->format = format;
  v->data   = data ? (char *)memcpy(v->ptr, data, datal) : NULL;

  param->vcnt++;
  return 1;
}

static int paramArrayAssign(
  PGconn *conn,
  char **buf,
  Oid **oids,
  char ***vals,
  int **lens,
  int **fmts)
{
  int n;
  char *b = *buf;
  PGparam *param = getparam(conn);

  /* no params to assign */
  if(!param || param->vcnt == 0)
    return 1;

  /* required memory size for the 3 param arrays */
  n = (int)((sizeof(void *) * param->vcnt) +
      ((sizeof(int) * 2) * param->vcnt) +
      (sizeof(Oid) * param->vcnt));

  /* required memory is too large for stack buffer, get some heap */
  if(n > PARAM_STACKSIZE)
  {
    char *p;

    if(!(p = (char *)malloc(n)))
    {
      printfPQExpBuffer(&conn->errorMessage,
        libpq_gettext("cannot allocate param arrays\n"));
      return 0;
    }

    *buf = p;
  }

  /* give arrays memory from buffer, which could be stack or heap. */
  *vals = (char **)b;
  *lens = (int *)(b + (sizeof(void *) * param->vcnt));
  *fmts = (*lens) + param->vcnt;
	*oids = (*fmts) + param->vcnt;

  /* loop param values and assign value, length and format to arrays. */
  for(n=0; n < param->vcnt; n++)
  {
    (*vals)[n] = param->vals[n].data;
    (*lens)[n] = param->vals[n].datal;
    (*fmts)[n] = param->vals[n].format;
		(*oids)[n] = param->vals[n].oid;
  }

  return 1;
}

static int putptarray(PGconn *conn, int npts, PGpoint *pts, Oid oid, int closed)
{
  int i;
  int hdr = (int)sizeof(int);
  char _stackbuffer[1+sizeof(int)+(512*sizeof(PGpoint))]; /* 512pts (8k) */
  unsigned int *data = (unsigned int *)_stackbuffer;
  unsigned int *d = data;

  /* pts is for a path, include 1 byte open/closed flag */
  if(oid == PATHOID)
    hdr++;

  /* If path is > 512 pts, allocate from the heap */
  if(npts > 512)
  {
    d = data = (unsigned int *)malloc(hdr + (npts * sizeof(PGpoint)));
    if(!d)
      return 0;
  }

  if(oid == PATHOID)
  {
    *(char *)data = closed ? 1 : 0; /* path open/closed flag */
    d = (unsigned int *)(((char *)data) + 1);
  }

  /* write the number of points as an int32 */
  d[0] = (unsigned int)htonl(npts);

  /* assign points to the data 'd' buffer */
  for(i=0, d++; i < npts; i++, d+=4)
  {
    pqSwapi8(d,     &pts[i].x, 1);
    pqSwapi8(d + 2, &pts[i].y, 1);
  }

  /* put the data buffer, MUST include 'hdr' in the len */
  i = PQput(conn, data, hdr + (npts * sizeof(PGpoint)), oid, 1);

  /* if allocated from the heap, free it */
  if((char *)data != _stackbuffer)
    free(data);

  return i;
}

unsigned int *pqSwapi8(void *outp, void *inp, int tonet)
{
  unsigned int *in = (unsigned int *)inp;
  unsigned int *out = (unsigned int *)outp;

  /* swap when needed */
  if(1 != htonl(1))
  {
    out[0] = tonet ? htonl(in[1]) : ntohl(in[1]);
    out[1] = tonet ? htonl(in[0]) : ntohl(in[0]);
  }
  else
  {
    out[0] = in[0];
    out[1] = in[1];
  }

  return outp;
}



