//Written in the D programming language

/++
    Module contains Date/Time functionality.

    See_Also:
        <a href="http://en.wikipedia.org/wiki/ISO_8601">ISO 8601</a>

    Copyright: Copyright Jonathan M Davis 2010
    License:   <a href="http://www.boost.org/LICENSE_1_0.txt">Boost License 1.0</a>.
    Authors:   Jonathan M Davis

             Copyright Jonathan M Davis 2010.
    Distributed under the Boost Software License, Version 1.0.
       (See accompanying file LICENSE_1_0.txt or copy at
             http://www.boost.org/LICENSE_1_0.txt)
+/
module datetime;

import std.exception;
import std.math;
import std.string;
import std.traits;
import std.typecons;

version(unittest)
{
import std.stdio;
import unittests;
}


alias short  Year;              /// Unit for holding a Gregorian year.
alias ubyte  MonthOfYear;       /// Unit for holding a Gregorian month
alias ubyte  DayOfMonth;        /// Unit for holding the day of a Gregorian month.
alias ushort DayOfYear;         /// Unit for holding the day of a Gregorian year.
alias long   DayOfGregorianCal; /// Unit for holding Xth day of the Gregorian Calendar.
alias ushort ISOWeek;           /// Unit for holding the ISOWeek of a Gregorian year.

alias ubyte Hour;        /// Unit for holding the hour of the day.
alias ubyte Minute;      /// Unit for holding the minute of the hour.
alias ubyte Second;      /// Unit for holding second of the minute.
alias int   Millisecond; /// Unit for holding milliseconds of a second.
alias int   Microsecond; /// Unit for holding microseconds of a second.
alias long  Tick;        /// Unit for holding 100 nanoseconds.

/++
    Represents the 12 months of the Gregorian year.
  +/
enum { january = 1,
       february,
       march,
       april,
       may,
       june,
       july,
       august,
       september,
       october,
       november,
       december }

/++
    Represents the 7 days of the Gregorian month.
  +/
enum DayOfWeek : ushort { sunday = 0,
                          monday,
                          tuesday,
                          wednesday,
                          thursday,
                          friday,
                          saturday }

/++
    Represents possible units of time.
  +/
enum TimeUnit { year, month, day, week, hour, minute, second, millisecond, microsecond, tick}

/++
    In some Date calculations, adding months or years can cause the Date to fall on
    a day of the month which is not valid (e.g. February 29th 2001 or June 31st 2000).
    If overlflow is allowed (as is the default), then the month will be incremented
    accordingly (so, February 29th 2001 would become March 1st 2001, and June 31st 2000
    would become July 1st 2000). If overlow is not allowed, then the day will be adjusted
    to the last valid day in that month (so, February 29th 2001 becomes February 28th 2001
    and June 31st 2000 becomes June 30th 2000).

    AllowDayOverflow only applies to calculations involving months or years. Adding any
    smaller units (such as days) would just add the appropriate amount and adjust the
    month and year accordingly.
  +/
enum AllowDayOverflow
{
    yes,
    no
}


/++
    Effectively a namespace to better organize time conversion functions.
    Its functions relate to converting between the different TimeUnits.
 +/
final class TimeUnitConv
{
public:

    /++
        Generic way of converting between two time units. Conversions
        to smaller unites use truncating division.

        Params:
            tu1   = The units of time to covert from.
            tu1   = The units of time to covert type.
            value = The value to convert.

        Examples:
        --------------------
        assert(to!(TimeUnit.year, TimeUnit.month)(1) == 12);
        assert(to!(TimeUnit.month, TimeUnit.year)(12) == 1);
       --------------------
      +/
    static long to(TimeUnit tu1, TimeUnit tu2)(long value)
        if((tu1 == TimeUnit.year || tu1 == TimeUnit.month) &&
           (tu2 == TimeUnit.year || tu2 == TimeUnit.month))
    {
        static if(tu1 == TimeUnit.year)
        {
            static if(tu2 == TimeUnit.month)
                return yearsToMonths(value);
            else
                static assert(0, "A generic month or year cannot be converted smaller units.");
        }
        else static if(tu1 == TimeUnit.month)
        {
            static if(tu2 == TimeUnit.year)
                return monthsToYears(value);
            else
                static assert(0, "A generic month or year cannot be converted smaller units.");
        }
        else
            static assert(0, "A generic month or year cannot be converted smaller units.");
    }

    unittest
    {
        static assert(!__traits(compiles, to!(TimeUnit.year, TimeUnit.week)(12)));
        static assert(!__traits(compiles, to!(TimeUnit.year, TimeUnit.day)(12)));
        static assert(!__traits(compiles, to!(TimeUnit.year, TimeUnit.hour)(12)));
        static assert(!__traits(compiles, to!(TimeUnit.year, TimeUnit.second)(12)));
        static assert(!__traits(compiles, to!(TimeUnit.year, TimeUnit.millisecond)(12)));
        static assert(!__traits(compiles, to!(TimeUnit.year, TimeUnit.microsecond)(12)));
        static assert(!__traits(compiles, to!(TimeUnit.year, TimeUnit.tick)(12)));

        static assert(!__traits(compiles, to!(TimeUnit.week, TimeUnit.year)(12)));
        static assert(!__traits(compiles, to!(TimeUnit.day, TimeUnit.year)(12)));
        static assert(!__traits(compiles, to!(TimeUnit.hour, TimeUnit.year)(12)));
        static assert(!__traits(compiles, to!(TimeUnit.second, TimeUnit.year)(12)));
        static assert(!__traits(compiles, to!(TimeUnit.millisecond, TimeUnit.year)(12)));
        static assert(!__traits(compiles, to!(TimeUnit.microsecond, TimeUnit.year)(12)));
        static assert(!__traits(compiles, to!(TimeUnit.tick, TimeUnit.year)(12)));

        //Verify Examples
        assert(to!(TimeUnit.year, TimeUnit.month)(1) == 12);
        assert(to!(TimeUnit.month, TimeUnit.year)(12) == 1);
    }

    /++
        Generic way of converting between two time units. Conversions
        to smaller unites use truncating division.

        Params:
            tu1   = The units of time to covert from.
            tu1   = The units of time to covert type.
            value = The value to convert.

        Examples:
        --------------------
        assert(to!(TimeUnit.week, TimeUnit.day)(1) == 7);
        assert(to!(TimeUnit.hour, TimeUnit.second)(1) == 3600);
        assert(to!(TimeUnit.second, TimeUnit.day)(1) == 0);
        assert(to!(TimeUnit.second, TimeUnit.day)(86_400) == 1);
       --------------------
      +/
    static long to(TimeUnit tu1, TimeUnit tu2)(long value)
        if((tu1 == TimeUnit.week        ||
            tu1 == TimeUnit.day         ||
            tu1 == TimeUnit.hour        ||
            tu1 == TimeUnit.minute      ||
            tu1 == TimeUnit.second      ||
            tu1 == TimeUnit.millisecond ||
            tu1 == TimeUnit.microsecond ||
            tu1 == TimeUnit.tick) &&
           (tu2 == TimeUnit.week        ||
            tu2 == TimeUnit.day         ||
            tu2 == TimeUnit.hour        ||
            tu2 == TimeUnit.minute      ||
            tu2 == TimeUnit.second      ||
            tu2 == TimeUnit.millisecond ||
            tu2 == TimeUnit.microsecond ||
            tu2 == TimeUnit.tick))
    {
        static if(tu1 == TimeUnit.week)
            immutable numTicks = weeksToTicks(value);
        else static if(tu1 == TimeUnit.day)
            immutable numTicks = daysToTicks(value);
        else static if(tu1 == TimeUnit.hour)
            immutable numTicks = hoursToTicks(value);
        else static if(tu1 == TimeUnit.minute)
            immutable numTicks = minToTicks(value);
        else static if(tu1 == TimeUnit.second)
            immutable numTicks = secToTicks(value);
        else static if(tu1 == TimeUnit.millisecond)
            immutable numTicks = msToTicks(value);
        else static if(tu1 == TimeUnit.microsecond)
            immutable numTicks = usToTicks(value);
        else static if(tu1 == TimeUnit.tick)
            immutable numTicks = value;
        else
            static assert(0);

        static if(tu2 == TimeUnit.week)
            return ticksToWeeks(numTicks);
        else static if(tu2 == TimeUnit.day)
            return ticksToDays(numTicks);
        else static if(tu2 == TimeUnit.hour)
            return ticksToHours(numTicks);
        else static if(tu2 == TimeUnit.minute)
            return ticksToMin(numTicks);
        else static if(tu2 == TimeUnit.second)
            return ticksToSec(numTicks);
        else static if(tu2 == TimeUnit.millisecond)
            return ticksToMS(numTicks);
        else static if(tu2 == TimeUnit.microsecond)
            return ticksToUS(numTicks);
        else static if(tu2 == TimeUnit.tick)
            return numTicks;
        else
            static assert(0);
    }

    unittest
    {
        assertEqual(to!(TimeUnit.week, TimeUnit.tick)(1), 6_048_000_000_000L);
        assertEqual(to!(TimeUnit.day, TimeUnit.tick)(1), 864_000_000_000L);
        assertEqual(to!(TimeUnit.hour, TimeUnit.tick)(1), 36_000_000_000L);
        assertEqual(to!(TimeUnit.minute, TimeUnit.tick)(1), 600_000_000L);
        assertEqual(to!(TimeUnit.second, TimeUnit.tick)(1), 10_000_000L);
        assertEqual(to!(TimeUnit.millisecond, TimeUnit.tick)(1), 10_000);
        assertEqual(to!(TimeUnit.microsecond, TimeUnit.tick)(1), 10);

        assertEqual(to!(TimeUnit.tick, TimeUnit.week)(6_048_000_000_000L), 1);
        assertEqual(to!(TimeUnit.tick, TimeUnit.day)(864_000_000_000L), 1);
        assertEqual(to!(TimeUnit.tick, TimeUnit.hour)(36_000_000_000L), 1);
        assertEqual(to!(TimeUnit.tick, TimeUnit.minute)(600_000_000L), 1);
        assertEqual(to!(TimeUnit.tick, TimeUnit.second)(10_000_000L), 1);
        assertEqual(to!(TimeUnit.tick, TimeUnit.millisecond)(10_000), 1);
        assertEqual(to!(TimeUnit.tick, TimeUnit.microsecond)(10), 1);

        assertEqual(to!(TimeUnit.week, TimeUnit.day)(1), 7);
        assertEqual(to!(TimeUnit.day, TimeUnit.week)(7), 1);

        assertEqual(to!(TimeUnit.day, TimeUnit.hour)(1), 24);
        assertEqual(to!(TimeUnit.hour, TimeUnit.day)(24), 1);

        assertEqual(to!(TimeUnit.hour, TimeUnit.minute)(1), 60);
        assertEqual(to!(TimeUnit.minute, TimeUnit.hour)(60), 1);

        assertEqual(to!(TimeUnit.minute, TimeUnit.second)(1), 60);
        assertEqual(to!(TimeUnit.second, TimeUnit.minute)(60), 1);

        assertEqual(to!(TimeUnit.second, TimeUnit.millisecond)(1), 1000);
        assertEqual(to!(TimeUnit.millisecond, TimeUnit.second)(1000), 1);

        assertEqual(to!(TimeUnit.millisecond, TimeUnit.microsecond)(1), 1000);
        assertEqual(to!(TimeUnit.microsecond, TimeUnit.millisecond)(1000), 1);

        assertEqual(to!(TimeUnit.microsecond, TimeUnit.tick)(1), 10);
        assertEqual(to!(TimeUnit.tick, TimeUnit.microsecond)(10), 1);

        //Verify Examples
        assert(to!(TimeUnit.week, TimeUnit.day)(1) == 7);
        assert(to!(TimeUnit.hour, TimeUnit.second)(1) == 3600);
        assert(to!(TimeUnit.second, TimeUnit.day)(1) == 0);
        assert(to!(TimeUnit.second, TimeUnit.day)(86_400) == 1);
    }


    /++
        Convert from TimeUnit.year to TimeUnit.month.
      +/
    static long yearsToMonths(long years)
    {
        return years * 12;
    }

    /++
        Convert from TimeUnit.month to TimeUnit.month.
        Uses truncating division.
      +/
    static long monthsToYears(long months)
    {
        return months / 12;
    }


    /++
        Convert from TimeUnit.week to TimeUnit.tick.

        Params:
            value = The number of weeks to convert to ticks.
      +/
    static long weeksToTicks(long weeks)
    {
        return weeks * 10_000_000L * 60 * 60 * 24 * 7;
    }

    /++
        Convert from TimeUnit.day to TimeUnit.tick.

        Params:
            value = The number of days to convert to ticks.
      +/
    static long daysToTicks(long days)
    {
        return days * 10_000_000L * 60 * 60 * 24;
    }

    /++
        Convert from TimeUnit.hour to TimeUnit.tick.

        Params:
            value = The number of hours to convert to ticks.
      +/
    static long hoursToTicks(long hours)
    {
        return hours * 10_000_000L * 60 * 60;
    }

    /++
        Convert from TimeUnit.minute to TimeUnit.tick.

        Params:
            value = The number of minutes to convert to ticks.
      +/
    static long minToTicks(long minutes)
    {
        return minutes * 10_000_000L * 60;
    }

    /++
        Convert from TimeUnit.second to TimeUnit.tick.

        Params:
            value = The number of seconds to convert to ticks.
      +/
    static long secToTicks(long seconds)
    {
        return seconds * 10_000_000L;
    }

    /++
        Convert from TimeUnit.millisecond to TimeUnit.tick.

        Params:
            value = The number of milliseconds to convert to ticks.
      +/
    static long msToTicks(long ms)
    {
        return ms * 10_000;
    }

    /++
        Convert from TimeUnit.microsecond to TimeUnit.tick.

        Params:
            value = The number of microseconds to convert to ticks.
      +/
    static long usToTicks(long us)
    {
        return us * 10;
    }


    /++
        Convert from TimeUnit.tick to TimeUnit.week.
        Uses truncating division.

        Params:
            value = The number of ticks to convert to weeks.
      +/
    static long ticksToWeeks(long ticks)
    {
        return ticks / (7 * 24 * 60 * 60 * 10_000_000L);
    }

    /++
        Convert from TimeUnit.tick to TimeUnit.day.
        Uses truncating division.

        Params:
            value = The number of ticks to convert to days.
      +/
    static long ticksToDays(long ticks)
    {
        return ticks / (24 * 60 * 60 * 10_000_000L);
    }

    /++
        Convert from TimeUnit.tick to TimeUnit.hour.
        Uses truncating division.

        Params:
            value = The number of ticks to convert to hours.
      +/
    static long ticksToHours(long ticks)
    {
        return ticks / (60 * 60 * 10_000_000L);
    }

    /++
        Convert from TimeUnit.tick to TimeUnit.minute.
        Uses truncating division.

        Params:
            value = The number of ticks to convert to minutes.
      +/
    static long ticksToMin(long ticks)
    {
        return ticks / (60 * 10_000_000L);
    }

    /++
        Convert from TimeUnit.tick to TimeUnit.second.
        Uses truncating division.

        Params:
            value = The number of ticks to convert to seconds.
      +/
    static long ticksToSec(long ticks)
    {
        return ticks / 10_000_000L;
    }

    /++
        Convert from TimeUnit.tick to TimeUnit.millisecond.
        Uses truncating division.

        Params:
            value = The number of ticks to convert to milliseconds.
      +/
    static long ticksToMS(long ticks)
    {
        return ticks / 10_000;
    }

    /++
        Convert from TimeUnit.tick to TimeUnit.microsecond.
        Uses truncating division.

        Params:
            value = The number of ticks to convert to microseconds.
      +/
    static long ticksToUS(long ticks)
    {
        return ticks / 10;
    }


private:
    @disable this() {}
}


/++
    Whether the long display name or the short display name should be used.
  +/
enum UseLongName
{
    yes,
    no
}


/++
    Returns the given month as a string.

    Params:
        month   = The month to get the string for.
        useLong = Whether the long or short version of the month name should be used.

    Examples:
    --------------------
    assert(monthOfYearToString(january) == "January");
    assert(monthOfYearToString(january, UseLongName.no) == "Jan");
    assert(monthOfYearToString(july) == "July");
    assert(monthOfYearToString(july, UseLongName.no) == "Jul");
   --------------------
  +/
string monthOfYearToString(MonthOfYear month, UseLongName useLong = UseLongName.yes)
{
    if(useLong == UseLongName.yes)
    {
        switch(month)
        {
            case january:
                return "January";
            case february:
                return "February";
            case march:
                return "March";
            case april:
                return "April";
            case may:
                return "May";
            case june:
                return "June";
            case july:
                return "July";
            case august:
                return "August";
            case september:
                return "September";
            case october:
                return "October";
            case november:
                return "November";
            case december:
                return "December";
            default:
                assert(0);
        }
    }
    else
    {
        switch(month)
        {
            case january:
                return "Jan";
            case february:
                return "Feb";
            case march:
                return "Mar";
            case april:
                return "Apr";
            case may:
                return "May";
            case june:
                return "Jun";
            case july:
                return "Jul";
            case august:
                return "Aug";
            case september:
                return "Sep";
            case october:
                return "Oct";
            case november:
                return "Nov";
            case december:
                return "Dec";
            default:
                assert(0);
        }
    }
}

unittest
{
    //Verify Examples
    assert(monthOfYearToString(january) == "January");
    assert(monthOfYearToString(january, UseLongName.no) == "Jan");
    assert(monthOfYearToString(july) == "July");
    assert(monthOfYearToString(july, UseLongName.no) == "Jul");
}


/++
    Represents a date in the propletic Gregorian Calendar ranging from 32,768 B.C. to 32,767 A.D.
    Positive years are A.D. Non-positive years are B.C.

    Year, month, and day are kept separately internally so that Date is optimized for
    calendar operations.

    Date uses the propletic Gregorian Calendar, so it assumes the leap year calculations
    for its entire length. As part of that, it also treats 1 B.C. as year 0, so 1 B.C. is
    0, 2 B.C. is -1, etc. Use yearBC if want B.C. as a positive integer with 1 B.C. being
    the year prior to 1 A.D. Year 0 is a leap year.
 +/
struct Date
{
public:

    /++
        Params:
            year  = Year of the Gregorian Calendar. Positive values are A.D. Non-positive
                    values are B.C. with year 0 being the year prior to 1 A.D.
            month = Month of the year.
            day   = Day of the month.
     +/
    this(Year year, MonthOfYear month, DayOfMonth day)
    out
    {
        assert(!isSpecial);
    }
    body
    {
        enforce(validDate(year, month, day), "Arguments would result in an invalid Date");
        _year = year;
        _month = month;
        _day = day;
    }

    unittest
    {
        assertEqual(Date(1, 1, 1), Date.init);

        static void testDate(in Date date, Year year, MonthOfYear month, DayOfMonth day, size_t line = __LINE__)
        {
            assertEqual(date._year, year, "", __FILE__, line);
            assertEqual(date._month, month, "", __FILE__, line);
            assertEqual(date._day, day, "", __FILE__, line);
        }


        testDate(Date(1999, 1 , 1), 1999, january, 1);
        testDate(Date(1999, 7 , 1), 1999, july, 1);
        testDate(Date(1999, 7 , 6), 1999, july, 6);

        //Test A.D.
        assertExcThrown!(Exception, Date)(LineInfo(), cast(Year)1, cast(MonthOfYear)0, cast(DayOfMonth)1);
        assertExcThrown!(Exception, Date)(LineInfo(), cast(Year)1, cast(MonthOfYear)1, cast(DayOfMonth)0);
        assertExcThrown!(Exception, Date)(LineInfo(), cast(Year)1999, cast(MonthOfYear)13, cast(DayOfMonth)1);
        assertExcThrown!(Exception, Date)(LineInfo(), cast(Year)1999, cast(MonthOfYear)1, cast(DayOfMonth)32);
        assertExcThrown!(Exception, Date)(LineInfo(), cast(Year)1999, cast(MonthOfYear)2, cast(DayOfMonth)29);
        assertExcThrown!(Exception, Date)(LineInfo(), cast(Year)2000, cast(MonthOfYear)2, cast(DayOfMonth)30);
        assertExcThrown!(Exception, Date)(LineInfo(), cast(Year)1999, cast(MonthOfYear)3, cast(DayOfMonth)32);
        assertExcThrown!(Exception, Date)(LineInfo(), cast(Year)1999, cast(MonthOfYear)4, cast(DayOfMonth)31);
        assertExcThrown!(Exception, Date)(LineInfo(), cast(Year)1999, cast(MonthOfYear)5, cast(DayOfMonth)32);
        assertExcThrown!(Exception, Date)(LineInfo(), cast(Year)1999, cast(MonthOfYear)6, cast(DayOfMonth)31);
        assertExcThrown!(Exception, Date)(LineInfo(), cast(Year)1999, cast(MonthOfYear)7, cast(DayOfMonth)32);
        assertExcThrown!(Exception, Date)(LineInfo(), cast(Year)1999, cast(MonthOfYear)8, cast(DayOfMonth)32);
        assertExcThrown!(Exception, Date)(LineInfo(), cast(Year)1999, cast(MonthOfYear)9, cast(DayOfMonth)31);
        assertExcThrown!(Exception, Date)(LineInfo(), cast(Year)1999, cast(MonthOfYear)10, cast(DayOfMonth)32);
        assertExcThrown!(Exception, Date)(LineInfo(), cast(Year)1999, cast(MonthOfYear)11, cast(DayOfMonth)31);
        assertExcThrown!(Exception, Date)(LineInfo(), cast(Year)1999, cast(MonthOfYear)12, cast(DayOfMonth)32);

        assertExcNotThrown!(Exception, Date)(LineInfo(), cast(Year)1999, cast(MonthOfYear)1, cast(DayOfMonth)31);
        assertExcNotThrown!(Exception, Date)(LineInfo(), cast(Year)1999, cast(MonthOfYear)2, cast(DayOfMonth)28);
        assertExcNotThrown!(Exception, Date)(LineInfo(), cast(Year)2000, cast(MonthOfYear)2, cast(DayOfMonth)29);
        assertExcNotThrown!(Exception, Date)(LineInfo(), cast(Year)1999, cast(MonthOfYear)3, cast(DayOfMonth)31);
        assertExcNotThrown!(Exception, Date)(LineInfo(), cast(Year)1999, cast(MonthOfYear)4, cast(DayOfMonth)30);
        assertExcNotThrown!(Exception, Date)(LineInfo(), cast(Year)1999, cast(MonthOfYear)5, cast(DayOfMonth)31);
        assertExcNotThrown!(Exception, Date)(LineInfo(), cast(Year)1999, cast(MonthOfYear)6, cast(DayOfMonth)30);
        assertExcNotThrown!(Exception, Date)(LineInfo(), cast(Year)1999, cast(MonthOfYear)7, cast(DayOfMonth)31);
        assertExcNotThrown!(Exception, Date)(LineInfo(), cast(Year)1999, cast(MonthOfYear)8, cast(DayOfMonth)31);
        assertExcNotThrown!(Exception, Date)(LineInfo(), cast(Year)1999, cast(MonthOfYear)9, cast(DayOfMonth)30);
        assertExcNotThrown!(Exception, Date)(LineInfo(), cast(Year)1999, cast(MonthOfYear)10, cast(DayOfMonth)31);
        assertExcNotThrown!(Exception, Date)(LineInfo(), cast(Year)1999, cast(MonthOfYear)11, cast(DayOfMonth)30);
        assertExcNotThrown!(Exception, Date)(LineInfo(), cast(Year)1999, cast(MonthOfYear)12, cast(DayOfMonth)31);

        //Test B.C.
        assertExcNotThrown!(Exception, Date)(LineInfo(), cast(Year)0, cast(MonthOfYear)1, cast(DayOfMonth)1);
        assertExcNotThrown!(Exception, Date)(LineInfo(), cast(Year)-1, cast(MonthOfYear)1, cast(DayOfMonth)1);
        assertExcNotThrown!(Exception, Date)(LineInfo(), cast(Year)-1, cast(MonthOfYear)12, cast(DayOfMonth)31);
        assertExcNotThrown!(Exception, Date)(LineInfo(), cast(Year)-1, cast(MonthOfYear)2, cast(DayOfMonth)28);
        assertExcNotThrown!(Exception, Date)(LineInfo(), cast(Year)-4, cast(MonthOfYear)2, cast(DayOfMonth)29);

        assertExcThrown!(Exception, Date)(LineInfo(), cast(Year)-1, cast(MonthOfYear)2, cast(DayOfMonth)29);
        assertExcThrown!(Exception, Date)(LineInfo(), cast(Year)-2, cast(MonthOfYear)2, cast(DayOfMonth)29);
        assertExcThrown!(Exception, Date)(LineInfo(), cast(Year)-3, cast(MonthOfYear)2, cast(DayOfMonth)29);
    }


    /++
       Compares this Date with the given Date.

       Returns:
        $(TABLE
        $(TR $(TD this &lt; rhs) $(TD &lt; 0))
        $(TR $(TD this == rhs) $(TD 0))
        $(TR $(TD this &gt; rhs) $(TD &gt; 0))
        )
     +/
    int opCmp(in Date rhs) const
    {
        if(isSpecial || rhs.isSpecial)
        {
            enforceIsADate();
            rhs.enforceIsADate();

            if(isPosInfinity)
            {
                if(rhs.isPosInfinity)
                    return 0;

                return 1;
            }

            if(isNegInfinity)
            {
                if(rhs.isNegInfinity)
                    return 0;

                return -1;
            }

            if(rhs.isPosInfinity)
                return -1;

            return 1;
        }

        if(_year < rhs._year)
            return -1;
        if(_year > rhs._year)
            return 1;

        if(_month < rhs._month)
            return -1;
        if(_month > rhs._month)
            return 1;

        if(_day < rhs._day)
            return -1;
        if(_day > rhs._day)
            return 1;

        return 0;
    }

    unittest
    {
        assertExcThrown!(Exception, (){notADate.opCmp(posInfinity);})(LineInfo());
        assertExcThrown!(Exception, (){notADate.opCmp(negInfinity);})(LineInfo());
        assertExcThrown!(Exception, (){notADate.opCmp(notADate);})(LineInfo());
        assertExcThrown!(Exception, (){notADate.opCmp(Date(1999, 7, 6));})(LineInfo());
        assertExcThrown!(Exception, (){posInfinity.opCmp(notADate);})(LineInfo());
        assertExcThrown!(Exception, (){negInfinity.opCmp(notADate);})(LineInfo());
        assertExcThrown!(Exception, (){Date(1999, 7, 6).opCmp(notADate);})(LineInfo());

        assertOpCmp!("==")(posInfinity, posInfinity);
        assertOpCmp!("==")(negInfinity, negInfinity);
        assertOpCmp!("<")(negInfinity, posInfinity);
        assertOpCmp!(">")(posInfinity, negInfinity);

        //Test A.D.
        assertOpCmp!("==")(Date(1, 1, 1), Date.init);

        assertOpCmp!("==")(Date(1999, 1, 1), Date(1999, 1, 1));
        assertOpCmp!("==")(Date(1, 7, 1), Date(1, 7, 1));
        assertOpCmp!("==")(Date(1, 1, 6), Date(1, 1, 6));

        assertOpCmp!("==")(Date(1999, 7, 1), Date(1999, 7, 1));
        assertOpCmp!("==")(Date(1999, 7, 6), Date(1999, 7, 6));

        assertOpCmp!("==")(Date(1, 7, 6), Date(1, 7, 6));

        assertOpCmp!("<")(Date(1999, 7, 6), Date(2000, 7, 6));
        assertOpCmp!(">")(Date(2000, 7, 6), Date(1999, 7, 6));
        assertOpCmp!("<")(Date(1999, 7, 6), Date(1999, 8, 6));
        assertOpCmp!(">")(Date(1999, 8, 6), Date(1999, 7, 6));
        assertOpCmp!("<")(Date(1999, 7, 6), Date(1999, 7, 7));
        assertOpCmp!(">")(Date(1999, 7, 7), Date(1999, 7, 6));

        assertOpCmp!("<")(Date(1999, 8, 7), Date(2000, 7, 6));
        assertOpCmp!(">")(Date(2000, 8, 6), Date(1999, 7, 7));
        assertOpCmp!("<")(Date(1999, 7, 7), Date(2000, 7, 6));
        assertOpCmp!(">")(Date(2000, 7, 6), Date(1999, 7, 7));
        assertOpCmp!("<")(Date(1999, 7, 7), Date(1999, 8, 6));
        assertOpCmp!(">")(Date(1999, 8, 6), Date(1999, 7, 7));

        assertOpCmp!("<")(Date(1999, 7, 6), posInfinity);
        assertOpCmp!(">")(posInfinity, Date(1999, 7, 6));
        assertOpCmp!("<")(negInfinity, Date(1999, 7, 6));
        assertOpCmp!(">")(Date(1999, 7, 6), negInfinity);

        //Test B.C.
        assertOpCmp!("==")(Date(0, 1, 1), Date(0, 1, 1));
        assertOpCmp!("==")(Date(-1, 1, 1), Date(-1, 1, 1));
        assertOpCmp!("==")(Date(-1, 7, 1), Date(-1, 7, 1));
        assertOpCmp!("==")(Date(-1, 1, 6), Date(-1, 1, 6));

        assertOpCmp!("==")(Date(-1999, 7, 1), Date(-1999, 7, 1));
        assertOpCmp!("==")(Date(-1999, 7, 6), Date(-1999, 7, 6));

        assertOpCmp!("==")(Date(-1, 7, 6), Date(-1, 7, 6));

        assertOpCmp!("<")(Date(-2000, 7, 6), Date(-1999, 7, 6));
        assertOpCmp!(">")(Date(-1999, 7, 6), Date(-2000, 7, 6));
        assertOpCmp!("<")(Date(-1999, 7, 6), Date(-1999, 8, 6));
        assertOpCmp!(">")(Date(-1999, 8, 6), Date(-1999, 7, 6));
        assertOpCmp!("<")(Date(-1999, 7, 6), Date(-1999, 7, 7));
        assertOpCmp!(">")(Date(-1999, 7, 7), Date(-1999, 7, 6));

        assertOpCmp!("<")(Date(-2000, 8, 6), Date(-1999, 7, 7));
        assertOpCmp!(">")(Date(-1999, 8, 7), Date(-2000, 7, 6));
        assertOpCmp!("<")(Date(-2000, 7, 6), Date(-1999, 7, 7));
        assertOpCmp!(">")(Date(-1999, 7, 7), Date(-2000, 7, 6));
        assertOpCmp!("<")(Date(-1999, 7, 7), Date(-1999, 8, 6));
        assertOpCmp!(">")(Date(-1999, 8, 6), Date(-1999, 7, 7));

        assertOpCmp!("<")(Date(-1999, 7, 6), posInfinity);
        assertOpCmp!(">")(posInfinity, Date(-1999, 7, 6));
        assertOpCmp!("<")(negInfinity, Date(-1999, 7, 6));
        assertOpCmp!(">")(Date(-1999, 7, 6), negInfinity);

        //Test Both
        assertOpCmp!("<")(Date(-1999, 7, 6), Date(1999, 7, 6));
        assertOpCmp!(">")(Date(1999, 7, 6), Date(-1999, 7, 6));

        assertOpCmp!("<")(Date(-1999, 8, 6), Date(1999, 7, 6));
        assertOpCmp!(">")(Date(1999, 7, 6), Date(-1999, 8, 6));

        assertOpCmp!("<")(Date(-1999, 7, 7), Date(1999, 7, 6));
        assertOpCmp!(">")(Date(1999, 7, 6), Date(-1999, 7, 7));

        assertOpCmp!("<")(Date(-1999, 8, 7), Date(1999, 7, 6));
        assertOpCmp!(">")(Date(1999, 7, 6), Date(-1999, 8, 7));

        assertOpCmp!("<")(Date(-1999, 8, 6), Date(1999, 6, 6));
        assertOpCmp!(">")(Date(1999, 6, 8), Date(-1999, 7, 6));

        const cdate = Date(1999, 7, 6);
        immutable idate = Date(1999, 7, 6);
        static assert(__traits(compiles, cdate.opCmp(idate)));
        static assert(__traits(compiles, idate.opCmp(cdate)));
    }


    /++
        Year of the Gregorian Calendar. Positive numbers are A.D. Non-positive are B.C.
     +/
    @property Year year() const
    {
        enforceNotSpecial();

        return _year;
    }

    unittest
    {
        assertEqual(Date.init.year, 1);
        assertEqual(Date(1999, 7, 6).year, 1999);
        assertEqual(Date(-1999, 7, 6).year, -1999);

        assertExcThrown!(Exception, (){posInfinity.year;})(LineInfo());
        assertExcThrown!(Exception, (){negInfinity.year;})(LineInfo());
        assertExcThrown!(Exception, (){notADate.year;})(LineInfo());

        const cdate = Date(1999, 7, 6);
        static assert(__traits(compiles, cdate.year == 1999));
        immutable idate = Date(1999, 7, 6);
        static assert(__traits(compiles, cdate.year == 1999));
    }

    /++
        Year of the Gregorian Calendar. Positive numbers are A.D. Non-positive are B.C.

        Params:
            year = The year to set this Date's year to.

        Examples:
        --------------------
        assert(Date(1999, 7, 6).year == 1999);
        assert(Date(2010, 10, 4).year == 2010);
        assert(Date(-7, 4, 5).year == -7);
        --------------------
     +/
    @property void year(Year year)
    {
        enforceNotSpecial();

        _year = year;
    }

    unittest
    {
        static void testDate(Date date, int year)
        {
            date.year = cast(Year)year;
        }

        {
            auto date = Date(1, 1, 1);
            date.year = 1999;
            assertEqual(date, Date(1999, 1, 1));
        }

        {
            auto date = Date(1, 1, 1);
            date.year = 0;
            assertEqual(date, Date(0, 1, 1));
        }

        {
            auto date = Date(1, 1, 1);
            date.year = -1999;
            assertEqual(date, Date(-1999, 1, 1));
        }

        assertExcThrown!(Exception, (){posInfinity.year = 7;})(LineInfo());
        assertExcThrown!(Exception, (){negInfinity.year = 7;})(LineInfo());
        assertExcThrown!(Exception, (){notADate.year = 7;})(LineInfo());

        const cdate = Date(1999, 7, 6);
        static assert(!__traits(compiles, cdate.year = 1999));
        immutable idate = Date(1999, 7, 6);
        static assert(!__traits(compiles, cdate.year = 1999));

        //Verify Examples
        assert(Date(1999, 7, 6).year == 1999);
        assert(Date(2010, 10, 4).year == 2010);
        assert(Date(-7, 4, 5).year == -7);
    }


    /++
        Year B.C. of the Gregorian Calendar counting year 0 as 1 B.C. An exception
        is thrown if isAD is true.

        Examples:
        --------------------
        assert(Date(0, 1, 1).yearBC == 1);
        assert(Date(-1, 1, 1).yearBC == 2);
        assert(Date(-100, 1, 1).yearBC == 101);
        --------------------
     +/
    @property Year yearBC() const
    {
        enforceNotSpecial();
        enforce(!isAD, format("%s is A.D.", this));

        return cast(Year)((_year * -1) + 1);
    }

    unittest
    {
        assertExcThrown!(Exception, (in Date date){date.yearBC;})(LineInfo(), Date(1, 1, 1));

        assertExcThrown!(Exception, (){posInfinity.yearBC;})(LineInfo());
        assertExcThrown!(Exception, (){negInfinity.yearBC;})(LineInfo());
        assertExcThrown!(Exception, (){notADate.yearBC;})(LineInfo());

        auto date = Date(0, 7, 6);
        const cdate = Date(0, 7, 6);
        immutable idate = Date(0, 7, 6);
        static assert(__traits(compiles, date.yearBC));
        static assert(__traits(compiles, cdate.yearBC));
        static assert(__traits(compiles, idate.yearBC));

        //Verify Examples
        assert(Date(0, 1, 1).yearBC == 1);
        assert(Date(-1, 1, 1).yearBC == 2);
        assert(Date(-100, 1, 1).yearBC == 101);
    }


    /++
        Year B.C. of the Gregorian Calendar counting year 0 as 1 B.C. An exception
        is thrown if a non-positive value is given.

        Params:
            year = The year B.C. to set this Date's year to.

        Examples:
        --------------------
        auto date = Date(2010, 1, 1);
        date.yearBC = 1;
        assert(date == Date(0, 1, 1));

        date.yearBC = 10;
        assert(date == Date(-9, 1, 1));
        --------------------
     +/
    @property void yearBC(Year year)
    {
        enforceNotSpecial();
        enforce(year > 0, "The given year is not a year B.C.");

        _year = cast(Year)((year - 1) * -1);
    }

    unittest
    {
        assertExcThrown!(Exception, (Date date){date.yearBC = -1;})(LineInfo(), Date(1, 1, 1));

        assertExcThrown!(Exception, (){posInfinity.yearBC = 1;})(LineInfo());
        assertExcThrown!(Exception, (){negInfinity.yearBC = 1;})(LineInfo());
        assertExcThrown!(Exception, (){notADate.yearBC = 1;})(LineInfo());

        {
            auto date = Date(0, 7, 6);
            const cdate = Date(0, 7, 6);
            immutable idate = Date(0, 7, 6);
            static assert(__traits(compiles, date.yearBC = 7));
            static assert(!__traits(compiles, cdate.yearBC = 7));
            static assert(!__traits(compiles, idate.yearBC = 7));
        }

        //Verify Examples
        {
            auto date = Date(2010, 1, 1);
            date.yearBC = 1;
            assert(date == Date(0, 1, 1));

            date.yearBC = 10;
            assert(date == Date(-9, 1, 1));
        }
    }


    /++
        Month of a Gregorian Year.

        Examples:
        --------------------
        assert(Date(1999, 7, 6).month == 7);
        assert(Date(2010, 10, 4).month == 10);
        assert(Date(-7, 4, 5).month == 4);
        --------------------
     +/
    @property MonthOfYear month() const
    {
        enforceNotSpecial();

        return _month;
    }

    unittest
    {
        assertEqual(Date.init.month, 1);
        assertEqual(Date(1999, 7, 6).month, 7);
        assertEqual(Date(-1999, 7, 6).month, 7);

        assertExcThrown!(Exception, (){posInfinity.month;})(LineInfo());
        assertExcThrown!(Exception, (){negInfinity.month;})(LineInfo());
        assertExcThrown!(Exception, (){notADate.month;})(LineInfo());

        const cdate = Date(1999, 7, 6);
        static assert(__traits(compiles, cdate.month == 7));
        immutable idate = Date(1999, 7, 6);
        static assert(__traits(compiles, cdate.month == 7));

        //Verify Examples
        assert(Date(1999, 7, 6).month == 7);
        assert(Date(2010, 10, 4).month == 10);
        assert(Date(-7, 4, 5).month == 4);
    }

    /++
        Month of a Gregorian Year.

        Params:
            mounth = The month to set this Date's month to.
     +/
    @property void month(MonthOfYear month)
    {
        enforceNotSpecial();
        enforce(validMonthOfYear(month), "Invalid month");

        _month = month;
    }

    unittest
    {
        static void testDate(Date date, int month)
        {
            date.month = cast(MonthOfYear)month;
        }

        assertExcThrown!(Exception, testDate)(LineInfo(), Date(1, 1, 1), 0);
        assertExcThrown!(Exception, testDate)(LineInfo(), Date(1, 1, 1), 13);

        {
            auto date = Date(1, 1, 1);
            date.month = 7;
            assertEqual(date, Date(1, 7, 1));
        }

        {
            auto date = Date(-1, 1, 1);
            date.month = 7;
            assertEqual(date, Date(-1, 7, 1));
        }

        assertExcThrown!(Exception, (){posInfinity.month = 4;})(LineInfo());
        assertExcThrown!(Exception, (){negInfinity.month = 4;})(LineInfo());
        assertExcThrown!(Exception, (){notADate.month = 4;})(LineInfo());

        const cdate = Date(1999, 7, 6);
        static assert(!__traits(compiles, cdate.month = 7));
        immutable idate = Date(1999, 7, 6);
        static assert(!__traits(compiles, cdate.month = 7));
    }


    /++
        Day of a Gregorian Month.

        Examples:
        --------------------
        assert(Date(1999, 7, 6).day == 6);
        assert(Date(2010, 10, 4).day == 4);
        assert(Date(-7, 4, 5).day == 5);
        --------------------
     +/
    @property DayOfMonth day() const
    {
        enforceNotSpecial();

        return _day;
    }

    unittest
    {
        assertEqual(Date.init.day, 1);
        assertEqual(Date(1999, 7, 6).day, 6);
        assertEqual(Date(-1999, 7, 6).day, 6);

        assertExcThrown!(Exception, (){posInfinity.day;})(LineInfo());
        assertExcThrown!(Exception, (){negInfinity.day;})(LineInfo());
        assertExcThrown!(Exception, (){notADate.day;})(LineInfo());

        const cdate = Date(1999, 7, 6);
        static assert(__traits(compiles, cdate.day == 6));
        immutable idate = Date(1999, 7, 6);
        static assert(__traits(compiles, cdate.day == 6));

        //Verify Examples
        assert(Date(1999, 7, 6).day == 6);
        assert(Date(2010, 10, 4).day == 4);
        assert(Date(-7, 4, 5).day == 5);
    }

    /++
        Day of a Gregorian Month.

        Params:
            day = The day of the month to set this Date's day to.
     +/
    @property void day(DayOfMonth day)
    {
        enforceNotSpecial();
        enforce(validDayOfMonth(_year, _month, day), "Invalid day");
        _day = day;
    }

    unittest
    {
        static void testDate(Date date, int day)
        {
            date.day = cast(DayOfMonth)day;
        }

        //Test A.D.
        assertExcThrown!(Exception, testDate)(LineInfo(), Date(1, 1, 1), 0);
        assertExcThrown!(Exception, testDate)(LineInfo(), Date(1, 1, 1), 32);
        assertExcThrown!(Exception, testDate)(LineInfo(), Date(1, 2, 1), 29);
        assertExcThrown!(Exception, testDate)(LineInfo(), Date(4, 2, 1), 30);
        assertExcThrown!(Exception, testDate)(LineInfo(), Date(1, 3, 1), 32);
        assertExcThrown!(Exception, testDate)(LineInfo(), Date(1, 4, 1), 31);
        assertExcThrown!(Exception, testDate)(LineInfo(), Date(1, 5, 1), 32);
        assertExcThrown!(Exception, testDate)(LineInfo(), Date(1, 6, 1), 31);
        assertExcThrown!(Exception, testDate)(LineInfo(), Date(1, 7, 1), 32);
        assertExcThrown!(Exception, testDate)(LineInfo(), Date(1, 8, 1), 32);
        assertExcThrown!(Exception, testDate)(LineInfo(), Date(1, 9, 1), 31);
        assertExcThrown!(Exception, testDate)(LineInfo(), Date(1, 10, 1), 32);
        assertExcThrown!(Exception, testDate)(LineInfo(), Date(1, 11, 1), 31);
        assertExcThrown!(Exception, testDate)(LineInfo(), Date(1, 12, 1), 32);

        assertExcNotThrown!(Exception, testDate)(LineInfo(), Date(1, 1, 1), 31);
        assertExcNotThrown!(Exception, testDate)(LineInfo(), Date(1, 2, 1), 28);
        assertExcNotThrown!(Exception, testDate)(LineInfo(), Date(4, 2, 1), 29);
        assertExcNotThrown!(Exception, testDate)(LineInfo(), Date(1, 3, 1), 31);
        assertExcNotThrown!(Exception, testDate)(LineInfo(), Date(1, 4, 1), 30);
        assertExcNotThrown!(Exception, testDate)(LineInfo(), Date(1, 5, 1), 31);
        assertExcNotThrown!(Exception, testDate)(LineInfo(), Date(1, 6, 1), 30);
        assertExcNotThrown!(Exception, testDate)(LineInfo(), Date(1, 7, 1), 31);
        assertExcNotThrown!(Exception, testDate)(LineInfo(), Date(1, 8, 1), 31);
        assertExcNotThrown!(Exception, testDate)(LineInfo(), Date(1, 9, 1), 30);
        assertExcNotThrown!(Exception, testDate)(LineInfo(), Date(1, 10, 1), 31);
        assertExcNotThrown!(Exception, testDate)(LineInfo(), Date(1, 11, 1), 30);
        assertExcNotThrown!(Exception, testDate)(LineInfo(), Date(1, 12, 1), 31);

        {
            auto date = Date(1, 1, 1);
            date.day = 6;
            assertEqual(date, Date(1, 1, 6));
        }

        //Test B.C.
        assertExcThrown!(Exception, testDate)(LineInfo(), Date(-1, 1, 1), 0);
        assertExcThrown!(Exception, testDate)(LineInfo(), Date(-1, 1, 1), 32);
        assertExcThrown!(Exception, testDate)(LineInfo(), Date(-1, 2, 1), 29);
        assertExcThrown!(Exception, testDate)(LineInfo(), Date(0, 2, 1), 30);
        assertExcThrown!(Exception, testDate)(LineInfo(), Date(-1, 3, 1), 32);
        assertExcThrown!(Exception, testDate)(LineInfo(), Date(-1, 4, 1), 31);
        assertExcThrown!(Exception, testDate)(LineInfo(), Date(-1, 5, 1), 32);
        assertExcThrown!(Exception, testDate)(LineInfo(), Date(-1, 6, 1), 31);
        assertExcThrown!(Exception, testDate)(LineInfo(), Date(-1, 7, 1), 32);
        assertExcThrown!(Exception, testDate)(LineInfo(), Date(-1, 8, 1), 32);
        assertExcThrown!(Exception, testDate)(LineInfo(), Date(-1, 9, 1), 31);
        assertExcThrown!(Exception, testDate)(LineInfo(), Date(-1, 10, 1), 32);
        assertExcThrown!(Exception, testDate)(LineInfo(), Date(-1, 11, 1), 31);
        assertExcThrown!(Exception, testDate)(LineInfo(), Date(-1, 12, 1), 32);

        assertExcNotThrown!(Exception, testDate)(LineInfo(), Date(-1, 1, 1), 31);
        assertExcNotThrown!(Exception, testDate)(LineInfo(), Date(-1, 2, 1), 28);
        assertExcNotThrown!(Exception, testDate)(LineInfo(), Date(0, 2, 1), 29);
        assertExcNotThrown!(Exception, testDate)(LineInfo(), Date(-1, 3, 1), 31);
        assertExcNotThrown!(Exception, testDate)(LineInfo(), Date(-1, 4, 1), 30);
        assertExcNotThrown!(Exception, testDate)(LineInfo(), Date(-1, 5, 1), 31);
        assertExcNotThrown!(Exception, testDate)(LineInfo(), Date(-1, 6, 1), 30);
        assertExcNotThrown!(Exception, testDate)(LineInfo(), Date(-1, 7, 1), 31);
        assertExcNotThrown!(Exception, testDate)(LineInfo(), Date(-1, 8, 1), 31);
        assertExcNotThrown!(Exception, testDate)(LineInfo(), Date(-1, 9, 1), 30);
        assertExcNotThrown!(Exception, testDate)(LineInfo(), Date(-1, 10, 1), 31);
        assertExcNotThrown!(Exception, testDate)(LineInfo(), Date(-1, 11, 1), 30);
        assertExcNotThrown!(Exception, testDate)(LineInfo(), Date(-1, 12, 1), 31);

        {
            auto date = Date(-1, 1, 1);
            date.day = 6;
            assertEqual(date, Date(-1, 1, 6));
        }

        assertExcThrown!(Exception, (){posInfinity.day = 5;})(LineInfo());
        assertExcThrown!(Exception, (){negInfinity.day = 5;})(LineInfo());
        assertExcThrown!(Exception, (){notADate.day = 5;})(LineInfo());

        const cdate = Date(1999, 7, 6);
        static assert(!__traits(compiles, cdate.day = 6));
        immutable idate = Date(1999, 7, 6);
        static assert(!__traits(compiles, cdate.day = 6));
    }


    /++
        Adds the given number of years to this Date. A negative number will subtract.

        Note that if day overflow is allowed, and the date is Febuary 29th of a leap year,
        and the new year is not a leap year, then the date is shifted to March 1st. If
        day overflow is not allowed, then the date is shifted to February 28th.

        Params:
            years         = The number of years to add to this Date.
            allowOverflow = Whether the days should be allowed to overflow, causing the month to increment.

        Examples:
        --------------------
        auto date1 = Date(2010, 1, 1);
        date1.addYears(1);
        assert(date1 == Date(2011, 1, 1));

        auto date2 = Date(2010, 1, 1);
        date2.addYears(-1);
        assert(date2 == Date(2009, 1, 1));

        auto date3 = Date(2000, 2, 29);
        date3.addYears(1);
        assert(date3 == Date(2001, 3, 1));

        auto date4 = Date(2000, 2, 29);
        date4.addYears(1, AllowDayOverflow.no);
        assert(date4 == Date(2001, 2, 28));
        --------------------
      +/
    void addYears(int years, AllowDayOverflow allowOverflow = AllowDayOverflow.yes)
    {
        enforceIsADate();

        if(isInfinity)
            return;

        immutable newYear = _year + years;

        _year += years;

        if(_month == february && _day == 29 && !yearIsLeapYear(_year))
        {
            if(allowOverflow == AllowDayOverflow.yes)
            {
                _month = march;
                _day = 1;
            }
            else
                _day = 28;
        }
    }

    //Test addYears with AllowDayOverlow.yes
    unittest
    {
        //Test A.D.
        {
            auto date = Date(1999, 7, 6);
            date.addYears(7);
            assertEqual(date, Date(2006, 7, 6));
            date.addYears(-9);
            assertEqual(date, Date(1997, 7, 6));
        }

        {
            auto date = Date(1999, 2, 28);
            date.addYears(1);
            assertEqual(date, Date(2000, 2, 28));
        }

        {
            auto date = Date(2000, 2, 29);
            date.addYears(-1);
            assertEqual(date, Date(1999, 3, 1));
        }

        //Test B.C.
        {
            auto date = Date(-1999, 7, 6);
            date.addYears(-7);
            assertEqual(date, Date(-2006, 7, 6));
            date.addYears(9);
            assertEqual(date, Date(-1997, 7, 6));
        }

        {
            auto date = Date(-1999, 2, 28);
            date.addYears(-1);
            assertEqual(date, Date(-2000, 2, 28));
        }

        {
            auto date = Date(-2000, 2, 29);
            date.addYears(1);
            assertEqual(date, Date(-1999, 3, 1));
        }

        //Test Both
        {
            auto date = Date(4, 7, 6);
            date.addYears(-5);
            assertEqual(date, Date(-1, 7, 6));
            date.addYears(5);
            assertEqual(date, Date(4, 7, 6));
        }

        {
            auto date = Date(-4, 7, 6);
            date.addYears(5);
            assertEqual(date, Date(1, 7, 6));
            date.addYears(-5);
            assertEqual(date, Date(-4, 7, 6));
        }

        {
            auto date = Date(4, 7, 6);
            date.addYears(-8);
            assertEqual(date, Date(-4, 7, 6));
            date.addYears(8);
            assertEqual(date, Date(4, 7, 6));
        }

        {
            auto date = Date(-4, 7, 6);
            date.addYears(8);
            assertEqual(date, Date(4, 7, 6));
            date.addYears(-8);
            assertEqual(date, Date(-4, 7, 6));
        }

        {
            auto date = Date(-4, 2, 29);
            date.addYears(5);
            assertEqual(date, Date(1, 3, 1));
        }

        {
            auto date = Date(4, 2, 29);
            date.addYears(-5);
            assertEqual(date, Date(-1, 3, 1));
        }

        assertExcNotThrown!(Exception, (){posInfinity.addYears(5);})(LineInfo());
        assertExcNotThrown!(Exception, (){negInfinity.addYears(5);})(LineInfo());
        assertExcThrown!(Exception, (){notADate.addYears(5);})(LineInfo());

        {
            auto date = posInfinity;
            posInfinity.addYears(5);
            assertEqual(date, posInfinity);
        }

        {
            auto date = negInfinity;
            negInfinity.addYears(5);
            assertEqual(date, negInfinity);
        }

        const cdate = Date(1999, 7, 6);
        static assert(!__traits(compiles, cdate.addYears(7)));
        immutable idate = Date(1999, 7, 6);
        static assert(!__traits(compiles, idate.addYears(7)));

        //Verify Examples
        auto date1 = Date(2010, 1, 1);
        date1.addYears(1);
        assert(date1 == Date(2011, 1, 1));

        auto date2 = Date(2010, 1, 1);
        date2.addYears(-1);
        assert(date2 == Date(2009, 1, 1));

        auto date3 = Date(2000, 2, 29);
        date3.addYears(1);
        assert(date3 == Date(2001, 3, 1));

        auto date4 = Date(2000, 2, 29);
        date4.addYears(1, AllowDayOverflow.no);
        assert(date4 == Date(2001, 2, 28));
    }

    //Test addYears with AllowDayOverlow.no
    unittest
    {
        //Test A.D.
        {
            auto date = Date(1999, 7, 6);
            date.addYears(7, AllowDayOverflow.no);
            assertEqual(date, Date(2006, 7, 6));
            date.addYears(-9, AllowDayOverflow.no);
            assertEqual(date, Date(1997, 7, 6));
        }

        {
            auto date = Date(1999, 2, 28);
            date.addYears(1, AllowDayOverflow.no);
            assertEqual(date, Date(2000, 2, 28));
        }

        {
            auto date = Date(2000, 2, 29);
            date.addYears(-1, AllowDayOverflow.no);
            assertEqual(date, Date(1999, 2, 28));
        }

        //Test B.C.
        {
            auto date = Date(-1999, 7, 6);
            date.addYears(-7, AllowDayOverflow.no);
            assertEqual(date, Date(-2006, 7, 6));
            date.addYears(9, AllowDayOverflow.no);
            assertEqual(date, Date(-1997, 7, 6));
        }

        {
            auto date = Date(-1999, 2, 28);
            date.addYears(-1, AllowDayOverflow.no);
            assertEqual(date, Date(-2000, 2, 28));
        }

        {
            auto date = Date(-2000, 2, 29);
            date.addYears(1, AllowDayOverflow.no);
            assertEqual(date, Date(-1999, 2, 28));
        }

        //Test Both
        {
            auto date = Date(4, 7, 6);
            date.addYears(-5, AllowDayOverflow.no);
            assertEqual(date, Date(-1, 7, 6));
            date.addYears(5, AllowDayOverflow.no);
            assertEqual(date, Date(4, 7, 6));
        }

        {
            auto date = Date(-4, 7, 6);
            date.addYears(5, AllowDayOverflow.no);
            assertEqual(date, Date(1, 7, 6));
            date.addYears(-5, AllowDayOverflow.no);
            assertEqual(date, Date(-4, 7, 6));
        }

        {
            auto date = Date(4, 7, 6);
            date.addYears(-8, AllowDayOverflow.no);
            assertEqual(date, Date(-4, 7, 6));
            date.addYears(8, AllowDayOverflow.no);
            assertEqual(date, Date(4, 7, 6));
        }

        {
            auto date = Date(-4, 7, 6);
            date.addYears(8, AllowDayOverflow.no);
            assertEqual(date, Date(4, 7, 6));
            date.addYears(-8, AllowDayOverflow.no);
            assertEqual(date, Date(-4, 7, 6));
        }

        {
            auto date = Date(-4, 2, 29);
            date.addYears(5, AllowDayOverflow.no);
            assertEqual(date, Date(1, 2, 28));
        }

        {
            auto date = Date(4, 2, 29);
            date.addYears(-5, AllowDayOverflow.no);
            assertEqual(date, Date(-1, 2, 28));
        }
    }

    /++
        Adds the given number of months to this Date. A negative number will subtract.

        The year will be adjusted along with the month if the number of months added (or subtracted)
        would overflow (or underflow) the current year.

        Note that if day overflow is allowed, and the date with the adjusted month overflows the number
        of days in the new month, then the month will be incremented by one, and the days set to the number
        of days overflowed. (e.g. if the day were 31 and the new month were June, then the month
        would be incremented to July, and the new day would be 1). If day overflow is not allowed,
        then the day will be set to the last valid day in the month (e.g. June 31st would become June 30th).

        Params:
            months        = The number of months to add to this Date.
            allowOverflow = Whether the days should be allowed to overflow, causing the month to increment.

        Examples:
        --------------------
        auto date1 = Date(2010, 1, 1);
        date1.addMonths(1);
        assert(date1 == Date(2010, 2, 1));

        auto date2 = Date(2010, 1, 1);
        date2.addMonths(-1);
        assert(date2 == Date(2009, 12, 1));

        auto date3 = Date(1999, 1, 29);
        date3.addMonths(1);
        assert(date3 == Date(1999, 3, 1));

        auto date4 = Date(1999, 1, 29);
        date4.addMonths(1, AllowDayOverflow.no);
        assert(date4 == Date(1999, 2, 28));
        --------------------
      +/
    void addMonths(int months, AllowDayOverflow allowOverflow = AllowDayOverflow.yes)
    {
        enforceIsADate();

        if(isInfinity)
            return;

        auto years = months / 12;
        months %= 12;
        auto newMonth = _month + months;

        if(months < 0)
        {
            if(newMonth < 1)
            {
                newMonth += 12;
                --years;
            }
        }
        else
        {
            if(newMonth > 12)
            {
                newMonth -= 12;
                ++years;
            }
        }

        _year += cast(Year)(years);
        _month = cast(MonthOfYear)newMonth;

        immutable currMaxDay = maxDay(_year, _month);
        immutable overflow = _day - currMaxDay;

        if(overflow > 0)
        {
            if(allowOverflow == AllowDayOverflow.yes)
            {
                ++_month;
                _day = cast(DayOfMonth)overflow;
            }
            else
                _day = cast(DayOfMonth)currMaxDay;
        }
    }

    //Test addYears with AllowDayOverlow.yes
    unittest
    {
        //Test A.D.
        {
            auto date = Date(1999, 7, 6);
            date.addMonths(3);
            assertEqual(date, Date(1999, 10, 6));
            date.addMonths(-4);
            assertEqual(date, Date(1999, 6, 6));
        }

        {
            auto date = Date(1999, 7, 6);
            date.addMonths(6);
            assertEqual(date, Date(2000, 1, 6));
            date.addMonths(-6);
            assertEqual(date, Date(1999, 7, 6));
        }

        {
            auto date = Date(1999, 7, 6);
            date.addMonths(27);
            assertEqual(date, Date(2001, 10, 6));
            date.addMonths(-28);
            assertEqual(date, Date(1999, 6, 6));
        }

        {
            auto date = Date(1999, 5, 31);
            date.addMonths(1);
            assertEqual(date, Date(1999, 7, 1));
        }

        {
            auto date = Date(1999, 5, 31);
            date.addMonths(-1);
            assertEqual(date, Date(1999, 5, 1));
        }

        {
            auto date = Date(1999, 2, 28);
            date.addMonths(12);
            assertEqual(date, Date(2000, 2, 28));
        }

        {
            auto date = Date(2000, 2, 29);
            date.addMonths(12);
            assertEqual(date, Date(2001, 3, 1));
        }

        {
            auto date = Date(1999, 7, 31);
            date.addMonths(1);
            assertEqual(date, Date(1999, 8, 31));
            date.addMonths(1);
            assertEqual(date, Date(1999, 10, 1));
        }

        {
            auto date = Date(1998, 8, 31);
            date.addMonths(13);
            assertEqual(date, Date(1999, 10, 1));
            date.addMonths(-13);
            assertEqual(date, Date(1998, 9, 1));
        }

        {
            auto date = Date(1997, 12, 31);
            date.addMonths(13);
            assertEqual(date, Date(1999, 1, 31));
            date.addMonths(-13);
            assertEqual(date, Date(1997, 12, 31));
        }

        {
            auto date = Date(1997, 12, 31);
            date.addMonths(14);
            assertEqual(date, Date(1999, 3, 3));
            date.addMonths(-14);
            assertEqual(date, Date(1998, 1, 3));
        }

        {
            auto date = Date(1998, 12, 31);
            date.addMonths(14);
            assertEqual(date, Date(2000, 3, 2));
            date.addMonths(-14);
            assertEqual(date, Date(1999, 1, 2));
        }

        {
            auto date = Date(1999, 12, 31);
            date.addMonths(14);
            assertEqual(date, Date(2001, 3, 3));
            date.addMonths(-14);
            assertEqual(date, Date(2000, 1, 3));
        }

        //Test B.C.
        {
            auto date = Date(-1999, 7, 6);
            date.addMonths(3);
            assertEqual(date, Date(-1999, 10, 6));
            date.addMonths(-4);
            assertEqual(date, Date(-1999, 6, 6));
        }

        {
            auto date = Date(-1999, 7, 6);
            date.addMonths(6);
            assertEqual(date, Date(-1998, 1, 6));
            date.addMonths(-6);
            assertEqual(date, Date(-1999, 7, 6));
        }

        {
            auto date = Date(-1999, 7, 6);
            date.addMonths(-27);
            assertEqual(date, Date(-2001, 4, 6));
            date.addMonths(28);
            assertEqual(date, Date(-1999, 8, 6));
        }

        {
            auto date = Date(-1999, 5, 31);
            date.addMonths(1);
            assertEqual(date, Date(-1999, 7, 1));
        }

        {
            auto date = Date(-1999, 5, 31);
            date.addMonths(-1);
            assertEqual(date, Date(-1999, 5, 1));
        }

        {
            auto date = Date(-1999, 2, 28);
            date.addMonths(-12);
            assertEqual(date, Date(-2000, 2, 28));
        }

        {
            auto date = Date(-2000, 2, 29);
            date.addMonths(-12);
            assertEqual(date, Date(-2001, 3, 1));
        }

        {
            auto date = Date(-1999, 7, 31);
            date.addMonths(1);
            assertEqual(date, Date(-1999, 8, 31));
            date.addMonths(1);
            assertEqual(date, Date(-1999, 10, 1));
        }

        {
            auto date = Date(-1998, 8, 31);
            date.addMonths(13);
            assertEqual(date, Date(-1997, 10, 1));
            date.addMonths(-13);
            assertEqual(date, Date(-1998, 9, 1));
        }

        {
            auto date = Date(-1997, 12, 31);
            date.addMonths(13);
            assertEqual(date, Date(-1995, 1, 31));
            date.addMonths(-13);
            assertEqual(date, Date(-1997, 12, 31));
        }

        {
            auto date = Date(-1997, 12, 31);
            date.addMonths(14);
            assertEqual(date, Date(-1995, 3, 3));
            date.addMonths(-14);
            assertEqual(date, Date(-1996, 1, 3));
        }

        {
            auto date = Date(-2002, 12, 31);
            date.addMonths(14);
            assertEqual(date, Date(-2000, 3, 2));
            date.addMonths(-14);
            assertEqual(date, Date(-2001, 1, 2));
        }

        {
            auto date = Date(-2001, 12, 31);
            date.addMonths(14);
            assertEqual(date, Date(-1999, 3, 3));
            date.addMonths(-14);
            assertEqual(date, Date(-2000, 1, 3));
        }

        //Test Both
        {
            auto date = Date(1, 1, 1);
            date.addMonths(-1);
            assertEqual(date, Date(0, 12, 1));
            date.addMonths(1);
            assertEqual(date, Date(1, 1, 1));
        }

        {
            auto date = Date(4, 1, 1);
            date.addMonths(-48);
            assertEqual(date, Date(0, 1, 1));
            date.addMonths(48);
            assertEqual(date, Date(4, 1, 1));
        }

        {
            auto date = Date(4, 3, 31);
            date.addMonths(-49);
            assertEqual(date, Date(0, 3, 2));
            date.addMonths(49);
            assertEqual(date, Date(4, 4, 2));
        }

        {
            auto date = Date(4, 3, 31);
            date.addMonths(-85);
            assertEqual(date, Date(-3, 3, 3));
            date.addMonths(85);
            assertEqual(date, Date(4, 4, 3));
        }

        assertExcNotThrown!(Exception, (){posInfinity.addMonths(5);})(LineInfo());
        assertExcNotThrown!(Exception, (){negInfinity.addMonths(5);})(LineInfo());
        assertExcThrown!(Exception, (){notADate.addMonths(5);})(LineInfo());

        {
            auto date = posInfinity;
            posInfinity.addMonths(5);
            assertEqual(date, posInfinity);
        }

        {
            auto date = negInfinity;
            negInfinity.addMonths(5);
            assertEqual(date, negInfinity);
        }

        const cdate = Date(1999, 7, 6);
        static assert(!__traits(compiles, cdate.addMonths(3)));
        immutable idate = Date(1999, 7, 6);
        static assert(!__traits(compiles, idate.addMonths(3)));

        //Verify Examples
        auto date1 = Date(2010, 1, 1);
        date1.addMonths(1);
        assert(date1 == Date(2010, 2, 1));

        auto date2 = Date(2010, 1, 1);
        date2.addMonths(-1);
        assert(date2 == Date(2009, 12, 1));

        auto date3 = Date(1999, 1, 29);
        date3.addMonths(1);
        assert(date3 == Date(1999, 3, 1));

        auto date4 = Date(1999, 1, 29);
        date4.addMonths(1, AllowDayOverflow.no);
        assert(date4 == Date(1999, 2, 28));
    }

    //Test addYears with AllowDayOverlow.no
    unittest
    {
        //Test A.D.
        {
            auto date = Date(1999, 7, 6);
            date.addMonths(3, AllowDayOverflow.no);
            assertEqual(date, Date(1999, 10, 6));
            date.addMonths(-4, AllowDayOverflow.no);
            assertEqual(date, Date(1999, 6, 6));
        }

        {
            auto date = Date(1999, 7, 6);
            date.addMonths(6, AllowDayOverflow.no);
            assertEqual(date, Date(2000, 1, 6));
            date.addMonths(-6, AllowDayOverflow.no);
            assertEqual(date, Date(1999, 7, 6));
        }

        {
            auto date = Date(1999, 7, 6);
            date.addMonths(27, AllowDayOverflow.no);
            assertEqual(date, Date(2001, 10, 6));
            date.addMonths(-28, AllowDayOverflow.no);
            assertEqual(date, Date(1999, 6, 6));
        }

        {
            auto date = Date(1999, 5, 31);
            date.addMonths(1, AllowDayOverflow.no);
            assertEqual(date, Date(1999, 6, 30));
        }

        {
            auto date = Date(1999, 5, 31);
            date.addMonths(-1, AllowDayOverflow.no);
            assertEqual(date, Date(1999, 4, 30));
        }

        {
            auto date = Date(1999, 2, 28);
            date.addMonths(12, AllowDayOverflow.no);
            assertEqual(date, Date(2000, 2, 28));
        }

        {
            auto date = Date(2000, 2, 29);
            date.addMonths(12, AllowDayOverflow.no);
            assertEqual(date, Date(2001, 2, 28));
        }

        {
            auto date = Date(1999, 7, 31);
            date.addMonths(1, AllowDayOverflow.no);
            assertEqual(date, Date(1999, 8, 31));
            date.addMonths(1, AllowDayOverflow.no);
            assertEqual(date, Date(1999, 9, 30));
        }

        {
            auto date = Date(1998, 8, 31);
            date.addMonths(13, AllowDayOverflow.no);
            assertEqual(date, Date(1999, 9, 30));
            date.addMonths(-13, AllowDayOverflow.no);
            assertEqual(date, Date(1998, 8, 30));
        }

        {
            auto date = Date(1997, 12, 31);
            date.addMonths(13, AllowDayOverflow.no);
            assertEqual(date, Date(1999, 1, 31));
            date.addMonths(-13, AllowDayOverflow.no);
            assertEqual(date, Date(1997, 12, 31));
        }

        {
            auto date = Date(1997, 12, 31);
            date.addMonths(14, AllowDayOverflow.no);
            assertEqual(date, Date(1999, 2, 28));
            date.addMonths(-14, AllowDayOverflow.no);
            assertEqual(date, Date(1997, 12, 28));
        }

        {
            auto date = Date(1998, 12, 31);
            date.addMonths(14, AllowDayOverflow.no);
            assertEqual(date, Date(2000, 2, 29));
            date.addMonths(-14, AllowDayOverflow.no);
            assertEqual(date, Date(1998, 12, 29));
        }

        {
            auto date = Date(1999, 12, 31);
            date.addMonths(14, AllowDayOverflow.no);
            assertEqual(date, Date(2001, 2, 28));
            date.addMonths(-14, AllowDayOverflow.no);
            assertEqual(date, Date(1999, 12, 28));
        }

        //Test B.C.
        {
            auto date = Date(-1999, 7, 6);
            date.addMonths(3, AllowDayOverflow.no);
            assertEqual(date, Date(-1999, 10, 6));
            date.addMonths(-4, AllowDayOverflow.no);
            assertEqual(date, Date(-1999, 6, 6));
        }

        {
            auto date = Date(-1999, 7, 6);
            date.addMonths(6, AllowDayOverflow.no);
            assertEqual(date, Date(-1998, 1, 6));
            date.addMonths(-6, AllowDayOverflow.no);
            assertEqual(date, Date(-1999, 7, 6));
        }

        {
            auto date = Date(-1999, 7, 6);
            date.addMonths(-27, AllowDayOverflow.no);
            assertEqual(date, Date(-2001, 4, 6));
            date.addMonths(28, AllowDayOverflow.no);
            assertEqual(date, Date(-1999, 8, 6));
        }

        {
            auto date = Date(-1999, 5, 31);
            date.addMonths(1, AllowDayOverflow.no);
            assertEqual(date, Date(-1999, 6, 30));
        }

        {
            auto date = Date(-1999, 5, 31);
            date.addMonths(-1, AllowDayOverflow.no);
            assertEqual(date, Date(-1999, 4, 30));
        }

        {
            auto date = Date(-1999, 2, 28);
            date.addMonths(-12, AllowDayOverflow.no);
            assertEqual(date, Date(-2000, 2, 28));
        }

        {
            auto date = Date(-2000, 2, 29);
            date.addMonths(-12, AllowDayOverflow.no);
            assertEqual(date, Date(-2001, 2, 28));
        }

        {
            auto date = Date(-1999, 7, 31);
            date.addMonths(1, AllowDayOverflow.no);
            assertEqual(date, Date(-1999, 8, 31));
            date.addMonths(1, AllowDayOverflow.no);
            assertEqual(date, Date(-1999, 9, 30));
        }

        {
            auto date = Date(-1998, 8, 31);
            date.addMonths(13, AllowDayOverflow.no);
            assertEqual(date, Date(-1997, 9, 30));
            date.addMonths(-13, AllowDayOverflow.no);
            assertEqual(date, Date(-1998, 8, 30));
        }

        {
            auto date = Date(-1997, 12, 31);
            date.addMonths(13, AllowDayOverflow.no);
            assertEqual(date, Date(-1995, 1, 31));
            date.addMonths(-13, AllowDayOverflow.no);
            assertEqual(date, Date(-1997, 12, 31));
        }

        {
            auto date = Date(-1997, 12, 31);
            date.addMonths(14, AllowDayOverflow.no);
            assertEqual(date, Date(-1995, 2, 28));
            date.addMonths(-14, AllowDayOverflow.no);
            assertEqual(date, Date(-1997, 12, 28));
        }

        {
            auto date = Date(-2002, 12, 31);
            date.addMonths(14, AllowDayOverflow.no);
            assertEqual(date, Date(-2000, 2, 29));
            date.addMonths(-14, AllowDayOverflow.no);
            assertEqual(date, Date(-2002, 12, 29));
        }

        {
            auto date = Date(-2001, 12, 31);
            date.addMonths(14, AllowDayOverflow.no);
            assertEqual(date, Date(-1999, 2, 28));
            date.addMonths(-14, AllowDayOverflow.no);
            assertEqual(date, Date(-2001, 12, 28));
        }

        //Test Both
        {
            auto date = Date(1, 1, 1);
            date.addMonths(-1, AllowDayOverflow.no);
            assertEqual(date, Date(0, 12, 1));
            date.addMonths(1, AllowDayOverflow.no);
            assertEqual(date, Date(1, 1, 1));
        }

        {
            auto date = Date(4, 1, 1);
            date.addMonths(-48, AllowDayOverflow.no);
            assertEqual(date, Date(0, 1, 1));
            date.addMonths(48, AllowDayOverflow.no);
            assertEqual(date, Date(4, 1, 1));
        }

        {
            auto date = Date(4, 3, 31);
            date.addMonths(-49, AllowDayOverflow.no);
            assertEqual(date, Date(0, 2, 29));
            date.addMonths(49, AllowDayOverflow.no);
            assertEqual(date, Date(4, 3, 29));
        }

        {
            auto date = Date(4, 3, 31);
            date.addMonths(-85, AllowDayOverflow.no);
            assertEqual(date, Date(-3, 2, 28));
            date.addMonths(85, AllowDayOverflow.no);
            assertEqual(date, Date(4, 3, 28));
        }
    }

    /++
        Adds the given number of weeks to this Date. A negative number will subtract.

        The day, month, and year will be adjusted accordingly.

        addWeeks(numWeeks) is effectively equivalent to addDays(numWeeks * 7).

        Note that addWeeks(52) is not equivalent to addYear(1) because a year does not have
        exactly 52 weeks.

        Params:
            weeks = The number of weeks to add to this Date.

        Examples:
        --------------------
        auto date1 = Date(2010, 1, 1);
        auto date2 = date1;

        date1.addWeeks(1);
        date2.addDays(7);
        assert(date1 == date2);

        date1.addWeeks(52);
        date2.addYears(1);
        assert(date1 != date2);
        assert(date1 == Date(2011, 1, 7));
        assert(date2 == Date(2011, 1, 8));
        --------------------
      +/
    void addWeeks(long weeks)
    {
        addDays(weeks * 7);
    }

    unittest
    {
        //Test A.D.
        {
            auto date = Date(1999, 2, 28);
            date.addWeeks(1);
            assertEqual(date, Date(1999, 3, 7));
            date.addWeeks(-1);
            assertEqual(date, Date(1999, 2, 28));
        }

        {
            auto date = Date(2000, 2, 28);
            date.addWeeks(1);
            assertEqual(date, Date(2000, 3, 6));
            date.addWeeks(1);
            assertEqual(date, Date(2000, 3, 13));
            date.addWeeks(-1);
            assertEqual(date, Date(2000, 3, 6));
        }

        {
            auto date = Date(1999, 2, 28);
            date.addWeeks(52);
            assertEqual(date, Date(2000, 2, 27));
            date.addWeeks(-52);
            assertEqual(date, Date(1999, 2, 28));
        }

        {
            auto date = Date(1999, 2, 28);
            date.addWeeks(53);
            assertEqual(date, Date(2000, 3, 5));
            date.addWeeks(-53);
            assertEqual(date, Date(1999, 2, 28));
        }

        //Test B.C.
        {
            auto date = Date(-1999, 2, 28);
            date.addWeeks(1);
            assertEqual(date, Date(-1999, 3, 7));
            date.addWeeks(-1);
            assertEqual(date, Date(-1999, 2, 28));
        }

        {
            auto date = Date(-2000, 2, 28);
            date.addWeeks(1);
            assertEqual(date, Date(-2000, 3, 6));
            date.addWeeks(1);
            assertEqual(date, Date(-2000, 3, 13));
            date.addWeeks(-1);
            assertEqual(date, Date(-2000, 3, 6));
        }

        {
            auto date = Date(-2001, 2, 28);
            date.addWeeks(52);
            assertEqual(date, Date(-2000, 2, 27));
            date.addWeeks(-52);
            assertEqual(date, Date(-2001, 2, 28));
        }

        {
            auto date = Date(-2001, 2, 28);
            date.addWeeks(53);
            assertEqual(date, Date(-2000, 3, 5));
            date.addWeeks(-53);
            assertEqual(date, Date(-2001, 2, 28));
        }

        //Test Both
        {
            auto date = Date(1, 1, 7);
            date.addWeeks(-1);
            assertEqual(date, Date(0, 12, 31));
            date.addWeeks(1);
            assertEqual(date, Date(1, 1, 7));
        }

        {
            auto date = Date(0, 12, 25);
            date.addWeeks(1);
            assertEqual(date, Date(1, 1, 1));
            date.addWeeks(-1);
            assertEqual(date, Date(0, 12, 25));
        }

        {
            auto date = Date(5, 1, 1);
            date.addWeeks(-468);
            assertEqual(date, Date(-4, 1, 13));
            date.addWeeks(468);
            assertEqual(date, Date(5, 1, 1));
        }

        assertExcNotThrown!(Exception, (){posInfinity.addWeeks(5);})(LineInfo());
        assertExcNotThrown!(Exception, (){negInfinity.addWeeks(5);})(LineInfo());
        assertExcThrown!(Exception, (){notADate.addWeeks(5);})(LineInfo());

        {
            auto date = posInfinity;
            posInfinity.addWeeks(5);
            assertEqual(date, posInfinity);
        }

        {
            auto date = negInfinity;
            negInfinity.addWeeks(5);
            assertEqual(date, negInfinity);
        }

        const cdate = Date(1999, 7, 6);
        static assert(!__traits(compiles, cdate.addWeeks(2)));
        immutable idate = Date(1999, 7, 6);
        static assert(!__traits(compiles, idate.addWeeks(2)));

        //Verify Examples
        {
            auto date1 = Date(2010, 1, 1);
            auto date2 = date1;

            date1.addWeeks(1);
            date2.addDays(7);
            assert(date1 == date2);

            date1.addWeeks(52);
            date2.addYears(1);
            assert(date1 != date2);
            assert(date1 == Date(2011, 1, 7));
            assert(date2 == Date(2011, 1, 8));
        }
    }


    /++
        Adds the given number of days to this Date. A negative number will subtract.

        The month will be adjusted along with the day if the number of days added (or subtracted)
        would overflow (or underflow) the current month. The year will be adjusted along with
        the month if the increase (or decrease) to the month would cause it to overflow (or underflow)
        the current year.

        addDays(numDays) is effectively equivalent to date.dayOfGregorianCal = date.dayOfGregorianCal + days.

        Params:
            days = The number of days to add to this Date.

        Examples:
        --------------------
        auto date = Date(2010, 1, 1);
        date.addDays(1);
        assert(date == Date(2010, 1, 2));
        date.addDays(365);
        assert(date == Date(2011, 1, 2));
        date.addDays(-32);
        assert(date == Date(2010, 12, 1));
        --------------------
      +/
    void addDays(long days)
    {
        enforceIsADate();

        if(isInfinity)
            return;

        dayOfGregorianCal = dayOfGregorianCal + days;
    }

    unittest
    {
        //Test A.D.
        {
            auto date = Date(1999, 2, 28);
            date.addDays(1);
            assertEqual(date, Date(1999, 3, 1));
            date.addDays(-1);
            assertEqual(date, Date(1999, 2, 28));
        }

        {
            auto date = Date(2000, 2, 28);
            date.addDays(1);
            assertEqual(date, Date(2000, 2, 29));
            date.addDays(1);
            assertEqual(date, Date(2000, 3, 1));
            date.addDays(-1);
            assertEqual(date, Date(2000, 2, 29));
        }

        {
            auto date = Date(1999, 6, 30);
            date.addDays(1);
            assertEqual(date, Date(1999, 7, 1));
            date.addDays(-1);
            assertEqual(date, Date(1999, 6, 30));
        }

        {
            auto date = Date(1999, 7, 31);
            date.addDays(1);
            assertEqual(date, Date(1999, 8, 1));
            date.addDays(-1);
            assertEqual(date, Date(1999, 7, 31));
        }

        {
            auto date = Date(1999, 1, 1);
            date.addDays(-1);
            assertEqual(date, Date(1998, 12, 31));
            date.addDays(1);
            assertEqual(date, Date(1999, 1, 1));
        }

        {
            auto date = Date(1999, 7, 6);
            date.addDays(9);
            assertEqual(date, Date(1999, 7, 15));
            date.addDays(-11);
            assertEqual(date, Date(1999, 7, 4));
            date.addDays(30);
            assertEqual(date, Date(1999, 8, 3));
            date.addDays(-3);
        }

        {
            auto date = Date(1999, 7, 6);
            date.addDays(365);
            assertEqual(date, Date(2000, 7, 5));
            date.addDays(-365);
            assertEqual(date, Date(1999, 7, 6));
            date.addDays(366);
            assertEqual(date, Date(2000, 7, 6));
            date.addDays(730);
            assertEqual(date, Date(2002, 7, 6));
            date.addDays(-1096);
            assertEqual(date, Date(1999, 7, 6));
        }

        //Test B.C.
        {
            auto date = Date(-1999, 2, 28);
            date.addDays(1);
            assertEqual(date, Date(-1999, 3, 1));
            date.addDays(-1);
            assertEqual(date, Date(-1999, 2, 28));
        }

        {
            auto date = Date(-2000, 2, 28);
            date.addDays(1);
            assertEqual(date, Date(-2000, 2, 29));
            date.addDays(1);
            assertEqual(date, Date(-2000, 3, 1));
            date.addDays(-1);
            assertEqual(date, Date(-2000, 2, 29));
        }

        {
            auto date = Date(-1999, 6, 30);
            date.addDays(1);
            assertEqual(date, Date(-1999, 7, 1));
            date.addDays(-1);
            assertEqual(date, Date(-1999, 6, 30));
        }

        {
            auto date = Date(-1999, 7, 31);
            date.addDays(1);
            assertEqual(date, Date(-1999, 8, 1));
            date.addDays(-1);
            assertEqual(date, Date(-1999, 7, 31));
        }

        {
            auto date = Date(-1999, 1, 1);
            date.addDays(-1);
            assertEqual(date, Date(-2000, 12, 31));
            date.addDays(1);
            assertEqual(date, Date(-1999, 1, 1));
        }

        {
            auto date = Date(-1999, 7, 6);
            date.addDays(9);
            assertEqual(date, Date(-1999, 7, 15));
            date.addDays(-11);
            assertEqual(date, Date(-1999, 7, 4));
            date.addDays(30);
            assertEqual(date, Date(-1999, 8, 3));
            date.addDays(-3);
        }

        {
            auto date = Date(-1999, 7, 6);
            date.addDays(365);
            assertEqual(date, Date(-1998, 7, 6));
            date.addDays(-365);
            assertEqual(date, Date(-1999, 7, 6));
            date.addDays(366);
            assertEqual(date, Date(-1998, 7, 7));
            date.addDays(730);
            assertEqual(date, Date(-1996, 7, 6));
            date.addDays(-1096);
            assertEqual(date, Date(-1999, 7, 6));
        }

        //Test Both
        {
            auto date = Date(1, 7, 6);
            date.addDays(-365);
            assertEqual(date, Date(0, 7, 6));
            date.addDays(365);
            assertEqual(date, Date(1, 7, 6));
            date.addDays(-731);
            assertEqual(date, Date(-1, 7, 6));
            date.addDays(730);
            assertEqual(date, Date(1, 7, 5));
        }

        assertExcNotThrown!(Exception, (){posInfinity.addDays(5);})(LineInfo());
        assertExcNotThrown!(Exception, (){negInfinity.addDays(5);})(LineInfo());
        assertExcThrown!(Exception, (){notADate.addWeeks(5);})(LineInfo());

        {
            auto date = posInfinity;
            posInfinity.addDays(5);
            assertEqual(date, posInfinity);
        }

        {
            auto date = negInfinity;
            negInfinity.addDays(5);
            assertEqual(date, negInfinity);
        }

        const cdate = Date(1999, 7, 6);
        static assert(!__traits(compiles, cdate.addDays(12)));
        immutable idate = Date(1999, 7, 6);
        static assert(!__traits(compiles, idate.addDays(12)));

        //Verify Examples
        auto date = Date(2010, 1, 1);
        date.addDays(1);
        assert(date == Date(2010, 1, 2));
        date.addDays(365);
        assert(date == Date(2011, 1, 2));
        date.addDays(-32);
        assert(date == Date(2010, 12, 1));
    }

    /++
        Function for adding time units generically. The given TimeUnit must
        be TimeUnit.year, TimeUnit.month, TimeUnit.week, or TimeUnit.day.

        Params:
            units         = The type of unit to add to.
            value         = The number of units to add to the given type of unit.
            allowOverflow = Whether the days should be allowed to overflow, causing
                            the month to increment (only really applies to TimeUnit.year
                            and TimeUnit.month, but addTimeUnit() is generic, so it's included;
                            It's ignore otherwise).

        Examples:
        --------------------
        addTimeUnit!(TimeUnit.year)(numYears);   //translates to addYears(numYears);
        addTimeUnit!(TimeUnit.month)(numMonths); //translates to addMonths(numMonths);
        addTimeUnit!(TimeUnit.week)(numWeeks);   //translates to addMonths(numWeeks);
        addTimeUnit!(TimeUnit.day)(numDays);     //translates to addMonths(numDays);
        --------------------
      +/
    void addTimeUnit(TimeUnit units)(long value, AllowDayOverflow allowOverflow = AllowDayOverflow.yes)
        if(units == TimeUnit.year  ||
           units == TimeUnit.month ||
           units == TimeUnit.week  ||
           units == TimeUnit.day)
    {
        static if(units == TimeUnit.year)
            addYears(cast(ParameterTypeTuple!(addYears)[0])value, allowOverflow);
        else static if(units == TimeUnit.month)
            addMonths(cast(ParameterTypeTuple!(addMonths)[0])value, allowOverflow);
        else static if(units == TimeUnit.week)
            addWeeks(value);
        else static if(units == TimeUnit.day)
            addDays(value);
        else
            static assert(0);
    }

    unittest
    {
        {
            auto date = Date(1999, 7, 6);
            date.addTimeUnit!(TimeUnit.year)(1);
            assertEqual(date, Date(2000, 7, 6));
            date.addTimeUnit!(TimeUnit.year)(-1);
            assertEqual(date, Date(1999, 7, 6));
        }

        {
            auto date = Date(1999, 7, 6);
            date.addTimeUnit!(TimeUnit.month)(1);
            assertEqual(date, Date(1999, 8, 6));
            date.addTimeUnit!(TimeUnit.month)(-1);
            assertEqual(date, Date(1999, 7, 6));
        }

        {
            auto date = Date(1999, 7, 6);
            date.addTimeUnit!(TimeUnit.week)(1);
            assertEqual(date, Date(1999, 7, 13));
            date.addTimeUnit!(TimeUnit.week)(-1);
            assertEqual(date, Date(1999, 7, 6));
        }

        {
            auto date = Date(1999, 7, 6);
            date.addTimeUnit!(TimeUnit.day)(1);
            assertEqual(date, Date(1999, 7, 7));
            date.addTimeUnit!(TimeUnit.day)(-1);
            assertEqual(date, Date(1999, 7, 6));
        }

        auto date = Date(1999, 7, 6);
        const cdate = Date(1999, 7, 6);
        immutable idate = Date(1999, 7, 6);
        static assert(__traits(compiles, date.addTimeUnit!(TimeUnit.year)(12)));
        static assert(!__traits(compiles, cdate.addTimeUnit!(TimeUnit.year)(12)));
        static assert(!__traits(compiles, idate.TimeUnit!(TimeUnit.year)(12)));

        static assert(__traits(compiles, date.addTimeUnit!(TimeUnit.month)(12)));
        static assert(!__traits(compiles, cdate.addTimeUnit!(TimeUnit.month)(12)));
        static assert(!__traits(compiles, idate.TimeUnit!(TimeUnit.month)(12)));

        static assert(__traits(compiles, date.addTimeUnit!(TimeUnit.week)(12)));
        static assert(!__traits(compiles, cdate.addTimeUnit!(TimeUnit.week)(12)));
        static assert(!__traits(compiles, idate.TimeUnit!(TimeUnit.week)(12)));

        static assert(__traits(compiles, date.addTimeUnit!(TimeUnit.day)(12)));
        static assert(!__traits(compiles, cdate.addTimeUnit!(TimeUnit.day)(12)));
        static assert(!__traits(compiles, idate.TimeUnit!(TimeUnit.day)(12)));


        static assert(!__traits(compiles, date.addTimeUnit!(TimeUnit.hour)(12)));
        static assert(!__traits(compiles, cdate.addTimeUnit!(TimeUnit.hour)(12)));
        static assert(!__traits(compiles, idate.TimeUnit!(TimeUnit.hour)(12)));

        static assert(!__traits(compiles, date.addTimeUnit!(TimeUnit.minute)(12)));
        static assert(!__traits(compiles, cdate.addTimeUnit!(TimeUnit.minute)(12)));
        static assert(!__traits(compiles, idate.TimeUnit!(TimeUnit.minute)(12)));

        static assert(!__traits(compiles, date.addTimeUnit!(TimeUnit.second)(12)));
        static assert(!__traits(compiles, cdate.addTimeUnit!(TimeUnit.second)(12)));
        static assert(!__traits(compiles, idate.TimeUnit!(TimeUnit.second)(12)));

        static assert(!__traits(compiles, date.addTimeUnit!(TimeUnit.millisecond)(12)));
        static assert(!__traits(compiles, cdate.addTimeUnit!(TimeUnit.millisecond)(12)));
        static assert(!__traits(compiles, idate.TimeUnit!(TimeUnit.millisecond)(12)));

        static assert(!__traits(compiles, date.addTimeUnit!(TimeUnit.microsecond)(12)));
        static assert(!__traits(compiles, cdate.addTimeUnit!(TimeUnit.microsecond)(12)));
        static assert(!__traits(compiles, idate.TimeUnit!(TimeUnit.microsecond)(12)));

        static assert(!__traits(compiles, date.addTimeUnit!(TimeUnit.tick)(12)));
        static assert(!__traits(compiles, cdate.addTimeUnit!(TimeUnit.tick)(12)));
        static assert(!__traits(compiles, idate.TimeUnit!(TimeUnit.tick)(12)));
    }


    /++
        Whether this Date is in a leap year.
     +/
    @property bool isLeapYear() const
    {
        enforceNotSpecial();

        return yearIsLeapYear(_year);
    }

    unittest
    {
        assertExcThrown!(Exception, (){posInfinity.isLeapYear;})(LineInfo());
        assertExcThrown!(Exception, (){negInfinity.isLeapYear;})(LineInfo());
        assertExcThrown!(Exception, (){notADate.isLeapYear;})(LineInfo());

        auto date = Date(1999, 7, 6);
        const cdate = Date(1999, 7, 6);
        immutable idate = Date(1999, 7, 6);
        static assert(!__traits(compiles, date.isLeapYear = true));
        static assert(!__traits(compiles, cdate.isLeapYear = true));
        static assert(!__traits(compiles, idate.isLeapYear = true));
    }


    /++
        Date as a tuple of year, month, and day.
      +/
    @property Tuple!(Year, MonthOfYear, DayOfMonth) yearMonthDay() const
    {
        enforceNotSpecial();

        return Tuple!(Year, MonthOfYear, DayOfMonth)(_year, _month, _day);
    }

    unittest
    {
        assertExcThrown!(Exception, (){posInfinity.yearMonthDay;})(LineInfo());
        assertExcThrown!(Exception, (){negInfinity.yearMonthDay;})(LineInfo());
        assertExcThrown!(Exception, (){notADate.yearMonthDay;})(LineInfo());

        auto date = Date(1999, 7, 6);
        auto dateTuple = date.yearMonthDay();
        assertEqual(date._year, dateTuple.field[0]);
        assertEqual(date._month, dateTuple.field[1]);
        assertEqual(date._day, dateTuple.field[2]);

        const cdate = Date(1999, 7, 6);
        immutable idate = Date(1999, 7, 6);
        static assert(__traits(compiles, cdate.yearMonthDay));
        static assert(__traits(compiles, idate.yearMonthDay));
    }


    /++
        Day of the week.
      +/
    @property DayOfWeek dayOfWeek() const
    {
        enforceNotSpecial();

        immutable day = dayOfGregorianCal;

        //January 1st, 1 A.D. was a Monday
        if(day >= 0)
            return cast(DayOfWeek)(day % 7);
        else
        {
            immutable doy = cast(DayOfWeek)((day % 7) + 7);

            if(doy == 7)
                return DayOfWeek.sunday;
            else
                return doy;
        }
    }

    unittest
    {
        //Test A.D.
        assertEqual(Date(1, 1, 1).dayOfWeek, DayOfWeek.monday);
        assertEqual(Date(1, 1, 2).dayOfWeek, DayOfWeek.tuesday);
        assertEqual(Date(1, 1, 3).dayOfWeek, DayOfWeek.wednesday);
        assertEqual(Date(1, 1, 4).dayOfWeek, DayOfWeek.thursday);
        assertEqual(Date(1, 1, 5).dayOfWeek, DayOfWeek.friday);
        assertEqual(Date(1, 1, 6).dayOfWeek, DayOfWeek.saturday);
        assertEqual(Date(1, 1, 7).dayOfWeek, DayOfWeek.sunday);
        assertEqual(Date(1, 1, 8).dayOfWeek, DayOfWeek.monday);
        assertEqual(Date(1, 1, 9).dayOfWeek, DayOfWeek.tuesday);
        assertEqual(Date(2, 1, 1).dayOfWeek, DayOfWeek.tuesday);
        assertEqual(Date(3, 1, 1).dayOfWeek, DayOfWeek.wednesday);
        assertEqual(Date(4, 1, 1).dayOfWeek, DayOfWeek.thursday);
        assertEqual(Date(5, 1, 1).dayOfWeek, DayOfWeek.saturday);
        assertEqual(Date(2000, 1, 1).dayOfWeek, DayOfWeek.saturday);
        assertEqual(Date(2010, 8, 22).dayOfWeek, DayOfWeek.sunday);
        assertEqual(Date(2010, 8, 23).dayOfWeek, DayOfWeek.monday);
        assertEqual(Date(2010, 8, 24).dayOfWeek, DayOfWeek.tuesday);
        assertEqual(Date(2010, 8, 25).dayOfWeek, DayOfWeek.wednesday);
        assertEqual(Date(2010, 8, 26).dayOfWeek, DayOfWeek.thursday);
        assertEqual(Date(2010, 8, 27).dayOfWeek, DayOfWeek.friday);
        assertEqual(Date(2010, 8, 28).dayOfWeek, DayOfWeek.saturday);
        assertEqual(Date(2010, 8, 29).dayOfWeek, DayOfWeek.sunday);

        //Test B.C.
        assertEqual(Date(0, 12, 31).dayOfWeek, DayOfWeek.sunday);
        assertEqual(Date(0, 12, 30).dayOfWeek, DayOfWeek.saturday);
        assertEqual(Date(0, 12, 29).dayOfWeek, DayOfWeek.friday);
        assertEqual(Date(0, 12, 28).dayOfWeek, DayOfWeek.thursday);
        assertEqual(Date(0, 12, 27).dayOfWeek, DayOfWeek.wednesday);
        assertEqual(Date(0, 12, 26).dayOfWeek, DayOfWeek.tuesday);
        assertEqual(Date(0, 12, 25).dayOfWeek, DayOfWeek.monday);
        assertEqual(Date(0, 12, 24).dayOfWeek, DayOfWeek.sunday);
        assertEqual(Date(0, 12, 23).dayOfWeek, DayOfWeek.saturday);

        assertExcThrown!(Exception, (){posInfinity.dayOfWeek;})(LineInfo());
        assertExcThrown!(Exception, (){negInfinity.dayOfWeek;})(LineInfo());
        assertExcThrown!(Exception, (){notADate.dayOfWeek;})(LineInfo());

        const cdate = Date(1999, 7, 6);
        static assert(__traits(compiles, cdate.dayOfWeek == DayOfWeek.sunday));
        static assert(!__traits(compiles, cdate.dayOfWeek = DayOfWeek.sunday));
        immutable idate = Date(1999, 7, 6);
        static assert(__traits(compiles, idate.dayOfWeek == DayOfWeek.sunday));
        static assert(!__traits(compiles, idate.dayOfWeek = DayOfWeek.sunday));
    }


    /++
        Day of the year this date is on.

        Examples:
        --------------------
        assert(Date(1999, 1, 1).dayOfYear == 1);
        assert(Date(1999, 12, 31).dayOfYear == 365);
        assert(Date(2000, 12, 31).dayOfYear == 366);
        --------------------
      +/
    @property DayOfYear dayOfYear() const
    {
        enforceNotSpecial();

        switch(_month)
        {
            case january:
                return _day;
            case february:
                return cast(DayOfYear)(31 + _day);
            case march:
                return cast(DayOfYear)((isLeapYear ? 60 : 59) + _day);
            case april:
                return cast(DayOfYear)((isLeapYear ? 91 : 90) + _day);
            case may:
                return cast(DayOfYear)((isLeapYear ? 121 : 120) + _day);
            case june:
                return cast(DayOfYear)((isLeapYear ? 152 : 151) + _day);
            case july:
                return cast(DayOfYear)((isLeapYear ? 182 : 181) + _day);
            case august:
                return cast(DayOfYear)((isLeapYear ? 213 : 212) + _day);
            case september:
                return cast(DayOfYear)((isLeapYear ? 244 : 243) + _day);
            case october:
                return cast(DayOfYear)((isLeapYear ? 274 : 273) + _day);
            case november:
                return cast(DayOfYear)((isLeapYear ? 305 : 304) + _day);
            case december:
                return cast(DayOfYear)((isLeapYear ? 335 : 334) + _day);
            default:
                assert(0);
        }
    }

    unittest
    {
        //Test A.D.
        assertEqual(Date(1999, 1, 1).dayOfYear, 1);
        assertEqual(Date(1999, 1, 2).dayOfYear, 2);
        assertEqual(Date(1999, 1, 3).dayOfYear, 3);
        assertEqual(Date(1999, 1, 31).dayOfYear, 31);
        assertEqual(Date(1999, 2, 1).dayOfYear, 32);
        assertEqual(Date(1999, 2, 28).dayOfYear, 59);
        assertEqual(Date(1999, 3, 1).dayOfYear, 60);
        assertEqual(Date(1999, 3, 31).dayOfYear, 90);
        assertEqual(Date(1999, 4, 1).dayOfYear, 91);
        assertEqual(Date(1999, 4, 30).dayOfYear, 120);
        assertEqual(Date(1999, 5, 1).dayOfYear, 121);
        assertEqual(Date(1999, 5, 31).dayOfYear, 151);
        assertEqual(Date(1999, 6, 1).dayOfYear, 152);
        assertEqual(Date(1999, 6, 30).dayOfYear, 181);
        assertEqual(Date(1999, 7, 1).dayOfYear, 182);
        assertEqual(Date(1999, 7, 31).dayOfYear, 212);
        assertEqual(Date(1999, 8, 1).dayOfYear, 213);
        assertEqual(Date(1999, 8, 31).dayOfYear, 243);
        assertEqual(Date(1999, 9, 1).dayOfYear, 244);
        assertEqual(Date(1999, 9, 30).dayOfYear, 273);
        assertEqual(Date(1999, 10, 1).dayOfYear, 274);
        assertEqual(Date(1999, 10, 31).dayOfYear, 304);
        assertEqual(Date(1999, 11, 1).dayOfYear, 305);
        assertEqual(Date(1999, 11, 30).dayOfYear, 334);
        assertEqual(Date(1999, 12, 1).dayOfYear, 335);
        assertEqual(Date(1999, 12, 31).dayOfYear, 365);

        assertEqual(Date(2000, 1, 1).dayOfYear, 1);
        assertEqual(Date(2000, 2, 1).dayOfYear, 32);
        assertEqual(Date(2000, 2, 29).dayOfYear, 60);
        assertEqual(Date(2000, 3, 1).dayOfYear, 61);
        assertEqual(Date(2000, 3, 31).dayOfYear, 91);
        assertEqual(Date(2000, 4, 1).dayOfYear, 92);
        assertEqual(Date(2000, 4, 30).dayOfYear, 121);
        assertEqual(Date(2000, 5, 1).dayOfYear, 122);
        assertEqual(Date(2000, 5, 31).dayOfYear, 152);
        assertEqual(Date(2000, 6, 1).dayOfYear, 153);
        assertEqual(Date(2000, 6, 30).dayOfYear, 182);
        assertEqual(Date(2000, 7, 1).dayOfYear, 183);
        assertEqual(Date(2000, 7, 31).dayOfYear, 213);
        assertEqual(Date(2000, 8, 1).dayOfYear, 214);
        assertEqual(Date(2000, 8, 31).dayOfYear, 244);
        assertEqual(Date(2000, 9, 1).dayOfYear, 245);
        assertEqual(Date(2000, 9, 30).dayOfYear, 274);
        assertEqual(Date(2000, 10, 1).dayOfYear, 275);
        assertEqual(Date(2000, 10, 31).dayOfYear, 305);
        assertEqual(Date(2000, 11, 1).dayOfYear, 306);
        assertEqual(Date(2000, 11, 30).dayOfYear, 335);
        assertEqual(Date(2000, 12, 1).dayOfYear, 336);
        assertEqual(Date(2000, 12, 31).dayOfYear, 366);

        //Test B.C.
        assertEqual(Date(0, 1, 1).dayOfYear, 1);
        assertEqual(Date(0, 12, 31).dayOfYear, 366);
        assertEqual(Date(1, 1, 1).dayOfYear, 1);
        assertEqual(Date(1, 12, 31).dayOfYear, 365);
        assertEqual(Date(-1, 1, 1).dayOfYear, 1);
        assertEqual(Date(-1, 12, 31).dayOfYear, 365);
        assertEqual(Date(4, 1, 1).dayOfYear, 1);
        assertEqual(Date(4, 12, 31).dayOfYear, 366);

        assertExcThrown!(Exception, (){posInfinity.dayOfYear;})(LineInfo());
        assertExcThrown!(Exception, (){negInfinity.dayOfYear;})(LineInfo());
        assertExcThrown!(Exception, (){notADate.dayOfYear;})(LineInfo());

        const cdate = Date(1999, 7, 6);
        static assert(__traits(compiles, cdate.dayOfYear == 187));
        immutable idate = Date(1999, 7, 6);
        static assert(__traits(compiles, idate.dayOfYear == 187));

        //Verify Examples
        assert(Date(1999, 1, 1).dayOfYear == 1);
        assert(Date(1999, 12, 31).dayOfYear == 365);
        assert(Date(2000, 12, 31).dayOfYear == 366);
    }

    /++
        Day of the year.

        Params:
            day = The day of the year to set which day of the year this Date is on.
      +/
    @property void dayOfYear(DayOfYear day)
    {
        enforceNotSpecial();

        if(isLeapYear)
        {
            enforce(day > 0 && day <= daysInLeapYear, "Invalid day of the year.");

            switch(day)
            {
                case 1: .. case 31:
                {
                    _month = january;
                    _day = cast(DayOfMonth)day;
                    break;
                }
                case 32: .. case 60:
                 {
                    _month = february;
                    _day = cast(DayOfMonth)(day - 31);
                    break;
                 }
                case 61: .. case 91:
                {
                    _month = march;
                    _day = cast(DayOfMonth)(day - 60);
                    break;
                }
                case 92: .. case 121:
                {
                    _month = april;
                    _day = cast(DayOfMonth)(day - 91);
                    break;
                }
                case 122: .. case 152:
                {
                    _month = may;
                    _day = cast(DayOfMonth)(day - 121);
                    break;
                }
                case 153: .. case 182:
                {
                    _month = june;
                    _day = cast(DayOfMonth)(day - 152);
                    break;
                }
                case 183: .. case 213:
                {
                    _month = july;
                    _day = cast(DayOfMonth)(day - 182);
                    break;
                }
                case 214: .. case 244:
                {
                    _month = august;
                    _day = cast(DayOfMonth)(day - 213);
                    break;
                }
                case 245: .. case 274:
                {
                    _month = september;
                    _day = cast(DayOfMonth)(day - 244);
                    break;
                }
                case 275: .. case 305:
                {
                    _month = october;
                    _day = cast(DayOfMonth)(day - 274);
                    break;
                }
                case 306: .. case 335:
                {
                    _month = november;
                    _day = cast(DayOfMonth)(day - 305);
                    break;
                }
                case 336: .. case 366:
                {
                    _month = december;
                    _day = cast(DayOfMonth)(day - 335);
                    break;
                }
                default:
                    assert(0);
            }
        }
        else
        {
            enforce(day > 0 && day <= daysInYear, "Invalid day of the year.");

            switch(day)
            {
                case 1: .. case 31:
                {
                    _month = january;
                    _day = cast(DayOfMonth)day;
                    break;
                }
                case 32: .. case 59:
                 {
                    _month = february;
                    _day = cast(DayOfMonth)(day - 31);
                    break;
                 }
                case 60: .. case 90:
                {
                    _month = march;
                    _day = cast(DayOfMonth)(day - 59);
                    break;
                }
                case 91: .. case 120:
                {
                    _month = april;
                    _day = cast(DayOfMonth)(day - 90);
                    break;
                }
                case 121: .. case 151:
                {
                    _month = may;
                    _day = cast(DayOfMonth)(day - 120);
                    break;
                }
                case 152: .. case 181:
                {
                    _month = june;
                    _day = cast(DayOfMonth)(day - 151);
                    break;
                }
                case 182: .. case 212:
                {
                    _month = july;
                    _day = cast(DayOfMonth)(day - 181);
                    break;
                }
                case 213: .. case 243:
                {
                    _month = august;
                    _day = cast(DayOfMonth)(day - 212);
                    break;
                }
                case 244: .. case 273:
                {
                    _month = september;
                    _day = cast(DayOfMonth)(day - 243);
                    break;
                }
                case 274: .. case 304:
                {
                    _month = october;
                    _day = cast(DayOfMonth)(day - 273);
                    break;
                }
                case 305: .. case 334:
                {
                    _month = november;
                    _day = cast(DayOfMonth)(day - 304);
                    break;
                }
                case 335: .. case 365:
                {
                    _month = december;
                    _day = cast(DayOfMonth)(day - 334);
                    break;
                }
                default:
                    assert(0);
            }
        }
    }

    unittest
    {
        auto date = Date(1999, 1, 1);

        void testDate(ushort day, in Date expected, size_t line = __LINE__)
        {
            date.dayOfYear = day;
            assertEqual(date, expected, "", __FILE__, line);
        }

        //Test A.D.
        testDate(1, Date(1999, 1, 1));
        testDate(2, Date(1999, 1, 2));
        testDate(3, Date(1999, 1, 3));
        testDate(31, Date(1999, 1, 31));

        testDate(32, Date(1999, 2, 1));
        testDate(59, Date(1999, 2, 28));
        testDate(60, Date(1999, 3, 1));
        testDate(90, Date(1999, 3, 31));
        testDate(91, Date(1999, 4, 1));
        testDate(120, Date(1999, 4, 30));
        testDate(121, Date(1999, 5, 1));
        testDate(151, Date(1999, 5, 31));
        testDate(152, Date(1999, 6, 1));
        testDate(181, Date(1999, 6, 30));
        testDate(182, Date(1999, 7, 1));
        testDate(212, Date(1999, 7, 31));
        testDate(213, Date(1999, 8, 1));
        testDate(243, Date(1999, 8, 31));
        testDate(244, Date(1999, 9, 1));
        testDate(273, Date(1999, 9, 30));
        testDate(274, Date(1999, 10, 1));
        testDate(304, Date(1999, 10, 31));
        testDate(305, Date(1999, 11, 1));
        testDate(334, Date(1999, 11, 30));
        testDate(335, Date(1999, 12, 1));
        testDate(365, Date(1999, 12, 31));

        date.year = 2000;
        testDate(1, Date(2000, 1, 1));
        testDate(32, Date(2000, 2, 1));
        testDate(60, Date(2000, 2, 29));
        testDate(61, Date(2000, 3, 1));
        testDate(91, Date(2000, 3, 31));
        testDate(92, Date(2000, 4, 1));
        testDate(121, Date(2000, 4, 30));
        testDate(122, Date(2000, 5, 1));
        testDate(152, Date(2000, 5, 31));
        testDate(153, Date(2000, 6, 1));
        testDate(182, Date(2000, 6, 30));
        testDate(183, Date(2000, 7, 1));
        testDate(213, Date(2000, 7, 31));
        testDate(214, Date(2000, 8, 1));
        testDate(244, Date(2000, 8, 31));
        testDate(245, Date(2000, 9, 1));
        testDate(274, Date(2000, 9, 30));
        testDate(275, Date(2000, 10, 1));
        testDate(305, Date(2000, 10, 31));
        testDate(306, Date(2000, 11, 1));
        testDate(335, Date(2000, 11, 30));
        testDate(336, Date(2000, 12, 1));
        testDate(366, Date(2000, 12, 31));

        //Test B.C.
        date.year = -1;
        testDate(1, Date(-1, 1, 1));
        testDate(2, Date(-1, 1, 2));
        testDate(3, Date(-1, 1, 3));
        testDate(31, Date(-1, 1, 31));

        testDate(32, Date(-1, 2, 1));
        testDate(59, Date(-1, 2, 28));
        testDate(60, Date(-1, 3, 1));
        testDate(90, Date(-1, 3, 31));
        testDate(91, Date(-1, 4, 1));
        testDate(120, Date(-1, 4, 30));
        testDate(121, Date(-1, 5, 1));
        testDate(151, Date(-1, 5, 31));
        testDate(152, Date(-1, 6, 1));
        testDate(181, Date(-1, 6, 30));
        testDate(182, Date(-1, 7, 1));
        testDate(212, Date(-1, 7, 31));
        testDate(213, Date(-1, 8, 1));
        testDate(243, Date(-1, 8, 31));
        testDate(244, Date(-1, 9, 1));
        testDate(273, Date(-1, 9, 30));
        testDate(274, Date(-1, 10, 1));
        testDate(304, Date(-1, 10, 31));
        testDate(305, Date(-1, 11, 1));
        testDate(334, Date(-1, 11, 30));
        testDate(335, Date(-1, 12, 1));
        testDate(365, Date(-1, 12, 31));

        date.year = 0;
        testDate(1, Date(0, 1, 1));
        testDate(32, Date(0, 2, 1));
        testDate(60, Date(0, 2, 29));
        testDate(61, Date(0, 3, 1));
        testDate(91, Date(0, 3, 31));
        testDate(92, Date(0, 4, 1));
        testDate(121, Date(0, 4, 30));
        testDate(122, Date(0, 5, 1));
        testDate(152, Date(0, 5, 31));
        testDate(153, Date(0, 6, 1));
        testDate(182, Date(0, 6, 30));
        testDate(183, Date(0, 7, 1));
        testDate(213, Date(0, 7, 31));
        testDate(214, Date(0, 8, 1));
        testDate(244, Date(0, 8, 31));
        testDate(245, Date(0, 9, 1));
        testDate(274, Date(0, 9, 30));
        testDate(275, Date(0, 10, 1));
        testDate(305, Date(0, 10, 31));
        testDate(306, Date(0, 11, 1));
        testDate(335, Date(0, 11, 30));
        testDate(336, Date(0, 12, 1));
        testDate(366, Date(0, 12, 31));

        assertExcThrown!(Exception, (){posInfinity.dayOfYear = 4;})(LineInfo());
        assertExcThrown!(Exception, (){negInfinity.dayOfYear = 4;})(LineInfo());
        assertExcThrown!(Exception, (){notADate.dayOfYear = 4;})(LineInfo());

        const cdate = Date(1999, 7, 6);
        static assert(!__traits(compiles, cdate.dayOfYear = 187));
        immutable idate = Date(1999, 7, 6);
        static assert(!__traits(compiles, idate.dayOfYear = 187));
    }


    /++
        The Xth day of the Gregorian Calendar.

        Examples:
        --------------------
        assert(Date(1, 1, 1).dayOfGregorianCal == 1);
        assert(Date(1, 12, 31).dayOfGregorianCal == 365);
        assert(Date(2, 1, 1).dayOfGregorianCal == 366);

        assert(Date(0, 12, 31).dayOfGregorianCal == 0);
        assert(Date(0, 1, 1).dayOfGregorianCal == -365);
        assert(Date(-1, 12, 31).dayOfGregorianCal == -366);

        assert(Date(2000, 1, 1).dayOfGregorianCal == 730_120);
        assert(Date(2010, 12, 31).dayOfGregorianCal == 734_137);
        --------------------
     +/
    @property DayOfGregorianCal dayOfGregorianCal() const
    {
        enforceNotSpecial();

        if(isAD)
        {
            if(_year == 1)
                return dayOfYear;

            long years = _year - 1;
            DayOfGregorianCal days = (years / 400) * daysIn400Years;
            years %= 400;

            days += (years / 100) * daysIn100Years;
            years %= 100;

            days += (years / 4) * daysIn4Years;
            years %= 4;

            days += years * daysInYear;

            days += dayOfYear;

            return days;
        }
        else if(_year == 0)
        {
            return dayOfYear - daysInLeapYear;
        }
        else
        {
            long years = _year;
            DayOfGregorianCal days = (years / 400) * daysIn400Years;
            years %= 400;

            days += (years / 100) * daysIn100Years;
            years %= 100;

            days += (years / 4) * daysIn4Years;
            years %= 4;

            if(years < 0)
            {
                days -= daysInLeapYear;
                ++years;

                days += years * daysInYear;

                days -= daysInYear - dayOfYear;
            }
            else
                days -= daysInLeapYear - dayOfYear;

            return days;
        }
    }

    unittest
    {
        //Test A.D.
        assertEqual(Date(1, 1, 1).dayOfGregorianCal, 1);
        assertEqual(Date(1, 1, 2).dayOfGregorianCal, 2);
        assertEqual(Date(1, 2, 1).dayOfGregorianCal, 32);
        assertEqual(Date(2, 1, 1).dayOfGregorianCal, 366);
        assertEqual(Date(3, 1, 1).dayOfGregorianCal, 731);
        assertEqual(Date(4, 1, 1).dayOfGregorianCal, 1096);
        assertEqual(Date(5, 1, 1).dayOfGregorianCal, 1462);
        assertEqual(Date(50, 1, 1).dayOfGregorianCal, 17_898);
        assertEqual(Date(97, 1, 1).dayOfGregorianCal, 35_065);
        assertEqual(Date(100, 1, 1).dayOfGregorianCal, 36_160);
        assertEqual(Date(101, 1, 1).dayOfGregorianCal, 36_525);
        assertEqual(Date(105, 1, 1).dayOfGregorianCal, 37_986);
        assertEqual(Date(200, 1, 1).dayOfGregorianCal, 72_684);
        assertEqual(Date(201, 1, 1).dayOfGregorianCal, 73_049);
        assertEqual(Date(300, 1, 1).dayOfGregorianCal, 109_208);
        assertEqual(Date(301, 1, 1).dayOfGregorianCal, 109_573);
        assertEqual(Date(400, 1, 1).dayOfGregorianCal, 145_732);
        assertEqual(Date(401, 1, 1).dayOfGregorianCal, 146_098);
        assertEqual(Date(500, 1, 1).dayOfGregorianCal, 182_257);
        assertEqual(Date(501, 1, 1).dayOfGregorianCal, 182_622);
        assertEqual(Date(1000, 1, 1).dayOfGregorianCal, 364_878);
        assertEqual(Date(1001, 1, 1).dayOfGregorianCal, 365_243);
        assertEqual(Date(1600, 1, 1).dayOfGregorianCal, 584_023);
        assertEqual(Date(1601, 1, 1).dayOfGregorianCal, 584_389);
        assertEqual(Date(1900, 1, 1).dayOfGregorianCal, 693_596);
        assertEqual(Date(1901, 1, 1).dayOfGregorianCal, 693_961);
        assertEqual(Date(1945, 11, 12).dayOfGregorianCal, 710_347);
        assertEqual(Date(1999, 1, 1).dayOfGregorianCal, 729_755);
        assertEqual(Date(2000, 1, 1).dayOfGregorianCal, 730_120);
        assertEqual(Date(2001, 1, 1).dayOfGregorianCal, 730_486);

        assertEqual(Date(2010, 1, 1).dayOfGregorianCal, 733_773);
        assertEqual(Date(2010, 1, 31).dayOfGregorianCal, 733_803);
        assertEqual(Date(2010, 2, 1).dayOfGregorianCal, 733_804);
        assertEqual(Date(2010, 2, 28).dayOfGregorianCal, 733_831);
        assertEqual(Date(2010, 3, 1).dayOfGregorianCal, 733_832);
        assertEqual(Date(2010, 3, 31).dayOfGregorianCal, 733_862);
        assertEqual(Date(2010, 4, 1).dayOfGregorianCal, 733_863);
        assertEqual(Date(2010, 4, 30).dayOfGregorianCal, 733_892);
        assertEqual(Date(2010, 5, 1).dayOfGregorianCal, 733_893);
        assertEqual(Date(2010, 5, 31).dayOfGregorianCal, 733_923);
        assertEqual(Date(2010, 6, 1).dayOfGregorianCal, 733_924);
        assertEqual(Date(2010, 6, 30).dayOfGregorianCal, 733_953);
        assertEqual(Date(2010, 7, 1).dayOfGregorianCal, 733_954);
        assertEqual(Date(2010, 7, 31).dayOfGregorianCal, 733_984);
        assertEqual(Date(2010, 8, 1).dayOfGregorianCal, 733_985);
        assertEqual(Date(2010, 8, 31).dayOfGregorianCal, 734_015);
        assertEqual(Date(2010, 9, 1).dayOfGregorianCal, 734_016);
        assertEqual(Date(2010, 9, 30).dayOfGregorianCal, 734_045);
        assertEqual(Date(2010, 10, 1).dayOfGregorianCal, 734_046);
        assertEqual(Date(2010, 10, 31).dayOfGregorianCal, 734_076);
        assertEqual(Date(2010, 11, 1).dayOfGregorianCal, 734_077);
        assertEqual(Date(2010, 11, 30).dayOfGregorianCal, 734_106);
        assertEqual(Date(2010, 12, 1).dayOfGregorianCal, 734_107);
        assertEqual(Date(2010, 12, 31).dayOfGregorianCal, 734_137);

        assertEqual(Date(2012, 2, 1).dayOfGregorianCal, 734_534);
        assertEqual(Date(2012, 2, 28).dayOfGregorianCal, 734_561);
        assertEqual(Date(2012, 2, 29).dayOfGregorianCal, 734_562);
        assertEqual(Date(2012, 3, 1).dayOfGregorianCal, 734_563);

        //Test B.C.
        assertEqual(Date(0, 12, 31).dayOfGregorianCal, 0);
        assertEqual(Date(0, 12, 30).dayOfGregorianCal, -1);
        assertEqual(Date(0, 12, 1).dayOfGregorianCal, -30);
        assertEqual(Date(0, 11, 30).dayOfGregorianCal, -31);

        assertEqual(Date(-1, 12, 31).dayOfGregorianCal, -366);
        assertEqual(Date(-1, 12, 30).dayOfGregorianCal, -367);
        assertEqual(Date(-1, 1, 1).dayOfGregorianCal, -730);
        assertEqual(Date(-2, 12, 31).dayOfGregorianCal, -731);
        assertEqual(Date(-2, 1, 1).dayOfGregorianCal, -1095);
        assertEqual(Date(-3, 12, 31).dayOfGregorianCal, -1096);
        assertEqual(Date(-3, 1, 1).dayOfGregorianCal, -1460);
        assertEqual(Date(-4, 12, 31).dayOfGregorianCal, -1461);
        assertEqual(Date(-4, 1, 1).dayOfGregorianCal, -1826);
        assertEqual(Date(-5, 12, 31).dayOfGregorianCal, -1827);
        assertEqual(Date(-5, 1, 1).dayOfGregorianCal, -2191);
        assertEqual(Date(-9, 1, 1).dayOfGregorianCal, -3652);

        assertEqual(Date(-49, 1, 1).dayOfGregorianCal, -18_262);
        assertEqual(Date(-50, 1, 1).dayOfGregorianCal, -18_627);
        assertEqual(Date(-97, 1, 1).dayOfGregorianCal, -35_794);
        assertEqual(Date(-99, 12, 31).dayOfGregorianCal, -36_160);
        assertEqual(Date(-99, 1, 1).dayOfGregorianCal, -36_524);
        assertEqual(Date(-100, 1, 1).dayOfGregorianCal, -36_889);
        assertEqual(Date(-101, 1, 1).dayOfGregorianCal, -37_254);
        assertEqual(Date(-105, 1, 1).dayOfGregorianCal, -38_715);
        assertEqual(Date(-200, 1, 1).dayOfGregorianCal, -73_413);
        assertEqual(Date(-201, 1, 1).dayOfGregorianCal, -73_778);
        assertEqual(Date(-300, 1, 1).dayOfGregorianCal, -109_937);
        assertEqual(Date(-301, 1, 1).dayOfGregorianCal, -110_302);
        assertEqual(Date(-400, 12, 31).dayOfGregorianCal, -146_097);
        assertEqual(Date(-400, 1, 1).dayOfGregorianCal, -146_462);
        assertEqual(Date(-401, 1, 1).dayOfGregorianCal, -146_827);
        assertEqual(Date(-499, 1, 1).dayOfGregorianCal, -182_621);
        assertEqual(Date(-500, 1, 1).dayOfGregorianCal, -182_986);
        assertEqual(Date(-501, 1, 1).dayOfGregorianCal, -183_351);
        assertEqual(Date(-1000, 1, 1).dayOfGregorianCal, -365_607);
        assertEqual(Date(-1001, 1, 1).dayOfGregorianCal, -365_972);
        assertEqual(Date(-1599, 1, 1).dayOfGregorianCal, -584_387);
        assertEqual(Date(-1600, 12, 31).dayOfGregorianCal, -584_388);
        assertEqual(Date(-1600, 1, 1).dayOfGregorianCal, -584_753);
        assertEqual(Date(-1601, 1, 1).dayOfGregorianCal, -585_118);
        assertEqual(Date(-1900, 1, 1).dayOfGregorianCal, -694_325);
        assertEqual(Date(-1901, 1, 1).dayOfGregorianCal, -694_690);
        assertEqual(Date(-1999, 1, 1).dayOfGregorianCal, -730_484);
        assertEqual(Date(-2000, 12, 31).dayOfGregorianCal, -730_485);
        assertEqual(Date(-2000, 1, 1).dayOfGregorianCal, -730_850);
        assertEqual(Date(-2001, 1, 1).dayOfGregorianCal, -731_215);

        assertEqual(Date(-2010, 1, 1).dayOfGregorianCal, -734_502);
        assertEqual(Date(-2010, 1, 31).dayOfGregorianCal, -734_472);
        assertEqual(Date(-2010, 2, 1).dayOfGregorianCal, -734_471);
        assertEqual(Date(-2010, 2, 28).dayOfGregorianCal, -734_444);
        assertEqual(Date(-2010, 3, 1).dayOfGregorianCal, -734_443);
        assertEqual(Date(-2010, 3, 31).dayOfGregorianCal, -734_413);
        assertEqual(Date(-2010, 4, 1).dayOfGregorianCal, -734_412);
        assertEqual(Date(-2010, 4, 30).dayOfGregorianCal, -734_383);
        assertEqual(Date(-2010, 5, 1).dayOfGregorianCal, -734_382);
        assertEqual(Date(-2010, 5, 31).dayOfGregorianCal, -734_352);
        assertEqual(Date(-2010, 6, 1).dayOfGregorianCal, -734_351);
        assertEqual(Date(-2010, 6, 30).dayOfGregorianCal, -734_322);
        assertEqual(Date(-2010, 7, 1).dayOfGregorianCal, -734_321);
        assertEqual(Date(-2010, 7, 31).dayOfGregorianCal, -734_291);
        assertEqual(Date(-2010, 8, 1).dayOfGregorianCal, -734_290);
        assertEqual(Date(-2010, 8, 31).dayOfGregorianCal, -734_260);
        assertEqual(Date(-2010, 9, 1).dayOfGregorianCal, -734_259);
        assertEqual(Date(-2010, 9, 30).dayOfGregorianCal, -734_230);
        assertEqual(Date(-2010, 10, 1).dayOfGregorianCal, -734_229);
        assertEqual(Date(-2010, 10, 31).dayOfGregorianCal, -734_199);
        assertEqual(Date(-2010, 11, 1).dayOfGregorianCal, -734_198);
        assertEqual(Date(-2010, 11, 30).dayOfGregorianCal, -734_169);
        assertEqual(Date(-2010, 12, 1).dayOfGregorianCal, -734_168);
        assertEqual(Date(-2010, 12, 31).dayOfGregorianCal, -734_138);

        assertEqual(Date(-2012, 2, 1).dayOfGregorianCal, -735_202);
        assertEqual(Date(-2012, 2, 28).dayOfGregorianCal, -735_175);
        assertEqual(Date(-2012, 2, 29).dayOfGregorianCal, -735_174);
        assertEqual(Date(-2012, 3, 1).dayOfGregorianCal, -735_173);

        assertEqual(Date(-3760, 9, 7).dayOfGregorianCal, -1_373_427); //Start of Hebrew Calendar

        assertExcThrown!(Exception, (){posInfinity.dayOfGregorianCal;})(LineInfo());
        assertExcThrown!(Exception, (){negInfinity.dayOfGregorianCal;})(LineInfo());
        assertExcThrown!(Exception, (){notADate.dayOfGregorianCal;})(LineInfo());

        const cdate = Date(1999, 7, 6);
        static assert(!__traits(compiles, cdate.dayOfGregorianCal = 187));
        immutable idate = Date(1999, 7, 6);
        static assert(!__traits(compiles, idate.GregorianCal = 187));

        //Verify Examples
        assert(Date(1, 1, 1).dayOfGregorianCal == 1);
        assert(Date(1, 12, 31).dayOfGregorianCal == 365);
        assert(Date(2, 1, 1).dayOfGregorianCal == 366);

        assert(Date(0, 12, 31).dayOfGregorianCal == 0);
        assert(Date(0, 1, 1).dayOfGregorianCal == -365);
        assert(Date(-1, 12, 31).dayOfGregorianCal == -366);

        assert(Date(2000, 1, 1).dayOfGregorianCal == 730_120);
        assert(Date(2010, 12, 31).dayOfGregorianCal == 734_137);
    }

    /++
        The Xth day of the Gregorian Calendar.

        Examples:
        --------------------
        auto date = Date.init;
        date.dayOfGregorianCal = 1;
        assert(date == Date(1, 1, 1));

        date.dayOfGregorianCal = 365;
        assert(date == Date(1, 12, 31));

        date.dayOfGregorianCal = 366;
        assert(date == Date(2, 1, 1));

        date.dayOfGregorianCal = 0;
        assert(date == Date(0, 12, 31));

        date.dayOfGregorianCal = -365;
        assert(date == Date(-0, 1, 1));

        date.dayOfGregorianCal = -366;
        assert(date == Date(-1, 12, 31));

        date.dayOfGregorianCal = 730_120;
        assert(date == Date(2000, 1, 1));

        date.dayOfGregorianCal = 734_137;
        assert(date == Date(2010, 12, 31));
        --------------------
     +/
    @property void dayOfGregorianCal(DayOfGregorianCal days)
    {
        //Since setting this sets the Date completely, we can allow
        //special dates to use this function. So that dayOfYear doesn't
        //throw, we reset the date before doing any calculations.
        this = Date.init;

        if(days > 0)
        {
            long years = (days / daysIn400Years) * 400 + 1;
            days %= daysIn400Years;

            years += (days / daysIn100Years) * 100;
            days %= daysIn100Years;

            years += (days / daysIn4Years) * 4;
            days %= daysIn4Years;

            years += days / daysInYear;
            days %= daysInYear;

            if(days == 0)
            {
                _year = cast(Year)(years - 1);
                _month = december;
                _day = 31;
            }
            else
            {
                _year = cast(Year)years;
                dayOfYear = cast(DayOfYear)days;
            }
        }
        else if(days <= 0 && -days < daysInLeapYear)
        {
            _year = 0;
            dayOfYear = cast(DayOfYear)(daysInLeapYear + days);
        }
        else
        {
            days += daysInLeapYear - 1;
            long years = (days / daysIn400Years) * 400 - 1;
            days %= daysIn400Years;

            years += (days / daysIn100Years) * 100;
            days %= daysIn100Years;

            years += (days / daysIn4Years) * 4;
            days %= daysIn4Years;

            years += days / daysInYear;
            days %= daysInYear;

            if(days == 0)
            {
                _year = cast(Year)(years + 1);
                _month = january;
                _day = 1;
            }
            else
            {
                _year = cast(Year)years;
                immutable newDoY = (yearIsLeapYear(_year) ? daysInLeapYear : daysInYear) + days + 1;
                dayOfYear = cast(DayOfYear)(newDoY);
            }
        }
    }

    unittest
    {
        {
            auto date = Date(1, 1, 1);

            void testDate(DayOfGregorianCal day, in Date expected, size_t line = __LINE__)
            {
                date.dayOfGregorianCal = day;
                assertEqual(date, expected, "", __FILE__, line);
            }

            //Test A.D.
            testDate(1, Date(1, 1, 1));
            testDate(2, Date(1, 1, 2));
            testDate(32, Date(1, 2, 1));
            testDate(366, Date(2, 1, 1));
            testDate(731, Date(3, 1, 1));
            testDate(1096, Date(4, 1, 1));
            testDate(1462, Date(5, 1, 1));
            testDate(17_898, Date(50, 1, 1));
            testDate(35_065, Date(97, 1, 1));
            testDate(36_160, Date(100, 1, 1));
            testDate(36_525, Date(101, 1, 1));
            testDate(37_986, Date(105, 1, 1));
            testDate(72_684, Date(200, 1, 1));
            testDate(73_049, Date(201, 1, 1));
            testDate(109_208, Date(300, 1, 1));
            testDate(109_573, Date(301, 1, 1));
            testDate(145_732, Date(400, 1, 1));
            testDate(146_098, Date(401, 1, 1));
            testDate(182_257, Date(500, 1, 1));
            testDate(182_622, Date(501, 1, 1));
            testDate(364_878, Date(1000, 1, 1));
            testDate(365_243, Date(1001, 1, 1));
            testDate(584_023, Date(1600, 1, 1));
            testDate(584_389, Date(1601, 1, 1));
            testDate(693_596, Date(1900, 1, 1));
            testDate(693_961, Date(1901, 1, 1));
            testDate(729_755, Date(1999, 1, 1));
            testDate(730_120, Date(2000, 1, 1));
            testDate(730_486, Date(2001, 1, 1));

            testDate(733_773, Date(2010, 1, 1));
            testDate(733_803, Date(2010, 1, 31));
            testDate(733_804, Date(2010, 2, 1));
            testDate(733_831, Date(2010, 2, 28));
            testDate(733_832, Date(2010, 3, 1));
            testDate(733_862, Date(2010, 3, 31));
            testDate(733_863, Date(2010, 4, 1));
            testDate(733_892, Date(2010, 4, 30));
            testDate(733_893, Date(2010, 5, 1));
            testDate(733_923, Date(2010, 5, 31));
            testDate(733_924, Date(2010, 6, 1));
            testDate(733_953, Date(2010, 6, 30));
            testDate(733_954, Date(2010, 7, 1));
            testDate(733_984, Date(2010, 7, 31));
            testDate(733_985, Date(2010, 8, 1));
            testDate(734_015, Date(2010, 8, 31));
            testDate(734_016, Date(2010, 9, 1));
            testDate(734_045, Date(2010, 9, 30));
            testDate(734_046, Date(2010, 10, 1));
            testDate(734_076, Date(2010, 10, 31));
            testDate(734_077, Date(2010, 11, 1));
            testDate(734_106, Date(2010, 11, 30));
            testDate(734_107, Date(2010, 12, 1));
            testDate(734_137, Date(2010, 12, 31));

            testDate(734_534, Date(2012, 2, 1));
            testDate(734_561, Date(2012, 2, 28));
            testDate(734_562, Date(2012, 2, 29));
            testDate(734_563, Date(2012, 3, 1));

            testDate(734_534,  Date(2012, 2, 1));

            testDate(734_561, Date(2012, 2, 28));
            testDate(734_562, Date(2012, 2, 29));
            testDate(734_563, Date(2012, 3, 1));

            //Test B.C.
            testDate(0, Date(0, 12, 31));
            testDate(-1, Date(0, 12, 30));
            testDate(-30, Date(0, 12, 1));
            testDate(-31, Date(0, 11, 30));

            testDate(-366, Date(-1, 12, 31));
            testDate(-367, Date(-1, 12, 30));
            testDate(-730, Date(-1, 1, 1));
            testDate(-731, Date(-2, 12, 31));
            testDate(-1095, Date(-2, 1, 1));
            testDate(-1096, Date(-3, 12, 31));
            testDate(-1460, Date(-3, 1, 1));
            testDate(-1461, Date(-4, 12, 31));
            testDate(-1826, Date(-4, 1, 1));
            testDate(-1827, Date(-5, 12, 31));
            testDate(-2191, Date(-5, 1, 1));
            testDate(-3652, Date(-9, 1, 1));

            testDate(-18_262, Date(-49, 1, 1));
            testDate(-18_627, Date(-50, 1, 1));
            testDate(-35_794, Date(-97, 1, 1));
            testDate(-36_160, Date(-99, 12, 31));
            testDate(-36_524, Date(-99, 1, 1));
            testDate(-36_889, Date(-100, 1, 1));
            testDate(-37_254, Date(-101, 1, 1));
            testDate(-38_715, Date(-105, 1, 1));
            testDate(-73_413, Date(-200, 1, 1));
            testDate(-73_778, Date(-201, 1, 1));
            testDate(-109_937, Date(-300, 1, 1));
            testDate(-110_302, Date(-301, 1, 1));
            testDate(-146_097, Date(-400, 12, 31));
            testDate(-146_462, Date(-400, 1, 1));
            testDate(-146_827, Date(-401, 1, 1));
            testDate(-182_621, Date(-499, 1, 1));
            testDate(-182_986, Date(-500, 1, 1));
            testDate(-183_351, Date(-501, 1, 1));
            testDate(-365_607, Date(-1000, 1, 1));
            testDate(-365_972, Date(-1001, 1, 1));
            testDate(-584_387, Date(-1599, 1, 1));
            testDate(-584_388, Date(-1600, 12, 31));
            testDate(-584_753, Date(-1600, 1, 1));
            testDate(-585_118, Date(-1601, 1, 1));
            testDate(-694_325, Date(-1900, 1, 1));
            testDate(-694_690, Date(-1901, 1, 1));
            testDate(-730_484, Date(-1999, 1, 1));
            testDate(-730_485, Date(-2000, 12, 31));
            testDate(-730_850, Date(-2000, 1, 1));
            testDate(-731_215, Date(-2001, 1, 1));

            testDate(-734_502, Date(-2010, 1, 1));
            testDate(-734_472, Date(-2010, 1, 31));
            testDate(-734_471, Date(-2010, 2, 1));
            testDate(-734_444, Date(-2010, 2, 28));
            testDate(-734_443, Date(-2010, 3, 1));
            testDate(-734_413, Date(-2010, 3, 31));
            testDate(-734_412, Date(-2010, 4, 1));
            testDate(-734_383, Date(-2010, 4, 30));
            testDate(-734_382, Date(-2010, 5, 1));
            testDate(-734_352, Date(-2010, 5, 31));
            testDate(-734_351, Date(-2010, 6, 1));
            testDate(-734_322, Date(-2010, 6, 30));
            testDate(-734_321, Date(-2010, 7, 1));
            testDate(-734_291, Date(-2010, 7, 31));
            testDate(-734_290, Date(-2010, 8, 1));
            testDate(-734_260, Date(-2010, 8, 31));
            testDate(-734_259, Date(-2010, 9, 1));
            testDate(-734_230, Date(-2010, 9, 30));
            testDate(-734_229, Date(-2010, 10, 1));
            testDate(-734_199, Date(-2010, 10, 31));
            testDate(-734_198, Date(-2010, 11, 1));
            testDate(-734_169, Date(-2010, 11, 30));
            testDate(-734_168, Date(-2010, 12, 1));
            testDate(-734_138, Date(-2010, 12, 31));

            testDate(-735_202, Date(-2012, 2, 1));
            testDate(-735_175, Date(-2012, 2, 28));
            testDate(-735_174, Date(-2012, 2, 29));
            testDate(-735_173, Date(-2012, 3, 1));

            testDate(-1_373_427, Date(-3760, 9, 7)); //Start of Hebrew Calendar
        }

        assertExcNotThrown!(Exception, (){posInfinity.dayOfGregorianCal = 4;})(LineInfo());
        assertExcNotThrown!(Exception, (){negInfinity.dayOfGregorianCal = 4;})(LineInfo());
        assertExcNotThrown!(Exception, (){notADate.dayOfGregorianCal = 4;})(LineInfo());

        {
            void testSpecial(Date date, size_t line = __LINE__)
            {
                date.dayOfGregorianCal = 17;
                assertEqual(date, Date(1, 1, 17), "", __FILE__, line);
            }

            testSpecial(posInfinity);
            testSpecial(negInfinity);
            testSpecial(notADate);
        }

        const cdate = Date(1999, 7, 6);
        immutable idate = Date(1999, 7, 6);
        static assert(!__traits(compiles, cdate.dayOfGregorianCal = 187));
        static assert(!__traits(compiles, idate.GregorianCal = 187));

        //Verify Examples
        {
            auto date = Date.init;
            date.dayOfGregorianCal = 1;
            assert(date == Date(1, 1, 1));

            date.dayOfGregorianCal = 365;
            assert(date == Date(1, 12, 31));

            date.dayOfGregorianCal = 366;
            assert(date == Date(2, 1, 1));

            date.dayOfGregorianCal = 0;
            assert(date == Date(0, 12, 31));

            date.dayOfGregorianCal = -365;
            assert(date == Date(-0, 1, 1));

            date.dayOfGregorianCal = -366;
            assert(date == Date(-1, 12, 31));

            date.dayOfGregorianCal = 730_120;
            assert(date == Date(2000, 1, 1));

            date.dayOfGregorianCal = 734_137;
            assert(date == Date(2010, 12, 31));
        }
    }


    /++
        The ISO 8601 week of the year that this Date is.

        See_Also:
            <a href="http://en.wikipedia.org/wiki/ISO_week_date">ISO Week Date</a>
      +/
    @property ISOWeek isoWeek() const
    {
        enforceNotSpecial();

        immutable weekday = dayOfWeek;
        immutable adjustedWeekday = weekday == DayOfWeek.sunday ? 7 : weekday;
        immutable day = dayOfYear;
        immutable week = (dayOfYear - adjustedWeekday + 10) / 7;

        if(week == 53)
        {
            switch(Date(cast(Year)(_year + 1), 1, 1).dayOfWeek)
            {
                case DayOfWeek.monday:
                case DayOfWeek.tuesday:
                case DayOfWeek.wednesday:
                case DayOfWeek.thursday:
                    return 1;
                case DayOfWeek.friday:
                case DayOfWeek.saturday:
                case DayOfWeek.sunday:
                    return 53;
                default:
                    assert(0);
            }
        }
        else if(week > 0)
            return cast(ISOWeek)week;
        else
            return Date(cast(Year)(_year - 1), 12, 31).isoWeek;
    }

    unittest
    {
        //Test A.D.
        assertEqual(Date(2009, 12, 28).isoWeek, 53);
        assertEqual(Date(2009, 12, 29).isoWeek, 53);
        assertEqual(Date(2009, 12, 30).isoWeek, 53);
        assertEqual(Date(2009, 12, 31).isoWeek, 53);
        assertEqual(Date(2010, 1, 1).isoWeek, 53);
        assertEqual(Date(2010, 1, 2).isoWeek, 53);
        assertEqual(Date(2010, 1, 3).isoWeek, 53);
        assertEqual(Date(2010, 1, 4).isoWeek, 1);
        assertEqual(Date(2010, 1, 5).isoWeek, 1);
        assertEqual(Date(2010, 1, 6).isoWeek, 1);
        assertEqual(Date(2010, 1, 7).isoWeek, 1);
        assertEqual(Date(2010, 1, 8).isoWeek, 1);
        assertEqual(Date(2010, 1, 9).isoWeek, 1);
        assertEqual(Date(2010, 1, 10).isoWeek, 1);
        assertEqual(Date(2010, 1, 11).isoWeek, 2);
        assertEqual(Date(2010, 12, 31).isoWeek, 52);

        assertEqual(Date(2004, 12, 26).isoWeek, 52);
        assertEqual(Date(2004, 12, 27).isoWeek, 53);
        assertEqual(Date(2004, 12, 28).isoWeek, 53);
        assertEqual(Date(2004, 12, 29).isoWeek, 53);
        assertEqual(Date(2004, 12, 30).isoWeek, 53);
        assertEqual(Date(2004, 12, 31).isoWeek, 53);
        assertEqual(Date(2005, 1, 1).isoWeek, 53);
        assertEqual(Date(2005, 1, 2).isoWeek, 53);

        assertEqual(Date(2005, 12, 31).isoWeek, 52);
        assertEqual(Date(2007, 1, 1).isoWeek, 1);

        assertEqual(Date(2007, 12, 30).isoWeek, 52);
        assertEqual(Date(2007, 12, 31).isoWeek, 1);
        assertEqual(Date(2008, 1, 1).isoWeek, 1);

        assertEqual(Date(2008, 12, 28).isoWeek, 52);
        assertEqual(Date(2008, 12, 29).isoWeek, 1);
        assertEqual(Date(2008, 12, 30).isoWeek, 1);
        assertEqual(Date(2008, 12, 31).isoWeek, 1);
        assertEqual(Date(2009, 1, 1).isoWeek, 1);
        assertEqual(Date(2009, 1, 2).isoWeek, 1);
        assertEqual(Date(2009, 1, 3).isoWeek, 1);
        assertEqual(Date(2009, 1, 4).isoWeek, 1);

        //Test B.C.
        //The algorithm should work identically for both A.D. and B.C. since
        //it doesn't really take the year into account, so B.C. testing
        //probably isn't really needed.
        assertEqual(Date(0, 12, 31).isoWeek, 52);
        assertEqual(Date(0, 1, 4).isoWeek, 1);
        assertEqual(Date(0, 1, 1).isoWeek, 52);

        assertExcThrown!(Exception, (){posInfinity.isoWeek;})(LineInfo());
        assertExcThrown!(Exception, (){negInfinity.isoWeek;})(LineInfo());
        assertExcThrown!(Exception, (){notADate.isoWeek;})(LineInfo());

        const cdate = Date(1999, 7, 6);
        static assert(__traits(compiles, cdate.isoWeek == 3));
        static assert(!__traits(compiles, cdate.isoWeek = 3));
        immutable idate = Date(1999, 7, 6);
        static assert(__traits(compiles, idate.isoWeek == 3));
        static assert(!__traits(compiles, idate.isoWeek = 3));
    }


    /++
        Date for the last day in the month that this Date is in.

        Examples:
        --------------------
        assert(Date(1999, 1, 6).endOfMonth == Date(1999, 1, 31));
        assert(Date(1999, 2, 7).endOfMonth == Date(1999, 2, 28));
        assert(Date(2000, 2, 7).endOfMonth == Date(1999, 2, 29));
        assert(Date(2000, 6, 4).endOfMonth == Date(1999, 6, 30));
        --------------------
      +/
    @property Date endOfMonth() const
    {
        enforceNotSpecial();

        return Date(_year, _month, maxDay(_year, _month));
    }

    unittest
    {
        //Test A.D.
        assertEqual(Date(1999, 1, 1).endOfMonth, Date(1999, 1, 31));
        assertEqual(Date(1999, 2, 1).endOfMonth, Date(1999, 2, 28));
        assertEqual(Date(2000, 2, 1).endOfMonth, Date(2000, 2, 29));
        assertEqual(Date(1999, 3, 1).endOfMonth, Date(1999, 3, 31));
        assertEqual(Date(1999, 4, 1).endOfMonth, Date(1999, 4, 30));
        assertEqual(Date(1999, 5, 1).endOfMonth, Date(1999, 5, 31));
        assertEqual(Date(1999, 6, 1).endOfMonth, Date(1999, 6, 30));
        assertEqual(Date(1999, 7, 1).endOfMonth, Date(1999, 7, 31));
        assertEqual(Date(1999, 8, 1).endOfMonth, Date(1999, 8, 31));
        assertEqual(Date(1999, 9, 1).endOfMonth, Date(1999, 9, 30));
        assertEqual(Date(1999, 10, 1).endOfMonth, Date(1999, 10, 31));
        assertEqual(Date(1999, 11, 1).endOfMonth, Date(1999, 11, 30));
        assertEqual(Date(1999, 12, 1).endOfMonth, Date(1999, 12, 31));

        //Test B.C.
        assertEqual(Date(-1999, 1, 1).endOfMonth, Date(-1999, 1, 31));
        assertEqual(Date(-1999, 2, 1).endOfMonth, Date(-1999, 2, 28));
        assertEqual(Date(-2000, 2, 1).endOfMonth, Date(-2000, 2, 29));
        assertEqual(Date(-1999, 3, 1).endOfMonth, Date(-1999, 3, 31));
        assertEqual(Date(-1999, 4, 1).endOfMonth, Date(-1999, 4, 30));
        assertEqual(Date(-1999, 5, 1).endOfMonth, Date(-1999, 5, 31));
        assertEqual(Date(-1999, 6, 1).endOfMonth, Date(-1999, 6, 30));
        assertEqual(Date(-1999, 7, 1).endOfMonth, Date(-1999, 7, 31));
        assertEqual(Date(-1999, 8, 1).endOfMonth, Date(-1999, 8, 31));
        assertEqual(Date(-1999, 9, 1).endOfMonth, Date(-1999, 9, 30));
        assertEqual(Date(-1999, 10, 1).endOfMonth, Date(-1999, 10, 31));
        assertEqual(Date(-1999, 11, 1).endOfMonth, Date(-1999, 11, 30));
        assertEqual(Date(-1999, 12, 1).endOfMonth, Date(-1999, 12, 31));

        assertExcThrown!(Exception, (){posInfinity.endOfMonth;})(LineInfo());
        assertExcThrown!(Exception, (){negInfinity.endOfMonth;})(LineInfo());
        assertExcThrown!(Exception, (){notADate.endOfMonth;})(LineInfo());

        const cdate = Date(1999, 7, 6);
        static assert(!__traits(compiles, cdate.endOfMonth = Date(1999, 7, 30)));
        immutable idate = Date(1999, 7, 6);
        static assert(!__traits(compiles, idate.endOfMonth = Date(1999, 7, 30)));

        //Verify Examples
        assert(Date(1999, 1, 6).endOfMonth == Date(1999, 1, 31));
        assert(Date(1999, 2, 7).endOfMonth == Date(1999, 2, 28));
        assert(Date(2000, 2, 7).endOfMonth == Date(2000, 2, 29));
        assert(Date(2000, 6, 4).endOfMonth == Date(2000, 6, 30));
    }


    /++
        The last day in the month that this Date is in.

        Examples:
        --------------------
        assert(Date(1999, 1, 6).endOfMonthDay == 31);
        assert(Date(1999, 2, 7).endOfMonthDay == 28);
        assert(Date(2000, 2, 7).endOfMonthDay == 29);
        assert(Date(2000, 6, 4).endOfMonthDay == 30);
        --------------------
      +/
    @property DayOfMonth endOfMonthDay() const
    {
        enforceNotSpecial();

        return maxDay(_year, _month);
    }

    unittest
    {
        //Test A.D.
        assertEqual(Date(1999, 1, 1).endOfMonthDay, 31);
        assertEqual(Date(1999, 2, 1).endOfMonthDay, 28);
        assertEqual(Date(2000, 2, 1).endOfMonthDay, 29);
        assertEqual(Date(1999, 3, 1).endOfMonthDay, 31);
        assertEqual(Date(1999, 4, 1).endOfMonthDay, 30);
        assertEqual(Date(1999, 5, 1).endOfMonthDay, 31);
        assertEqual(Date(1999, 6, 1).endOfMonthDay, 30);
        assertEqual(Date(1999, 7, 1).endOfMonthDay, 31);
        assertEqual(Date(1999, 8, 1).endOfMonthDay, 31);
        assertEqual(Date(1999, 9, 1).endOfMonthDay, 30);
        assertEqual(Date(1999, 10, 1).endOfMonthDay, 31);
        assertEqual(Date(1999, 11, 1).endOfMonthDay, 30);
        assertEqual(Date(1999, 12, 1).endOfMonthDay, 31);

        //Test B.C.
        assertEqual(Date(-1999, 1, 1).endOfMonthDay, 31);
        assertEqual(Date(-1999, 2, 1).endOfMonthDay, 28);
        assertEqual(Date(-2000, 2, 1).endOfMonthDay, 29);
        assertEqual(Date(-1999, 3, 1).endOfMonthDay, 31);
        assertEqual(Date(-1999, 4, 1).endOfMonthDay, 30);
        assertEqual(Date(-1999, 5, 1).endOfMonthDay, 31);
        assertEqual(Date(-1999, 6, 1).endOfMonthDay, 30);
        assertEqual(Date(-1999, 7, 1).endOfMonthDay, 31);
        assertEqual(Date(-1999, 8, 1).endOfMonthDay, 31);
        assertEqual(Date(-1999, 9, 1).endOfMonthDay, 30);
        assertEqual(Date(-1999, 10, 1).endOfMonthDay, 31);
        assertEqual(Date(-1999, 11, 1).endOfMonthDay, 30);
        assertEqual(Date(-1999, 12, 1).endOfMonthDay, 31);

        assertExcThrown!(Exception, (){posInfinity.endOfMonthDay;})(LineInfo());
        assertExcThrown!(Exception, (){negInfinity.endOfMonthDay;})(LineInfo());
        assertExcThrown!(Exception, (){notADate.endOfMonthDay;})(LineInfo());

        const cdate = Date(1999, 7, 6);
        static assert(!__traits(compiles, cdate.endOfMonthDay = 30));
        immutable idate = Date(1999, 7, 6);
        static assert(!__traits(compiles, idate.endOfMonthDay = 30));

        //Verify Examples
        assert(Date(1999, 1, 6).endOfMonthDay == 31);
        assert(Date(1999, 2, 7).endOfMonthDay == 28);
        assert(Date(2000, 2, 7).endOfMonthDay == 29);
        assert(Date(2000, 6, 4).endOfMonthDay == 30);
    }


    /++
        Whether the current year is a date in A.D.

        Examples:
        --------------------
        assert(Date(1, 1, 1).isAD);
        assert(Date(2010, 12, 31).isAD);
        assert(!Date(0, 12, 31).isAD);
        assert(!Date(-2010, 1, 1).isAD);
        --------------------
      +/
    @property bool isAD() const
    {
        enforceIsADate();

        return _year > 0;
    }

    unittest
    {
        assertExcNotThrown!(Exception, (){posInfinity.isAD;})(LineInfo());
        assertExcNotThrown!(Exception, (){negInfinity.isAD;})(LineInfo());
        assertExcThrown!(Exception, (){notADate.isAD;})(LineInfo());

        assert(posInfinity.isAD);
        assert(!negInfinity.isAD);

        auto date = Date(1999, 7, 6);
        const cdate = Date(1999, 7, 6);
        immutable idate = Date(1999, 7, 6);
        static assert(__traits(compiles, date.isAD));
        static assert(__traits(compiles, cdate.isAD));
        static assert(__traits(compiles, idate.isAD));

        //Verify Examples
        assert(Date(1, 1, 1).isAD);
        assert(Date(2010, 12, 31).isAD);
        assert(!Date(0, 12, 31).isAD);
        assert(!Date(-2010, 1, 1).isAD);
    }


    /++
        Returns true if isInfinity or isNAD are true.
      +/
    @property bool isSpecial() const
    {
        return _month == 0;
    }

    unittest
    {
        assertExcNotThrown!(Exception, (){posInfinity.isSpecial;})(LineInfo());
        assertExcNotThrown!(Exception, (){negInfinity.isSpecial;})(LineInfo());
        assertExcNotThrown!(Exception, (){notADate.isSpecial;})(LineInfo());

        assert(posInfinity.isSpecial);
        assert(negInfinity.isSpecial);
        assert(notADate.isSpecial);
        assert(!Date(1999, 7, 6).isSpecial);

        auto date = Date(1999, 7, 6);
        const cdate = Date(1999, 7, 6);
        immutable idate = Date(1999, 7, 6);
        static assert(__traits(compiles, date.isSpecial));
        static assert(__traits(compiles, cdate.isSpecial));
        static assert(__traits(compiles, idate.isSpecial));
    }


    /++
        Returns true if this date represents positive or negative infinity.
      +/
    @property bool isInfinity() const
    {
        return _month == 0 && (_day == 1 || _day == 2);
    }

    unittest
    {
        assert(posInfinity.isInfinity);
        assert(negInfinity.isInfinity);
        assert(!notADate.isInfinity);
        assert(!Date(0, 1, 1).isInfinity);
        assert(!Date(1, 1, 1).isInfinity);
        assert(!Date(1999, 7, 6).isInfinity);
        assert(!max.isInfinity);
        assert(!min.isInfinity);

        assertExcNotThrown!(Exception, (){posInfinity.isInfinity;})(LineInfo());
        assertExcNotThrown!(Exception, (){negInfinity.isInfinity;})(LineInfo());
        assertExcNotThrown!(Exception, (){notADate.isInfinity;})(LineInfo());

        auto date = Date(1999, 7, 6);
        const cdate = Date(1999, 7, 6);
        immutable idate = Date(1999, 7, 6);
        static assert(__traits(compiles, date.isInfinity));
        static assert(__traits(compiles, cdate.isInfinity));
        static assert(__traits(compiles, idate.isInfinity));
    }


    /++
        Returns true if this date represents positive.
      +/
    @property bool isPosInfinity() const
    {
        return _month == 0 && _day == 2;
    }

    unittest
    {
        assert(posInfinity.isPosInfinity);
        assert(!negInfinity.isPosInfinity);
        assert(!notADate.isPosInfinity);
        assert(!Date(0, 1, 1).isPosInfinity);
        assert(!Date(1, 1, 1).isPosInfinity);
        assert(!Date(1999, 7, 6).isPosInfinity);
        assert(!max.isPosInfinity);
        assert(!min.isPosInfinity);

        assertExcNotThrown!(Exception, (){posInfinity.isPosInfinity;})(LineInfo());
        assertExcNotThrown!(Exception, (){negInfinity.isPosInfinity;})(LineInfo());
        assertExcNotThrown!(Exception, (){notADate.isPosInfinity;})(LineInfo());

        auto date = Date(1999, 7, 6);
        const cdate = Date(1999, 7, 6);
        immutable idate = Date(1999, 7, 6);
        static assert(__traits(compiles, date.isPosInfinity));
        static assert(__traits(compiles, cdate.isPosInfinity));
        static assert(__traits(compiles, idate.isPosInfinity));
    }


    /++
        Returns true if this date represents negative infinity.
      +/
    @property bool isNegInfinity() const
    {
        return _month == 0 && _day == 1;
    }

    unittest
    {
        assert(!posInfinity.isNegInfinity);
        assert(negInfinity.isNegInfinity);
        assert(!notADate.isNegInfinity);
        assert(!Date(0, 1, 1).isNegInfinity);
        assert(!Date(1, 1, 1).isNegInfinity);
        assert(!Date(1999, 7, 6).isNegInfinity);
        assert(!max.isNegInfinity);
        assert(!min.isNegInfinity);

        assertExcNotThrown!(Exception, (){posInfinity.isNegInfinity;})(LineInfo());
        assertExcNotThrown!(Exception, (){negInfinity.isNegInfinity;})(LineInfo());
        assertExcNotThrown!(Exception, (){notADate.isNegInfinity;})(LineInfo());

        auto date = Date(1999, 7, 6);
        const cdate = Date(1999, 7, 6);
        immutable idate = Date(1999, 7, 6);
        static assert(__traits(compiles, date.isNegInfinity));
        static assert(__traits(compiles, cdate.isNegInfinity));
        static assert(__traits(compiles, idate.isNegInfinity));
    }


    /++
        Returns true if this date is NAD (Not-a-Date), the Date equivalent of NAN.
      +/
    @property bool isNAD() const
    {
        return _month == 0 && _day == 0;
    }

    unittest
    {
        assert(!posInfinity.isNAD);
        assert(!negInfinity.isNAD);
        assert(notADate.isNAD);
        assert(!Date(0, 1, 1).isNAD);
        assert(!Date(1, 1, 1).isNAD);
        assert(!Date(1999, 7, 6).isNAD);
        assert(!max.isNAD);
        assert(!min.isNAD);

        assertExcNotThrown!(Exception, (){posInfinity.isNAD;})(LineInfo());
        assertExcNotThrown!(Exception, (){negInfinity.isNAD;})(LineInfo());
        assertExcNotThrown!(Exception, (){notADate.isNAD;})(LineInfo());

        auto date = Date(1999, 7, 6);
        const cdate = Date(1999, 7, 6);
        immutable idate = Date(1999, 7, 6);
        static assert(__traits(compiles, date.isNAD));
        static assert(__traits(compiles, cdate.isNAD));
        static assert(__traits(compiles, idate.isNAD));
    }


    /++
        The julian day for this Date at noon (since the julian day changes at noon).
      +/
    @property long julianDay() const
    {
        enforceNotSpecial();

        return dayOfGregorianCal + 1_721_425;
    }

    unittest
    {
        assertEqual(Date(-4713, 11, 24).julianDay, 0);
        assertEqual(Date(0, 12, 31).julianDay, 1_721_425);
        assertEqual(Date(1, 1, 1).julianDay, 1_721_426);
        assertEqual(Date(1582, 10, 15).julianDay, 2_299_161);
        assertEqual(Date(1858, 11, 17).julianDay, 2_400_001);
        assertEqual(Date(1982, 1, 4).julianDay, 2_444_974);
        assertEqual(Date(1996, 3, 31).julianDay, 2_450_174);
        assertEqual(Date(2010, 8, 24).julianDay, 2_455_433);

        assertExcThrown!(Exception, (){posInfinity.julianDay;})(LineInfo());
        assertExcThrown!(Exception, (){negInfinity.julianDay;})(LineInfo());
        assertExcThrown!(Exception, (){notADate.julianDay;})(LineInfo());

        auto date = Date(1999, 7, 6);
        const cdate = Date(1999, 7, 6);
        immutable idate = Date(1999, 7, 6);
        static assert(__traits(compiles, date.julianDay));
        static assert(__traits(compiles, cdate.julianDay));
        static assert(__traits(compiles, idate.julianDay));
    }


    /++
        The modified julian day for this Date at any time (since, the modified julian day changes at midnight).
      +/
    @property long modJulianDay() const
    {
        enforceNotSpecial();

        return julianDay - 2_400_001;
    }

    unittest
    {
        assertEqual(Date(1858, 11, 17).modJulianDay, 0);
        assertEqual(Date(2010, 8, 24).modJulianDay, 55_432);

        assertExcThrown!(Exception, (){posInfinity.modJulianDay;})(LineInfo());
        assertExcThrown!(Exception, (){negInfinity.modJulianDay;})(LineInfo());
        assertExcThrown!(Exception, (){notADate.modJulianDay;})(LineInfo());

        auto date = Date(1999, 7, 6);
        const cdate = Date(1999, 7, 6);
        immutable idate = Date(1999, 7, 6);
        static assert(__traits(compiles, date.modJulianDay));
        static assert(__traits(compiles, cdate.modJulianDay));
        static assert(__traits(compiles, idate.modJulianDay));
    }


    /++
        Converts this Date to a string with the format YYYYMMDD.

        Examples:
        --------------------
        assert(Date(2010, 7, 4).toISOString() == "20100704");
        assert(Date(1998, 12, 25).toISOString() == "19981225");
        assert(Date(0, 1, 5).toISOString() == "00000105");
        assert(Date(-4, 1, 5).toISOString() == "-00040105");
        --------------------
      +/
    string toISOString() const
    {
        if(isSpecial)
            return specialToString();

        if(_year >= 0)
        {
            if(_year < 10_000)
                return format("%04d%02d%02d", _year, _month, _day);
            else
                return format("+%05d%02d%02d", _year, _month, _day);
        }
        else if(_year > -10_000)
            return format("%05d%02d%02d", _year, _month, _day);
        else
            return format("%06d%02d%02d", _year, _month, _day);
    }

    unittest
    {
        //Test A.D.
        assertEqual(Date(9, 12, 4).toISOString(), "00091204");
        assertEqual(Date(99, 12, 4).toISOString(), "00991204");
        assertEqual(Date(999, 12, 4).toISOString(), "09991204");
        assertEqual(Date(9999, 7, 4).toISOString(), "99990704");
        assertEqual(Date(10000, 10, 20).toISOString(), "+100001020");

        //Test B.C.
        assertEqual(Date(0, 12, 4).toISOString(), "00001204");
        assertEqual(Date(-9, 12, 4).toISOString(), "-00091204");
        assertEqual(Date(-99, 12, 4).toISOString(), "-00991204");
        assertEqual(Date(-999, 12, 4).toISOString(), "-09991204");
        assertEqual(Date(-9999, 7, 4).toISOString(), "-99990704");
        assertEqual(Date(-10000, 10, 20).toISOString(), "-100001020");

        assertExcNotThrown!(Exception, (){posInfinity.toISOString;})(LineInfo());
        assertExcNotThrown!(Exception, (){negInfinity.toISOString;})(LineInfo());
        assertExcNotThrown!(Exception, (){notADate.toISOString;})(LineInfo());

        assertEqual(posInfinity.toISOString(), posInfinity.specialToString());
        assertEqual(negInfinity.toISOString(), negInfinity.specialToString());
        assertEqual(notADate.toISOString(), notADate.specialToString());

        auto date = Date(1999, 7, 6);
        const cdate = Date(1999, 7, 6);
        immutable idate = Date(1999, 7, 6);
        static assert(__traits(compiles, date.toISOString()));
        static assert(__traits(compiles, cdate.toISOString()));
        static assert(__traits(compiles, idate.toISOString()));

        //Verify Examples
        assert(Date(2010, 7, 4).toISOString() == "20100704");
        assert(Date(1998, 12, 25).toISOString() == "19981225");
        assert(Date(0, 1, 5).toISOString() == "00000105");
        assert(Date(-4, 1, 5).toISOString() == "-00040105");
    }

    /++
        Converts this Date to a string with the format YYYY-MM-DD.

        Examples:
        --------------------
        assert(Date(2010, 7, 4).toISOExtendedString() == "2010-07-04");
        assert(Date(1998, 12, 25).toISOExtendedString() == "1998-12-25");
        assert(Date(0, 1, 5).toISOExtendedString() == "0000-01-05");
        assert(Date(-4, 1, 5).toISOExtendedString() == "-0004-01-05");
        --------------------
      +/
    string toISOExtendedString() const
    {
        if(isSpecial)
            return specialToString();

        if(_year >= 0)
        {
            if(_year < 10_000)
                return format("%04d-%02d-%02d", _year, _month, _day);
            else
                return format("+%05d-%02d-%02d", _year, _month, _day);
        }
        else if(_year > -10_000)
            return format("%05d-%02d-%02d", _year, _month, _day);
        else
            return format("%06d-%02d-%02d", _year, _month, _day);
    }

    unittest
    {
        //Test A.D.
        assertEqual(Date(9, 12, 4).toISOExtendedString(), "0009-12-04");
        assertEqual(Date(99, 12, 4).toISOExtendedString(), "0099-12-04");
        assertEqual(Date(999, 12, 4).toISOExtendedString(), "0999-12-04");
        assertEqual(Date(9999, 7, 4).toISOExtendedString(), "9999-07-04");
        assertEqual(Date(10000, 10, 20).toISOExtendedString(), "+10000-10-20");

        //Test B.C.
        assertEqual(Date(0, 12, 4).toISOExtendedString(), "0000-12-04");
        assertEqual(Date(-9, 12, 4).toISOExtendedString(), "-0009-12-04");
        assertEqual(Date(-99, 12, 4).toISOExtendedString(), "-0099-12-04");
        assertEqual(Date(-999, 12, 4).toISOExtendedString(), "-0999-12-04");
        assertEqual(Date(-9999, 7, 4).toISOExtendedString(), "-9999-07-04");
        assertEqual(Date(-10000, 10, 20).toISOExtendedString(), "-10000-10-20");

        assertExcNotThrown!(Exception, (){posInfinity.toISOExtendedString;})(LineInfo());
        assertExcNotThrown!(Exception, (){negInfinity.toISOExtendedString;})(LineInfo());
        assertExcNotThrown!(Exception, (){notADate.toISOExtendedString;})(LineInfo());

        assertEqual(posInfinity.toISOExtendedString(), posInfinity.specialToString());
        assertEqual(negInfinity.toISOExtendedString(), negInfinity.specialToString());
        assertEqual(notADate.toISOExtendedString(), notADate.specialToString());

        auto date = Date(1999, 7, 6);
        const cdate = Date(1999, 7, 6);
        immutable idate = Date(1999, 7, 6);
        static assert(__traits(compiles, date.toISOExtendedString()));
        static assert(__traits(compiles, cdate.toISOExtendedString()));
        static assert(__traits(compiles, idate.toISOExtendedString()));

        //Verify Examples
        assert(Date(2010, 7, 4).toISOExtendedString() == "2010-07-04");
        assert(Date(1998, 12, 25).toISOExtendedString() == "1998-12-25");
        assert(Date(0, 1, 5).toISOExtendedString() == "0000-01-05");
        assert(Date(-4, 1, 5).toISOExtendedString() == "-0004-01-05");
    }

    /++
        Converts this Date to a string with the format YYYY-MonthShortName-DD.

        Examples:
        --------------------
        assert(Date(2010, 7, 4).toSimpleString() == "2010-Jul-04");
        assert(Date(1998, 12, 25).toSimpleString() == "1998-Dec-25");
        assert(Date(0, 1, 5).toSimpleString() == "0000-Jan-05");
        assert(Date(-4, 1, 5).toSimpleString() == "-0004-Jan-05");
        --------------------
      +/
    string toSimpleString() const
    {
        if(isSpecial)
            return specialToString();

        if(_year >= 0)
        {
            if(_year < 10_000)
                return format("%04d-%s-%02d", _year, monthOfYearToString(_month, UseLongName.no), _day);
            else
                return format("+%05d-%s-%02d", _year, monthOfYearToString(_month, UseLongName.no), _day);
        }
        else if(_year > -10_000)
            return format("%05d-%s-%02d", _year, monthOfYearToString(_month, UseLongName.no), _day);
        else
            return format("%06d-%s-%02d", _year, monthOfYearToString(_month, UseLongName.no), _day);
    }

    unittest
    {
        //Test A.D.
        assertEqual(Date(9, 12, 4).toSimpleString(), "0009-Dec-04");
        assertEqual(Date(99, 12, 4).toSimpleString(), "0099-Dec-04");
        assertEqual(Date(999, 12, 4).toSimpleString(), "0999-Dec-04");
        assertEqual(Date(9999, 7, 4).toSimpleString(), "9999-Jul-04");
        assertEqual(Date(10000, 10, 20).toSimpleString(), "+10000-Oct-20");

        //Test B.C.
        assertEqual(Date(0, 12, 4).toSimpleString(), "0000-Dec-04");
        assertEqual(Date(-9, 12, 4).toSimpleString(), "-0009-Dec-04");
        assertEqual(Date(-99, 12, 4).toSimpleString(), "-0099-Dec-04");
        assertEqual(Date(-999, 12, 4).toSimpleString(), "-0999-Dec-04");
        assertEqual(Date(-9999, 7, 4).toSimpleString(), "-9999-Jul-04");
        assertEqual(Date(-10000, 10, 20).toSimpleString(), "-10000-Oct-20");

        assertExcNotThrown!(Exception, (){posInfinity.toSimpleString;})(LineInfo());
        assertExcNotThrown!(Exception, (){negInfinity.toSimpleString;})(LineInfo());
        assertExcNotThrown!(Exception, (){notADate.toSimpleString;})(LineInfo());

        assertEqual(posInfinity.toSimpleString(), posInfinity.specialToString());
        assertEqual(negInfinity.toSimpleString(), negInfinity.specialToString());
        assertEqual(notADate.toSimpleString(), notADate.specialToString());

        auto date = Date(1999, 7, 6);
        const cdate = Date(1999, 7, 6);
        immutable idate = Date(1999, 7, 6);
        static assert(__traits(compiles, date.toSimpleString()));
        static assert(__traits(compiles, cdate.toSimpleString()));
        static assert(__traits(compiles, idate.toSimpleString()));

        //Verify Examples
        assert(Date(2010, 7, 4).toSimpleString() == "2010-Jul-04");
        assert(Date(1998, 12, 25).toSimpleString() == "1998-Dec-25");
        assert(Date(0, 1, 5).toSimpleString() == "0000-Jan-05");
        assert(Date(-4, 1, 5).toSimpleString() == "-0004-Jan-05");
    }

    /++
        Converts this Date to a string.
      +/
    //For some reason, the const version doesn't work with format(), so I'm defining
    //it twice - one const and one non-const.
    string toString()
    {
        return toSimpleString();
    }

    /++
        Converts this Date to a string.
      +/
    string toString() const
    {
        return toSimpleString();
    }

    unittest
    {
        auto date = Date(1999, 7, 6);
        const cdate = Date(1999, 7, 6);
        immutable idate = Date(1999, 7, 6);
        static assert(__traits(compiles, date.toString()));
        static assert(__traits(compiles, cdate.toString()));
        static assert(__traits(compiles, idate.toString()));
    }


    /++
        Whether the given Year is a leap year.
     +/
    static bool yearIsLeapYear(Year year)
    {
        if(year % 400 == 0)
            return true;

        if(year % 100 == 0)
            return false;

        return year % 4 == 0;
    }

    unittest
    {
        //Test A.D.
        assert(!yearIsLeapYear(1));
        assert(!yearIsLeapYear(2));
        assert(!yearIsLeapYear(3));
        assert(yearIsLeapYear(4));
        assert(!yearIsLeapYear(5));
        assert(!yearIsLeapYear(6));
        assert(!yearIsLeapYear(7));
        assert(yearIsLeapYear(8));

        assert(!yearIsLeapYear(100));
        assert(!yearIsLeapYear(200));
        assert(!yearIsLeapYear(300));
        assert(yearIsLeapYear(400));
        assert(!yearIsLeapYear(500));
        assert(!yearIsLeapYear(600));
        assert(!yearIsLeapYear(700));
        assert(yearIsLeapYear(800));

        assert(yearIsLeapYear(1996));
        assert(!yearIsLeapYear(1997));
        assert(!yearIsLeapYear(1998));
        assert(!yearIsLeapYear(1999));
        assert(yearIsLeapYear(2000));
        assert(!yearIsLeapYear(2001));
        assert(!yearIsLeapYear(2002));
        assert(!yearIsLeapYear(2003));
        assert(yearIsLeapYear(2004));
        assert(!yearIsLeapYear(2005));
        assert(!yearIsLeapYear(2006));
        assert(!yearIsLeapYear(2007));
        assert(yearIsLeapYear(2008));
        assert(!yearIsLeapYear(2009));
        assert(!yearIsLeapYear(2010));

        //Test B.C.
        assert(yearIsLeapYear(0));
        assert(!yearIsLeapYear(-1));
        assert(!yearIsLeapYear(-2));
        assert(!yearIsLeapYear(-3));
        assert(yearIsLeapYear(-4));
        assert(!yearIsLeapYear(-5));
        assert(!yearIsLeapYear(-6));
        assert(!yearIsLeapYear(-7));
        assert(yearIsLeapYear(-8));

        assert(!yearIsLeapYear(-100));
        assert(!yearIsLeapYear(-200));
        assert(!yearIsLeapYear(-300));
        assert(yearIsLeapYear(-400));
        assert(!yearIsLeapYear(-500));
        assert(!yearIsLeapYear(-600));
        assert(!yearIsLeapYear(-700));
        assert(yearIsLeapYear(-800));

        assert(yearIsLeapYear(-1996));
        assert(!yearIsLeapYear(-1997));
        assert(!yearIsLeapYear(-1998));
        assert(!yearIsLeapYear(-1999));
        assert(yearIsLeapYear(-2000));
        assert(!yearIsLeapYear(-2001));
        assert(!yearIsLeapYear(-2002));
        assert(!yearIsLeapYear(-2003));
        assert(yearIsLeapYear(-2004));
        assert(!yearIsLeapYear(-2005));
        assert(!yearIsLeapYear(-2006));
        assert(!yearIsLeapYear(-2007));
        assert(yearIsLeapYear(-2008));
        assert(!yearIsLeapYear(-2009));
        assert(!yearIsLeapYear(-2010));
    }


    /++
        Returns a Date for which isPosInfinity is true.
      +/
    @property static Date posInfinity()
    out(result)
    {
        assert(result.isPosInfinity);
        assert(result.isSpecial);
    }
    body
    {
        auto date = Date.init;
        date._year = Year.max;
        date._month = 0;
        date._day = 2;

        return date;
    }

    /++
        Returns a Date for which isNegInfinity is true.
      +/
    @property static Date negInfinity()
    out(result)
    {
        assert(result.isNegInfinity);
        assert(result.isSpecial);
    }
    body
    {
        auto date = Date.init;
        date._year = Year.min;
        date._month = 0;
        date._day = 1;

        return date;
    }

    /++
        Returns a Date for which isNAD is true.
      +/
    @property static Date notADate()
    out(result)
    {
        assert(result.isNAD);
        assert(result.isSpecial);
    }
    body
    {
        auto date = Date.init;
        date._month = 0;
        date._day = 0;

        return date;
    }


    /++
        Returns the Date farthest in the past which is representable by Date.
      +/
    @property static Date min()
    out(result)
    {
        assert(!result.isSpecial);
    }
    body
    {
        return Date(Year.min, 1, 1);
    }

    /++
        Returns the Date farthest in the future which is representable by Date.
      +/
    @property static Date max()
    out(result)
    {
        assert(!result.isSpecial);
    }
    body
    {
        return Date(Year.max, december, 31);
    }


    /++
        The lowest time resolution that Date has.
      +/
    @property static TimeUnit minResolution()
    {
        return TimeUnit.year;
    }


    /++
        The highest time resolution that Date has.
      +/
    @property static TimeUnit maxResolution()
    {
        return TimeUnit.day;
    }


private:

    /++
        Returns the tring representation of special dates.
      +/
    string specialToString() const
    {
        assert(isSpecial);

        if(isNAD)
            return "Not-A-Date";

        if(isPosInfinity)
            return "∞";

        return "-∞";
    }

    unittest
    {
        assertEqual(notADate.specialToString(), "Not-A-Date");
        assertEqual(posInfinity.specialToString(), "∞");
        assertEqual(negInfinity.specialToString(), "-∞");
    }


    /++
        Throws exception if date is special.
      +/
    void enforceNotSpecial(size_t line = __LINE__) const
    {
        if(isSpecial)
            throw new Exception("Invalid operation for a special date.", __FILE__, line);
    }


    /++
        Throws exception if date is Not-A-Date.
      +/
    void enforceIsADate(size_t line = __LINE__) const
    {
        if(isNAD)
            throw new Exception("Date is not a Not-A-Date.", __FILE__, line);
    }


    /++
        Whether the given month is a valid Gregorian month.
     +/
    static bool validMonthOfYear(MonthOfYear month)
    {
        return month >= january && month <= december;
    }


    /++
        Whether the given day is a valid day of the given Gregorian Year and Gregorian Month.
     +/
    static bool validDayOfMonth(Year year, MonthOfYear month, DayOfMonth day)
    {
        return day > 0 && day <= maxDay(year, month);
    }


    /++
        Whether the given values form a valid date.
     +/
    static bool validDate(Year year, MonthOfYear month, DayOfMonth day)
    {
        if(!validMonthOfYear(month))
            return false;

        return validDayOfMonth(year, month, day);
    }


    /++
        The maximum valid Day in the given month in the given year.
     +/
    static DayOfMonth maxDay(Year year, MonthOfYear month)
    in
    {
        assert(month >= january && month <= december);
    }
    body
    {
        switch(month)
        {
            case january, march, may, july, august, october, december:
                return 31;
            case february:
                return yearIsLeapYear(year) ? 29 : 28;
            case april, june, september, november:
                return 30;
            default:
                assert(0);
        }
    }

    unittest
    {
        //Test A.D.
        assertEqual(maxDay(1999, 1), 31);
        assertEqual(maxDay(1999, 2), 28);
        assertEqual(maxDay(1999, 3), 31);
        assertEqual(maxDay(1999, 4), 30);
        assertEqual(maxDay(1999, 5), 31);
        assertEqual(maxDay(1999, 6), 30);
        assertEqual(maxDay(1999, 7), 31);
        assertEqual(maxDay(1999, 8), 31);
        assertEqual(maxDay(1999, 9), 30);
        assertEqual(maxDay(1999, 10), 31);
        assertEqual(maxDay(1999, 11), 30);
        assertEqual(maxDay(1999, 12), 31);

        assertEqual(maxDay(2000, 1), 31);
        assertEqual(maxDay(2000, 2), 29);
        assertEqual(maxDay(2000, 3), 31);
        assertEqual(maxDay(2000, 4), 30);
        assertEqual(maxDay(2000, 5), 31);
        assertEqual(maxDay(2000, 6), 30);
        assertEqual(maxDay(2000, 7), 31);
        assertEqual(maxDay(2000, 8), 31);
        assertEqual(maxDay(2000, 9), 30);
        assertEqual(maxDay(2000, 10), 31);
        assertEqual(maxDay(2000, 11), 30);
        assertEqual(maxDay(2000, 12), 31);

        //Test B.C.
        assertEqual(maxDay(-1999, 1), 31);
        assertEqual(maxDay(-1999, 2), 28);
        assertEqual(maxDay(-1999, 3), 31);
        assertEqual(maxDay(-1999, 4), 30);
        assertEqual(maxDay(-1999, 5), 31);
        assertEqual(maxDay(-1999, 6), 30);
        assertEqual(maxDay(-1999, 7), 31);
        assertEqual(maxDay(-1999, 8), 31);
        assertEqual(maxDay(-1999, 9), 30);
        assertEqual(maxDay(-1999, 10), 31);
        assertEqual(maxDay(-1999, 11), 30);
        assertEqual(maxDay(-1999, 12), 31);

        assertEqual(maxDay(-2000, 1), 31);
        assertEqual(maxDay(-2000, 2), 29);
        assertEqual(maxDay(-2000, 3), 31);
        assertEqual(maxDay(-2000, 4), 30);
        assertEqual(maxDay(-2000, 5), 31);
        assertEqual(maxDay(-2000, 6), 30);
        assertEqual(maxDay(-2000, 7), 31);
        assertEqual(maxDay(-2000, 8), 31);
        assertEqual(maxDay(-2000, 9), 30);
        assertEqual(maxDay(-2000, 10), 31);
        assertEqual(maxDay(-2000, 11), 30);
        assertEqual(maxDay(-2000, 12), 31);
    }


    invariant()
    {
        immutable special = _month == 0 && (_day >= 0 && _day <= 2);

        assert(special || validDate(_year, _month, _day));
    }


    Year        _year  = 1;
    MonthOfYear _month = january;
    DayOfMonth  _day   = 1;


    enum daysInYear     = 365;
    enum daysInLeapYear = 366;
    enum daysIn4Years   = daysInYear * 3 + daysInLeapYear;
    enum daysIn100Years = daysIn4Years * 25 - 1;
    enum daysIn400Years = daysIn100Years * 4 + 1;
}



/++
    Represents a time of day with hours, minutes, and seconds. It uses 24 hour time.
+/
struct TimeOfDay
{
public:

    /++
        Params:
            hours   = Hour of the day [0 - 24&#41;.
            minutes = Minute of the hour [0 - 60&#41;.
            seconds = Second of the minute [0 - 60&#41;.
     +/
    this(Hour hours, Minute minutes, Second seconds = 0)
    {
        enforce(hours <= maxHours, "Invalid hour of the day");
        enforce(minutes <= maxMinutes, "Invalid minute of an hour");
        enforce(seconds <= maxSeconds, "Invalid second of a minute");

        _hours = hours;
        _minutes = minutes;
        _seconds = seconds;
    }

    unittest
    {
        assert(TimeOfDay(0, 0) == TimeOfDay.init);

        {
            auto tod = TimeOfDay(0, 0);
            assertEqual(tod._hours, 0);
            assertEqual(tod._minutes, 0);
            assertEqual(tod._seconds, 0);
        }

        {
            auto tod = TimeOfDay(12, 30, 33);
            assertEqual(tod._hours, 12);
            assertEqual(tod._minutes, 30);
            assertEqual(tod._seconds, 33);
        }

        {
            auto tod = TimeOfDay(23, 59, 59);
            assertEqual(tod._hours, 23);
            assertEqual(tod._minutes, 59);
            assertEqual(tod._seconds, 59);
        }

        assertExcThrown!(Exception, TimeOfDay)(LineInfo(), cast(Hour)24, cast(Minute)0, cast(Second)0);
        assertExcThrown!(Exception, TimeOfDay)(LineInfo(), cast(Hour)0, cast(Minute)60, cast(Second)0);
        assertExcThrown!(Exception, TimeOfDay)(LineInfo(), cast(Hour)0, cast(Minute)0, cast(Second)60);
    }


    /++
       Compares this TimeOfDay with the given TimeOfDay.

       Returns:
        $(TABLE
        $(TR $(TD this &lt; rhs) $(TD &lt; 0))
        $(TR $(TD this == rhs) $(TD 0))
        $(TR $(TD this &gt; rhs) $(TD &gt; 0))
        )
     +/
    int opCmp(in TimeOfDay rhs) const
    {
        if(_hours < rhs._hours)
            return -1;
        if(_hours > rhs._hours)
            return 1;

        if(_minutes < rhs._minutes)
            return -1;
        if(_minutes > rhs._minutes)
            return 1;

        if(_seconds < rhs._seconds)
            return -1;
        if(_seconds > rhs._seconds)
            return 1;

        return 0;
    }

    unittest
    {
        assertOpCmp!("==")(TimeOfDay(0, 0, 0), TimeOfDay.init);

        assertOpCmp!("==")(TimeOfDay(0, 0, 0), TimeOfDay(0, 0, 0));
        assertOpCmp!("==")(TimeOfDay(12, 0, 0), TimeOfDay(12, 0, 0));
        assertOpCmp!("==")(TimeOfDay(0, 30, 0), TimeOfDay(0, 30, 0));
        assertOpCmp!("==")(TimeOfDay(0, 0, 33), TimeOfDay(0, 0, 33));

        assertOpCmp!("==")(TimeOfDay(12, 30, 0), TimeOfDay(12, 30, 0));
        assertOpCmp!("==")(TimeOfDay(12, 30, 33), TimeOfDay(12, 30, 33));

        assertOpCmp!("==")(TimeOfDay(0, 30, 33), TimeOfDay(0, 30, 33));
        assertOpCmp!("==")(TimeOfDay(0, 0, 33), TimeOfDay(0, 0, 33));

        assertOpCmp!("<")(TimeOfDay(12, 30, 33), TimeOfDay(13, 30, 33));
        assertOpCmp!(">")(TimeOfDay(13, 30, 33), TimeOfDay(12, 30, 33));
        assertOpCmp!("<")(TimeOfDay(12, 30, 33), TimeOfDay(12, 31, 33));
        assertOpCmp!(">")(TimeOfDay(12, 31, 33), TimeOfDay(12, 30, 33));
        assertOpCmp!("<")(TimeOfDay(12, 30, 33), TimeOfDay(12, 30, 34));
        assertOpCmp!(">")(TimeOfDay(12, 30, 34), TimeOfDay(12, 30, 33));

        assertOpCmp!(">")(TimeOfDay(13, 30, 33), TimeOfDay(12, 30, 34));
        assertOpCmp!("<")(TimeOfDay(12, 30, 34), TimeOfDay(13, 30, 33));
        assertOpCmp!(">")(TimeOfDay(13, 30, 33), TimeOfDay(12, 31, 33));
        assertOpCmp!("<")(TimeOfDay(12, 31, 33), TimeOfDay(13, 30, 33));

        assertOpCmp!(">")(TimeOfDay(12, 31, 33), TimeOfDay(12, 30, 34));
        assertOpCmp!("<")(TimeOfDay(12, 30, 34), TimeOfDay(12, 31, 33));

        const ctod = TimeOfDay(12, 30, 33);
        immutable itod = TimeOfDay(12, 30, 33);
        static assert(__traits(compiles, ctod.opCmp(itod)));
        static assert(__traits(compiles, itod.opCmp(ctod)));
    }


    /++
        Hours past midnight.
     +/
    @property Hour hours() const
    {
        return _hours;
    }

    unittest
    {
        assertEqual(TimeOfDay.init.hours, 0);
        assertEqual(TimeOfDay(12, 0, 0).hours, 12);

        const ctod = TimeOfDay(12, 0, 0);
        static assert(__traits(compiles, ctod.hours == 12));
        immutable itod = TimeOfDay(12, 0, 0);
        static assert(__traits(compiles, itod.hours == 12));
    }


    /++
        Hours past midnight.

        Params:
            hours = The hours to set this TimeOfDay's hours to.
     +/
    @property void hours(Hour hours)
    {
        enforce(hours <= maxHours, "Invalid hour of the day");
        _hours = hours;
    }

    unittest
    {
        try
        {
            auto tod = TimeOfDay(0, 0, 0);
            tod.hours = 24;
            assert(0, "Exception not thrown");
        }
        catch(Exception e)
        {
        }

        auto tod = TimeOfDay(0, 0, 0);
        tod.hours = 12;
        assertEqual(tod, TimeOfDay(12, 0, 0));

        const ctod = TimeOfDay(0, 0, 0);
        static assert(!__traits(compiles, ctod.hours = 12));
        immutable itod = TimeOfDay(0, 0, 0);
        static assert(!__traits(compiles, itod.hours = 12));
    }


    /++
        Minutes past the current hour.
     +/
    @property Minute minutes() const
    {
        return _minutes;
    }

    unittest
    {
        assertEqual(TimeOfDay.init.minutes, 0);
        assertEqual(TimeOfDay(0, 30, 0).minutes, 30);

        const ctod = TimeOfDay(0, 30, 0);
        static assert(__traits(compiles, ctod.minutes == 30));
        immutable itod = TimeOfDay(0, 30, 0);
        static assert(__traits(compiles, itod.minutes == 30));
    }


    /++
        Minutes past the current hour.

        Params:
            min = The minutes to set this TimeOfDay's minutes to.
     +/
    @property void minutes(Minute minutes)
    {
        enforce(minutes <= maxMinutes, "Invalid minute of an hour");
        _minutes = minutes;
    }

    unittest
    {
        try
        {
            auto tod = TimeOfDay(0, 0, 0);
            tod.minutes = 60;
            assert(0, "Exception not thrown");
        }
        catch(Exception e)
        {
        }

        auto tod = TimeOfDay(0, 0, 0);
        tod.minutes = 30;
        assertEqual(tod, TimeOfDay(0, 30, 0));

        const ctod = TimeOfDay(0, 0, 0);
        static assert(!__traits(compiles, ctod.minutes = 30));
        immutable itod = TimeOfDay(0, 0, 0);
        static assert(!__traits(compiles, itod.minutes = 30));
    }


    /++
        Seconds past the current minute.
     +/
    @property Second seconds() const
    {
        return _seconds;
    }

    unittest
    {
        assertEqual(TimeOfDay.init.seconds, 0);
        assertEqual(TimeOfDay(0, 0, 33).seconds, 33);

        const ctod = TimeOfDay(0, 0, 33);
        static assert(__traits(compiles, ctod.seconds == 33));
        immutable itod = TimeOfDay(0, 0, 33);
        static assert(__traits(compiles, itod.seconds == 33));
    }


    /++
        Seconds past the current minute.

        Params:
            sec = The seconds to set this TimeOfDay's seconds to.
     +/
    @property void seconds(Second seconds)
    {
        enforce(seconds <= maxSeconds, "Invalid second of a minute");
        _seconds = seconds;
    }

    unittest
    {
        try
        {
            auto tod = TimeOfDay(0, 0, 0);
            tod.seconds = 60;
            assert(0, "Exception not thrown");
        }
        catch(Exception e)
        {
        }

        auto tod = TimeOfDay(0, 0, 0);
        tod.seconds = 33;
        assertEqual(tod, TimeOfDay(0, 0, 33));

        const ctod = TimeOfDay(0, 0, 0);
        static assert(!__traits(compiles, ctod.seconds = 33));
        immutable itod = TimeOfDay(0, 0, 0);
        static assert(!__traits(compiles, itod.seconds = 33));
    }


    /++
        Add hours to the time of day. Negative values will subtract. If the number of hours
        overflows or underflows, then the hours will wrap (e.g. adding 3 hours to 23:00 would
        result in 02:00).

        Params:
            hours = The number of hours to add to this TimeOfDay.
      +/
    void addHours(int hours)
    {
        hours %= 24;

        if(hours < 0)
            _hours += 24;

        _hours += hours;
        _hours %= 24;
    }


    unittest
    {
        static void testTOD(TimeOfDay orig, int hours, in TimeOfDay expected, size_t line = __LINE__)
        {
            orig.addHours(hours);
            assertEqual(orig, expected, "", __FILE__, line);
        }

        testTOD(TimeOfDay(12, 30, 33), 0, TimeOfDay(12, 30, 33));
        testTOD(TimeOfDay(12, 30, 33), 1, TimeOfDay(13, 30, 33));
        testTOD(TimeOfDay(12, 30, 33), 2, TimeOfDay(14, 30, 33));
        testTOD(TimeOfDay(12, 30, 33), 3, TimeOfDay(15, 30, 33));
        testTOD(TimeOfDay(12, 30, 33), 4, TimeOfDay(16, 30, 33));
        testTOD(TimeOfDay(12, 30, 33), 5, TimeOfDay(17, 30, 33));
        testTOD(TimeOfDay(12, 30, 33), 6, TimeOfDay(18, 30, 33));
        testTOD(TimeOfDay(12, 30, 33), 7, TimeOfDay(19, 30, 33));
        testTOD(TimeOfDay(12, 30, 33), 8, TimeOfDay(20, 30, 33));
        testTOD(TimeOfDay(12, 30, 33), 9, TimeOfDay(21, 30, 33));
        testTOD(TimeOfDay(12, 30, 33), 10, TimeOfDay(22, 30, 33));
        testTOD(TimeOfDay(12, 30, 33), 11, TimeOfDay(23, 30, 33));
        testTOD(TimeOfDay(12, 30, 33), 12, TimeOfDay(0, 30, 33));
        testTOD(TimeOfDay(12, 30, 33), 13, TimeOfDay(1, 30, 33));
        testTOD(TimeOfDay(12, 30, 33), 14, TimeOfDay(2, 30, 33));
        testTOD(TimeOfDay(12, 30, 33), 15, TimeOfDay(3, 30, 33));
        testTOD(TimeOfDay(12, 30, 33), 16, TimeOfDay(4, 30, 33));
        testTOD(TimeOfDay(12, 30, 33), 17, TimeOfDay(5, 30, 33));
        testTOD(TimeOfDay(12, 30, 33), 18, TimeOfDay(6, 30, 33));
        testTOD(TimeOfDay(12, 30, 33), 19, TimeOfDay(7, 30, 33));
        testTOD(TimeOfDay(12, 30, 33), 20, TimeOfDay(8, 30, 33));
        testTOD(TimeOfDay(12, 30, 33), 21, TimeOfDay(9, 30, 33));
        testTOD(TimeOfDay(12, 30, 33), 22, TimeOfDay(10, 30, 33));
        testTOD(TimeOfDay(12, 30, 33), 23, TimeOfDay(11, 30, 33));
        testTOD(TimeOfDay(12, 30, 33), 24, TimeOfDay(12, 30, 33));
        testTOD(TimeOfDay(12, 30, 33), 25, TimeOfDay(13, 30, 33));

        testTOD(TimeOfDay(12, 30, 33), -1, TimeOfDay(11, 30, 33));
        testTOD(TimeOfDay(12, 30, 33), -2, TimeOfDay(10, 30, 33));
        testTOD(TimeOfDay(12, 30, 33), -3, TimeOfDay(9, 30, 33));
        testTOD(TimeOfDay(12, 30, 33), -4, TimeOfDay(8, 30, 33));
        testTOD(TimeOfDay(12, 30, 33), -5, TimeOfDay(7, 30, 33));
        testTOD(TimeOfDay(12, 30, 33), -6, TimeOfDay(6, 30, 33));
        testTOD(TimeOfDay(12, 30, 33), -7, TimeOfDay(5, 30, 33));
        testTOD(TimeOfDay(12, 30, 33), -8, TimeOfDay(4, 30, 33));
        testTOD(TimeOfDay(12, 30, 33), -9, TimeOfDay(3, 30, 33));
        testTOD(TimeOfDay(12, 30, 33), -10, TimeOfDay(2, 30, 33));
        testTOD(TimeOfDay(12, 30, 33), -11, TimeOfDay(1, 30, 33));
        testTOD(TimeOfDay(12, 30, 33), -12, TimeOfDay(0, 30, 33));
        testTOD(TimeOfDay(12, 30, 33), -13, TimeOfDay(23, 30, 33));
        testTOD(TimeOfDay(12, 30, 33), -14, TimeOfDay(22, 30, 33));
        testTOD(TimeOfDay(12, 30, 33), -15, TimeOfDay(21, 30, 33));
        testTOD(TimeOfDay(12, 30, 33), -16, TimeOfDay(20, 30, 33));
        testTOD(TimeOfDay(12, 30, 33), -17, TimeOfDay(19, 30, 33));
        testTOD(TimeOfDay(12, 30, 33), -18, TimeOfDay(18, 30, 33));
        testTOD(TimeOfDay(12, 30, 33), -19, TimeOfDay(17, 30, 33));
        testTOD(TimeOfDay(12, 30, 33), -20, TimeOfDay(16, 30, 33));
        testTOD(TimeOfDay(12, 30, 33), -21, TimeOfDay(15, 30, 33));
        testTOD(TimeOfDay(12, 30, 33), -22, TimeOfDay(14, 30, 33));
        testTOD(TimeOfDay(12, 30, 33), -23, TimeOfDay(13, 30, 33));
        testTOD(TimeOfDay(12, 30, 33), -24, TimeOfDay(12, 30, 33));
        testTOD(TimeOfDay(12, 30, 33), -25, TimeOfDay(11, 30, 33));

        testTOD(TimeOfDay(0, 30, 33), 1, TimeOfDay(1, 30, 33));
        testTOD(TimeOfDay(0, 30, 33), 0, TimeOfDay(0, 30, 33));
        testTOD(TimeOfDay(0, 30, 33), -1, TimeOfDay(23, 30, 33));

        testTOD(TimeOfDay(23, 30, 33), 1, TimeOfDay(0, 30, 33));
        testTOD(TimeOfDay(23, 30, 33), 0, TimeOfDay(23, 30, 33));
        testTOD(TimeOfDay(23, 30, 33), -1, TimeOfDay(22, 30, 33));

        const ctod = TimeOfDay(0, 0, 0);
        static assert(!__traits(compiles, ctod.addHours(7)));
        immutable itod = TimeOfDay(0, 0, 0);
        static assert(!__traits(compiles, itod.addHours(7)));
    }

    /++
        Add minutes to the time of day. Negative values will subtract. If the number of minutes
        overflows (or underflows), then the minutes will wrap, increasing (or decreasing) the
        number of hours accordingly. If the number of hours overflows (or underflows), then the
        hours will wrap. (e.g. adding 90 minutes to 23:30 would result in 01:00).

        Params:
            minutes = The numbef or Minutes to add to this TimeOfDay.
      +/
    void addMinutes(int minutes)
    {
        auto hours = minutes / 60;
        minutes %= 60;
        auto newMinute = cast(Minute)(_minutes) + minutes;

        if(minutes < 0)
        {
            if(newMinute < 0)
            {
                newMinute += 60;
                --hours;
            }
        }
        else
        {
            if(newMinute >= 60)
            {
                newMinute -= 60;
                ++hours;
            }
        }

        _minutes = cast(Minute)newMinute;
        addHours(hours);
    }

    unittest
    {
        static void testTOD(TimeOfDay orig, int minutes, in TimeOfDay expected, size_t line = __LINE__)
        {
            orig.addMinutes(minutes);
            assertEqual(orig, expected, "", __FILE__, line);
        }

        testTOD(TimeOfDay(12, 30, 33), 0, TimeOfDay(12, 30, 33));
        testTOD(TimeOfDay(12, 30, 33), 1, TimeOfDay(12, 31, 33));
        testTOD(TimeOfDay(12, 30, 33), 2, TimeOfDay(12, 32, 33));
        testTOD(TimeOfDay(12, 30, 33), 3, TimeOfDay(12, 33, 33));
        testTOD(TimeOfDay(12, 30, 33), 4, TimeOfDay(12, 34, 33));
        testTOD(TimeOfDay(12, 30, 33), 5, TimeOfDay(12, 35, 33));
        testTOD(TimeOfDay(12, 30, 33), 10, TimeOfDay(12, 40, 33));
        testTOD(TimeOfDay(12, 30, 33), 15, TimeOfDay(12, 45, 33));
        testTOD(TimeOfDay(12, 30, 33), 29, TimeOfDay(12, 59, 33));
        testTOD(TimeOfDay(12, 30, 33), 30, TimeOfDay(13, 0, 33));
        testTOD(TimeOfDay(12, 30, 33), 45, TimeOfDay(13, 15, 33));
        testTOD(TimeOfDay(12, 30, 33), 60, TimeOfDay(13, 30, 33));
        testTOD(TimeOfDay(12, 30, 33), 75, TimeOfDay(13, 45, 33));
        testTOD(TimeOfDay(12, 30, 33), 90, TimeOfDay(14, 0, 33));
        testTOD(TimeOfDay(12, 30, 33), 100, TimeOfDay(14, 10, 33));

        testTOD(TimeOfDay(12, 30, 33), 689, TimeOfDay(23, 59, 33));
        testTOD(TimeOfDay(12, 30, 33), 690, TimeOfDay(0, 0, 33));
        testTOD(TimeOfDay(12, 30, 33), 691, TimeOfDay(0, 1, 33));
        testTOD(TimeOfDay(12, 30, 33), 960, TimeOfDay(4, 30, 33));
        testTOD(TimeOfDay(12, 30, 33), 1439, TimeOfDay(12, 29, 33));
        testTOD(TimeOfDay(12, 30, 33), 1440, TimeOfDay(12, 30, 33));
        testTOD(TimeOfDay(12, 30, 33), 1441, TimeOfDay(12, 31, 33));
        testTOD(TimeOfDay(12, 30, 33), 2880, TimeOfDay(12, 30, 33));

        testTOD(TimeOfDay(12, 30, 33), -1, TimeOfDay(12, 29, 33));
        testTOD(TimeOfDay(12, 30, 33), -2, TimeOfDay(12, 28, 33));
        testTOD(TimeOfDay(12, 30, 33), -3, TimeOfDay(12, 27, 33));
        testTOD(TimeOfDay(12, 30, 33), -4, TimeOfDay(12, 26, 33));
        testTOD(TimeOfDay(12, 30, 33), -5, TimeOfDay(12, 25, 33));
        testTOD(TimeOfDay(12, 30, 33), -10, TimeOfDay(12, 20, 33));
        testTOD(TimeOfDay(12, 30, 33), -15, TimeOfDay(12, 15, 33));
        testTOD(TimeOfDay(12, 30, 33), -29, TimeOfDay(12, 1, 33));
        testTOD(TimeOfDay(12, 30, 33), -30, TimeOfDay(12, 0, 33));
        testTOD(TimeOfDay(12, 30, 33), -45, TimeOfDay(11, 45, 33));
        testTOD(TimeOfDay(12, 30, 33), -60, TimeOfDay(11, 30, 33));
        testTOD(TimeOfDay(12, 30, 33), -75, TimeOfDay(11, 15, 33));
        testTOD(TimeOfDay(12, 30, 33), -90, TimeOfDay(11, 0, 33));
        testTOD(TimeOfDay(12, 30, 33), -100, TimeOfDay(10, 50, 33));

        testTOD(TimeOfDay(12, 30, 33), -749, TimeOfDay(0, 1, 33));
        testTOD(TimeOfDay(12, 30, 33), -750, TimeOfDay(0, 0, 33));
        testTOD(TimeOfDay(12, 30, 33), -751, TimeOfDay(23, 59, 33));
        testTOD(TimeOfDay(12, 30, 33), -960, TimeOfDay(20, 30, 33));
        testTOD(TimeOfDay(12, 30, 33), -1439, TimeOfDay(12, 31, 33));
        testTOD(TimeOfDay(12, 30, 33), -1440, TimeOfDay(12, 30, 33));
        testTOD(TimeOfDay(12, 30, 33), -1441, TimeOfDay(12, 29, 33));
        testTOD(TimeOfDay(12, 30, 33), -2880, TimeOfDay(12, 30, 33));

        testTOD(TimeOfDay(12, 0, 33), 1, TimeOfDay(12, 1, 33));
        testTOD(TimeOfDay(12, 0, 33), 0, TimeOfDay(12, 0, 33));
        testTOD(TimeOfDay(12, 0, 33), -1, TimeOfDay(11, 59, 33));

        testTOD(TimeOfDay(11, 59, 33), 1, TimeOfDay(12, 0, 33));
        testTOD(TimeOfDay(11, 59, 33), 0, TimeOfDay(11, 59, 33));
        testTOD(TimeOfDay(11, 59, 33), -1, TimeOfDay(11, 58, 33));

        testTOD(TimeOfDay(0, 0, 33), 1, TimeOfDay(0, 1, 33));
        testTOD(TimeOfDay(0, 0, 33), 0, TimeOfDay(0, 0, 33));
        testTOD(TimeOfDay(0, 0, 33), -1, TimeOfDay(23, 59, 33));

        testTOD(TimeOfDay(23, 59, 33), 1, TimeOfDay(0, 0, 33));
        testTOD(TimeOfDay(23, 59, 33), 0, TimeOfDay(23, 59, 33));
        testTOD(TimeOfDay(23, 59, 33), -1, TimeOfDay(23, 58, 33));

        const ctod = TimeOfDay(0, 0, 0);
        static assert(!__traits(compiles, ctod.addMinutes(7)));
        immutable itod = TimeOfDay(0, 0, 0);
        static assert(!__traits(compiles, itod.addMinutes(7)));
    }


    /++
        Add seconds to the time of day. Negative values will subtract. If the number of seconds
        overflows (or underflows), then the seconds will wrap, increasing (or decreasing) the
        number of minutes accordingly. If the number of minutes overflows (or underflows), then the
        minutes will wrap. If the number of minutes overflows(or underflows), then the hour will
        wrap. (e.g. adding 90 seconds to 23:59:00 would result in 00:00:30).

        Params:
            seconds = The number of seconds to add to this TimeOfDay.
      +/
    void addSeconds(int seconds)
    {
        auto minutes = seconds / 60;
        seconds %= 60;
        auto newSecond = cast(Second)(_seconds) + seconds;

        if(seconds < 0)
        {
            if(newSecond < 0)
            {
                newSecond += 60;
                --minutes;
            }
        }
        else
        {
            if(newSecond >= 60)
            {
                newSecond -= 60;
                ++minutes;
            }
        }

        _seconds = cast(Second)newSecond;
        addMinutes(minutes);
    }

    unittest
    {
        static void testTOD(TimeOfDay orig, int seconds, in TimeOfDay expected, size_t line = __LINE__)
        {
            orig.addSeconds(seconds);
            assertEqual(orig, expected, "", __FILE__, line);
        }

        testTOD(TimeOfDay(12, 30, 33), 0, TimeOfDay(12, 30, 33));
        testTOD(TimeOfDay(12, 30, 33), 1, TimeOfDay(12, 30, 34));
        testTOD(TimeOfDay(12, 30, 33), 2, TimeOfDay(12, 30, 35));
        testTOD(TimeOfDay(12, 30, 33), 3, TimeOfDay(12, 30, 36));
        testTOD(TimeOfDay(12, 30, 33), 4, TimeOfDay(12, 30, 37));
        testTOD(TimeOfDay(12, 30, 33), 5, TimeOfDay(12, 30, 38));
        testTOD(TimeOfDay(12, 30, 33), 10, TimeOfDay(12, 30, 43));
        testTOD(TimeOfDay(12, 30, 33), 15, TimeOfDay(12, 30, 48));
        testTOD(TimeOfDay(12, 30, 33), 26, TimeOfDay(12, 30, 59));
        testTOD(TimeOfDay(12, 30, 33), 27, TimeOfDay(12, 31, 0));
        testTOD(TimeOfDay(12, 30, 33), 30, TimeOfDay(12, 31, 3));
        testTOD(TimeOfDay(12, 30, 33), 59, TimeOfDay(12, 31, 32));
        testTOD(TimeOfDay(12, 30, 33), 60, TimeOfDay(12, 31, 33));
        testTOD(TimeOfDay(12, 30, 33), 61, TimeOfDay(12, 31, 34));

        testTOD(TimeOfDay(12, 30, 33), 1766, TimeOfDay(12, 59, 59));
        testTOD(TimeOfDay(12, 30, 33), 1767, TimeOfDay(13, 0, 0));
        testTOD(TimeOfDay(12, 30, 33), 1768, TimeOfDay(13, 0, 1));
        testTOD(TimeOfDay(12, 30, 33), 2007, TimeOfDay(13, 4, 0));
        testTOD(TimeOfDay(12, 30, 33), 3599, TimeOfDay(13, 30, 32));
        testTOD(TimeOfDay(12, 30, 33), 3600, TimeOfDay(13, 30, 33));
        testTOD(TimeOfDay(12, 30, 33), 3601, TimeOfDay(13, 30, 34));
        testTOD(TimeOfDay(12, 30, 33), 7200, TimeOfDay(14, 30, 33));

        testTOD(TimeOfDay(12, 30, 33), -1, TimeOfDay(12, 30, 32));
        testTOD(TimeOfDay(12, 30, 33), -2, TimeOfDay(12, 30, 31));
        testTOD(TimeOfDay(12, 30, 33), -3, TimeOfDay(12, 30, 30));
        testTOD(TimeOfDay(12, 30, 33), -4, TimeOfDay(12, 30, 29));
        testTOD(TimeOfDay(12, 30, 33), -5, TimeOfDay(12, 30, 28));
        testTOD(TimeOfDay(12, 30, 33), -10, TimeOfDay(12, 30, 23));
        testTOD(TimeOfDay(12, 30, 33), -15, TimeOfDay(12, 30, 18));
        testTOD(TimeOfDay(12, 30, 33), -33, TimeOfDay(12, 30, 0));
        testTOD(TimeOfDay(12, 30, 33), -34, TimeOfDay(12, 29, 59));
        testTOD(TimeOfDay(12, 30, 33), -35, TimeOfDay(12, 29, 58));
        testTOD(TimeOfDay(12, 30, 33), -59, TimeOfDay(12, 29, 34));
        testTOD(TimeOfDay(12, 30, 33), -60, TimeOfDay(12, 29, 33));
        testTOD(TimeOfDay(12, 30, 33), -61, TimeOfDay(12, 29, 32));

        testTOD(TimeOfDay(12, 30, 0), 1, TimeOfDay(12, 30, 1));
        testTOD(TimeOfDay(12, 30, 0), 0, TimeOfDay(12, 30, 0));
        testTOD(TimeOfDay(12, 30, 0), -1, TimeOfDay(12, 29, 59));

        testTOD(TimeOfDay(12, 0, 0), 1, TimeOfDay(12, 0, 1));
        testTOD(TimeOfDay(12, 0, 0), 0, TimeOfDay(12, 0, 0));
        testTOD(TimeOfDay(12, 0, 0), -1, TimeOfDay(11, 59, 59));

        testTOD(TimeOfDay(0, 0, 0), 1, TimeOfDay(0, 0, 1));
        testTOD(TimeOfDay(0, 0, 0), 0, TimeOfDay(0, 0, 0));
        testTOD(TimeOfDay(0, 0, 0), -1, TimeOfDay(23, 59, 59));

        testTOD(TimeOfDay(23, 59, 59), 1, TimeOfDay(0, 0, 0));
        testTOD(TimeOfDay(23, 59, 59), 0, TimeOfDay(23, 59, 59));
        testTOD(TimeOfDay(23, 59, 59), -1, TimeOfDay(23, 59, 58));

        const ctod = TimeOfDay(0, 0, 0);
        static assert(!__traits(compiles, ctod.addSeconds(7)));
        immutable itod = TimeOfDay(0, 0, 0);
        static assert(!__traits(compiles, itod.addSeconds(7)));
    }

    /++
        Function for adding time units generically. The given TimeUnit must
        be TimeUnit.hour, TimeUnit. minute, or TimeUnit.second;

        Params:
            units         = The type of unit to add to.
            value         = The number of units to add to the given type of unit.
            allowOverflow = Whether the days should be allowed to overflow, causing
                            the month to increment (only really applies to TimeUnit.year
                            and TimeUnit.month, but addTimeUnit() is generic, so it's included;
                            However, it's ignored for TimeOfDay).

        Examples:
        --------------------
        addTimeUnit!(TimeUnit.hour)(numHours);     //translates to addHours(numHours);
        addTimeUnit!(TimeUnit.minute)(numMinutes); //translates to addMinutes(numMinutes);
        addTimeUnit!(TimeUnit.second)(numSeconds); //translates to addSeconds(numSeconds);
        --------------------
      +/
    void addTimeUnit(TimeUnit units)(long value, AllowDayOverflow allowOverflow = AllowDayOverflow.yes)
        if(units == TimeUnit.hour   ||
           units == TimeUnit.minute ||
           units == TimeUnit.second)
    {
        static if(units == TimeUnit.hour)
            addHours(cast(ParameterTypeTuple!(addHours)[0])value);
        else static if(units == TimeUnit.minute)
            addMinutes(cast(ParameterTypeTuple!(addMinutes)[0])value);
        else static if(units == TimeUnit.second)
            addSeconds(cast(ParameterTypeTuple!(addSeconds)[0])value);
        else
            static assert(0);
    }

    unittest
    {
        {
            auto tod = TimeOfDay(12, 30, 33);
            tod.addTimeUnit!(TimeUnit.hour)(1);
            assertEqual(tod, TimeOfDay(13, 30, 33));
            tod.addTimeUnit!(TimeUnit.hour)(-1);
            assertEqual(tod, TimeOfDay(12, 30, 33));
        }

        {
            auto tod = TimeOfDay(12, 30, 33);
            tod.addTimeUnit!(TimeUnit.minute)(1);
            assertEqual(tod, TimeOfDay(12, 31, 33));
            tod.addTimeUnit!(TimeUnit.minute)(-1);
            assertEqual(tod, TimeOfDay(12, 30, 33));
        }

        {
            auto tod = TimeOfDay(12, 30, 33);
            tod.addTimeUnit!(TimeUnit.second)(1);
            assertEqual(tod, TimeOfDay(12, 30, 34));
            tod.addTimeUnit!(TimeUnit.second)(-1);
            assertEqual(tod, TimeOfDay(12, 30, 33));
        }

        auto tod = TimeOfDay(12, 30, 33);
        const ctod = TimeOfDay(12, 30, 33);
        immutable itod = TimeOfDay(12, 30, 33);
        static assert(!__traits(compiles, tod.addTimeUnit!(TimeUnit.year)(12)));
        static assert(!__traits(compiles, ctod.addTimeUnit!(TimeUnit.year)(12)));
        static assert(!__traits(compiles, itod.TimeUnit!(TimeUnit.year)(12)));

        static assert(!__traits(compiles, tod.addTimeUnit!(TimeUnit.month)(12)));
        static assert(!__traits(compiles, ctod.addTimeUnit!(TimeUnit.month)(12)));
        static assert(!__traits(compiles, itod.TimeUnit!(TimeUnit.month)(12)));

        static assert(!__traits(compiles, tod.addTimeUnit!(TimeUnit.week)(12)));
        static assert(!__traits(compiles, ctod.addTimeUnit!(TimeUnit.week)(12)));
        static assert(!__traits(compiles, itod.TimeUnit!(TimeUnit.week)(12)));

        static assert(!__traits(compiles, tod.addTimeUnit!(TimeUnit.day)(12)));
        static assert(!__traits(compiles, ctod.addTimeUnit!(TimeUnit.day)(12)));
        static assert(!__traits(compiles, itod.TimeUnit!(TimeUnit.day)(12)));


        static assert(__traits(compiles, tod.addTimeUnit!(TimeUnit.hour)(12)));
        static assert(!__traits(compiles, ctod.addTimeUnit!(TimeUnit.hour)(12)));
        static assert(!__traits(compiles, itod.TimeUnit!(TimeUnit.hour)(12)));

        static assert(__traits(compiles, tod.addTimeUnit!(TimeUnit.minute)(12)));
        static assert(!__traits(compiles, ctod.addTimeUnit!(TimeUnit.minute)(12)));
        static assert(!__traits(compiles, itod.TimeUnit!(TimeUnit.minute)(12)));

        static assert(__traits(compiles, tod.addTimeUnit!(TimeUnit.second)(12)));
        static assert(!__traits(compiles, ctod.addTimeUnit!(TimeUnit.second)(12)));
        static assert(!__traits(compiles, itod.TimeUnit!(TimeUnit.second)(12)));


        static assert(!__traits(compiles, tod.addTimeUnit!(TimeUnit.millisecond)(12)));
        static assert(!__traits(compiles, ctod.addTimeUnit!(TimeUnit.millisecond)(12)));
        static assert(!__traits(compiles, itod.TimeUnit!(TimeUnit.millisecond)(12)));

        static assert(!__traits(compiles, tod.addTimeUnit!(TimeUnit.microsecond)(12)));
        static assert(!__traits(compiles, ctod.addTimeUnit!(TimeUnit.microsecond)(12)));
        static assert(!__traits(compiles, itod.TimeUnit!(TimeUnit.microsecond)(12)));

        static assert(!__traits(compiles, tod.addTimeUnit!(TimeUnit.tick)(12)));
        static assert(!__traits(compiles, ctod.addTimeUnit!(TimeUnit.tick)(12)));
        static assert(!__traits(compiles, itod.TimeUnit!(TimeUnit.tick)(12)));
    }


    /++
        TimeOfDay as a tuple of hour, minute, and second.
      +/
    @property Tuple!(Hour, Minute, Second) hourMinSec() const
    {
        return Tuple!(Hour, Minute, Second) (_hours, _minutes, seconds);
    }


    /++
        Converts this TimeOfDay to a string with the format HHMMSS.

        Examples:
        --------------------
        assert(TimeOfDay(0, 0, 0).toISOString() == "000000");
        assert(TimeOfDay(12, 30, 33).toISOString() == "123033");
        --------------------
      +/
    string toISOString() const
    {
        return format("%02d%02d%02d", _hours, _minutes, _seconds);
    }

    unittest
    {
        //Verify Examples
        assert(TimeOfDay(0, 0, 0).toISOString() == "000000");
        assert(TimeOfDay(12, 30, 33).toISOString() == "123033");
    }


    /++
        Converts this TimeOfDay to a string with the format HH:MM:SS.

        Examples:
        --------------------
        assert(TimeOfDay(0, 0, 0).toISOExtendedString() == "000000");
        assert(TimeOfDay(12, 30, 33).toISOExtendedString() == "123033");
        --------------------
      +/
    string toISOExtendedString() const
    {
        return format("%02d:%02d:%02d", _hours, _minutes, _seconds);
    }


    unittest
    {
        //Verify Examples
        assert(TimeOfDay(0, 0, 0).toISOExtendedString() == "00:00:00");
        assert(TimeOfDay(12, 30, 33).toISOExtendedString() == "12:30:33");
    }


    /++
        Converts this TimeOfDay to a string.
      +/
    //For some reason, the const version doesn't work with format(), so I'm defining
    //it twice - one const and one non-const.
    string toString()
    {
        return toISOExtendedString();
    }


    /++
        Converts this TimeOfDay to a string.
      +/
    string toString() const
    {
        return toISOExtendedString();
    }

    unittest
    {
        auto tod = TimeOfDay(12, 30, 33);
        const ctod = TimeOfDay(12, 30, 33);
        immutable itod = TimeOfDay(12, 30, 33);
        static assert(__traits(compiles, tod.toString()));
        static assert(__traits(compiles, ctod.toString()));
        static assert(__traits(compiles, itod.toString()));
    }


    /++
        Returns midnight.
      +/
    @property static TimeOfDay min()
    {
        return TimeOfDay(0, 0, 0);
    }


    /++
        Returns one second short of midnight.
      +/
    @property static TimeOfDay max()
    {
        return TimeOfDay(23, 59, 59);
    }


    /++
        The lowest time resolution that TimeOfDay has.
      +/
    @property static TimeUnit minResolution()
    {
        return TimeUnit.hour;
    }


    /++
        The highest time resolution that TimeOfDay has.
      +/
    @property static TimeUnit maxResolution()
    {
        return TimeUnit.second;
    }


private:

    invariant()
    {
        assert(_hours <= maxHours);
        assert(_minutes <= maxMinutes);
        assert(_seconds <= maxSeconds);
    }

    Hour       _hours;
    Minute     _minutes;
    Second     _seconds;


    enum Hour        maxHours   = 24 - 1;
    enum Minute      maxMinutes = 60 - 1;
    enum Second      maxSeconds    = 60 - 1;
}


/++
   Combines Date and TimeOfDay structs to give you an object which holds both the date
   and the time. It is optimized for calendar operations and has no concept of time zone.
   If you want an object which is optimized for time operations based on the system time,
   then use SysTime. SysTime has a concept of time zone and has much higher precision (ticks).
   DateTime is intended primarily for calendar-based uses rather than precise time operations.
  +/
struct DateTime
{
public:

    /++
        Params:
            date = The date portion of DateTime.
            tod  = The time portion of DateTime.
      +/
    this(in Date date, in TimeOfDay tod = TimeOfDay.init)
    {
        _date = date;
        _tod = tod;
    }

    unittest
    {
        {
            auto dt = DateTime.init;
            assertEqual(dt._date, Date.init);
            assertEqual(dt._tod, TimeOfDay.init);
        }

        {
            auto dt = DateTime(Date(1999, 7 ,6));
            assertEqual(dt._date, Date(1999, 7, 6));
            assertEqual(dt._tod, TimeOfDay.init);
        }

        {
            auto dt = DateTime(Date(1999, 7 ,6), TimeOfDay(12, 30, 33));
            assertEqual(dt._date, Date(1999, 7, 6));
            assertEqual(dt._tod, TimeOfDay(12, 30, 33));
        }
    }


    /++
        Params:
            year    = The year portion of the date.
            month   = The month portion of the date.
            day     = The day portion of the date.
            hours   = The hour portion of the time;
            minutes = The minute portion of the time;
            seconds = The second portion of the time;
      +/
    this(Year year, MonthOfYear month, DayOfMonth day,
         Hour hours = 0, Minute minutes = 0, Second seconds = 0)
    {
        _date = Date(year, month, day);
        _tod = TimeOfDay(hours, minutes, seconds);
    }

    unittest
    {
        {
            auto dt = DateTime(1999, 7 ,6);
            assertEqual(dt._date, Date(1999, 7, 6));
            assertEqual(dt._tod, TimeOfDay.init);
        }

        {
            auto dt = DateTime(1999, 7 ,6, 12, 30, 33);
            assertEqual(dt._date, Date(1999, 7, 6));
            assertEqual(dt._tod, TimeOfDay(12, 30, 33));
        }
    }


    /++
       Compares this DateTime with the given DateTime.
       Returns:
        $(TABLE
        $(TR $(TD this &lt; rhs) $(TD &lt; 0))
        $(TR $(TD this == rhs) $(TD 0))
        $(TR $(TD this &gt; rhs) $(TD &gt; 0))
        )
     +/
    int opCmp(in DateTime rhs) const
    {
        immutable result = _date.opCmp(rhs._date);

        if(result < 0)
            return -1;
        if(result > 0)
            return 1;

        return _tod.opCmp(rhs._tod);
    }

    unittest
    {
        assertExcThrown!(Exception, (){notADateTime.opCmp(posInfinity);})(LineInfo());
        assertExcThrown!(Exception, (){notADateTime.opCmp(negInfinity);})(LineInfo());
        assertExcThrown!(Exception, (){notADateTime.opCmp(notADateTime);})(LineInfo());
        assertExcThrown!(Exception, (){notADateTime.opCmp(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)));})(LineInfo());
        assertExcThrown!(Exception, (){posInfinity.opCmp(notADateTime);})(LineInfo());
        assertExcThrown!(Exception, (){negInfinity.opCmp(notADateTime);})(LineInfo());
        assertExcThrown!(Exception, (){DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)).opCmp(notADateTime);})(LineInfo());

        assertOpCmp!("==")(posInfinity, posInfinity);
        assertOpCmp!("==")(negInfinity, negInfinity);
        assertOpCmp!("<")(negInfinity, posInfinity);
        assertOpCmp!(">")(posInfinity, negInfinity);

        //Test A.D.
        assertOpCmp!("==")(DateTime(Date.init, TimeOfDay.init), DateTime.init);

        assertOpCmp!("==")(DateTime(Date(1999, 1, 1)), DateTime(Date(1999, 1, 1)));
        assertOpCmp!("==")(DateTime(Date(1, 7, 1)), DateTime(Date(1, 7, 1)));
        assertOpCmp!("==")(DateTime(Date(1, 1, 6)), DateTime(Date(1, 1, 6)));

        assertOpCmp!("==")(DateTime(Date(1999, 7, 1)), DateTime(Date(1999, 7, 1)));
        assertOpCmp!("==")(DateTime(Date(1999, 7, 6)), DateTime(Date(1999, 7, 6)));

        assertOpCmp!("==")(DateTime(Date(1, 7, 6)), DateTime(Date(1, 7, 6)));

        assertOpCmp!("<")(DateTime(Date(1999, 7, 6)), DateTime(Date(2000, 7, 6)));
        assertOpCmp!(">")(DateTime(Date(2000, 7, 6)), DateTime(Date(1999, 7, 6)));
        assertOpCmp!("<")(DateTime(Date(1999, 7, 6)), DateTime(Date(1999, 8, 6)));
        assertOpCmp!(">")(DateTime(Date(1999, 8, 6)), DateTime(Date(1999, 7, 6)));
        assertOpCmp!("<")(DateTime(Date(1999, 7, 6)), DateTime(Date(1999, 7, 7)));
        assertOpCmp!(">")(DateTime(Date(1999, 7, 7)), DateTime(Date(1999, 7, 6)));

        assertOpCmp!("<")(DateTime(Date(1999, 8, 7)), DateTime(Date(2000, 7, 6)));
        assertOpCmp!(">")(DateTime(Date(2000, 8, 6)), DateTime(Date(1999, 7, 7)));
        assertOpCmp!("<")(DateTime(Date(1999, 7, 7)), DateTime(Date(2000, 7, 6)));
        assertOpCmp!(">")(DateTime(Date(2000, 7, 6)), DateTime(Date(1999, 7, 7)));
        assertOpCmp!("<")(DateTime(Date(1999, 7, 7)), DateTime(Date(1999, 8, 6)));
        assertOpCmp!(">")(DateTime(Date(1999, 8, 6)), DateTime(Date(1999, 7, 7)));


        assertOpCmp!("==")(DateTime(Date(1999, 7, 6), TimeOfDay(0, 0, 0)),
                           DateTime(Date(1999, 7, 6), TimeOfDay(0, 0, 0)));
        assertOpCmp!("==")(DateTime(Date(1999, 7, 6), TimeOfDay(12, 0, 0)),
                           DateTime(Date(1999, 7, 6), TimeOfDay(12, 0, 0)));
        assertOpCmp!("==")(DateTime(Date(1999, 7, 6), TimeOfDay(0, 30, 0)),
                           DateTime(Date(1999, 7, 6), TimeOfDay(0, 30, 0)));
        assertOpCmp!("==")(DateTime(Date(1999, 7, 6), TimeOfDay(0, 0, 33)),
                           DateTime(Date(1999, 7, 6), TimeOfDay(0, 0, 33)));

        assertOpCmp!("==")(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 0)),
                           DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 0)));
        assertOpCmp!("==")(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)),
                           DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)));

        assertOpCmp!("==")(DateTime(Date(1999, 7, 6), TimeOfDay(0, 30, 33)),
                           DateTime(Date(1999, 7, 6), TimeOfDay(0, 30, 33)));
        assertOpCmp!("==")(DateTime(Date(1999, 7, 6), TimeOfDay(0, 0, 33)),
                           DateTime(Date(1999, 7, 6), TimeOfDay(0, 0, 33)));

        assertOpCmp!("<")(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)),
                          DateTime(Date(1999, 7, 6), TimeOfDay(13, 30, 33)));
        assertOpCmp!(">")(DateTime(Date(1999, 7, 6), TimeOfDay(13, 30, 33)),
                          DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)));
        assertOpCmp!("<")(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)),
                          DateTime(Date(1999, 7, 6), TimeOfDay(12, 31, 33)));
        assertOpCmp!(">")(DateTime(Date(1999, 7, 6), TimeOfDay(12, 31, 33)),
                          DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)));
        assertOpCmp!("<")(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)),
                          DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 34)));
        assertOpCmp!(">")(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 34)),
                          DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)));

        assertOpCmp!(">")(DateTime(Date(1999, 7, 6), TimeOfDay(13, 30, 33)),
                          DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 34)));
        assertOpCmp!("<")(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 34)),
                          DateTime(Date(1999, 7, 6), TimeOfDay(13, 30, 33)));
        assertOpCmp!(">")(DateTime(Date(1999, 7, 6), TimeOfDay(13, 30, 33)),
                          DateTime(Date(1999, 7, 6), TimeOfDay(12, 31, 33)));
        assertOpCmp!("<")(DateTime(Date(1999, 7, 6), TimeOfDay(12, 31, 33)),
                          DateTime(Date(1999, 7, 6), TimeOfDay(13, 30, 33)));

        assertOpCmp!(">")(DateTime(Date(1999, 7, 6), TimeOfDay(12, 31, 33)),
                          DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 34)));
        assertOpCmp!("<")(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 34)),
                          DateTime(Date(1999, 7, 6), TimeOfDay(12, 31, 33)));

        assertOpCmp!("<")(DateTime(Date(1999, 7, 6), TimeOfDay(13, 30, 33)),
                          DateTime(Date(2000, 7, 6), TimeOfDay(12, 30, 33)));
        assertOpCmp!(">")(DateTime(Date(2000, 7, 6), TimeOfDay(12, 30, 33)),
                          DateTime(Date(1999, 7, 6), TimeOfDay(13, 30, 33)));
        assertOpCmp!("<")(DateTime(Date(1999, 7, 6), TimeOfDay(12, 31, 33)),
                          DateTime(Date(2000, 7, 6), TimeOfDay(12, 30, 33)));
        assertOpCmp!(">")(DateTime(Date(2000, 7, 6), TimeOfDay(12, 30, 33)),
                          DateTime(Date(1999, 7, 6), TimeOfDay(12, 31, 33)));
        assertOpCmp!("<")(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 34)),
                          DateTime(Date(2000, 7, 6), TimeOfDay(12, 30, 33)));
        assertOpCmp!(">")(DateTime(Date(2000, 7, 6), TimeOfDay(12, 30, 33)),
                          DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 34)));

        assertOpCmp!("<")(DateTime(Date(1999, 7, 6), TimeOfDay(13, 30, 33)),
                          DateTime(Date(1999, 8, 6), TimeOfDay(12, 30, 33)));
        assertOpCmp!(">")(DateTime(Date(1999, 8, 6), TimeOfDay(12, 30, 33)),
                          DateTime(Date(1999, 7, 6), TimeOfDay(13, 30, 33)));
        assertOpCmp!("<")(DateTime(Date(1999, 7, 6), TimeOfDay(12, 31, 33)),
                          DateTime(Date(1999, 8, 6), TimeOfDay(12, 30, 33)));
        assertOpCmp!(">")(DateTime(Date(1999, 8, 6), TimeOfDay(12, 30, 33)),
                          DateTime(Date(1999, 7, 6), TimeOfDay(12, 31, 33)));
        assertOpCmp!("<")(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 34)),
                          DateTime(Date(1999, 8, 6), TimeOfDay(12, 30, 33)));
        assertOpCmp!(">")(DateTime(Date(1999, 8, 6), TimeOfDay(12, 30, 33)),
                          DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 34)));

        assertOpCmp!("<")(DateTime(Date(1999, 7, 6), TimeOfDay(13, 30, 33)),
                          DateTime(Date(1999, 7, 7), TimeOfDay(12, 30, 33)));
        assertOpCmp!(">")(DateTime(Date(1999, 7, 7), TimeOfDay(12, 30, 33)),
                          DateTime(Date(1999, 7, 6), TimeOfDay(13, 30, 33)));
        assertOpCmp!("<")(DateTime(Date(1999, 7, 6), TimeOfDay(12, 31, 33)),
                          DateTime(Date(1999, 7, 7), TimeOfDay(12, 31, 33)));
        assertOpCmp!(">")(DateTime(Date(1999, 7, 7), TimeOfDay(12, 30, 33)),
                          DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)));
        assertOpCmp!("<")(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 34)),
                          DateTime(Date(1999, 7, 7), TimeOfDay(12, 30, 33)));
        assertOpCmp!(">")(DateTime(Date(1999, 7, 7), TimeOfDay(12, 30, 33)),
                          DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 34)));

        assertOpCmp!("<")(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)),
                          posInfinity);
        assertOpCmp!(">")(posInfinity,
                          DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)));
        assertOpCmp!("<")(negInfinity,
                          DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)));
        assertOpCmp!(">")(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)),
                          negInfinity);

        //Test B.C.
        assertOpCmp!("==")(DateTime(Date(-1, 1, 1), TimeOfDay(12, 30, 33)),
                           DateTime(Date(-1, 1, 1), TimeOfDay(12, 30, 33)));
        assertOpCmp!("==")(DateTime(Date(-1, 7, 1), TimeOfDay(12, 30, 33)),
                           DateTime(Date(-1, 7, 1), TimeOfDay(12, 30, 33)));
        assertOpCmp!("==")(DateTime(Date(-1, 1, 6), TimeOfDay(12, 30, 33)),
                           DateTime(Date(-1, 1, 6), TimeOfDay(12, 30, 33)));

        assertOpCmp!("==")(DateTime(Date(-1999, 7, 1), TimeOfDay(12, 30, 33)),
                           DateTime(Date(-1999, 7, 1), TimeOfDay(12, 30, 33)));
        assertOpCmp!("==")(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)),
                           DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)));

        assertOpCmp!("==")(DateTime(Date(-1, 7, 6), TimeOfDay(12, 30, 33)),
                           DateTime(Date(-1, 7, 6), TimeOfDay(12, 30, 33)));

        assertOpCmp!("<")(DateTime(Date(-2000, 7, 6), TimeOfDay(12, 30, 33)),
                          DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)));
        assertOpCmp!(">")(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)),
                          DateTime(Date(-2000, 7, 6), TimeOfDay(12, 30, 33)));
        assertOpCmp!("<")(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)),
                          DateTime(Date(-1999, 8, 6), TimeOfDay(12, 30, 33)));
        assertOpCmp!(">")(DateTime(Date(-1999, 8, 6), TimeOfDay(12, 30, 33)),
                          DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)));
        assertOpCmp!("<")(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)),
                          DateTime(Date(-1999, 7, 7), TimeOfDay(12, 30, 33)));
        assertOpCmp!(">")(DateTime(Date(-1999, 7, 7), TimeOfDay(12, 30, 33)),
                          DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)));

        assertOpCmp!("<")(DateTime(Date(-2000, 8, 6), TimeOfDay(12, 30, 33)),
                          DateTime(Date(-1999, 7, 7), TimeOfDay(12, 30, 33)));
        assertOpCmp!(">")(DateTime(Date(-1999, 8, 7), TimeOfDay(12, 30, 33)),
                          DateTime(Date(-2000, 7, 6), TimeOfDay(12, 30, 33)));
        assertOpCmp!("<")(DateTime(Date(-2000, 7, 6), TimeOfDay(12, 30, 33)),
                          DateTime(Date(-1999, 7, 7), TimeOfDay(12, 30, 33)));
        assertOpCmp!(">")(DateTime(Date(-1999, 7, 7), TimeOfDay(12, 30, 33)),
                          DateTime(Date(-2000, 7, 6), TimeOfDay(12, 30, 33)));
        assertOpCmp!("<")(DateTime(Date(-1999, 7, 7), TimeOfDay(12, 30, 33)),
                          DateTime(Date(-1999, 8, 6), TimeOfDay(12, 30, 33)));
        assertOpCmp!(">")(DateTime(Date(-1999, 8, 6), TimeOfDay(12, 30, 33)),
                          DateTime(Date(-1999, 7, 7), TimeOfDay(12, 30, 33)));

        assertOpCmp!("<")(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), posInfinity);
        assertOpCmp!(">")(posInfinity, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)));
        assertOpCmp!("<")(negInfinity, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)));
        assertOpCmp!(">")(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), negInfinity);

        //Test Both
        assertOpCmp!("<")(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)),
                          DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)));
        assertOpCmp!(">")(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)),
                          DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)));

        assertOpCmp!("<")(DateTime(Date(-1999, 8, 6), TimeOfDay(12, 30, 33)),
                          DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)));
        assertOpCmp!(">")(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)),
                          DateTime(Date(-1999, 8, 6), TimeOfDay(12, 30, 33)));

        assertOpCmp!("<")(DateTime(Date(-1999, 7, 7), TimeOfDay(12, 30, 33)),
                          DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)));
        assertOpCmp!(">")(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)),
                          DateTime(Date(-1999, 7, 7), TimeOfDay(12, 30, 33)));

        assertOpCmp!("<")(DateTime(Date(-1999, 8, 7), TimeOfDay(12, 30, 33)),
                          DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)));
        assertOpCmp!(">")(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)),
                          DateTime(Date(-1999, 8, 7), TimeOfDay(12, 30, 33)));

        assertOpCmp!("<")(DateTime(Date(-1999, 8, 6), TimeOfDay(12, 30, 33)),
                          DateTime(Date(1999, 6, 6), TimeOfDay(12, 30, 33)));
        assertOpCmp!(">")(DateTime(Date(1999, 6, 8), TimeOfDay(12, 30, 33)),
                          DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)));

        const cdt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 33, 30));
        immutable idt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 33, 30));
        static assert(__traits(compiles, cdt.opCmp(idt)));
        static assert(__traits(compiles, idt.opCmp(cdt)));
    }


    /++
        The date portion of DateTime.
      +/
    @property Date date() const
    {
        return _date;
    }

    unittest
    {
        {
            auto dt = DateTime.init;
            assertEqual(dt.date, Date.init);
        }

        {
            auto dt = DateTime(Date(1999, 7, 6));
            assertEqual(dt.date, Date(1999, 7, 6));
        }

        assertExcNotThrown!(Exception, (){posInfinity.date;})(LineInfo());
        assertExcNotThrown!(Exception, (){negInfinity.date;})(LineInfo());
        assertExcNotThrown!(Exception, (){notADateTime.date;})(LineInfo());

        const ctd = DateTime(1999, 7, 6);
        immutable itd = DateTime(1999, 7, 6);
        static assert(__traits(compiles, ctd.date == Date(2010, 1, 1)));
        static assert(__traits(compiles, itd.date == Date(2010, 1, 1)));
    }


    /++
        The date portion of DateTime.

        Params:
            date = The Date to set this DateTime's Date portion to.
      +/
    @property void date(in Date date)
    {
        if(isSpecial)
            _tod = TimeOfDay.init;

        _date = date;
    }

    unittest
    {
        {
            auto dt = DateTime.init;
            dt.date = Date(1999, 7, 6);
            assertEqual(dt._date, Date(1999, 7, 6));
            assertEqual(dt._tod, TimeOfDay.init);
        }

        {
            auto dt = notADateTime;
            dt._tod = TimeOfDay(12, 12, 12);
            dt.date = Date(1, 1, 4);
            assertEqual(dt, DateTime(Date(1, 1, 4), TimeOfDay.init));
        }

        assertExcNotThrown!(Exception, (){posInfinity.date = Date(1, 7, 6);})(LineInfo());
        assertExcNotThrown!(Exception, (){negInfinity.date = Date(1, 7, 6);})(LineInfo());
        assertExcNotThrown!(Exception, (){notADateTime.date = Date(1, 7, 6);})(LineInfo());

        const ctd = DateTime(1999, 7, 6);
        immutable itd = DateTime(1999, 7, 6);
        static assert(!__traits(compiles, ctd.date = Date(2010, 1, 1)));
        static assert(!__traits(compiles, itd.date = Date(2010, 1, 1)));
    }


    /++
        The time portion of DateTime.
      +/
    @property TimeOfDay timeOfDay() const
    {
        _date.enforceNotSpecial();

        return _tod;
    }

    unittest
    {
        {
            auto dt = DateTime.init;
            assertEqual(dt.timeOfDay, TimeOfDay.init);
        }

        {
            auto dt = DateTime(Date.init, TimeOfDay(12, 30, 33));
            assertEqual(dt.timeOfDay, TimeOfDay(12, 30, 33));
        }

        assertExcThrown!(Exception, (){posInfinity.timeOfDay;})(LineInfo());
        assertExcThrown!(Exception, (){negInfinity.timeOfDay;})(LineInfo());
        assertExcThrown!(Exception, (){notADateTime.timeOfDay;})(LineInfo());

        const ctd = DateTime(1999, 7, 6, 12, 30, 33);
        immutable itd = DateTime(1999, 7, 6, 12, 30, 33);
        static assert(__traits(compiles, ctd.timeOfDay == TimeOfDay(12, 30, 33)));
        static assert(__traits(compiles, itd.timeOfDay == TimeOfDay(12, 30, 33)));
    }


    /++
        The time portion of DateTime.

        Params:
            tod = The TimeOfDay to set this DateTime's TimeOfDay portion to.
      +/
    @property void timeOfDay(in TimeOfDay tod)
    {
        _date.enforceNotSpecial();

        _tod = tod;
    }

    unittest
    {
        auto dt = DateTime.init;
        dt.timeOfDay = TimeOfDay(12, 30, 33);
        assertEqual(dt._date, date.init);
        assertEqual(dt._tod, TimeOfDay(12, 30, 33));

        assertExcThrown!(Exception, (){posInfinity.timeOfDay = TimeOfDay(0, 12, 11);})(LineInfo());
        assertExcThrown!(Exception, (){negInfinity.timeOfDay = TimeOfDay(0, 12, 11);})(LineInfo());
        assertExcThrown!(Exception, (){notADateTime.timeOfDay = TimeOfDay(0, 12, 11);})(LineInfo());

        const ctd = DateTime(1999, 7, 6, 12, 30, 33);
        immutable itd = DateTime(1999, 7, 6, 12, 30, 33);
        static assert(!__traits(compiles, ctd.timeOfDay = TimeOfDay(12, 30, 33)));
        static assert(!__traits(compiles, itd.timeOfDay = TimeOfDay(12, 30, 33)));
    }


    /++
        See_Also:
            Date.year().
     +/
    @property Year year() const
    {
        return _date.year;
    }

    unittest
    {
        assertExcThrown!(Exception, (){posInfinity.year;})(LineInfo());
        assertExcThrown!(Exception, (){negInfinity.year;})(LineInfo());
        assertExcThrown!(Exception, (){notADateTime.year;})(LineInfo());

        auto td = DateTime(1999, 7, 6, 12, 30, 33);
        const ctd = DateTime(1999, 7, 6, 12, 30, 33);
        immutable itd = DateTime(1999, 7, 6, 12, 30, 33);
        static assert(__traits(compiles, td.year));
        static assert(__traits(compiles, itd.year));
        static assert(__traits(compiles, itd.year));
    }


    /++
        See_Also:
            Date.year().
     +/
    @property void year(Year year)
    {
        _date.year = year;
    }

    unittest
    {
        assertExcThrown!(Exception, (){posInfinity.year = 7;})(LineInfo());
        assertExcThrown!(Exception, (){negInfinity.year = 7;})(LineInfo());
        assertExcThrown!(Exception, (){notADateTime.year = 7;})(LineInfo());

        auto td = DateTime(1999, 7, 6, 12, 30, 33);
        const ctd = DateTime(1999, 7, 6, 12, 30, 33);
        immutable itd = DateTime(1999, 7, 6, 12, 30, 33);
        static assert(__traits(compiles, td.year = 7));
        static assert(!__traits(compiles, ctd.year = 7));
        static assert(!__traits(compiles, itd.year = 7));
    }


    @property MonthOfYear month() const
    {
        return _date.month;
    }

    unittest
    {
        assertExcThrown!(Exception, (){posInfinity.month;})(LineInfo());
        assertExcThrown!(Exception, (){negInfinity.month;})(LineInfo());
        assertExcThrown!(Exception, (){notADateTime.month;})(LineInfo());

        auto td = DateTime(1999, 7, 6, 12, 30, 33);
        const ctd = DateTime(1999, 7, 6, 12, 30, 33);
        immutable itd = DateTime(1999, 7, 6, 12, 30, 33);
        static assert(__traits(compiles, td.month));
        static assert(__traits(compiles, ctd.month));
        static assert(__traits(compiles, itd.month));
    }


    /++
        Year B.C. of the Gregorian Calendar counting year 0 as 1 B.C. An exception
        is thrown if isAD is true.

        Examples:
        --------------------
        assert(DateTime(Date(0, 1, 1), TimeOfDay(12, 30, 33)).yearBC == 1);
        assert(DateTime(Date(-1, 1, 1), TimeOfDay(10, 7, 2)).yearBC == 2);
        assert(DateTime(Date(-100, 1, 1), TimeOfDay(4, 59, 0)).yearBC == 101);
        --------------------
     +/
    @property Year yearBC() const
    {
        return _date.yearBC;
    }

    unittest
    {
        assertExcThrown!(Exception, (in Date date){date.yearBC;})(LineInfo(), Date(1, 1, 1));

        assertExcThrown!(Exception, (){posInfinity.yearBC;})(LineInfo());
        assertExcThrown!(Exception, (){negInfinity.yearBC;})(LineInfo());
        assertExcThrown!(Exception, (){notADateTime.yearBC;})(LineInfo());

        auto td = DateTime(1999, 7, 6, 12, 30, 33);
        const ctd = DateTime(1999, 7, 6, 12, 30, 33);
        immutable itd = DateTime(1999, 7, 6, 12, 30, 33);
        static assert(__traits(compiles, td.yearBC = 12));
        static assert(!__traits(compiles, ctd.yearBC = 12));
        static assert(!__traits(compiles, itd.yearBC = 12));

        //Verify Examples
        assert(DateTime(Date(0, 1, 1), TimeOfDay(12, 30, 33)).yearBC == 1);
        assert(DateTime(Date(-1, 1, 1), TimeOfDay(10, 7, 2)).yearBC == 2);
        assert(DateTime(Date(-100, 1, 1), TimeOfDay(4, 59, 0)).yearBC == 101);
    }


    /++
        Year B.C. of the Gregorian Calendar counting year 0 as 1 B.C. An exception
        is thrown if a non-positive value is given.

        Params:
            year = The year B.C. to set this Date's year to.

        Examples:
        --------------------
        auto dt = DateTime(Date(2010, 1, 1), TimeOfDay(7, 30, 0));
        dt.yearBC = 1;
        assert(dt == DateTime(Date(0, 1, 1), TimeOfDay(7, 30, 0)));

        dt.yearBC = 10;
        assert(dt == DateTime(Date(-9, 1, 1), TimeOfDay(7, 30, 0)));
        --------------------
     +/
    @property void yearBC(Year year)
    {
        _date.yearBC = year;
    }

    unittest
    {
        assertExcThrown!(Exception, (Date date){date.yearBC = -1;})(LineInfo(), Date(1, 1, 1));

        assertExcThrown!(Exception, (){posInfinity.yearBC = 1;})(LineInfo());
        assertExcThrown!(Exception, (){negInfinity.yearBC = 1;})(LineInfo());
        assertExcThrown!(Exception, (){notADateTime.yearBC = 1;})(LineInfo());

        {
            auto td = DateTime(1999, 7, 6, 12, 30, 33);
            const ctd = DateTime(1999, 7, 6, 12, 30, 33);
            immutable itd = DateTime(1999, 7, 6, 12, 30, 33);
            static assert(__traits(compiles, td.yearBC = 12));
            static assert(!__traits(compiles, ctd.yearBC = 12));
            static assert(!__traits(compiles, itd.yearBC = 12));
        }

        //Verify Examples
        {
            auto dt = DateTime(Date(2010, 1, 1), TimeOfDay(7, 30, 0));
            dt.yearBC = 1;
            assert(dt == DateTime(Date(0, 1, 1), TimeOfDay(7, 30, 0)));

            dt.yearBC = 10;
            assert(dt == DateTime(Date(-9, 1, 1), TimeOfDay(7, 30, 0)));
        }
    }


    /++
        See_Also:
            Date.month().
     +/
    @property void month(MonthOfYear month)
    {
        _date.month = month;
    }

    unittest
    {
        assertExcThrown!(Exception, (){posInfinity.month = 7;})(LineInfo());
        assertExcThrown!(Exception, (){negInfinity.month = 7;})(LineInfo());
        assertExcThrown!(Exception, (){notADateTime.month = 7;})(LineInfo());

        auto td = DateTime(1999, 7, 6, 12, 30, 33);
        const ctd = DateTime(1999, 7, 6, 12, 30, 33);
        immutable itd = DateTime(1999, 7, 6, 12, 30, 33);
        static assert(__traits(compiles, td.month = 12));
        static assert(!__traits(compiles, ctd.month = 12));
        static assert(!__traits(compiles, itd.month = 12));
    }


    /++
        See_Also:
            Date.day().
     +/
    @property DayOfMonth day() const
    {
        return _date.day;
    }

    unittest
    {
        assertExcThrown!(Exception, (){posInfinity.day;})(LineInfo());
        assertExcThrown!(Exception, (){negInfinity.day;})(LineInfo());
        assertExcThrown!(Exception, (){notADateTime.day;})(LineInfo());

        auto td = DateTime(1999, 7, 6, 12, 30, 33);
        const ctd = DateTime(1999, 7, 6, 12, 30, 33);
        immutable itd = DateTime(1999, 7, 6, 12, 30, 33);
        static assert(__traits(compiles, td.day));
        static assert(__traits(compiles, ctd.day));
        static assert(__traits(compiles, itd.day));
    }


    /++
        See_Also:
            Date.addYears().
     +/
    @property void day(DayOfMonth day)
    {

        _date.day = day;
    }

    unittest
    {
        assertExcThrown!(Exception, (){posInfinity.day = 7;})(LineInfo());
        assertExcThrown!(Exception, (){negInfinity.day = 7;})(LineInfo());
        assertExcThrown!(Exception, (){notADateTime.day = 7;})(LineInfo());

        auto td = DateTime(1999, 7, 6, 12, 30, 33);
        const ctd = DateTime(1999, 7, 6, 12, 30, 33);
        immutable itd = DateTime(1999, 7, 6, 12, 30, 33);
        static assert(__traits(compiles, td.day = 27));
        static assert(!__traits(compiles, ctd.day = 27));
        static assert(!__traits(compiles, itd.day = 27));
    }


    /++
        See_Also:
            TimeOfDay.hours().
     +/
    @property Hour hours() const
    {
        _date.enforceNotSpecial();

        return _tod.hours;
    }

    unittest
    {
        assertExcThrown!(Exception, (){posInfinity.hours;})(LineInfo());
        assertExcThrown!(Exception, (){negInfinity.hours;})(LineInfo());
        assertExcThrown!(Exception, (){notADateTime.hours;})(LineInfo());

        auto td = DateTime(1999, 7, 6, 12, 30, 33);
        const ctd = DateTime(1999, 7, 6, 12, 30, 33);
        immutable itd = DateTime(1999, 7, 6, 12, 30, 33);
        static assert(__traits(compiles, td.hours));
        static assert(__traits(compiles, ctd.hours));
        static assert(__traits(compiles, itd.hours));
    }


    /++
        See_Also:
            TimeOfDay.hours().
     +/
    @property void hours(Hour hours)
    {
        _date.enforceNotSpecial();

        _tod.hours = hours;
    }

    unittest
    {
        assertExcThrown!(Exception, (){posInfinity.hours = 7;})(LineInfo());
        assertExcThrown!(Exception, (){negInfinity.hours = 7;})(LineInfo());
        assertExcThrown!(Exception, (){notADateTime.hours = 7;})(LineInfo());

        auto td = DateTime(1999, 7, 6, 12, 30, 33);
        const ctd = DateTime(1999, 7, 6, 12, 30, 33);
        immutable itd = DateTime(1999, 7, 6, 12, 30, 33);
        static assert(__traits(compiles, td.hours = 27));
        static assert(!__traits(compiles, ctd.hours = 27));
        static assert(!__traits(compiles, itd.hours = 27));
    }


    /++
        See_Also:
            TimeOfDay.minutes().
     +/
    @property Minute minutes() const
    {
        _date.enforceNotSpecial();

        return _tod.minutes;
    }

    unittest
    {
        assertExcThrown!(Exception, (){posInfinity.minutes;})(LineInfo());
        assertExcThrown!(Exception, (){negInfinity.minutes;})(LineInfo());
        assertExcThrown!(Exception, (){notADateTime.minutes;})(LineInfo());

        auto td = DateTime(1999, 7, 6, 12, 30, 33);
        const ctd = DateTime(1999, 7, 6, 12, 30, 33);
        immutable itd = DateTime(1999, 7, 6, 12, 30, 33);
        static assert(__traits(compiles, td.minutes));
        static assert(__traits(compiles, ctd.minutes));
        static assert(__traits(compiles, itd.minutes));
    }


    /++
        See_Also:
            TimeOfDay.minutes().
     +/
    @property void minutes(Minute minutes)
    {
        _date.enforceNotSpecial();

        _tod.minutes = minutes;
    }

    unittest
    {
        assertExcThrown!(Exception, (){posInfinity.minutes = 7;})(LineInfo());
        assertExcThrown!(Exception, (){negInfinity.minutes = 7;})(LineInfo());
        assertExcThrown!(Exception, (){notADateTime.minutes = 7;})(LineInfo());

        auto td = DateTime(1999, 7, 6, 12, 30, 33);
        const ctd = DateTime(1999, 7, 6, 12, 30, 33);
        immutable itd = DateTime(1999, 7, 6, 12, 30, 33);
        static assert(__traits(compiles, td.minutes = 27));
        static assert(!__traits(compiles, ctd.minutes = 27));
        static assert(!__traits(compiles, itd.minutes = 27));
    }


    /++
        See_Also:
            TimeOfDay.seconds().
     +/
    @property Second seconds() const
    {
        _date.enforceNotSpecial();

        return _tod.seconds;
    }

    unittest
    {
        assertExcThrown!(Exception, (){posInfinity.seconds;})(LineInfo());
        assertExcThrown!(Exception, (){negInfinity.seconds;})(LineInfo());
        assertExcThrown!(Exception, (){notADateTime.seconds;})(LineInfo());

        auto td = DateTime(1999, 7, 6, 12, 30, 33);
        const ctd = DateTime(1999, 7, 6, 12, 30, 33);
        immutable itd = DateTime(1999, 7, 6, 12, 30, 33);
        static assert(__traits(compiles, td.seconds));
        static assert(__traits(compiles, ctd.seconds));
        static assert(__traits(compiles, itd.seconds));
    }


    /++
        See_Also:
            TimeOfDay.seconds().
     +/
    @property void seconds(Second seconds)
    {
        _date.enforceNotSpecial();

        _tod.seconds = seconds;
    }

    unittest
    {
        assertExcThrown!(Exception, (){posInfinity.seconds = 7;})(LineInfo());
        assertExcThrown!(Exception, (){negInfinity.seconds = 7;})(LineInfo());
        assertExcThrown!(Exception, (){notADateTime.seconds = 7;})(LineInfo());

        auto td = DateTime(1999, 7, 6, 12, 30, 33);
        const ctd = DateTime(1999, 7, 6, 12, 30, 33);
        immutable itd = DateTime(1999, 7, 6, 12, 30, 33);
        static assert(__traits(compiles, td.seconds = 27));
        static assert(!__traits(compiles, ctd.seconds = 27));
        static assert(!__traits(compiles, itd.seconds = 27));
    }


    /++
        See_Also:
            Date.addYears().
      +/
    void addYears(short years, AllowDayOverflow allowOverflow = AllowDayOverflow.yes)
    {
        _date.addYears(years, allowOverflow);
    }

    unittest
    {
        assertExcNotThrown!(Exception, (){posInfinity.addYears(4);})(LineInfo());
        assertExcNotThrown!(Exception, (){negInfinity.addYears(4);})(LineInfo());
        assertExcThrown!(Exception, (){notADateTime.addYears(4);})(LineInfo());

        {
            auto dt = posInfinity;
            posInfinity.addYears(5);
            assertEqual(dt, posInfinity);
        }

        {
            auto dt = negInfinity;
            negInfinity.addYears(5);
            assertEqual(dt, negInfinity);
        }

        auto td = DateTime(1999, 7, 6, 12, 30, 33);
        const ctd = DateTime(1999, 7, 6, 12, 30, 33);
        immutable itd = DateTime(1999, 7, 6, 12, 30, 33);
        static assert(__traits(compiles, td.addYears(4)));
        static assert(!__traits(compiles, ctd.addYears(4)));
        static assert(!__traits(compiles, itd.addYears(4)));
    }


    /++
        See_Also:
            Date.addMonths().
      +/
    void addMonths(int months, AllowDayOverflow allowOverflow = AllowDayOverflow.yes)
    {
        _date.addMonths(months, allowOverflow);
    }

    unittest
    {
        assertExcNotThrown!(Exception, (){posInfinity.addMonths(4);})(LineInfo());
        assertExcNotThrown!(Exception, (){negInfinity.addMonths(4);})(LineInfo());
        assertExcThrown!(Exception, (){notADateTime.addMonths(4);})(LineInfo());

        {
            auto dt = posInfinity;
            posInfinity.addMonths(5);
            assertEqual(dt, posInfinity);
        }

        {
            auto dt = negInfinity;
            negInfinity.addMonths(5);
            assertEqual(dt, negInfinity);
        }

        auto td = DateTime(1999, 7, 6, 12, 30, 33);
        const ctd = DateTime(1999, 7, 6, 12, 30, 33);
        immutable itd = DateTime(1999, 7, 6, 12, 30, 33);
        static assert(__traits(compiles, td.addMonths(4)));
        static assert(!__traits(compiles, ctd.addMonths(4)));
        static assert(!__traits(compiles, itd.addMonths(4)));
    }


    /++
        See_Also:
            Date.addWeeks().
      +/
    void addWeeks(long weeks)
    {
        _date.addWeeks(weeks);
    }

    unittest
    {
        assertExcNotThrown!(Exception, (){posInfinity.addWeeks(4);})(LineInfo());
        assertExcNotThrown!(Exception, (){negInfinity.addWeeks(4);})(LineInfo());
        assertExcThrown!(Exception, (){notADateTime.addWeeks(4);})(LineInfo());

        {
            auto dt = posInfinity;
            posInfinity.addWeeks(5);
            assertEqual(dt, posInfinity);
        }

        {
            auto dt = negInfinity;
            negInfinity.addWeeks(5);
            assertEqual(dt, negInfinity);
        }

        auto td = DateTime(1999, 7, 6, 12, 30, 33);
        const ctd = DateTime(1999, 7, 6, 12, 30, 33);
        immutable itd = DateTime(1999, 7, 6, 12, 30, 33);
        static assert(__traits(compiles, td.addWeeks(4)));
        static assert(!__traits(compiles, ctd.addWeeks(4)));
        static assert(!__traits(compiles, itd.addWeeks(4)));
    }


    /++
        See_Also:
            Date.addDays().
      +/
    void addDays(long days)
    {
        _date.addDays(days);
    }

    unittest
    {
        assertExcNotThrown!(Exception, (){posInfinity.addDays(4);})(LineInfo());
        assertExcNotThrown!(Exception, (){negInfinity.addDays(4);})(LineInfo());
        assertExcThrown!(Exception, (){notADateTime.addDays(4);})(LineInfo());

        {
            auto dt = posInfinity;
            posInfinity.addDays(5);
            assertEqual(dt, posInfinity);
        }

        {
            auto dt = negInfinity;
            negInfinity.addDays(5);
            assertEqual(dt, negInfinity);
        }

        auto td = DateTime(1999, 7, 6, 12, 30, 33);
        const ctd = DateTime(1999, 7, 6, 12, 30, 33);
        immutable itd = DateTime(1999, 7, 6, 12, 30, 33);
        static assert(__traits(compiles, td.addDays(4)));
        static assert(!__traits(compiles, ctd.addDays(4)));
        static assert(!__traits(compiles, itd.addDays(4)));
    }


    /++
        Add hours to the time of day. Negative values will subtract. If the number of hours
        overflows or underflows, then the hours will wrap (e.g. adding 3 hours to 23:00 would
        result in 02:00). Also, if the number of hours wrap, then the day is incremented
        (or decremented) appropriately as well as any greater units if necessary.

        Params:
            hours = The number of hours to add to this DateTime.
      +/
    void addHours(long hours)
    {
        _date.enforceIsADate();

        if(isInfinity)
            return;

        immutable days = TimeUnitConv.to!(TimeUnit.hour, TimeUnit.day)(hours);
        immutable quotient = TimeUnitConv.to!(TimeUnit.day, TimeUnit.hour)(days);
        immutable remainder = hours - quotient;
        immutable newHours = _tod._hours + remainder;

        if(newHours < 0)
            _date.addDays(days - 1);
        else
        {
            immutable overflow = newHours - TimeOfDay.maxHours;

            _date.addDays(overflow > 0 ? days + 1 : days);
        }

        _tod.addHours(cast(ParameterTypeTuple!(_tod.addHours)[0])remainder);
    }

    unittest
    {
        static void testDT(DateTime orig, int hours, in DateTime expected, size_t line = __LINE__)
        {
            orig.addHours(hours);
            assertEqual(orig, expected, "", __FILE__, line);
        }

        //Test A.D.
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 0, DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 1, DateTime(Date(1999, 7, 6), TimeOfDay(13, 30, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 2, DateTime(Date(1999, 7, 6), TimeOfDay(14, 30, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 3, DateTime(Date(1999, 7, 6), TimeOfDay(15, 30, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 4, DateTime(Date(1999, 7, 6), TimeOfDay(16, 30, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 5, DateTime(Date(1999, 7, 6), TimeOfDay(17, 30, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 6, DateTime(Date(1999, 7, 6), TimeOfDay(18, 30, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 7, DateTime(Date(1999, 7, 6), TimeOfDay(19, 30, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 8, DateTime(Date(1999, 7, 6), TimeOfDay(20, 30, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 9, DateTime(Date(1999, 7, 6), TimeOfDay(21, 30, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 10, DateTime(Date(1999, 7, 6), TimeOfDay(22, 30, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 11, DateTime(Date(1999, 7, 6), TimeOfDay(23, 30, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 12, DateTime(Date(1999, 7, 7), TimeOfDay(0, 30, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 13, DateTime(Date(1999, 7, 7), TimeOfDay(1, 30, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 14, DateTime(Date(1999, 7, 7), TimeOfDay(2, 30, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 15, DateTime(Date(1999, 7, 7), TimeOfDay(3, 30, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 16, DateTime(Date(1999, 7, 7), TimeOfDay(4, 30, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 17, DateTime(Date(1999, 7, 7), TimeOfDay(5, 30, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 18, DateTime(Date(1999, 7, 7), TimeOfDay(6, 30, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 19, DateTime(Date(1999, 7, 7), TimeOfDay(7, 30, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 20, DateTime(Date(1999, 7, 7), TimeOfDay(8, 30, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 21, DateTime(Date(1999, 7, 7), TimeOfDay(9, 30, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 22, DateTime(Date(1999, 7, 7), TimeOfDay(10, 30, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 23, DateTime(Date(1999, 7, 7), TimeOfDay(11, 30, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 24, DateTime(Date(1999, 7, 7), TimeOfDay(12, 30, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 25, DateTime(Date(1999, 7, 7), TimeOfDay(13, 30, 33)));

        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -1, DateTime(Date(1999, 7, 6), TimeOfDay(11, 30, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -2, DateTime(Date(1999, 7, 6), TimeOfDay(10, 30, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -3, DateTime(Date(1999, 7, 6), TimeOfDay(9, 30, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -4, DateTime(Date(1999, 7, 6), TimeOfDay(8, 30, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -5, DateTime(Date(1999, 7, 6), TimeOfDay(7, 30, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -6, DateTime(Date(1999, 7, 6), TimeOfDay(6, 30, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -7, DateTime(Date(1999, 7, 6), TimeOfDay(5, 30, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -8, DateTime(Date(1999, 7, 6), TimeOfDay(4, 30, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -9, DateTime(Date(1999, 7, 6), TimeOfDay(3, 30, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -10, DateTime(Date(1999, 7, 6), TimeOfDay(2, 30, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -11, DateTime(Date(1999, 7, 6), TimeOfDay(1, 30, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -12, DateTime(Date(1999, 7, 6), TimeOfDay(0, 30, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -13, DateTime(Date(1999, 7, 5), TimeOfDay(23, 30, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -14, DateTime(Date(1999, 7, 5), TimeOfDay(22, 30, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -15, DateTime(Date(1999, 7, 5), TimeOfDay(21, 30, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -16, DateTime(Date(1999, 7, 5), TimeOfDay(20, 30, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -17, DateTime(Date(1999, 7, 5), TimeOfDay(19, 30, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -18, DateTime(Date(1999, 7, 5), TimeOfDay(18, 30, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -19, DateTime(Date(1999, 7, 5), TimeOfDay(17, 30, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -20, DateTime(Date(1999, 7, 5), TimeOfDay(16, 30, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -21, DateTime(Date(1999, 7, 5), TimeOfDay(15, 30, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -22, DateTime(Date(1999, 7, 5), TimeOfDay(14, 30, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -23, DateTime(Date(1999, 7, 5), TimeOfDay(13, 30, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -24, DateTime(Date(1999, 7, 5), TimeOfDay(12, 30, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -25, DateTime(Date(1999, 7, 5), TimeOfDay(11, 30, 33)));

        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(0, 30, 33)), 1, DateTime(Date(1999, 7, 6), TimeOfDay(1, 30, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(0, 30, 33)), 0, DateTime(Date(1999, 7, 6), TimeOfDay(0, 30, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(0, 30, 33)), -1, DateTime(Date(1999, 7, 5), TimeOfDay(23, 30, 33)));

        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(23, 30, 33)), 1, DateTime(Date(1999, 7, 7), TimeOfDay(0, 30, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(23, 30, 33)), 0, DateTime(Date(1999, 7, 6), TimeOfDay(23, 30, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(23, 30, 33)), -1, DateTime(Date(1999, 7, 6), TimeOfDay(22, 30, 33)));

        testDT(DateTime(Date(1999, 7, 31), TimeOfDay(23, 30, 33)), 1, DateTime(Date(1999, 8, 1), TimeOfDay(0, 30, 33)));
        testDT(DateTime(Date(1999, 8, 1), TimeOfDay(0, 30, 33)), -1, DateTime(Date(1999, 7, 31), TimeOfDay(23, 30, 33)));

        testDT(DateTime(Date(1999, 12, 31), TimeOfDay(23, 30, 33)), 1, DateTime(Date(2000, 1, 1), TimeOfDay(0, 30, 33)));
        testDT(DateTime(Date(2000, 1, 1), TimeOfDay(0, 30, 33)), -1, DateTime(Date(1999, 12, 31), TimeOfDay(23, 30, 33)));

        testDT(DateTime(Date(1999, 2, 28), TimeOfDay(23, 30, 33)), 25, DateTime(Date(1999, 3, 2), TimeOfDay(0, 30, 33)));
        testDT(DateTime(Date(1999, 3, 2), TimeOfDay(0, 30, 33)), -25, DateTime(Date(1999, 2, 28), TimeOfDay(23, 30, 33)));

        testDT(DateTime(Date(2000, 2, 28), TimeOfDay(23, 30, 33)), 25, DateTime(Date(2000, 3, 1), TimeOfDay(0, 30, 33)));
        testDT(DateTime(Date(2000, 3, 1), TimeOfDay(0, 30, 33)), -25, DateTime(Date(2000, 2, 28), TimeOfDay(23, 30, 33)));

        //Test B.C.
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 0, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 1, DateTime(Date(-1999, 7, 6), TimeOfDay(13, 30, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 2, DateTime(Date(-1999, 7, 6), TimeOfDay(14, 30, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 3, DateTime(Date(-1999, 7, 6), TimeOfDay(15, 30, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 4, DateTime(Date(-1999, 7, 6), TimeOfDay(16, 30, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 5, DateTime(Date(-1999, 7, 6), TimeOfDay(17, 30, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 6, DateTime(Date(-1999, 7, 6), TimeOfDay(18, 30, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 7, DateTime(Date(-1999, 7, 6), TimeOfDay(19, 30, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 8, DateTime(Date(-1999, 7, 6), TimeOfDay(20, 30, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 9, DateTime(Date(-1999, 7, 6), TimeOfDay(21, 30, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 10, DateTime(Date(-1999, 7, 6), TimeOfDay(22, 30, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 11, DateTime(Date(-1999, 7, 6), TimeOfDay(23, 30, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 12, DateTime(Date(-1999, 7, 7), TimeOfDay(0, 30, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 13, DateTime(Date(-1999, 7, 7), TimeOfDay(1, 30, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 14, DateTime(Date(-1999, 7, 7), TimeOfDay(2, 30, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 15, DateTime(Date(-1999, 7, 7), TimeOfDay(3, 30, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 16, DateTime(Date(-1999, 7, 7), TimeOfDay(4, 30, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 17, DateTime(Date(-1999, 7, 7), TimeOfDay(5, 30, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 18, DateTime(Date(-1999, 7, 7), TimeOfDay(6, 30, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 19, DateTime(Date(-1999, 7, 7), TimeOfDay(7, 30, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 20, DateTime(Date(-1999, 7, 7), TimeOfDay(8, 30, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 21, DateTime(Date(-1999, 7, 7), TimeOfDay(9, 30, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 22, DateTime(Date(-1999, 7, 7), TimeOfDay(10, 30, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 23, DateTime(Date(-1999, 7, 7), TimeOfDay(11, 30, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 24, DateTime(Date(-1999, 7, 7), TimeOfDay(12, 30, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 25, DateTime(Date(-1999, 7, 7), TimeOfDay(13, 30, 33)));

        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -1, DateTime(Date(-1999, 7, 6), TimeOfDay(11, 30, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -2, DateTime(Date(-1999, 7, 6), TimeOfDay(10, 30, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -3, DateTime(Date(-1999, 7, 6), TimeOfDay(9, 30, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -4, DateTime(Date(-1999, 7, 6), TimeOfDay(8, 30, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -5, DateTime(Date(-1999, 7, 6), TimeOfDay(7, 30, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -6, DateTime(Date(-1999, 7, 6), TimeOfDay(6, 30, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -7, DateTime(Date(-1999, 7, 6), TimeOfDay(5, 30, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -8, DateTime(Date(-1999, 7, 6), TimeOfDay(4, 30, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -9, DateTime(Date(-1999, 7, 6), TimeOfDay(3, 30, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -10, DateTime(Date(-1999, 7, 6), TimeOfDay(2, 30, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -11, DateTime(Date(-1999, 7, 6), TimeOfDay(1, 30, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -12, DateTime(Date(-1999, 7, 6), TimeOfDay(0, 30, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -13, DateTime(Date(-1999, 7, 5), TimeOfDay(23, 30, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -14, DateTime(Date(-1999, 7, 5), TimeOfDay(22, 30, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -15, DateTime(Date(-1999, 7, 5), TimeOfDay(21, 30, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -16, DateTime(Date(-1999, 7, 5), TimeOfDay(20, 30, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -17, DateTime(Date(-1999, 7, 5), TimeOfDay(19, 30, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -18, DateTime(Date(-1999, 7, 5), TimeOfDay(18, 30, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -19, DateTime(Date(-1999, 7, 5), TimeOfDay(17, 30, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -20, DateTime(Date(-1999, 7, 5), TimeOfDay(16, 30, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -21, DateTime(Date(-1999, 7, 5), TimeOfDay(15, 30, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -22, DateTime(Date(-1999, 7, 5), TimeOfDay(14, 30, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -23, DateTime(Date(-1999, 7, 5), TimeOfDay(13, 30, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -24, DateTime(Date(-1999, 7, 5), TimeOfDay(12, 30, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -25, DateTime(Date(-1999, 7, 5), TimeOfDay(11, 30, 33)));

        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(0, 30, 33)), 1, DateTime(Date(-1999, 7, 6), TimeOfDay(1, 30, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(0, 30, 33)), 0, DateTime(Date(-1999, 7, 6), TimeOfDay(0, 30, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(0, 30, 33)), -1, DateTime(Date(-1999, 7, 5), TimeOfDay(23, 30, 33)));

        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(23, 30, 33)), 1, DateTime(Date(-1999, 7, 7), TimeOfDay(0, 30, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(23, 30, 33)), 0, DateTime(Date(-1999, 7, 6), TimeOfDay(23, 30, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(23, 30, 33)), -1, DateTime(Date(-1999, 7, 6), TimeOfDay(22, 30, 33)));

        testDT(DateTime(Date(-1999, 7, 31), TimeOfDay(23, 30, 33)), 1, DateTime(Date(-1999, 8, 1), TimeOfDay(0, 30, 33)));
        testDT(DateTime(Date(-1999, 8, 1), TimeOfDay(0, 30, 33)), -1, DateTime(Date(-1999, 7, 31), TimeOfDay(23, 30, 33)));

        testDT(DateTime(Date(-2001, 12, 31), TimeOfDay(23, 30, 33)), 1, DateTime(Date(-2000, 1, 1), TimeOfDay(0, 30, 33)));
        testDT(DateTime(Date(-2000, 1, 1), TimeOfDay(0, 30, 33)), -1, DateTime(Date(-2001, 12, 31), TimeOfDay(23, 30, 33)));

        testDT(DateTime(Date(-2001, 2, 28), TimeOfDay(23, 30, 33)), 25, DateTime(Date(-2001, 3, 2), TimeOfDay(0, 30, 33)));
        testDT(DateTime(Date(-2001, 3, 2), TimeOfDay(0, 30, 33)), -25, DateTime(Date(-2001, 2, 28), TimeOfDay(23, 30, 33)));

        testDT(DateTime(Date(-2000, 2, 28), TimeOfDay(23, 30, 33)), 25, DateTime(Date(-2000, 3, 1), TimeOfDay(0, 30, 33)));
        testDT(DateTime(Date(-2000, 3, 1), TimeOfDay(0, 30, 33)), -25, DateTime(Date(-2000, 2, 28), TimeOfDay(23, 30, 33)));

        //Test Both
        testDT(DateTime(Date(-1, 1, 1), TimeOfDay(11, 30, 33)), 17_546, DateTime(Date(1, 1, 1), TimeOfDay(13, 30, 33)));
        testDT(DateTime(Date(1, 1, 1), TimeOfDay(13, 30, 33)), -17_546, DateTime(Date(-1, 1, 1), TimeOfDay(11, 30, 33)));

        assertExcNotThrown!(Exception, (){posInfinity.addMinutes(4);})(LineInfo());
        assertExcNotThrown!(Exception, (){negInfinity.addMinutes(4);})(LineInfo());
        assertExcThrown!(Exception, (){notADateTime.addMinutes(4);})(LineInfo());

        {
            auto dt = posInfinity;
            posInfinity.addMinutes(5);
            assertEqual(dt, posInfinity);
        }

        {
            auto dt = negInfinity;
            negInfinity.addMinutes(5);
            assertEqual(dt, negInfinity);
        }

        const ctod = TimeOfDay(0, 0, 0);
        static assert(!__traits(compiles, ctod.addHours(7)));
        immutable itod = TimeOfDay(0, 0, 0);
        static assert(!__traits(compiles, itod.addHours(7)));
    }


    /++
        Add minutes to the time of day. Negative values will subtract. If the number of minutes
        overflows (or underflows), then the minutes will wrap, increasing (or decreasing) the
        number of hours accordingly. If the number of hours overflows (or underflows), then the
        hours will wrap. (e.g. adding 90 minutes to 23:30 would result in 01:00). Also, if the
        number of hours wrap, then the day is incremented (or decremented) appropriately as well
        as any greater units if necessary.

        Params:
            minutes = The numbef or minutes to add to this DateTime.
      +/
    void addMinutes(long minutes)
    {
        _date.enforceIsADate();

        if(isInfinity)
            return;

        immutable hours = TimeUnitConv.to!(TimeUnit.minute, TimeUnit.hour)(minutes);
        immutable quotient = TimeUnitConv.to!(TimeUnit.hour, TimeUnit.minute)(hours);
        immutable remainder = minutes - quotient;
        immutable newMinutes = _tod._minutes + remainder;

        if(newMinutes < 0)
        {
            addHours(hours - 1);
            _tod._minutes = cast(Minute)(_tod.maxMinutes + 1 + newMinutes);
        }
        else if(newMinutes >= 60)
        {
            addHours(hours + 1);
            _tod._minutes = cast(Minute)(newMinutes % (_tod.maxMinutes + 1));
        }
        else
        {
            addHours(hours);
            _tod._minutes = cast(Minute)newMinutes;
        }
    }

    unittest
    {
        static void testDT(DateTime orig, int minutes, in DateTime expected, size_t line = __LINE__)
        {
            orig.addMinutes(minutes);
            assertEqual(orig, expected, "", __FILE__, line);
        }

        //Test A.D.
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 0, DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 1, DateTime(Date(1999, 7, 6), TimeOfDay(12, 31, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 2, DateTime(Date(1999, 7, 6), TimeOfDay(12, 32, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 3, DateTime(Date(1999, 7, 6), TimeOfDay(12, 33, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 4, DateTime(Date(1999, 7, 6), TimeOfDay(12, 34, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 5, DateTime(Date(1999, 7, 6), TimeOfDay(12, 35, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 10, DateTime(Date(1999, 7, 6), TimeOfDay(12, 40, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 15, DateTime(Date(1999, 7, 6), TimeOfDay(12, 45, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 29, DateTime(Date(1999, 7, 6), TimeOfDay(12, 59, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 30, DateTime(Date(1999, 7, 6), TimeOfDay(13, 0, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 45, DateTime(Date(1999, 7, 6), TimeOfDay(13, 15, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 60, DateTime(Date(1999, 7, 6), TimeOfDay(13, 30, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 75, DateTime(Date(1999, 7, 6), TimeOfDay(13, 45, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 90, DateTime(Date(1999, 7, 6), TimeOfDay(14, 0, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 100, DateTime(Date(1999, 7, 6), TimeOfDay(14, 10, 33)));

        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 689, DateTime(Date(1999, 7, 6), TimeOfDay(23, 59, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 690, DateTime(Date(1999, 7, 7), TimeOfDay(0, 0, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 691, DateTime(Date(1999, 7, 7), TimeOfDay(0, 1, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 960, DateTime(Date(1999, 7, 7), TimeOfDay(4, 30, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 1439, DateTime(Date(1999, 7, 7), TimeOfDay(12, 29, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 1440, DateTime(Date(1999, 7, 7), TimeOfDay(12, 30, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 1441, DateTime(Date(1999, 7, 7), TimeOfDay(12, 31, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 2880, DateTime(Date(1999, 7, 8), TimeOfDay(12, 30, 33)));

        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -1, DateTime(Date(1999, 7, 6), TimeOfDay(12, 29, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -2, DateTime(Date(1999, 7, 6), TimeOfDay(12, 28, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -3, DateTime(Date(1999, 7, 6), TimeOfDay(12, 27, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -4, DateTime(Date(1999, 7, 6), TimeOfDay(12, 26, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -5, DateTime(Date(1999, 7, 6), TimeOfDay(12, 25, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -10, DateTime(Date(1999, 7, 6), TimeOfDay(12, 20, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -15, DateTime(Date(1999, 7, 6), TimeOfDay(12, 15, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -29, DateTime(Date(1999, 7, 6), TimeOfDay(12, 1, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -30, DateTime(Date(1999, 7, 6), TimeOfDay(12, 0, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -45, DateTime(Date(1999, 7, 6), TimeOfDay(11, 45, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -60, DateTime(Date(1999, 7, 6), TimeOfDay(11, 30, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -75, DateTime(Date(1999, 7, 6), TimeOfDay(11, 15, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -90, DateTime(Date(1999, 7, 6), TimeOfDay(11, 0, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -100, DateTime(Date(1999, 7, 6), TimeOfDay(10, 50, 33)));

        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -749, DateTime(Date(1999, 7, 6), TimeOfDay(0, 1, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -750, DateTime(Date(1999, 7, 6), TimeOfDay(0, 0, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -751, DateTime(Date(1999, 7, 5), TimeOfDay(23, 59, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -960, DateTime(Date(1999, 7, 5), TimeOfDay(20, 30, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -1439, DateTime(Date(1999, 7, 5), TimeOfDay(12, 31, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -1440, DateTime(Date(1999, 7, 5), TimeOfDay(12, 30, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -1441, DateTime(Date(1999, 7, 5), TimeOfDay(12, 29, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -2880, DateTime(Date(1999, 7, 4), TimeOfDay(12, 30, 33)));

        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 0, 33)), 1, DateTime(Date(1999, 7, 6), TimeOfDay(12, 1, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 0, 33)), 0, DateTime(Date(1999, 7, 6), TimeOfDay(12, 0, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 0, 33)), -1, DateTime(Date(1999, 7, 6), TimeOfDay(11, 59, 33)));

        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(11, 59, 33)), 1, DateTime(Date(1999, 7, 6), TimeOfDay(12, 0, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(11, 59, 33)), 0, DateTime(Date(1999, 7, 6), TimeOfDay(11, 59, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(11, 59, 33)), -1, DateTime(Date(1999, 7, 6), TimeOfDay(11, 58, 33)));

        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(0, 0, 33)), 1, DateTime(Date(1999, 7, 6), TimeOfDay(0, 1, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(0, 0, 33)), 0, DateTime(Date(1999, 7, 6), TimeOfDay(0, 0, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(0, 0, 33)), -1, DateTime(Date(1999, 7, 5), TimeOfDay(23, 59, 33)));

        testDT(DateTime(Date(1999, 7, 5), TimeOfDay(23, 59, 33)), 1, DateTime(Date(1999, 7, 6), TimeOfDay(0, 0, 33)));
        testDT(DateTime(Date(1999, 7, 5), TimeOfDay(23, 59, 33)), 0, DateTime(Date(1999, 7, 5), TimeOfDay(23, 59, 33)));
        testDT(DateTime(Date(1999, 7, 5), TimeOfDay(23, 59, 33)), -1, DateTime(Date(1999, 7, 5), TimeOfDay(23, 58, 33)));

        testDT(DateTime(Date(1998, 12, 31), TimeOfDay(23, 59, 33)), 1, DateTime(Date(1999, 1, 1), TimeOfDay(0, 0, 33)));
        testDT(DateTime(Date(1998, 12, 31), TimeOfDay(23, 59, 33)), 0, DateTime(Date(1998, 12, 31), TimeOfDay(23, 59, 33)));
        testDT(DateTime(Date(1998, 12, 31), TimeOfDay(23, 59, 33)), -1, DateTime(Date(1998, 12, 31), TimeOfDay(23, 58, 33)));

        //Test B.C.
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 0, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 1, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 31, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 2, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 32, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 3, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 33, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 4, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 34, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 5, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 35, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 10, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 40, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 15, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 45, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 29, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 59, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 30, DateTime(Date(-1999, 7, 6), TimeOfDay(13, 0, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 45, DateTime(Date(-1999, 7, 6), TimeOfDay(13, 15, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 60, DateTime(Date(-1999, 7, 6), TimeOfDay(13, 30, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 75, DateTime(Date(-1999, 7, 6), TimeOfDay(13, 45, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 90, DateTime(Date(-1999, 7, 6), TimeOfDay(14, 0, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 100, DateTime(Date(-1999, 7, 6), TimeOfDay(14, 10, 33)));

        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 689, DateTime(Date(-1999, 7, 6), TimeOfDay(23, 59, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 690, DateTime(Date(-1999, 7, 7), TimeOfDay(0, 0, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 691, DateTime(Date(-1999, 7, 7), TimeOfDay(0, 1, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 960, DateTime(Date(-1999, 7, 7), TimeOfDay(4, 30, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 1439, DateTime(Date(-1999, 7, 7), TimeOfDay(12, 29, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 1440, DateTime(Date(-1999, 7, 7), TimeOfDay(12, 30, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 1441, DateTime(Date(-1999, 7, 7), TimeOfDay(12, 31, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 2880, DateTime(Date(-1999, 7, 8), TimeOfDay(12, 30, 33)));

        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -1, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 29, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -2, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 28, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -3, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 27, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -4, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 26, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -5, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 25, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -10, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 20, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -15, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 15, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -29, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 1, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -30, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 0, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -45, DateTime(Date(-1999, 7, 6), TimeOfDay(11, 45, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -60, DateTime(Date(-1999, 7, 6), TimeOfDay(11, 30, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -75, DateTime(Date(-1999, 7, 6), TimeOfDay(11, 15, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -90, DateTime(Date(-1999, 7, 6), TimeOfDay(11, 0, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -100, DateTime(Date(-1999, 7, 6), TimeOfDay(10, 50, 33)));

        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -749, DateTime(Date(-1999, 7, 6), TimeOfDay(0, 1, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -750, DateTime(Date(-1999, 7, 6), TimeOfDay(0, 0, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -751, DateTime(Date(-1999, 7, 5), TimeOfDay(23, 59, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -960, DateTime(Date(-1999, 7, 5), TimeOfDay(20, 30, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -1439, DateTime(Date(-1999, 7, 5), TimeOfDay(12, 31, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -1440, DateTime(Date(-1999, 7, 5), TimeOfDay(12, 30, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -1441, DateTime(Date(-1999, 7, 5), TimeOfDay(12, 29, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -2880, DateTime(Date(-1999, 7, 4), TimeOfDay(12, 30, 33)));

        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 0, 33)), 1, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 1, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 0, 33)), 0, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 0, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 0, 33)), -1, DateTime(Date(-1999, 7, 6), TimeOfDay(11, 59, 33)));

        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(11, 59, 33)), 1, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 0, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(11, 59, 33)), 0, DateTime(Date(-1999, 7, 6), TimeOfDay(11, 59, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(11, 59, 33)), -1, DateTime(Date(-1999, 7, 6), TimeOfDay(11, 58, 33)));

        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(0, 0, 33)), 1, DateTime(Date(-1999, 7, 6), TimeOfDay(0, 1, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(0, 0, 33)), 0, DateTime(Date(-1999, 7, 6), TimeOfDay(0, 0, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(0, 0, 33)), -1, DateTime(Date(-1999, 7, 5), TimeOfDay(23, 59, 33)));

        testDT(DateTime(Date(-1999, 7, 5), TimeOfDay(23, 59, 33)), 1, DateTime(Date(-1999, 7, 6), TimeOfDay(0, 0, 33)));
        testDT(DateTime(Date(-1999, 7, 5), TimeOfDay(23, 59, 33)), 0, DateTime(Date(-1999, 7, 5), TimeOfDay(23, 59, 33)));
        testDT(DateTime(Date(-1999, 7, 5), TimeOfDay(23, 59, 33)), -1, DateTime(Date(-1999, 7, 5), TimeOfDay(23, 58, 33)));

        testDT(DateTime(Date(-2000, 12, 31), TimeOfDay(23, 59, 33)), 1, DateTime(Date(-1999, 1, 1), TimeOfDay(0, 0, 33)));
        testDT(DateTime(Date(-2000, 12, 31), TimeOfDay(23, 59, 33)), 0, DateTime(Date(-2000, 12, 31), TimeOfDay(23, 59, 33)));
        testDT(DateTime(Date(-2000, 12, 31), TimeOfDay(23, 59, 33)), -1, DateTime(Date(-2000, 12, 31), TimeOfDay(23, 58, 33)));

        //Test Both
        testDT(DateTime(Date(1, 1, 1), TimeOfDay(0, 0, 0)), -1, DateTime(Date(0, 12, 31), TimeOfDay(23, 59, 0)));
        testDT(DateTime(Date(0, 12, 31), TimeOfDay(23, 59, 0)), 1, DateTime(Date(1, 1, 1), TimeOfDay(0, 0, 0)));

        testDT(DateTime(Date(0, 1, 1), TimeOfDay(0, 0, 0)), -1, DateTime(Date(-1, 12, 31), TimeOfDay(23, 59, 0)));
        testDT(DateTime(Date(-1, 12, 31), TimeOfDay(23, 59, 0)), 1, DateTime(Date(0, 1, 1), TimeOfDay(0, 0, 0)));

        testDT(DateTime(Date(-1, 1, 1), TimeOfDay(11, 30, 33)), 1_052_760, DateTime(Date(1, 1, 1), TimeOfDay(13, 30, 33)));
        testDT(DateTime(Date(1, 1, 1), TimeOfDay(13, 30, 33)), -1_052_760, DateTime(Date(-1, 1, 1), TimeOfDay(11, 30, 33)));

        testDT(DateTime(Date(-1, 1, 1), TimeOfDay(11, 30, 33)), 1_052_782, DateTime(Date(1, 1, 1), TimeOfDay(13, 52, 33)));
        testDT(DateTime(Date(1, 1, 1), TimeOfDay(13, 52, 33)), -1_052_782, DateTime(Date(-1, 1, 1), TimeOfDay(11, 30, 33)));


        assertExcNotThrown!(Exception, (){posInfinity.addMinutes(4);})(LineInfo());
        assertExcNotThrown!(Exception, (){negInfinity.addMinutes(4);})(LineInfo());
        assertExcThrown!(Exception, (){notADateTime.addMinutes(4);})(LineInfo());

        {
            auto dt = posInfinity;
            posInfinity.addMinutes(5);
            assertEqual(dt, posInfinity);
        }

        {
            auto dt = negInfinity;
            negInfinity.addMinutes(5);
            assertEqual(dt, negInfinity);
        }

        auto td = DateTime(1999, 7, 6, 12, 30, 33);
        const ctd = DateTime(1999, 7, 6, 12, 30, 33);
        immutable itd = DateTime(1999, 7, 6, 12, 30, 33);
        static assert(__traits(compiles, td.addMinutes(4)));
        static assert(!__traits(compiles, ctd.addMinutes(4)));
        static assert(!__traits(compiles, itd.addMinutes(4)));
    }


    /++
        Add seconds to the time of day. Negative values will subtract. If the number of seconds
        overflows (or underflows), then the seconds will wrap, increasing (or decreasing) the
        number of minutes accordingly. If the number of minutes overflows (or underflows), then the
        minutes will wrap. If the number of minutes overflows(or underflows), then the hour will
        wrap. (e.g. adding 90 seconds to 23:59:00 would result in 00:00:30). Also, if the number
        of hours wrap, then the day is incremented (or decremented) appropriately as well as
        any greater units if necessary.

        Params:
            seconds = The numbef or seconds to add to this DateTime.
      +/
    void addSeconds(long seconds)
    {
        _date.enforceIsADate();

        if(isInfinity)
            return;

        immutable minutes = TimeUnitConv.to!(TimeUnit.second, TimeUnit.minute)(seconds);
        immutable quotient = TimeUnitConv.to!(TimeUnit.minute, TimeUnit.second)(minutes);
        immutable remainder = seconds - quotient;
        immutable newSeconds = _tod._seconds + remainder;

        if(newSeconds < 0)
        {
            addMinutes(minutes - 1);
            _tod._seconds = cast(Minute)(_tod.maxSeconds + 1 + newSeconds);
        }
        else if(newSeconds >= 60)
        {
            addMinutes(minutes + 1);
            _tod._seconds = cast(Minute)(newSeconds % (_tod.maxSeconds + 1));
        }
        else
        {
            addMinutes(minutes);
            _tod._seconds = cast(Minute)newSeconds;
        }
    }

    unittest
    {
        static void testDT(DateTime orig, int seconds, in DateTime expected, size_t line = __LINE__)
        {
            orig.addSeconds(seconds);
            assertEqual(orig, expected, "", __FILE__, line);
        }

        //Test A.D.
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 0, DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 1, DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 34)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 2, DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 35)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 3, DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 36)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 4, DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 37)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 5, DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 38)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 10, DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 43)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 15, DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 48)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 26, DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 59)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 27, DateTime(Date(1999, 7, 6), TimeOfDay(12, 31, 0)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 30, DateTime(Date(1999, 7, 6), TimeOfDay(12, 31, 3)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 59, DateTime(Date(1999, 7, 6), TimeOfDay(12, 31, 32)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 60, DateTime(Date(1999, 7, 6), TimeOfDay(12, 31, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 61, DateTime(Date(1999, 7, 6), TimeOfDay(12, 31, 34)));

        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 1766, DateTime(Date(1999, 7, 6), TimeOfDay(12, 59, 59)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 1767, DateTime(Date(1999, 7, 6), TimeOfDay(13, 0, 0)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 1768, DateTime(Date(1999, 7, 6), TimeOfDay(13, 0, 1)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 2007, DateTime(Date(1999, 7, 6), TimeOfDay(13, 4, 0)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 3599, DateTime(Date(1999, 7, 6), TimeOfDay(13, 30, 32)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 3600, DateTime(Date(1999, 7, 6), TimeOfDay(13, 30, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 3601, DateTime(Date(1999, 7, 6), TimeOfDay(13, 30, 34)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 7200, DateTime(Date(1999, 7, 6), TimeOfDay(14, 30, 33)));

        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -1, DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 32)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -2, DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 31)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -3, DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 30)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -4, DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 29)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -5, DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 28)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -10, DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 23)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -15, DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 18)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -33, DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 0)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -34, DateTime(Date(1999, 7, 6), TimeOfDay(12, 29, 59)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -35, DateTime(Date(1999, 7, 6), TimeOfDay(12, 29, 58)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -59, DateTime(Date(1999, 7, 6), TimeOfDay(12, 29, 34)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -60, DateTime(Date(1999, 7, 6), TimeOfDay(12, 29, 33)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -61, DateTime(Date(1999, 7, 6), TimeOfDay(12, 29, 32)));

        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 0)), 1, DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 1)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 0)), 0, DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 0)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 0)), -1, DateTime(Date(1999, 7, 6), TimeOfDay(12, 29, 59)));

        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 0, 0)), 1, DateTime(Date(1999, 7, 6), TimeOfDay(12, 0, 1)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 0, 0)), 0, DateTime(Date(1999, 7, 6), TimeOfDay(12, 0, 0)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 0, 0)), -1, DateTime(Date(1999, 7, 6), TimeOfDay(11, 59, 59)));

        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(0, 0, 0)), 1, DateTime(Date(1999, 7, 6), TimeOfDay(0, 0, 1)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(0, 0, 0)), 0, DateTime(Date(1999, 7, 6), TimeOfDay(0, 0, 0)));
        testDT(DateTime(Date(1999, 7, 6), TimeOfDay(0, 0, 0)), -1, DateTime(Date(1999, 7, 5), TimeOfDay(23, 59, 59)));

        testDT(DateTime(Date(1999, 7, 5), TimeOfDay(23, 59, 59)), 1, DateTime(Date(1999, 7, 6), TimeOfDay(0, 0, 0)));
        testDT(DateTime(Date(1999, 7, 5), TimeOfDay(23, 59, 59)), 0, DateTime(Date(1999, 7, 5), TimeOfDay(23, 59, 59)));
        testDT(DateTime(Date(1999, 7, 5), TimeOfDay(23, 59, 59)), -1, DateTime(Date(1999, 7, 5), TimeOfDay(23, 59, 58)));

        testDT(DateTime(Date(1998, 12, 31), TimeOfDay(23, 59, 59)), 1, DateTime(Date(1999, 1, 1), TimeOfDay(0, 0, 0)));
        testDT(DateTime(Date(1998, 12, 31), TimeOfDay(23, 59, 59)), 0, DateTime(Date(1998, 12, 31), TimeOfDay(23, 59, 59)));
        testDT(DateTime(Date(1998, 12, 31), TimeOfDay(23, 59, 59)), -1, DateTime(Date(1998, 12, 31), TimeOfDay(23, 59, 58)));

        //Test B.C.
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 0, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 1, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 34)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 2, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 35)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 3, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 36)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 4, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 37)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 5, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 38)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 10, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 43)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 15, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 48)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 26, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 59)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 27, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 31, 0)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 30, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 31, 3)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 59, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 31, 32)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 60, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 31, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 61, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 31, 34)));

        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 1766, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 59, 59)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 1767, DateTime(Date(-1999, 7, 6), TimeOfDay(13, 0, 0)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 1768, DateTime(Date(-1999, 7, 6), TimeOfDay(13, 0, 1)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 2007, DateTime(Date(-1999, 7, 6), TimeOfDay(13, 4, 0)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 3599, DateTime(Date(-1999, 7, 6), TimeOfDay(13, 30, 32)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 3600, DateTime(Date(-1999, 7, 6), TimeOfDay(13, 30, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 3601, DateTime(Date(-1999, 7, 6), TimeOfDay(13, 30, 34)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 7200, DateTime(Date(-1999, 7, 6), TimeOfDay(14, 30, 33)));

        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -1, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 32)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -2, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 31)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -3, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 30)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -4, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 29)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -5, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 28)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -10, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 23)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -15, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 18)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -33, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 0)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -34, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 29, 59)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -35, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 29, 58)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -59, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 29, 34)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -60, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 29, 33)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -61, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 29, 32)));

        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 0)), 1, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 1)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 0)), 0, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 0)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 0)), -1, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 29, 59)));

        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 0, 0)), 1, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 0, 1)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 0, 0)), 0, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 0, 0)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 0, 0)), -1, DateTime(Date(-1999, 7, 6), TimeOfDay(11, 59, 59)));

        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(0, 0, 0)), 1, DateTime(Date(-1999, 7, 6), TimeOfDay(0, 0, 1)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(0, 0, 0)), 0, DateTime(Date(-1999, 7, 6), TimeOfDay(0, 0, 0)));
        testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(0, 0, 0)), -1, DateTime(Date(-1999, 7, 5), TimeOfDay(23, 59, 59)));

        testDT(DateTime(Date(-1999, 7, 5), TimeOfDay(23, 59, 59)), 1, DateTime(Date(-1999, 7, 6), TimeOfDay(0, 0, 0)));
        testDT(DateTime(Date(-1999, 7, 5), TimeOfDay(23, 59, 59)), 0, DateTime(Date(-1999, 7, 5), TimeOfDay(23, 59, 59)));
        testDT(DateTime(Date(-1999, 7, 5), TimeOfDay(23, 59, 59)), -1, DateTime(Date(-1999, 7, 5), TimeOfDay(23, 59, 58)));

        testDT(DateTime(Date(-2000, 12, 31), TimeOfDay(23, 59, 59)), 1, DateTime(Date(-1999, 1, 1), TimeOfDay(0, 0, 0)));
        testDT(DateTime(Date(-2000, 12, 31), TimeOfDay(23, 59, 59)), 0, DateTime(Date(-2000, 12, 31), TimeOfDay(23, 59, 59)));
        testDT(DateTime(Date(-2000, 12, 31), TimeOfDay(23, 59, 59)), -1, DateTime(Date(-2000, 12, 31), TimeOfDay(23, 59, 58)));

        //Test Both
        testDT(DateTime(Date(1, 1, 1), TimeOfDay(0, 0, 0)), -1, DateTime(Date(0, 12, 31), TimeOfDay(23, 59, 59)));
        testDT(DateTime(Date(0, 12, 31), TimeOfDay(23, 59, 59)), 1, DateTime(Date(1, 1, 1), TimeOfDay(0, 0, 0)));

        testDT(DateTime(Date(0, 1, 1), TimeOfDay(0, 0, 0)), -1, DateTime(Date(-1, 12, 31), TimeOfDay(23, 59, 59)));
        testDT(DateTime(Date(-1, 12, 31), TimeOfDay(23, 59, 59)), 1, DateTime(Date(0, 1, 1), TimeOfDay(0, 0, 0)));

        testDT(DateTime(Date(-1, 1, 1), TimeOfDay(11, 30, 33)), 63_165_600L, DateTime(Date(1, 1, 1), TimeOfDay(13, 30, 33)));
        testDT(DateTime(Date(1, 1, 1), TimeOfDay(13, 30, 33)), -63_165_600L, DateTime(Date(-1, 1, 1), TimeOfDay(11, 30, 33)));

        testDT(DateTime(Date(-1, 1, 1), TimeOfDay(11, 30, 33)), 63_165_617L, DateTime(Date(1, 1, 1), TimeOfDay(13, 30, 50)));
        testDT(DateTime(Date(1, 1, 1), TimeOfDay(13, 30, 50)), -63_165_617L, DateTime(Date(-1, 1, 1), TimeOfDay(11, 30, 33)));


        assertExcNotThrown!(Exception, (){posInfinity.addSeconds(4);})(LineInfo());
        assertExcNotThrown!(Exception, (){negInfinity.addSeconds(4);})(LineInfo());
        assertExcThrown!(Exception, (){notADateTime.addSeconds(4);})(LineInfo());

        {
            auto dt = posInfinity;
            posInfinity.addSeconds(5);
            assertEqual(dt, posInfinity);
        }

        {
            auto dt = negInfinity;
            negInfinity.addSeconds(5);
            assertEqual(dt, negInfinity);
        }

        auto td = DateTime(1999, 7, 6, 12, 30, 33);
        const ctd = DateTime(1999, 7, 6, 12, 30, 33);
        immutable itd = DateTime(1999, 7, 6, 12, 30, 33);
        static assert(__traits(compiles, td.addSeconds(4)));
        static assert(!__traits(compiles, ctd.addSeconds(4)));
        static assert(!__traits(compiles, itd.addSeconds(4)));
    }


    /++
        Function for adding time units generically. The given TimeUnit must
        be TimeUnit.year, TimeUnit.month, TimeUnit.week, TimeUnit.day,
        TimeUnit.hour, TimeUnit. minute, or TimeUnit.second;

        Params:
            units         = The type of unit to add to.
            value         = The number of units to add to the given type of unit.
            allowOverflow = Whether the days should be allowed to overflow, causing
                            the month to increment (only really applies to TimeUnit.year
                            and TimeUnit.month, but addTimeUnit() is generic, so it's included;
                            It's ignore otherwise).

        Examples:
        --------------------
        addTimeUnit!(TimeUnit.year)(numYears);     //translates to addYears(numYears);
        addTimeUnit!(TimeUnit.month)(numMonths);   //translates to addMonths(numMonths);
        addTimeUnit!(TimeUnit.week)(numWeeks);     //translates to addMonths(numWeeks);
        addTimeUnit!(TimeUnit.day)(numDays);       //translates to addDays(numDays);
        addTimeUnit!(TimeUnit.hour)(numHours);     //translates to addHours(numHours);
        addTimeUnit!(TimeUnit.minute)(numMinutes); //translates to addMinutes(numMinutes);
        addTimeUnit!(TimeUnit.second)(numSeconds); //translates to addSeconds(numSeconds);
        --------------------
      +/
    void addTimeUnit(TimeUnit units)(long value, AllowDayOverflow allowOverflow = AllowDayOverflow.yes)
        if(units == TimeUnit.year   ||
           units == TimeUnit.month  ||
           units == TimeUnit.week   ||
           units == TimeUnit.day    ||
           units == TimeUnit.hour   ||
           units == TimeUnit.minute ||
           units == TimeUnit.second)
    {
        static if(units == TimeUnit.year)
            addYears(cast(ParameterTypeTuple!(addYears)[0])value, allowOverflow);
        else static if(units == TimeUnit.month)
            addMonths(cast(ParameterTypeTuple!(addMonths)[0])value, allowOverflow);
        else static if(units == TimeUnit.week)
            addWeeks(value);
        else static if(units == TimeUnit.day)
            addDays(value);
        else static if(units == TimeUnit.hour)
            addHours(cast(ParameterTypeTuple!(addHours)[0])value);
        else static if(units == TimeUnit.minute)
            addMinutes(cast(ParameterTypeTuple!(addMinutes)[0])value);
        else static if(units == TimeUnit.second)
            addSeconds(cast(ParameterTypeTuple!(addSeconds)[0])value);
        else
            static assert(0);
    }

    unittest
    {
        {
            auto dt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
            dt.addTimeUnit!(TimeUnit.year)(1);
            assertEqual(dt, DateTime(Date(2000, 7, 6), TimeOfDay(12, 30, 33)));
            dt.addTimeUnit!(TimeUnit.year)(-1);
            assertEqual(dt, DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)));
        }

        {
            auto dt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
            dt.addTimeUnit!(TimeUnit.month)(1);
            assertEqual(dt, DateTime(Date(1999, 8, 6), TimeOfDay(12, 30, 33)));
            dt.addTimeUnit!(TimeUnit.month)(-1);
            assertEqual(dt, DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)));
        }

        {
            auto dt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
            dt.addTimeUnit!(TimeUnit.week)(1);
            assertEqual(dt, DateTime(Date(1999, 7, 13), TimeOfDay(12, 30, 33)));
            dt.addTimeUnit!(TimeUnit.week)(-1);
            assertEqual(dt, DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)));
        }

        {
            auto dt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
            dt.addTimeUnit!(TimeUnit.day)(1);
            assertEqual(dt, DateTime(Date(1999, 7, 7), TimeOfDay(12, 30, 33)));
            dt.addTimeUnit!(TimeUnit.day)(-1);
            assertEqual(dt, DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)));
        }

        {
            auto dt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
            dt.addTimeUnit!(TimeUnit.hour)(1);
            assertEqual(dt, DateTime(Date(1999, 7, 6), TimeOfDay(13, 30, 33)));
            dt.addTimeUnit!(TimeUnit.hour)(-1);
            assertEqual(dt, DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)));
        }

        {
            auto dt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
            dt.addTimeUnit!(TimeUnit.minute)(1);
            assertEqual(dt, DateTime(Date(1999, 7, 6), TimeOfDay(12, 31, 33)));
            dt.addTimeUnit!(TimeUnit.minute)(-1);
            assertEqual(dt, DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)));
        }

        {
            auto dt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
            dt.addTimeUnit!(TimeUnit.second)(1);
            assertEqual(dt, DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 34)));
            dt.addTimeUnit!(TimeUnit.second)(-1);
            assertEqual(dt, DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)));
        }

        auto dt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
        const cdt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
        immutable idt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
        static assert(__traits(compiles, dt.addTimeUnit!(TimeUnit.year)(12)));
        static assert(!__traits(compiles, cdt.addTimeUnit!(TimeUnit.year)(12)));
        static assert(!__traits(compiles, idt.TimeUnit!(TimeUnit.year)(12)));

        static assert(__traits(compiles, dt.addTimeUnit!(TimeUnit.month)(12)));
        static assert(!__traits(compiles, cdt.addTimeUnit!(TimeUnit.month)(12)));
        static assert(!__traits(compiles, idt.TimeUnit!(TimeUnit.month)(12)));

        static assert(__traits(compiles, dt.addTimeUnit!(TimeUnit.week)(12)));
        static assert(!__traits(compiles, cdt.addTimeUnit!(TimeUnit.week)(12)));
        static assert(!__traits(compiles, idt.TimeUnit!(TimeUnit.week)(12)));

        static assert(__traits(compiles, dt.addTimeUnit!(TimeUnit.day)(12)));
        static assert(!__traits(compiles, cdt.addTimeUnit!(TimeUnit.day)(12)));
        static assert(!__traits(compiles, idt.TimeUnit!(TimeUnit.day)(12)));

        static assert(__traits(compiles, dt.addTimeUnit!(TimeUnit.hour)(12)));
        static assert(!__traits(compiles, cdt.addTimeUnit!(TimeUnit.hour)(12)));
        static assert(!__traits(compiles, idt.TimeUnit!(TimeUnit.hour)(12)));

        static assert(__traits(compiles, dt.addTimeUnit!(TimeUnit.minute)(12)));
        static assert(!__traits(compiles, cdt.addTimeUnit!(TimeUnit.minute)(12)));
        static assert(!__traits(compiles, idt.TimeUnit!(TimeUnit.minute)(12)));

        static assert(__traits(compiles, dt.addTimeUnit!(TimeUnit.second)(12)));
        static assert(!__traits(compiles, cdt.addTimeUnit!(TimeUnit.second)(12)));
        static assert(!__traits(compiles, idt.TimeUnit!(TimeUnit.second)(12)));


        static assert(!__traits(compiles, dt.addTimeUnit!(TimeUnit.millisecond)(12)));
        static assert(!__traits(compiles, cdt.addTimeUnit!(TimeUnit.millisecond)(12)));
        static assert(!__traits(compiles, idt.TimeUnit!(TimeUnit.millisecond)(12)));

        static assert(!__traits(compiles, dt.addTimeUnit!(TimeUnit.microsecond)(12)));
        static assert(!__traits(compiles, cdt.addTimeUnit!(TimeUnit.microsecond)(12)));
        static assert(!__traits(compiles, idt.TimeUnit!(TimeUnit.microsecond)(12)));

        static assert(!__traits(compiles, dt.addTimeUnit!(TimeUnit.tick)(12)));
        static assert(!__traits(compiles, cdt.addTimeUnit!(TimeUnit.tick)(12)));
        static assert(!__traits(compiles, idt.TimeUnit!(TimeUnit.tick)(12)));
    }


    /++
        See_Also:
            Date.isLeapYear().
     +/
    @property bool isLeapYear() const
    {
        return _date.isLeapYear;
    }

    unittest
    {
        assertExcThrown!(Exception, (){posInfinity.isLeapYear;})(LineInfo());
        assertExcThrown!(Exception, (){negInfinity.isLeapYear;})(LineInfo());
        assertExcThrown!(Exception, (){notADateTime.isLeapYear;})(LineInfo());

        auto dt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
        const cdt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
        immutable idt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
        static assert(__traits(compiles, dt.isLeapYear));
        static assert(__traits(compiles, cdt.isLeapYear));
        static assert(__traits(compiles, idt.isLeapYear));
    }


    /++
        DateTime as a tuple of year, month, day, hour, minute, and second.
      +/
    @property Tuple!(Year, MonthOfYear, DayOfMonth, Hour, Minute, Second) yearMonthDayHourMinSec() const
    {
        _date.enforceNotSpecial();

        return Tuple!(Year, MonthOfYear, DayOfMonth, Hour, Minute, Second)
                     (_date._year, _date._month, _date._day,
                      _tod._hours, _tod._minutes, _tod._seconds);
    }

    unittest
    {
        assertExcThrown!(Exception, (){posInfinity.yearMonthDayHourMinSec;})(LineInfo());
        assertExcThrown!(Exception, (){negInfinity.yearMonthDayHourMinSec;})(LineInfo());
        assertExcThrown!(Exception, (){notADateTime.yearMonthDayHourMinSec;})(LineInfo());

        auto dt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
        const cdt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
        immutable idt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
        static assert(__traits(compiles, dt.yearMonthDayHourMinSec));
        static assert(__traits(compiles, cdt.yearMonthDayHourMinSec));
        static assert(__traits(compiles, idt.yearMonthDayHourMinSec));
    }


    /++
        See_Also:
            Date.dayOfWeek().
     +/
    @property DayOfWeek dayOfWeek() const
    {
        return _date.dayOfWeek;
    }

    unittest
    {
        assertExcThrown!(Exception, (){posInfinity.dayOfWeek;})(LineInfo());
        assertExcThrown!(Exception, (){negInfinity.dayOfWeek;})(LineInfo());
        assertExcThrown!(Exception, (){notADateTime.dayOfWeek;})(LineInfo());

        auto dt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
        const cdt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
        immutable idt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
        static assert(__traits(compiles, dt.dayOfWeek));
        static assert(__traits(compiles, cdt.dayOfWeek));
        static assert(__traits(compiles, idt.dayOfWeek));
    }


    /++
        See_Also:
            Date.dayOfYear().
     +/
    @property DayOfYear dayOfYear() const
    {
        return _date.dayOfYear;
    }

    unittest
    {
        assertExcThrown!(Exception, (){posInfinity.dayOfYear;})(LineInfo());
        assertExcThrown!(Exception, (){negInfinity.dayOfYear;})(LineInfo());
        assertExcThrown!(Exception, (){notADateTime.dayOfYear;})(LineInfo());

        auto dt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
        const cdt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
        immutable idt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
        static assert(__traits(compiles, dt.dayOfYear));
        static assert(__traits(compiles, cdt.dayOfYear));
        static assert(__traits(compiles, idt.dayOfYear));
    }


    /++
        See_Also:
            Date.dayOfYear().
      +/
    @property void dayOfYear(DayOfYear day)
    {
        _date.dayOfYear = day;
    }

    unittest
    {
        assertExcThrown!(Exception, (){posInfinity.dayOfYear;})(LineInfo());
        assertExcThrown!(Exception, (){negInfinity.dayOfYear;})(LineInfo());
        assertExcThrown!(Exception, (){notADateTime.dayOfYear;})(LineInfo());

        auto dt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
        const cdt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
        immutable idt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
        static assert(__traits(compiles, dt.dayOfYear = 12));
        static assert(!__traits(compiles, cdt.dayOfYear = 12));
        static assert(!__traits(compiles, idt.dayOfYear = 12));
    }


    /++
        See_Also:
            Date.dayOfGregorianCal().
     +/
    @property DayOfGregorianCal dayOfGregorianCal() const
    {
        return _date.dayOfGregorianCal;
    }

    unittest
    {
        assertExcThrown!(Exception, (){posInfinity.dayOfGregorianCal;})(LineInfo());
        assertExcThrown!(Exception, (){negInfinity.dayOfGregorianCal;})(LineInfo());
        assertExcThrown!(Exception, (){notADateTime.dayOfGregorianCal;})(LineInfo());

        auto dt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
        const cdt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
        immutable idt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
        static assert(__traits(compiles, dt.dayOfGregorianCal));
        static assert(__traits(compiles, cdt.dayOfGregorianCal));
        static assert(__traits(compiles, idt.dayOfGregorianCal));
    }


    /++
        See_Also:
            Date.dayOfGregorianCal().
     +/
    @property void dayOfGregorianCal(DayOfGregorianCal days)
    {
        if(isSpecial)
            _tod = TimeOfDay.init;

        _date.dayOfGregorianCal = days;
    }

    unittest
    {
        assertExcNotThrown!(Exception, (){posInfinity.dayOfGregorianCal = 15;})(LineInfo());
        assertExcNotThrown!(Exception, (){negInfinity.dayOfGregorianCal = 15;})(LineInfo());
        assertExcNotThrown!(Exception, (){notADateTime.dayOfGregorianCal = 15;})(LineInfo());

        {
            auto dt = notADateTime;
            dt._tod = TimeOfDay(12, 12, 12);
            dt.dayOfGregorianCal = 4;
            assertEqual(dt, DateTime(Date(1, 1, 4), TimeOfDay.init));
        }

        auto dt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
        const cdt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
        immutable idt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
        static assert(__traits(compiles, dt.dayOfGregorianCal = 7));
        static assert(!__traits(compiles, cdt.dayOfGregorianCal = 7));
        static assert(!__traits(compiles, idt.dayOfGregorianCal = 7));
    }


    /++
        See_Also:
            Date.isoWeek().
     +/
    @property ISOWeek isoWeek() const
    {
        return _date.isoWeek;
    }

    unittest
    {
        assertExcThrown!(Exception, (){posInfinity.isoWeek;})(LineInfo());
        assertExcThrown!(Exception, (){negInfinity.isoWeek;})(LineInfo());
        assertExcThrown!(Exception, (){notADateTime.isoWeek;})(LineInfo());

        auto dt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
        const cdt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
        immutable idt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
        static assert(__traits(compiles, dt.isoWeek));
        static assert(__traits(compiles, cdt.isoWeek));
        static assert(__traits(compiles, idt.isoWeek));
    }

    /++
        TimeOfDay portion of endOfMonth is 23:59:59.

        See_Also:
            Date.endOfMonth().
      +/
    @property DateTime endOfMonth() const
    {
        return DateTime(_date.endOfMonth(), TimeOfDay(23, 59, 59));
    }

    unittest
    {
        assertEqual(DateTime(Date(2000, 2, 1), TimeOfDay.init).endOfMonth,
                    DateTime(Date(2000, 2, 29), TimeOfDay(23, 59, 59)));

        assertExcThrown!(Exception, (){posInfinity.endOfMonth;})(LineInfo());
        assertExcThrown!(Exception, (){negInfinity.endOfMonth;})(LineInfo());
        assertExcThrown!(Exception, (){notADateTime.endOfMonth;})(LineInfo());

        auto dt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
        const cdt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
        immutable idt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
        static assert(__traits(compiles, dt.endOfMonth));
        static assert(__traits(compiles, cdt.endOfMonth));
        static assert(__traits(compiles, idt.endOfMonth));
    }


    /++
        See_Also:
            Date.endOfMonthDay().
      +/
    @property DayOfMonth endOfMonthDay() const
    {
        return _date.endOfMonthDay;
    }

    unittest
    {
        assertExcThrown!(Exception, (){posInfinity.endOfMonthDay;})(LineInfo());
        assertExcThrown!(Exception, (){negInfinity.endOfMonthDay;})(LineInfo());
        assertExcThrown!(Exception, (){notADateTime.endOfMonthDay;})(LineInfo());

        auto dt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
        const cdt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
        immutable idt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
        static assert(__traits(compiles, dt.endOfMonthDay));
        static assert(__traits(compiles, cdt.endOfMonthDay));
        static assert(__traits(compiles, idt.endOfMonthDay));
    }


    /++
        See_Also:
            Date.isAD().
     +/
    @property bool isAD() const
    {
        return _date.isAD;
    }

    unittest
    {
        assertExcNotThrown!(Exception, (){posInfinity.isAD;})(LineInfo());
        assertExcNotThrown!(Exception, (){negInfinity.isAD;})(LineInfo());
        assertExcThrown!(Exception, (){notADateTime.isAD;})(LineInfo());

        auto dt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
        const cdt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
        immutable idt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
        static assert(__traits(compiles, dt.isAD));
        static assert(__traits(compiles, cdt.isAD));
        static assert(__traits(compiles, idt.isAD));
    }


    /++
        Note that it is possible to set the Date portion of a special DateTime and make it so that
        it is a normal DateTime. However, you cannot set the TimeOfDay portion of a special DateTime.
        If you set the DateTime portion of a special DateTime, it's TimeOfDay portion will be TimeOfDay.init.
        It is also possible to make a normal DateTime special by setting its Date portion to a special date.
        Regardless, you cannot do anything with the TimeOfDay portion of a special Date.

        See_Also:
            Date.isSpecial().
      +/
    @property bool isSpecial() const
    {
        return _date.isSpecial;
    }

    unittest
    {
        assertExcNotThrown!(Exception, (){posInfinity.isSpecial;})(LineInfo());
        assertExcNotThrown!(Exception, (){negInfinity.isSpecial;})(LineInfo());
        assertExcNotThrown!(Exception, (){notADateTime.isSpecial;})(LineInfo());

        assert(posInfinity.isSpecial);
        assert(negInfinity.isSpecial);
        assert(notADateTime.isSpecial);
        assert(!DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)).isSpecial);

        auto dt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
        const cdt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
        immutable idt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
        static assert(__traits(compiles, dt.isSpecial));
        static assert(__traits(compiles, cdt.isSpecial));
        static assert(__traits(compiles, idt.isSpecial));
    }


    /++
        See_Also:
            Date.isInfinity().
      +/
    @property bool isInfinity() const
    {
        return _date.isInfinity;
    }

    unittest
    {
        assert(posInfinity.isInfinity);
        assert(negInfinity.isInfinity);
        assert(!notADateTime.isInfinity);
        assert(!DateTime(Date(0, 1, 1), TimeOfDay(12, 30, 33)).isInfinity);
        assert(!DateTime(Date(1, 1, 1), TimeOfDay(12, 30, 33)).isInfinity);
        assert(!DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)).isInfinity);
        assert(!max.isInfinity);
        assert(!min.isInfinity);

        assertExcNotThrown!(Exception, (){posInfinity.isInfinity;})(LineInfo());
        assertExcNotThrown!(Exception, (){negInfinity.isInfinity;})(LineInfo());
        assertExcNotThrown!(Exception, (){notADateTime.isInfinity;})(LineInfo());

        auto dt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
        const cdt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
        immutable idt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
        static assert(__traits(compiles, dt.isInfinity));
        static assert(__traits(compiles, cdt.isInfinity));
        static assert(__traits(compiles, idt.isInfinity));
    }


    /++
        See_Also:
            Date.isPosInfinity().
      +/
    @property bool isPosInfinity() const
    {
        return _date.isPosInfinity;
    }

    unittest
    {
        assert(posInfinity.isPosInfinity);
        assert(!negInfinity.isPosInfinity);
        assert(!notADateTime.isPosInfinity);
        assert(!DateTime(Date(0, 1, 1), TimeOfDay(12, 30, 33)).isPosInfinity);
        assert(!DateTime(Date(1, 1, 1), TimeOfDay(12, 30, 33)).isPosInfinity);
        assert(!DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)).isPosInfinity);
        assert(!max.isPosInfinity);
        assert(!min.isPosInfinity);

        assertExcNotThrown!(Exception, (){posInfinity.isPosInfinity;})(LineInfo());
        assertExcNotThrown!(Exception, (){negInfinity.isPosInfinity;})(LineInfo());
        assertExcNotThrown!(Exception, (){notADateTime.isPosInfinity;})(LineInfo());

        auto dt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
        const cdt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
        immutable idt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
        static assert(__traits(compiles, dt.isPosInfinity));
        static assert(__traits(compiles, cdt.isPosInfinity));
        static assert(__traits(compiles, idt.isPosInfinity));
    }


    /++
        See_Also:
            Date.isNegInfinity().
      +/
    @property bool isNegInfinity() const
    {
        return _date.isNegInfinity;
    }

    unittest
    {
        assert(!posInfinity.isNegInfinity);
        assert(negInfinity.isNegInfinity);
        assert(!notADateTime.isNegInfinity);
        assert(!DateTime(Date(0, 1, 1), TimeOfDay(12, 30, 33)).isNegInfinity);
        assert(!DateTime(Date(1, 1, 1), TimeOfDay(12, 30, 33)).isNegInfinity);
        assert(!DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)).isNegInfinity);
        assert(!max.isNegInfinity);
        assert(!min.isNegInfinity);

        assertExcNotThrown!(Exception, (){posInfinity.isNegInfinity;})(LineInfo());
        assertExcNotThrown!(Exception, (){negInfinity.isNegInfinity;})(LineInfo());
        assertExcNotThrown!(Exception, (){notADateTime.isNegInfinity;})(LineInfo());

        auto dt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
        const cdt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
        immutable idt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
        static assert(__traits(compiles, dt.isNegInfinity));
        static assert(__traits(compiles, cdt.isNegInfinity));
        static assert(__traits(compiles, idt.isNegInfinity));
    }


    /++
        Returns true if this DateTime is NADT (Not-a-DateTime), the DateTime equivalent of NAN.
      +/
    @property bool isNADT() const
    {
        return _date.isNAD;
    }

    unittest
    {
        assert(!posInfinity.isNADT);
        assert(!negInfinity.isNADT);
        assert(notADateTime.isNADT);
        assert(!DateTime(Date(0, 1, 1), TimeOfDay(12, 30, 33)).isNADT);
        assert(!DateTime(Date(1, 1, 1), TimeOfDay(12, 30, 33)).isNADT);
        assert(!DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)).isNADT);
        assert(!max.isNADT);
        assert(!min.isNADT);

        assertExcNotThrown!(Exception, (){posInfinity.isNADT;})(LineInfo());
        assertExcNotThrown!(Exception, (){negInfinity.isNADT;})(LineInfo());
        assertExcNotThrown!(Exception, (){notADateTime.isNADT;})(LineInfo());

        auto dt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
        const cdt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
        immutable idt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
        static assert(__traits(compiles, dt.isNADT));
        static assert(__traits(compiles, cdt.isNADT));
        static assert(__traits(compiles, idt.isNADT));
    }


    /++
        The julian day for this Date at the given time. For example, prior to noon, 1996-03-31 would be the
        julian day number 2_450_173, so this function returns 2_450_173, while from noon onward, it
        the julian day number would be 2_450_174, so this function returns 2_450_174.
      +/
    @property long julianDay() const
    {
        _date.enforceNotSpecial();

        if(_tod._hours < 12)
            return _date.julianDay - 1;
        else
            return _date.julianDay;
    }

    unittest
    {
        assertEqual(DateTime(Date(-4713, 11, 24), TimeOfDay(0, 0, 0)).julianDay, -1);
        assertEqual(DateTime(Date(-4713, 11, 24), TimeOfDay(12, 0, 0)).julianDay, 0);

        assertEqual(DateTime(Date(0, 12, 31), TimeOfDay(0, 0, 0)).julianDay, 1_721_424);
        assertEqual(DateTime(Date(0, 12, 31), TimeOfDay(12, 0, 0)).julianDay, 1_721_425);

        assertEqual(DateTime(Date(1, 1, 1), TimeOfDay(0, 0, 0)).julianDay, 1_721_425);
        assertEqual(DateTime(Date(1, 1, 1), TimeOfDay(12, 0, 0)).julianDay, 1_721_426);

        assertEqual(DateTime(Date(1582, 10, 15), TimeOfDay(0, 0, 0)).julianDay, 2_299_160);
        assertEqual(DateTime(Date(1582, 10, 15), TimeOfDay(12, 0, 0)).julianDay, 2_299_161);

        assertEqual(DateTime(Date(1858, 11, 17), TimeOfDay(0, 0, 0)).julianDay, 2_400_000);
        assertEqual(DateTime(Date(1858, 11, 17), TimeOfDay(12, 0, 0)).julianDay, 2_400_001);

        assertEqual(DateTime(Date(1982, 1, 4), TimeOfDay(0, 0, 0)).julianDay, 2_444_973);
        assertEqual(DateTime(Date(1982, 1, 4), TimeOfDay(12, 0, 0)).julianDay, 2_444_974);

        assertEqual(DateTime(Date(1996, 3, 31), TimeOfDay(0, 0, 0)).julianDay, 2_450_173);
        assertEqual(DateTime(Date(1996, 3, 31), TimeOfDay(12, 0, 0)).julianDay, 2_450_174);

        assertEqual(DateTime(Date(2010, 8, 24), TimeOfDay(0, 0, 0)).julianDay, 2_455_432);
        assertEqual(DateTime(Date(2010, 8, 24), TimeOfDay(12, 0, 0)).julianDay, 2_455_433);

        assertExcThrown!(Exception, (){posInfinity.julianDay;})(LineInfo());
        assertExcThrown!(Exception, (){negInfinity.julianDay;})(LineInfo());
        assertExcThrown!(Exception, (){notADateTime.julianDay;})(LineInfo());

        auto dt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
        const cdt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
        immutable idt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
        static assert(__traits(compiles, dt.julianDay));
        static assert(__traits(compiles, cdt.julianDay));
        static assert(__traits(compiles, idt.julianDay));
    }


    /++
        See_Also:
            Date.modJulianDay().
      +/
    @property long modJulianDay() const
    {
        return _date.modJulianDay;
    }

    unittest
    {
        assertEqual(DateTime(Date(1858, 11, 17), TimeOfDay(0, 0, 0)).modJulianDay, 0);
        assertEqual(DateTime(Date(1858, 11, 17), TimeOfDay(12, 0, 0)).modJulianDay, 0);

        assertEqual(DateTime(Date(2010, 8, 24), TimeOfDay(0, 0, 0)).modJulianDay, 55_432);
        assertEqual(DateTime(Date(2010, 8, 24), TimeOfDay(12, 0, 0)).modJulianDay, 55_432);

        assertExcThrown!(Exception, (){posInfinity.modJulianDay;})(LineInfo());
        assertExcThrown!(Exception, (){negInfinity.modJulianDay;})(LineInfo());
        assertExcThrown!(Exception, (){notADateTime.modJulianDay;})(LineInfo());

        auto dt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
        const cdt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
        immutable idt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
        static assert(__traits(compiles, dt.modJulianDay));
        static assert(__traits(compiles, cdt.modJulianDay));
        static assert(__traits(compiles, idt.modJulianDay));
    }


    /++
        Converts this DateTime to a string with the format YYYYMMDDTHHMMSS.

        Examples:
        --------------------
        assert(DateTime(Date(2010, 7, 4), TimeOfDay(7, 6, 12)).toISOString() == "20100704T070612");
        assert(DateTime(Date(1998, 12, 25), TimeOfDay(2, 15, 0)).toISOString() == "19981225T021500");
        assert(DateTime(Date(0, 1, 5), TimeOfDay(23, 9, 59)).toISOString() == "00000105T230959");
        assert(DateTime(Date(-4, 1, 5), TimeOfDay(0, 0, 2)).toISOString() == "-00040105T000002");
        --------------------
      +/
    string toISOString() const
    {
        if(isSpecial)
            return _date.specialToString();

        return format("%sT%s", _date.toISOString(), _tod.toISOString());
    }

    unittest
    {
        //Test A.D.
        assertEqual(DateTime(Date(9, 12, 4), TimeOfDay(0, 0, 0)).toISOString(), "00091204T000000");
        assertEqual(DateTime(Date(99, 12, 4), TimeOfDay(5, 6, 12)).toISOString(), "00991204T050612");
        assertEqual(DateTime(Date(999, 12, 4), TimeOfDay(13, 44, 59)).toISOString(), "09991204T134459");
        assertEqual(DateTime(Date(9999, 7, 4), TimeOfDay(23, 59, 59)).toISOString(), "99990704T235959");
        assertEqual(DateTime(Date(10000, 10, 20), TimeOfDay(1, 1, 1)).toISOString(), "+100001020T010101");

        //Test B.C.
        assertEqual(DateTime(Date(0, 12, 4), TimeOfDay(0, 12, 4)).toISOString(), "00001204T001204");
        assertEqual(DateTime(Date(-9, 12, 4), TimeOfDay(0, 0, 0)).toISOString(), "-00091204T000000");
        assertEqual(DateTime(Date(-99, 12, 4), TimeOfDay(5, 6, 12)).toISOString(), "-00991204T050612");
        assertEqual(DateTime(Date(-999, 12, 4), TimeOfDay(13, 44, 59)).toISOString(), "-09991204T134459");
        assertEqual(DateTime(Date(-9999, 7, 4), TimeOfDay(23, 59, 59)).toISOString(), "-99990704T235959");
        assertEqual(DateTime(Date(-10000, 10, 20), TimeOfDay(1, 1, 1)).toISOString(), "-100001020T010101");

        assertExcNotThrown!(Exception, (){posInfinity.toISOString;})(LineInfo());
        assertExcNotThrown!(Exception, (){negInfinity.toISOString;})(LineInfo());
        assertExcNotThrown!(Exception, (){notADateTime.toISOString;})(LineInfo());

        assertEqual(posInfinity.toISOString(), Date.posInfinity.specialToString());
        assertEqual(negInfinity.toISOString(), Date.negInfinity.specialToString());
        assertEqual(notADateTime.toISOString(), Date.notADate.specialToString());

        auto date = Date(1999, 7, 6);
        const cdate = Date(1999, 7, 6);
        immutable idate = Date(1999, 7, 6);
        static assert(__traits(compiles, date.toISOString()));
        static assert(__traits(compiles, cdate.toISOString()));
        static assert(__traits(compiles, idate.toISOString()));

        //Verify Examples
        assert(DateTime(Date(2010, 7, 4), TimeOfDay(7, 6, 12)).toISOString() == "20100704T070612");
        assert(DateTime(Date(1998, 12, 25), TimeOfDay(2, 15, 0)).toISOString() == "19981225T021500");
        assert(DateTime(Date(0, 1, 5), TimeOfDay(23, 9, 59)).toISOString() == "00000105T230959");
        assert(DateTime(Date(-4, 1, 5), TimeOfDay(0, 0, 2)).toISOString() == "-00040105T000002");
    }



    /++
        Converts this DateTime to a string with the format YYYY-MM-DDTHH:MM:SS.

        Examples:
        --------------------
        assert(DateTime(Date(2010, 7, 4), TimeOfDay(7, 6, 12)).toISOExtendedString() == "2010-07-04T07:06:12");
        assert(DateTime(Date(1998, 12, 25), TimeOfDay(2, 15, 0)).toISOExtendedString() == "1998-12-25T02:15:00");
        assert(DateTime(Date(0, 1, 5), TimeOfDay(23, 9, 59)).toISOExtendedString() == "0000-01-05T23:09:59");
        assert(DateTime(Date(-4, 1, 5), TimeOfDay(0, 0, 2)).toISOExtendedString() == "-0004-01-05T00:00:02");
        --------------------
      +/
    string toISOExtendedString() const
    {
        if(isSpecial)
            return _date.specialToString();

        return format("%sT%s", _date.toISOExtendedString(), _tod.toISOExtendedString());
    }

    unittest
    {
        //Test A.D.
        assertEqual(DateTime(Date(9, 12, 4), TimeOfDay(0, 0, 0)).toISOExtendedString(), "0009-12-04T00:00:00");
        assertEqual(DateTime(Date(99, 12, 4), TimeOfDay(5, 6, 12)).toISOExtendedString(), "0099-12-04T05:06:12");
        assertEqual(DateTime(Date(999, 12, 4), TimeOfDay(13, 44, 59)).toISOExtendedString(), "0999-12-04T13:44:59");
        assertEqual(DateTime(Date(9999, 7, 4), TimeOfDay(23, 59, 59)).toISOExtendedString(), "9999-07-04T23:59:59");
        assertEqual(DateTime(Date(10000, 10, 20), TimeOfDay(1, 1, 1)).toISOExtendedString(), "+10000-10-20T01:01:01");

        //Test B.C.
        assertEqual(DateTime(Date(0, 12, 4), TimeOfDay(0, 12, 4)).toISOExtendedString(), "0000-12-04T00:12:04");
        assertEqual(DateTime(Date(-9, 12, 4), TimeOfDay(0, 0, 0)).toISOExtendedString(), "-0009-12-04T00:00:00");
        assertEqual(DateTime(Date(-99, 12, 4), TimeOfDay(5, 6, 12)).toISOExtendedString(), "-0099-12-04T05:06:12");
        assertEqual(DateTime(Date(-999, 12, 4), TimeOfDay(13, 44, 59)).toISOExtendedString(), "-0999-12-04T13:44:59");
        assertEqual(DateTime(Date(-9999, 7, 4), TimeOfDay(23, 59, 59)).toISOExtendedString(), "-9999-07-04T23:59:59");
        assertEqual(DateTime(Date(-10000, 10, 20), TimeOfDay(1, 1, 1)).toISOExtendedString(), "-10000-10-20T01:01:01");

        assertExcNotThrown!(Exception, (){posInfinity.toISOExtendedString;})(LineInfo());
        assertExcNotThrown!(Exception, (){negInfinity.toISOExtendedString;})(LineInfo());
        assertExcNotThrown!(Exception, (){notADateTime.toISOExtendedString;})(LineInfo());

        assertEqual(posInfinity.toISOExtendedString(), Date.posInfinity.specialToString());
        assertEqual(negInfinity.toISOExtendedString(), Date.negInfinity.specialToString());
        assertEqual(notADateTime.toISOExtendedString(), Date.notADate.specialToString());

        auto date = Date(1999, 7, 6);
        const cdate = Date(1999, 7, 6);
        immutable idate = Date(1999, 7, 6);
        static assert(__traits(compiles, date.toISOExtendedString()));
        static assert(__traits(compiles, cdate.toISOExtendedString()));
        static assert(__traits(compiles, idate.toISOExtendedString()));

        //Verify Examples
        assert(DateTime(Date(2010, 7, 4), TimeOfDay(7, 6, 12)).toISOExtendedString() == "2010-07-04T07:06:12");
        assert(DateTime(Date(1998, 12, 25), TimeOfDay(2, 15, 0)).toISOExtendedString() == "1998-12-25T02:15:00");
        assert(DateTime(Date(0, 1, 5), TimeOfDay(23, 9, 59)).toISOExtendedString() == "0000-01-05T23:09:59");
        assert(DateTime(Date(-4, 1, 5), TimeOfDay(0, 0, 2)).toISOExtendedString() == "-0004-01-05T00:00:02");
    }

    /++
        Converts this DateTime to a string with the format YYYY-MonthShortName-DD HH:MM:SS.

        Examples:
        --------------------
        assert(DateTime(Date(2010, 7, 4), TimeOfDay(7, 6, 12)).toSimpleString() == "2010-Jul-04 07:06:12");
        assert(DateTime(Date(1998, 12, 25), TimeOfDay(2, 15, 0)).toSimpleString() == "1998-Dec-25 02:15:00");
        assert(DateTime(Date(0, 1, 5), TimeOfDay(23, 9, 59)).toSimpleString() == "0000-Jan-05 23:09:59");
        assert(DateTime(Date(-4, 1, 5), TimeOfDay(0, 0, 2)).toSimpleString() == "-0004-Jan-05 00:00:02");
        --------------------
      +/
    string toSimpleString() const
    {
        if(isSpecial)
            return _date.specialToString();

        return format("%s %s", _date.toSimpleString(), _tod.toString());
    }

    unittest
    {
        //Test A.D.
        assertEqual(DateTime(Date(9, 12, 4), TimeOfDay(0, 0, 0)).toSimpleString(), "0009-Dec-04 00:00:00");
        assertEqual(DateTime(Date(99, 12, 4), TimeOfDay(5, 6, 12)).toSimpleString(), "0099-Dec-04 05:06:12");
        assertEqual(DateTime(Date(999, 12, 4), TimeOfDay(13, 44, 59)).toSimpleString(), "0999-Dec-04 13:44:59");
        assertEqual(DateTime(Date(9999, 7, 4), TimeOfDay(23, 59, 59)).toSimpleString(), "9999-Jul-04 23:59:59");
        assertEqual(DateTime(Date(10000, 10, 20), TimeOfDay(1, 1, 1)).toSimpleString(), "+10000-Oct-20 01:01:01");

        //Test B.C.
        assertEqual(DateTime(Date(0, 12, 4), TimeOfDay(0, 12, 4)).toSimpleString(), "0000-Dec-04 00:12:04");
        assertEqual(DateTime(Date(-9, 12, 4), TimeOfDay(0, 0, 0)).toSimpleString(), "-0009-Dec-04 00:00:00");
        assertEqual(DateTime(Date(-99, 12, 4), TimeOfDay(5, 6, 12)).toSimpleString(), "-0099-Dec-04 05:06:12");
        assertEqual(DateTime(Date(-999, 12, 4), TimeOfDay(13, 44, 59)).toSimpleString(), "-0999-Dec-04 13:44:59");
        assertEqual(DateTime(Date(-9999, 7, 4), TimeOfDay(23, 59, 59)).toSimpleString(), "-9999-Jul-04 23:59:59");
        assertEqual(DateTime(Date(-10000, 10, 20), TimeOfDay(1, 1, 1)).toSimpleString(), "-10000-Oct-20 01:01:01");

        assertExcNotThrown!(Exception, (){posInfinity.toSimpleString;})(LineInfo());
        assertExcNotThrown!(Exception, (){negInfinity.toSimpleString;})(LineInfo());
        assertExcNotThrown!(Exception, (){notADateTime.toSimpleString;})(LineInfo());

        assertEqual(posInfinity.toSimpleString(), Date.posInfinity.specialToString());
        assertEqual(negInfinity.toSimpleString(), Date.negInfinity.specialToString());
        assertEqual(notADateTime.toSimpleString(), Date.notADate.specialToString());

        auto date = Date(1999, 7, 6);
        const cdate = Date(1999, 7, 6);
        immutable idate = Date(1999, 7, 6);
        static assert(__traits(compiles, date.toSimpleString()));
        static assert(__traits(compiles, cdate.toSimpleString()));
        static assert(__traits(compiles, idate.toSimpleString()));

        //Verify Examples
        assert(DateTime(Date(2010, 7, 4), TimeOfDay(7, 6, 12)).toSimpleString() == "2010-Jul-04 07:06:12");
        assert(DateTime(Date(1998, 12, 25), TimeOfDay(2, 15, 0)).toSimpleString() == "1998-Dec-25 02:15:00");
        assert(DateTime(Date(0, 1, 5), TimeOfDay(23, 9, 59)).toSimpleString() == "0000-Jan-05 23:09:59");
        assert(DateTime(Date(-4, 1, 5), TimeOfDay(0, 0, 2)).toSimpleString() == "-0004-Jan-05 00:00:02");
    }


    /++
        Converts this DateTime to a string.
      +/
    //For some reason, the const version doesn't work with format(), so I'm defining
    //it twice - one const and one non-const.
    string toString()
    {
        return toSimpleString();
    }

    /++
        Converts this DateTime to a string.
      +/
    string toString() const
    {
        return toSimpleString();
    }

    unittest
    {
        auto dt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
        const cdt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
        immutable idt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
        static assert(__traits(compiles, dt.toString()));
        static assert(__traits(compiles, cdt.toString()));
        static assert(__traits(compiles, idt.toString()));
    }

    /++
        Returns a DateTime for which isPosInfinity is true.
      +/
    @property static DateTime posInfinity()
    out(result)
    {
        assert(result.isPosInfinity);
        assert(result.isSpecial);
    }
    body
    {
        return DateTime(Date.posInfinity, TimeOfDay.init);
    }

    /++
        Returns a DateTime for which isNegInfinity is true.
      +/
    @property static DateTime negInfinity()
    out(result)
    {
        assert(result.isNegInfinity);
        assert(result.isSpecial);
    }
    body
    {
        return DateTime(Date.negInfinity, TimeOfDay.init);
    }

    /++
        Returns a DateTime for which isNADT is true.
      +/
    @property static DateTime notADateTime()
    out(result)
    {
        assert(result.isNADT);
        assert(result.isSpecial);
    }
    body
    {
        return DateTime(Date.notADate, TimeOfDay.init);
    }


    /++
        Returns the DateTime farthest in the past which is representable by DateTime.
      +/
    @property static DateTime min()
    out(result)
    {
        assert(!result.isSpecial);
    }
    body
    {
        return DateTime(Date.min, TimeOfDay.min);
    }


    /++
        Returns the DateTime farthest in the future which is representable by DateTime.
      +/
    @property static DateTime max()
    out(result)
    {
        assert(!result.isSpecial);
    }
    body
    {
        return DateTime(Date.max, TimeOfDay.max);
    }


    /++
        The lowest time resolution that DateTime has.
      +/
    @property static TimeUnit minResolution()
    {
        return TimeUnit.day;
    }


    /++
        The highest time resolution that DateTime has.
      +/
    @property static TimeUnit maxResolution()
    {
        return TimeUnit.second;
    }


private:

    Date      _date;
    TimeOfDay _tod;
}

