/*
 *  Gazebo - Outdoor Multi-Robot Simulator
 *  Copyright (C) 2003  
 *     Nate Koenig & Andrew Howard
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
/*
 * Desc: Factory for creating models
 * Author: Andrew Howard
 * Date: 14 Jul 2004
 * CVS info: $Id: ExampleModel.cc,v 1.5 2004/11/16 19:22:53 inspectorg Exp $
 */
#include <stdio.h>

#include <gazebo.h>
#include <gazebo/BoxGeom.hh>
#include <gazebo/ModelFactory.hh>
#include <gazebo/WorldFile.hh>
#include <gazebo/World.hh>

#include "immovableObject.hh"


/////////////////////////////////////////////////////////////////////////////
// Register the model
GZ_REGISTER_PLUGIN("ImmovableObject", ImmovableObject)


//////////////////////////////////////////////////////////////////////////////
// Constructor
ImmovableObject::ImmovableObject( World *world )
    : Model( world )
{
  return;
}


//////////////////////////////////////////////////////////////////////////////
// Destructor
ImmovableObject::~ImmovableObject()
{
  return;
}


//////////////////////////////////////////////////////////////////////////////
// Load the sensor
int ImmovableObject::Load( WorldFile *file, WorldFileNode *node )
{
  Geom *geom;

  // Create a dummy body using NULL, since this object is immovable
  this->body = new Body( this->world, NULL, true);
  // add as canonical body
  this->AddBody( this->body, true );
  // note body is dummy so don't set position (do that with geom->SetRelativePosition)

  // get object (box) size and position
  double sx, sy, sz;
  sx = node->GetTupleLength("size", 0, 0.5);
  sy = node->GetTupleLength("size", 1, 0.5);
  sz = node->GetTupleLength("size", 2, 0.5);
  GzVector initialPos = node->GetPosition("xyz", GzVectorZero()); 
  
  // Create a simple box
  geom = new BoxGeom( this->body, this->modelSpaceId, sx, sy, sz );
  geom->SetRelativePosition( GzVectorSet(initialPos.x, initialPos.y, sz / 2));
  geom->SetMass( node->GetDouble("mass", 1, 0) );
  geom->SetColor( node->GetColor("color", GzColor(1.0, 1.0, 1.0)) );

  // set rotation (nb. must set rotation using geom since body is null)
  GzQuatern rot = node->GetRotation("rpy", GzQuaternIdent());
  GzQuatern unit = GzQuaternIdent();
  geom->SetRelativeRotation( node->GetRotation("rpy", GzQuaternIdent()));
  
  return 0;
}


//////////////////////////////////////////////////////////////////////////////
// Initialize the sensor
int ImmovableObject::Init( WorldFile *file, WorldFileNode *node )
{
  return 0;
}


//////////////////////////////////////////////////////////////////////////////
// Finalize the sensor
int ImmovableObject::Fini()
{
  return 0;
}


//////////////////////////////////////////////////////////////////////////////
// Get commands from the external interface
void ImmovableObject::IfaceGetCmd()
{
  // Receive commands from external interface

  return;
}


//////////////////////////////////////////////////////////////////////////////
// Update external interface
void ImmovableObject::IfacePutData()
{
  // Update external interfaces
  
  return;
}


//////////////////////////////////////////////////////////////////////////////
// Update the sensor information; returns non-zero if sensor data has
// been updated
void ImmovableObject::Update( double step )
{
  // Get commands from the external interface
  this->IfaceGetCmd();

  // Command any actuators
  
  // Update any sensors

  // Update the external interface with the new data
  IfacePutData();

  return;
}


