/*
    hlproxy - A Half-Life Proxy Daemon to avoid WON ID authentication from
              clients connecting to it and the same time acting as a 
              legitimate Half-Life game server by authenticating and
              registering to Half-Life Master Servers. 

    Copyright (C) 2001-2002 fooler@skyinet.net

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <string.h>
#include <syslog.h>
#include <varargs.h>
#include <sys/param.h>
#include <netdb.h>
#include <unistd.h>
#include <sys/stat.h>
#include <pwd.h>
#include <signal.h>
#include <sys/time.h>
#include <errno.h>

/*
   next line after this comment is a simple trick to determine
   if the OS is linux or freebsd without using auto configuration
   in order to know the syslog's level for daemon's messages
   like error messages and others in /var/log/messages. linux
   default entry in /etc/syslog.conf is *.INFO while freebsd
   default is *.NOTICE. the reference file is <syslog.h>,
   freebsd has extra log level that linux dont have which is
   LOG_SECURITY
*/
#ifdef LOG_SECURITY
	#define LOG_LEVEL	LOG_NOTICE	/*its freebsd */
#else
	#define	LOG_LEVEL	LOG_INFO	/*its linux */
#endif

static const char *apcHLMasterServers[] =
{	"half-life.east.won.net",
	"half-life.west.won.net",
	"half-life.central.won.net"
};

#define NOQUIT		0
#define QUIT		1
#define NONROOT		"nobody"
#define TIMEOUT		60		/* 60 seconds udp connection timeout */
#define HEARTBEAT	(300 / TIMEOUT)	/* 5 minutes heartbeat to master servers */
#define BUFFSIZE	1472		/* udp buffer [MTU - IP - UDP] */
#define ADDRLEN		(sizeof(unsigned short int) + sizeof(struct in_addr))
#define HLMLEN		(sizeof(apcHLMasterServers) / sizeof(char *))
#define HLMPORT		27010
#define HLM_Q		'\x71'
#define HLM_QLEN	1
#define HLM_S		"\xff\xff\xff\xff\x73\x0a\x00\x00\x00\x00"
#define HLM_SLEN	10
#define HLM_SOFFSET	6
#define HLM_CHALLENGE	"\x30\x0a\\protocol\\45\\challenge\\"
#define HLM_PLAYERS	"\\players\\0\\max\\"
#define HLM_GAMEDIR	"\\gamedir\\"
#define HLM_MAP		"\\map\\"
#define HLM_TYPE	"\\type\\d\\password\\0\\os\\l\\lan\\0"
#define SYSLOG(iLevel, pcMessage, iQuit)	HLSyslog((iLevel), (pcMessage), (iQuit), __FILE__, (__LINE__ - 1))

static const char *apcErrMsg[] =
{	"out of memory",
	"cant fork",
	"cant setsid",
	"cant chdir",
	"cant change user",
	"cant setgid",
	"cant setuid",
	"cant signal",
	"cant create socket",
	"cant bind socket",
	"incomplete sending data",
	"unknown select interrupt",
	"invalid reply by HL master server",
	"unknown recvfrom interrupt"
};

#define ERRMEMORY	0
#define ERRFORK		1
#define ERRSETSID	2
#define ERRCHDIR	3
#define ERRNOBODY	4
#define ERRSETGID	5
#define ERRSETUID	6
#define ERRSIGNAL	7
#define ERRSOCKET	8
#define ERRBIND		9
#define ERRSENDTO	10
#define	ERRSELECT	11
#define ERRHLMS		12
#define ERRRECVFROM	13

static int iTimeout;

struct hlmaster {
	struct sockaddr_in sinMaster;
	struct hlmaster *phlmNext;
};

struct sockdata {
	int iFD;
	unsigned long int uliUpdated;
	struct sockaddr_in sinRemote;
	struct sockdata *psdNext;
};

static void HLSyslog(int iLevel, char *pcMessage, int iQuit, char *pcFilename, int iLineNo) {

	if (iLevel == LOG_ERR) {
		syslog(LOG_LEVEL, "%s: %s @ %s:%d", "ERROR", pcMessage, pcFilename, iLineNo);
	} else {
		syslog(LOG_LEVEL, "%s: %s", "INFO", pcMessage);
	}
	if (iQuit) {
		exit(1);
	}
}

static void ShowUsage(char *pcErrMsg, char *pcProgName) {

	fprintf(stderr,	"Error: %s\n"
			"Usage: %s <ip address> <udp port> <gamedir> <map> <maxplayers>\n"
			, pcErrMsg, pcProgName);
	exit(1);
}

static void IntegerToAscii(char *pcBuffer, unsigned long int uliNumber) {
	char acTempBuffer[16];
	char *pcIndex = acTempBuffer + sizeof(acTempBuffer) - 1;

	*pcIndex-- = '\x0';
	while (uliNumber) {
		*pcIndex-- = '0' + (uliNumber % 10);
		uliNumber = uliNumber / 10;
	}
	strcpy(pcBuffer, pcIndex + 1);
}

static int ChallengeKey(char *pcBuffer, char *pcHlmStr) {
	unsigned long int uliKey = *((unsigned long int *) (pcBuffer + HLM_SOFFSET));

	strcpy(pcBuffer, HLM_CHALLENGE);
	IntegerToAscii(pcBuffer + strlen(pcBuffer), uliKey);
	strcpy(pcBuffer + strlen(pcBuffer), pcHlmStr);

	return (strlen(pcBuffer));
}

static void InitSystem(int argc, char *argv[], struct sockaddr_in *psinServer, struct sockaddr_in *psinTarget, struct sockaddr_in *psinLocal, struct hlmaster **phlmList, char **pcHlmStr) {
	char *pcProgName, *pcTemp;
	struct in_addr inaIPaddr;
	unsigned short int usiUDPport;
	int iMaxPlayers, iIndex, iList;
	struct hostent *pheMasterServer;
	struct hlmaster *phlmTemp;
	char acBuffer[BUFFSIZE];

	pcProgName = argv[0];
	if (argc != 6) {
		ShowUsage("Lack of parameters", pcProgName);
	}
	if (! inet_aton(argv[1], &inaIPaddr)) {
		ShowUsage("Invalid IP address", pcProgName);
	}
	if ((usiUDPport = atoi(argv[2])) < 1024) {
		ShowUsage("Invalid UDP port", pcProgName);
	}
	if ((iMaxPlayers = atoi(argv[5])) < 2) {
		ShowUsage("Invalid maxplayers", pcProgName);
	}

	if ((pcTemp = strrchr(pcProgName, '/'))) {
		pcProgName = ++pcTemp;
	}

	openlog(pcProgName, LOG_NDELAY | LOG_PID, LOG_USER);

	memset(psinLocal, 0, sizeof(struct sockaddr));
	psinLocal->sin_family = AF_INET;
	psinLocal->sin_addr.s_addr = htonl(INADDR_LOOPBACK);

	memcpy(psinTarget, psinLocal, sizeof(struct sockaddr));
	psinTarget->sin_port = htons(usiUDPport);

	memcpy(psinServer, psinTarget, sizeof(struct sockaddr));
	psinServer->sin_addr = inaIPaddr;
 
	*phlmList = NULL;
	for (iIndex = HLMLEN; iIndex; iIndex--) {
		if (! (pheMasterServer = gethostbyname(apcHLMasterServers[iIndex - 1]))) {
			SYSLOG(LOG_ERR, (char *) apcHLMasterServers[iIndex - 1], QUIT);
		}
		for (iList = 0; pheMasterServer->h_addr_list[iList]; iList++) {
			if (! (phlmTemp = (struct hlmaster *) malloc(sizeof(struct hlmaster)))) {
				SYSLOG(LOG_ERR, (char *) apcErrMsg[ERRMEMORY], QUIT);
			}
			memcpy(&(phlmTemp->sinMaster), psinLocal, sizeof(struct sockaddr));
			phlmTemp->sinMaster.sin_port = htons(HLMPORT);
			memcpy(&(phlmTemp->sinMaster.sin_addr), pheMasterServer->h_addr_list[iList], sizeof(struct in_addr));
			phlmTemp->phlmNext = *phlmList;
			*phlmList = phlmTemp;
		}
	}

	strcpy(acBuffer, HLM_PLAYERS);
	IntegerToAscii(acBuffer + strlen(acBuffer), iMaxPlayers);
	strcpy(acBuffer + strlen(acBuffer), HLM_GAMEDIR);
	strcpy(acBuffer + strlen(acBuffer), argv[3]);
	strcpy(acBuffer + strlen(acBuffer), HLM_MAP);
	strcpy(acBuffer + strlen(acBuffer), argv[4]);
	strcpy(acBuffer + strlen(acBuffer), HLM_TYPE);
	if (! (*pcHlmStr = (char *) malloc(strlen(acBuffer) + 1))) {
		SYSLOG(LOG_ERR, (char *) apcErrMsg[ERRMEMORY], QUIT);
	}
	strcpy(*pcHlmStr, acBuffer);
}

static void InitDaemon(void) {
	pid_t	ptPID;
	struct passwd *ppwNobody;

	if ((ptPID = fork()) == -1) {
		SYSLOG(LOG_ERR, (char *) apcErrMsg[ERRFORK], QUIT);
	} else if (ptPID) {
		exit(0);
	}
	if (setsid() == -1) {
		SYSLOG(LOG_ERR, (char *) apcErrMsg[ERRSETSID], QUIT);
	}
	if (chdir("/") == -1) {
		SYSLOG(LOG_ERR, (char *) apcErrMsg[ERRCHDIR], QUIT);
	}
	umask(0);
	close(0); /* close standard input */
	close(1); /* close standard output */
	close(2); /* close standard error */
	if (! getuid()) {
		if (! (ppwNobody = getpwnam(NONROOT))) {
			SYSLOG(LOG_ERR, (char *) apcErrMsg[ERRNOBODY], QUIT);
		}
		if (setgid(ppwNobody->pw_gid) == -1) {
			SYSLOG(LOG_ERR, (char *) apcErrMsg[ERRSETGID], QUIT);
		}
		if (setuid(ppwNobody->pw_uid) == -1) {
			SYSLOG(LOG_ERR, (char *) apcErrMsg[ERRSETUID], QUIT);
		}
	}
	SYSLOG(LOG_INFO, "Daemon is now running!", NOQUIT);
}

static void SignalAlarm(int iSigNo) {

	iTimeout = 1;
	if (signal(iSigNo, SignalAlarm) == SIG_ERR) {
		SYSLOG(LOG_ERR, (char *) apcErrMsg[ERRSIGNAL], QUIT);
	}
}

static void SetAlarm(void) {

	iTimeout = 0;
	alarm(TIMEOUT);
}

int main(int argc, char *argv[]) {
	struct sockaddr_in sinServer, sinTarget, sinLocal, sinRemote;
	struct hlmaster *phlmList, *phlmIndex;
	char *pcHlmStr;
	int iFDserver, iNready, iLen, iTemp;
	struct sockdata *psdActive, *psdIndex, *psdSearch, *psdPrev;
	fd_set fdsMaster, fdsTemp;
	char acBuffer[BUFFSIZE];
	static int iFDmaxplus1;
	static unsigned long int uliUpdated;

	InitSystem(argc, argv, &sinServer, &sinTarget, &sinLocal, &phlmList, &pcHlmStr);
	InitDaemon();

	if ((iFDserver = socket(AF_INET, SOCK_DGRAM, 0)) == -1) {
		SYSLOG(LOG_ERR, (char *) apcErrMsg[ERRSOCKET], QUIT);
	}
	if (bind(iFDserver, (struct sockaddr *) &sinServer, sizeof(struct sockaddr)) == -1) {
		SYSLOG(LOG_ERR, (char *) apcErrMsg[ERRBIND], QUIT);
	}	
	if (! (psdActive = (struct sockdata *) malloc(sizeof(struct sockdata)))) {
		SYSLOG(LOG_ERR, (char *) apcErrMsg[ERRMEMORY], QUIT);
	} 
	psdActive->iFD = iFDserver;
	psdActive->psdNext = NULL;

	FD_ZERO(&fdsMaster);
	FD_SET(iFDserver, &fdsMaster);

	SignalAlarm(SIGALRM);
	/* iTimeout = 1 */
	/* uliUpdated = 0 */

	for ( ; ; ) {
		if (iTimeout) {
			for (iFDmaxplus1 = iFDserver + 1, psdPrev = psdActive, psdIndex = psdActive->psdNext; psdIndex; psdPrev = psdIndex, psdIndex = psdIndex->psdNext) {
				if (psdIndex->uliUpdated == uliUpdated) {
					if (psdIndex->iFD >= iFDmaxplus1) {
						iFDmaxplus1 = psdIndex->iFD + 1;
					}
				} else {
					FD_CLR(psdIndex->iFD, &fdsMaster);
					close(psdIndex->iFD);
					psdPrev->psdNext = psdIndex->psdNext;
					free(psdIndex);
					psdIndex = psdPrev;
				}
			}
			if (! (uliUpdated % HEARTBEAT)) {
				acBuffer[0] = HLM_Q;
				iLen = HLM_QLEN;
				for (phlmIndex = phlmList; phlmIndex; phlmIndex = phlmIndex->phlmNext) {
					if (sendto(iFDserver, acBuffer, iLen, 0, (struct sockaddr *) &(phlmIndex->sinMaster), sizeof(struct sockaddr)) != iLen) {
						SYSLOG(LOG_ERR, (char *) apcErrMsg[ERRSENDTO], NOQUIT);
					}
				}
			}
			uliUpdated++;
			SetAlarm();
		}
		fdsTemp = fdsMaster;
		if ((iNready = select(iFDmaxplus1, &fdsTemp, NULL, NULL, NULL)) == -1) {
			if (errno != EINTR) {
				SYSLOG(LOG_ERR, (char *) apcErrMsg[ERRSELECT], QUIT);
			}
			continue;
		}
		for (psdIndex = psdActive; iNready && psdIndex; psdIndex = psdIndex->psdNext) {
			if (FD_ISSET(psdIndex->iFD, &fdsTemp)) {
				iNready--;
				iTemp = sizeof(struct sockaddr);
				if ((iLen = recvfrom(psdIndex->iFD, acBuffer, BUFFSIZE, 0, (struct sockaddr *) &sinRemote, (socklen_t *) &iTemp)) == -1) {
					if ((errno != EAGAIN) && (errno != EINTR)) {
						SYSLOG(LOG_ERR, (char *) apcErrMsg[ERRRECVFROM], QUIT);
					}
					break;
				}
				if (psdIndex->iFD == iFDserver) {
					for (psdSearch = psdActive->psdNext; psdSearch && memcmp(&(sinRemote.sin_port), &(psdSearch->sinRemote.sin_port), ADDRLEN); psdSearch = psdSearch->psdNext);
					if (! psdSearch) {
						for (phlmIndex = phlmList; phlmIndex && memcmp(&(sinRemote.sin_port), &(phlmIndex->sinMaster.sin_port), ADDRLEN); phlmIndex = phlmIndex->phlmNext);
						if (phlmIndex) {
							if (strncmp(acBuffer, HLM_S, HLM_SOFFSET)) {
								SYSLOG(LOG_ERR, (char *) apcErrMsg[ERRHLMS], NOQUIT);
							} else {
								iLen = ChallengeKey(acBuffer, pcHlmStr);
								if (sendto(iFDserver, acBuffer, iLen, 0, (struct sockaddr *) &(phlmIndex->sinMaster), sizeof(struct sockaddr)) != iLen) {
									SYSLOG(LOG_ERR, (char *) apcErrMsg[ERRSENDTO], NOQUIT);
								}
							}
							continue;
						}
						if ((iTemp = socket(AF_INET, SOCK_DGRAM, 0)) == -1) {
							SYSLOG(LOG_ERR, (char *) apcErrMsg[ERRSOCKET], NOQUIT);
							continue;
						}
						if (! (psdSearch = (struct sockdata *) malloc(sizeof(struct sockdata)))) {
							SYSLOG(LOG_ERR, (char *) apcErrMsg[ERRMEMORY], NOQUIT);
							close(iTemp);
							continue;
						}
						if (bind(iTemp, (struct sockaddr *) &sinLocal, sizeof(struct sockaddr)) == -1) {
							SYSLOG(LOG_ERR, (char *) apcErrMsg[ERRBIND], NOQUIT);
							free(psdSearch);
							close(iTemp);
							continue;
						}
						if (iTemp >= iFDmaxplus1) {
							iFDmaxplus1 = iTemp + 1;
						}
						FD_SET(iTemp, &fdsMaster);
						psdSearch->iFD = iTemp;
						psdSearch->sinRemote = sinRemote;
						psdSearch->psdNext = psdActive->psdNext;
						psdActive->psdNext = psdIndex = psdSearch;
					}
					psdSearch->uliUpdated = uliUpdated;
					if (sendto(psdSearch->iFD, acBuffer, iLen, 0, (struct sockaddr *) &sinTarget, sizeof(struct sockaddr)) != iLen) {
						SYSLOG(LOG_ERR, (char *) apcErrMsg[ERRSENDTO], NOQUIT);
					}
				} else {
					psdIndex->uliUpdated = uliUpdated;
					if (sendto(iFDserver, acBuffer, iLen, 0, (struct sockaddr *) &(psdIndex->sinRemote), sizeof(struct sockaddr)) != iLen) {
						SYSLOG(LOG_ERR, (char *) apcErrMsg[ERRSENDTO],NOQUIT);
					}
				}
			}
		}
	}
	/* NOT REACH */
	return (0);
}
