/*
 *
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 * 
 *   http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 *
 */

/**
 *  topic_listener.cpp:
 *
 *  This program is one of three programs designed to be used
 *  together. These programs use the topic exchange.
 *  
 *    topic_config_queues.cpp:
 *
 *      Creates a queue on a broker, binding a routing key to route
 *      messages to that queue.
 *
 *    topic_publisher.cpp:
 *
 *      Publishes to a broker, specifying a routing key.
 *
 *    topic_listener.cpp (this program):
 *
 *      Reads from a queue on the broker using a message listener.
 *
 */

#include <qpid/client/Connection.h>
#include <qpid/client/Session.h>
#include <qpid/client/Message.h>
#include <qpid/client/MessageListener.h>
#include <qpid/client/Queue.h>
#include "SubscriptionManager.h"

#include <unistd.h>
#include <cstdlib>
#include <iostream>
#include <set>

using namespace qpid::client;
using namespace qpid::framing;


class Listener : public MessageListener {
  private:
    Session& session;
    SubscriptionManager subscriptions;
  public:
    Listener(Session& session);
    virtual void prepareQueue(std::string queue, std::string routing_key);
    virtual void received(Message& message);
    virtual void listen();
    ~Listener() { };
};


/*
 *  Listener::Listener
 *
 *  Subscribe to the queue, route it to a client destination for the
 *  listener. (The destination name merely identifies the destination
 *  in the listener, you can use any name as long as you use the same
 *  name for the listener).
 */

Listener::Listener(Session& session) : 
        session(session),
        subscriptions(session)
{
}


void Listener::prepareQueue(std::string queue, std::string routing_key) {

    /* Create a unique queue name for this consumer by concatenating
     * the Session ID and the destination.
     */

    std::ostringstream uniqueQueueName;
    uniqueQueueName << queue << "-" << session.getId();

    std::string _queue = uniqueQueueName.str();

    std::cout << "Declaring queue: " << _queue <<  std::endl;
   
    /* Declare an exclusive queue on the broker
     */

    session.queueDeclare_(queue=_queue, exclusive=true);

    /* Route messages to the new queue if they match the routing key.
     *
     * Also route any messages to with the "control" routing key to
     * this queue so we know when it's time to stop. A publisher sends
     * a message with the content "That's all, Folks!", using the
     * "control" routing key, when it is finished.
     */

    session.queueBind_(exchange="amq.topic", queue=_queue, routingKey=routing_key);
    session.queueBind_(exchange="amq.topic", queue=_queue, routingKey="control");

    /*
     * subscribe to the queue using the subscription manager.
     */

    std::cout << "Subscribing to queue " << _queue << std::endl;
    destination = subscriptions.subscribe(_queue, *this);
}

void Listener::received(Message& message) {
    std::cout << "Message: " << message.getData() << " from " << message.getDestination() << std::endl;

    if (message.getData() == "That's all, folks!") {
        std::cout << "Shutting down listener for " << message.getDestination() << std::endl;
        subscriptions.cancel(message.getDestination());

        if (!subscriptions.subscriptionCount()) {
            std::cout << "No more destinations to listen to, shutting down!" << std::endl;
            subscriptions.stop();
        }
    }
}

void Listener::listen() {
  subscriptions.run();
}

int main() {
    Connection connection;
    try {
        connection.open("127.0.0.1", 5672);
        Session session =  connection.newSession();

        //--------- Main body of program --------------------------------------------

	// Create a listener for the session

        Listener listener(session);

        // Subscribe to messages on the queues we are interested in

	listener.prepareQueue("usa", "usa.#");
	listener.prepareQueue("europe", "europe.#");
	listener.prepareQueue("news", "#.news");
	listener.prepareQueue("weather", "#.weather");

	std::cout << "Listening for messages ..." << std::endl;

        // Give up control and receive messages
        listener.listen();


        //-----------------------------------------------------------------------------

        connection.close();
        return 0;
    } catch(const std::exception& error) {
        std::cout << error.what() << std::endl;
    }
    return 1;   
}


