/*
 *  arch-tag: Implementation of the Album-Cover Widget
 *
 *  Copyright (C) 2004 Joergen Scheibengruber <mfcn@gmx.de>
 *  Copyright (C) 2004,2005 Marc Pavot <m.pavot@laposte.net>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 */

#include <gtk/gtk.h>
#include <libgnomevfs/gnome-vfs.h>
#include <libgnome/gnome-i18n.h>
#include <string.h>

#include "rb-album-cover.h"
#include "rb-file-helpers.h"
#include "rb-glade-helpers.h"
#include "eel-gconf-extensions.h"
#include "rb-debug.h"
#include "rb-preferences.h"
#include "rb-cover.h"
#include "rb-library-source.h"

#define MAX_COVER_SIZE 500

static void rb_album_cover_class_init (RBAlbumCoverClass *klass);
static void rb_album_cover_init (RBAlbumCover *album_cover);
static void rb_album_cover_finalize (GObject *object);
static void rb_album_cover_set_property (GObject *object,
					 guint prop_id,
					 const GValue *value,
					 GParamSpec *pspec);
static void rb_album_cover_get_property (GObject *object,
					 guint prop_id,
					 GValue *value,
					 GParamSpec *pspec);
static void rb_album_cover_sync (RBAlbumCover *album_cover);
static void rb_album_cover_search (RBAlbumCover *album_cover,
				   const char *artist,
				   const char *album,
				   const char *location);
static void rb_album_cover_search_end (RBAlbumCover *album_cover);
static void rb_album_cover_set_uri (RBAlbumCover *album_cover, 
				    const char *uri);
static void rb_album_cover_set_nocover (RBAlbumCover *album_cover);
static void rb_album_cover_add_searchcover (RBAlbumCover *album_cover, 
					    const char *artist,
					    const char *album,
					    const char *location);
static void rb_album_cover_reload (RBAlbumCover *album_cover);
static void rb_album_cover_redraw (RBAlbumCover *album_cover);
static void rb_album_cover_find_amazon_uri (RBAlbumCover *album_cover,
					    const char *artist,
					    const char *album);
static void rb_album_cover_load_xml_async (RBAlbumCover *album_cover, 
					   const char *xml_uri);
static void rb_album_cover_load_uri_async (RBAlbumCover *album_cover);
static GdkPixbuf *rb_album_cover_get_pixbuf (RBAlbumCover *album_cover);
static void rb_album_cover_save_cover (RBAlbumCover *album_cover);
void rb_album_cover_preferences_local_check_changed_cb (GtkCheckButton *butt,
							RBAlbumCover *album_cover);
void rb_album_cover_preferences_amazon_check_changed_cb (GtkCheckButton *butt,
							 RBAlbumCover *album_cover);
void rb_album_cover_preferences_covertree_check_changed_cb (GtkCheckButton *butt,
							    RBAlbumCover *album_cover);
void rb_album_cover_preferences_amazon_country_changed_cb (GtkComboBoxEntry *combobox,
							   RBAlbumCover *album_cover);
static void rb_album_cover_cover_changed_cb (RBSource *source,
					     RBAlbumCover *album_cover);
static void rb_album_cover_show_no_cover (RBAlbumCover *album_cover);
static void rb_album_cover_size_changed_cb (GConfClient *client,
					    guint cnxn_id,
					    GConfEntry *entry,
					    RBAlbumCover *album_cover);

static const char *amazon_countries[] = {
	"com",
	"fr",
	"de",
	"uk",
	"ca",
	"jp",
	NULL
};

struct RBAlbumCoverPrivate
{	
	GtkWidget *image;

	GtkWidget *config_widget;
	GtkWidget *local_check;
	GtkWidget *amazon_check;
	GtkWidget *covertree_check;
	GtkWidget *amazon_country;

	char *uri;

	RhythmDBEntry *entry;
	RhythmDBEntry *next_entry;

	char *current_artist;
	char *current_album;
	char *current_location;

	char *next_artist;
	char *next_album;
	char *next_location;

	gboolean searching_cover;
	gboolean search_queued;

	gboolean is_amazon_cover;

	char *imagedata;
	gint size;

	RBCoverReadFileHandle *read_handle;

	RBSource *source;
};

enum
{
	PROP_0,
	PROP_SOURCE,
	PROP_URI,
	PROP_CURRENTARTIST,
	PROP_CURRENTALBUM,
	PROP_CURRENTLOCATION,
	PROP_NEXTARTIST,
	PROP_NEXTALBUM,
	PROP_NEXTLOCATION
};

static GObjectClass *parent_class = NULL;

GType
rb_album_cover_get_type (void)
{

	static GType type = 0;

	if (!type) {
		static const GTypeInfo our_info =
		{
			sizeof (RBAlbumCoverClass),
			NULL,
			NULL,
			(GClassInitFunc) rb_album_cover_class_init,
			NULL,
			NULL,
			sizeof (RBAlbumCover),
			0,
			(GInstanceInitFunc) rb_album_cover_init
		};

		type = g_type_register_static (GTK_TYPE_EVENT_BOX,
					       "RBAlbumCover",
						&our_info, 0);
	}
	return type;
}

static void
rb_album_cover_class_init (RBAlbumCoverClass *klass)
{

	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	parent_class = g_type_class_peek_parent (klass);

	object_class->finalize = rb_album_cover_finalize;

	object_class->set_property = rb_album_cover_set_property;
	object_class->get_property = rb_album_cover_get_property;

	g_object_class_install_property (object_class,
					 PROP_SOURCE,
					 g_param_spec_object ("source",
							      "RBSource",
							      "RBSource",
							      RB_TYPE_SOURCE,
							      G_PARAM_READWRITE));
	g_object_class_install_property (object_class,
					 PROP_URI,
					 g_param_spec_string ("uri",
							      "uri",
							      "uri",
							      NULL,
							      G_PARAM_READWRITE));
	g_object_class_install_property (object_class,
					 PROP_CURRENTARTIST,
					 g_param_spec_string ("current_artist",
							      "current_artist",
							      "current_artist",
							      NULL,
							      G_PARAM_READWRITE));
	g_object_class_install_property (object_class,
					 PROP_CURRENTALBUM,
					 g_param_spec_string ("current_album",
							      "current_album",
							      "current_album",
							      NULL,
							      G_PARAM_READWRITE));
	g_object_class_install_property (object_class,
					 PROP_CURRENTLOCATION,
					 g_param_spec_string ("current_location",
							      "current_location",
							      "current_location",
							      NULL,
							      G_PARAM_READWRITE));
	g_object_class_install_property (object_class,
					 PROP_NEXTARTIST,
					 g_param_spec_string ("next_artist",
							      "next_artist",
							      "next_arti	GdkPixbuf *pixbuf;st",
							      NULL,
							      G_PARAM_READWRITE));
	g_object_class_install_property (object_class,
					 PROP_NEXTALBUM,
					 g_param_spec_string ("next_album",
							      "next_album",
							      "next_album",
							      NULL,
							      G_PARAM_READWRITE));
	g_object_class_install_property (object_class,
					 PROP_NEXTLOCATION,
					 g_param_spec_string ("next_location",
							      "next_location",
							      "next_location",
							      NULL,
							      G_PARAM_READWRITE));
}

static void
rb_album_cover_init (RBAlbumCover *album_cover)
{
	album_cover->priv = g_new0 (RBAlbumCoverPrivate, 1);

	album_cover->priv->image = gtk_image_new ();
	gtk_container_add (GTK_CONTAINER (album_cover), 
			   album_cover->priv->image);
	
	album_cover->priv->current_artist = g_strdup ("");
	album_cover->priv->current_album = g_strdup ("");

	eel_gconf_notification_add (CONF_COVER_SIZE,
				    (GConfClientNotifyFunc) rb_album_cover_size_changed_cb,
				    album_cover);
}

static void
rb_album_cover_finalize (GObject *object)
{
	RBAlbumCover *album_cover;

	g_return_if_fail (object != NULL);
	g_return_if_fail (RB_IS_ALBUM_COVER (object));

	album_cover = RB_ALBUM_COVER (object);

	g_return_if_fail (album_cover->priv != NULL);

	g_free (album_cover->priv->uri);
	g_free (album_cover->priv->current_artist);
	g_free (album_cover->priv->current_album);
	g_free (album_cover->priv->next_artist);
	g_free (album_cover->priv->next_album);
	g_free (album_cover->priv->next_location);
	g_free (album_cover->priv);

	G_OBJECT_CLASS (parent_class)->finalize (object);
}

static void
rb_album_cover_set_property (GObject *object,
			     guint prop_id,
			     const GValue *value,
			     GParamSpec *pspec)
{
	RBAlbumCover *album_cover = RB_ALBUM_COVER (object);
	const char *uri;
	const char *current_artist;
	const char *current_album;
	const char *current_location;
	const char *next_artist;
	const char *next_album;
	const char *next_location;
		
	switch (prop_id) {
	case PROP_SOURCE:
		album_cover->priv->source = g_value_get_object (value);
		if (RB_IS_LIBRARY_SOURCE (album_cover->priv->source))
			g_signal_connect_object (G_OBJECT (album_cover->priv->source),
						 "cover_changed", G_CALLBACK (rb_album_cover_cover_changed_cb),
						 album_cover, 0);
		break;	
	case PROP_URI:
		uri = g_value_get_string (value);
		g_free (album_cover->priv->uri);
		album_cover->priv->uri = g_strdup (uri);
		rb_album_cover_sync (album_cover);
		break;
	case PROP_CURRENTARTIST:
		current_artist = g_value_get_string (value);
		g_free (album_cover->priv->current_artist);
		album_cover->priv->current_artist = g_strdup (current_artist);
		break;
	case PROP_CURRENTALBUM:
		current_album = g_value_get_string (value);
		g_free (album_cover->priv->current_album);
		album_cover->priv->current_album = g_strdup (current_album);
		break;
	case PROP_CURRENTLOCATION:
		current_location = g_value_get_string (value);
		g_free (album_cover->priv->current_location);
		album_cover->priv->current_location = g_strdup (current_location);
		break;
	case PROP_NEXTARTIST:
		next_artist = g_value_get_string (value);
		g_free (album_cover->priv->next_artist);
		album_cover->priv->next_artist = g_strdup (next_artist);
		break;
	case PROP_NEXTALBUM:
		next_album = g_value_get_string (value);
		g_free (album_cover->priv->next_album);
		album_cover->priv->next_album = g_strdup (next_album);
		break;
	case PROP_NEXTLOCATION:
		next_location = g_value_get_string (value);
		g_free (album_cover->priv->next_location);
		album_cover->priv->next_location = g_strdup (next_location);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
		break;
	}
}

static void 
rb_album_cover_get_property (GObject *object,
			guint prop_id,
			GValue *value,
			GParamSpec *pspec)
{
	RBAlbumCover *album_cover = RB_ALBUM_COVER (object);

	switch (prop_id) {
	case PROP_SOURCE:
		g_value_set_object (value, album_cover->priv->source);
		break;
	case PROP_URI:
		g_value_set_string (value, album_cover->priv->uri);
		break;
	case PROP_CURRENTARTIST:
		g_value_set_string (value, album_cover->priv->current_artist);
		break;
	case PROP_CURRENTALBUM:
		g_value_set_string (value, album_cover->priv->current_album);
		break;
	case PROP_CURRENTLOCATION:
		g_value_set_string (value, album_cover->priv->current_location);
		break;
	case PROP_NEXTARTIST:
		g_value_set_string (value, album_cover->priv->next_artist);
		break;
	case PROP_NEXTALBUM:
		g_value_set_string (value, album_cover->priv->next_album);
		break;
	case PROP_NEXTLOCATION:
		g_value_set_string (value, album_cover->priv->next_location);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
		break;
	}
}

GtkWidget *
rb_album_cover_new (void)
{
	RBAlbumCover *album_cover;

	album_cover = g_object_new (RB_TYPE_ALBUM_COVER,
				    NULL);

	g_return_val_if_fail (album_cover->priv != NULL, NULL);

	return GTK_WIDGET (album_cover);
}

static void
rb_album_cover_sync (RBAlbumCover *album_cover)
{
	if (album_cover->priv->uri) 
		rb_album_cover_load_uri_async (album_cover);
	else
		rb_album_cover_set_nocover (album_cover);
}

void
rb_album_cover_set_source (RBAlbumCover *album_cover,
			   RBSource *source)
{
	g_object_set (G_OBJECT (album_cover), "source", source, NULL);
}

static void
rb_album_cover_set_uri (RBAlbumCover *album_cover, const char *uri)
{
	g_object_set (G_OBJECT (album_cover), "uri", uri, NULL);
}

static int
rb_album_cover_get_cover_size (void)
{
	int cover_size;

	cover_size = eel_gconf_get_integer (CONF_COVER_SIZE);

	/* We limit the size of the cover to avoid side effects */
	if (cover_size > MAX_COVER_SIZE)
		cover_size = MAX_COVER_SIZE;

	return cover_size;
}

static void
rb_album_cover_show_no_cover (RBAlbumCover *album_cover)
{
	gtk_widget_hide (GTK_WIDGET (album_cover->priv->image));	
}

static void
rb_album_cover_set_nocover (RBAlbumCover *album_cover)
{
	rb_debug ("No cover found");
	rb_album_cover_show_no_cover (album_cover);
	rb_album_cover_search_end (album_cover);
}

static void rb_album_cover_redraw (RBAlbumCover *album_cover)
{
	GdkPixbuf *pixbuf;

	pixbuf = rb_album_cover_get_pixbuf (album_cover);
	if (pixbuf) {
		gtk_image_set_from_pixbuf (GTK_IMAGE (album_cover->priv->image), pixbuf);
		g_object_unref (G_OBJECT (pixbuf));
	}
}

static void 
rb_album_cover_reload (RBAlbumCover *album_cover)
{
	gchar *artist = g_strdup (album_cover->priv->current_artist);
	gchar *album = g_strdup (album_cover->priv->current_album);
	gchar *location = g_strdup (album_cover->priv->current_location);

	rb_album_cover_add_searchcover (album_cover, 
					artist,
					album,
					location);
	g_free (artist);
	g_free (album);
	g_free (location);
}

static void
rb_album_cover_size_changed_cb (GConfClient *client,
				 guint cnxn_id,
				 GConfEntry *entry,
				 RBAlbumCover *album_cover)
{
	rb_album_cover_redraw (album_cover);
}

static void
rb_album_cover_cover_changed_cb (RBSource *source,
				 RBAlbumCover *album_cover)
{
	rb_album_cover_reload (album_cover);
}

void
rb_album_cover_set_playing_entry (RBAlbumCover *album_cover, 
				  RhythmDBEntry *entry)
{
	const gchar *artist, *album, *location;

	/* This function can be called with entry==NULL
	 * to say that rb isn't playing a song */
	if (entry == NULL) {
		rb_album_cover_show_no_cover (album_cover);
		return;
	}

	artist = rb_refstring_get (entry->artist);
	album = rb_refstring_get (entry->album);
	location = (gchar *) rhythmdb_entry_get_string (entry,
					                RHYTHMDB_PROP_LOCATION);

	if (!artist || !album || !location) {
		rb_album_cover_show_no_cover (album_cover);
		return;
	}

	/* Same artist, same album. We don't change anything*/
	if (!strcmp (album_cover->priv->current_artist, artist) && !strcmp (album_cover->priv->current_album, album))
		return;

	rb_album_cover_add_searchcover (album_cover, 
					artist,
					album,
					location);
}

static void
rb_album_cover_add_searchcover (RBAlbumCover *album_cover, 
				const char *artist,
				const char *album,
				const char *location)
{
	if (!artist || !album || !location) {
		rb_album_cover_show_no_cover (album_cover);
		return;
	}

	/* If we are already searching for a cover, we put the new search in a queue */
	if (album_cover->priv->searching_cover) {
		album_cover->priv->search_queued = TRUE;

		g_object_set (G_OBJECT (album_cover), "next_artist", artist, NULL);
		g_object_set (G_OBJECT (album_cover), "next_album", album, NULL);
		g_object_set (G_OBJECT (album_cover), "next_location", location, NULL);
	/* Else, we search for this cover */
	} else {
		album_cover->priv->searching_cover = TRUE;

		rb_album_cover_search (album_cover,
				       artist,
				       album,
				       location);
	}
}

static void
rb_album_cover_search_end (RBAlbumCover *album_cover)
{
	/* If a search has been queued, we search for a new cover */
	if (album_cover->priv->search_queued) {
		album_cover->priv->search_queued = FALSE;

		rb_album_cover_search (album_cover,
				       album_cover->priv->next_artist,
				       album_cover->priv->next_album,
				       album_cover->priv->next_location);
	/* Else, we wait for a new search */
	} else {
		album_cover->priv->searching_cover = FALSE;
	}
}

static void
rb_album_cover_search (RBAlbumCover *album_cover,
		       const char *artist,
		       const char *album,
		       const char *location)
{
	char *cover_uri = NULL;

	if (!artist || !album || !location) {
		rb_album_cover_set_nocover (album_cover);
		return;
	}

	rb_debug ("Searching for a cover");

	g_object_set (G_OBJECT (album_cover), "current_artist", artist, NULL);
	g_object_set (G_OBJECT (album_cover), "current_album", album, NULL);
	g_object_set (G_OBJECT (album_cover), "current_location", location, NULL);

	/* By default, we consider that it isn't an amazon cover. This will be
	 * used to save informations about the cover in the covers.xml file */
	album_cover->priv->is_amazon_cover = FALSE;

	/* If a cover exists in the ~/.gnome2./rhythmbox/covers/ directory, we show it */
	if (rb_cover_cover_exists (artist, album))
		cover_uri = rb_cover_make_cover_path (artist, album, NORMAL_COVER);
	else {
		/* The user has said that we don't have to search for this cover */
		if (rb_cover_get_do_not_search (artist, album)) {
			rb_album_cover_set_nocover (album_cover);
			return;
		}
	}

	/* If there is no cover in ~/.gnome2./rhythmbox/covers/ and if the user,
	 * want it we search for a cover in the song directory */
	if (!cover_uri && location && eel_gconf_get_boolean (CONF_COVER_LOCAL))
		cover_uri = rb_cover_find_local_uri (location);

	/* If there is no cover in the song directory and if the user want it,
	 * we search for a cover on amazon. We can't use something like 
	 * cover_uri =  rb_cover_find_amazon_uri (artist, album) because 
	 * we have to download an XML file asynchronously to get the url*/
	if (!cover_uri) {
		if (eel_gconf_get_boolean (CONF_COVER_AMAZON)) {
			rb_album_cover_find_amazon_uri (album_cover, artist, album);
			return;
		} else {
			rb_album_cover_set_nocover (album_cover);
			return;
		}
	}

	/* If a cover was found in ~/.gnome2./rhythmbox/covers/ or in the song
	 * directory, we use it */
	rb_debug ("Cover found locally : %s", cover_uri);
	rb_album_cover_set_uri (album_cover, cover_uri);
	g_free (cover_uri);
}

static void
rb_album_cover_find_amazon_uri (RBAlbumCover *album_cover,
				const char *artist,
				const char *album)
{
	char *xml_uri;

	rb_debug ("Searching for a cover on Amazon");
	xml_uri = rb_cover_make_amazon_xml_uri (artist,
						album);

	/* We load an XML file from amazon containing the url of covers */
	rb_album_cover_load_xml_async (album_cover, xml_uri);
	g_free (xml_uri);
}

static void
rb_album_cover_load_xml_async (RBAlbumCover *album_cover, 
			       const char *xml_uri)
{
	rb_debug ("Loading XML file from Amazon : %s", xml_uri);
	
	if (!xml_uri) {
		rb_album_cover_set_nocover (album_cover);
		return;
	}

	album_cover->priv->read_handle = rb_cover_read_entire_file_async (xml_uri, 0,
									  rb_album_cover_xml_loaded_cb, album_cover);
}

void
rb_album_cover_xml_loaded_cb (GnomeVFSResult result,
                    GnomeVFSFileSize file_size,
                    char *file_contents,
                    gpointer callback_data)
{
	GList *cover_uris = NULL;
	RBAlbumCover *album_cover = callback_data;

	album_cover->priv->read_handle = NULL;

	if (result != GNOME_VFS_OK) {
		g_free (file_contents);
		return;
	}

	rb_debug ("XML file loaded");

	/* We parse tha XML file to get the url of a cover. We only
	 * we want the url of the first result sent by amazon. */
	cover_uris = rb_cover_parse_amazon_xml_file (file_contents,
						     file_size,
						     GET_FIRST_COVER,
						     AMAZON_MEDIUM_COVER);

	g_free (file_contents);

	/* If amazon didn't find any cover, we stop here */
	if (!cover_uris) {
		rb_album_cover_set_nocover (album_cover);
		return;
	}

	/* If a cover was found on amazon, we use it. */
	rb_debug ("Cover found on amazon : %s", cover_uris->data);
	rb_album_cover_set_uri (album_cover, cover_uris->data);
	album_cover->priv->is_amazon_cover = TRUE;

	g_list_foreach (cover_uris, (GFunc) g_free, NULL);
	g_list_free (cover_uris);
}

static void
rb_album_cover_load_uri_async (RBAlbumCover *album_cover)
{
	rb_debug ("Loading the cover %s", album_cover->priv->uri);

	/* We load the cover asynchronously because it can be on a slow network
	 * or on the Internet (amazon) */
	album_cover->priv->read_handle = rb_cover_read_entire_file_async (album_cover->priv->uri, 0,
									  rb_album_cover_uri_loaded_cb, album_cover);
}

void
rb_album_cover_uri_loaded_cb (GnomeVFSResult result,
			      GnomeVFSFileSize file_size,
			      char *file_contents,
			      gpointer callback_data)
{
	GdkPixbuf *pixbuf;

	RBAlbumCover *album_cover = callback_data;

	album_cover->priv->read_handle = NULL;

	if (result != GNOME_VFS_OK) {
		g_free (file_contents);
		return;
	}

	/* If the cover is too big or too small (blank amazon image), we don't use it */
	if (!rb_cover_size_is_valid (file_size)) {
		g_free (file_contents);
		rb_album_cover_set_nocover (album_cover);
		return;
	}

	rb_debug ("Cover loaded");

	g_free (album_cover->priv->imagedata);
	album_cover->priv->imagedata = file_contents;
	album_cover->priv->size = file_size;

	pixbuf = rb_album_cover_get_pixbuf (album_cover);
	if (pixbuf) {
		rb_debug ("Cover shown");
		gtk_image_set_from_pixbuf (GTK_IMAGE (album_cover->priv->image), pixbuf);
		gtk_widget_show (GTK_WIDGET (album_cover->priv->image));
		g_object_unref (G_OBJECT (pixbuf));
		rb_album_cover_save_cover (album_cover);
	} else {
		rb_album_cover_set_nocover (album_cover);
	}
}

static GdkPixbuf *
rb_album_cover_get_pixbuf (RBAlbumCover *album_cover)
{
	GdkPixbufLoader *loader;
	GdkPixbuf *pixbuf, *resized_pixbuf;
	int width, height;
	int cover_size;

	loader = gdk_pixbuf_loader_new ();

	if (!album_cover->priv->imagedata)
		return NULL;

	if (gdk_pixbuf_loader_write (loader,
				     (const guchar *)album_cover->priv->imagedata,
				     album_cover->priv->size,
				     NULL)) {
		gdk_pixbuf_loader_close (loader, NULL);

		pixbuf = gdk_pixbuf_loader_get_pixbuf (loader);

		width = gdk_pixbuf_get_width (pixbuf);
		height = gdk_pixbuf_get_height (pixbuf);

		cover_size = rb_album_cover_get_cover_size ();

		/* We need to use different width and height for covers who aren't square */
		if (width > height) {
			resized_pixbuf = gdk_pixbuf_scale_simple (pixbuf,
								  cover_size,
								  height * cover_size / width,
								  GDK_INTERP_BILINEAR);
		} else {
			resized_pixbuf = gdk_pixbuf_scale_simple (pixbuf,
								  width * cover_size / height,
								  cover_size,
								  GDK_INTERP_BILINEAR);
		}

		g_object_unref (G_OBJECT (pixbuf));
		return resized_pixbuf;
	} else {
		return NULL;
	}
}

static void 
rb_album_cover_save_cover (RBAlbumCover *album_cover)
{
	char *cover_path;
	RBCoverOrigin origin;

	/* This will be used to save informations on the cover in the covers.xml file */
	if (album_cover->priv->is_amazon_cover)
		origin = AMAZON_COVER;
	else
		origin = LOCAL_COVER;

	/* We save the cover in the ~/.gnome2/rhythmbox/covers/ directory.
	 * We can do it asynchronously but it lead to an horrible and (IMO) useless code. */
	rb_cover_save_cover (album_cover->priv->current_artist,
			     album_cover->priv->current_album,
			     album_cover->priv->imagedata,
			     album_cover->priv->size,
			     origin,
			     album_cover->priv->uri,
			     OVERWRITE_MODE_SKIP);

	/* We update the current uri to use the cover in ~/.gnome2/rhythmbox/covers/ */
	cover_path = rb_cover_make_cover_path (album_cover->priv->current_artist,
					       album_cover->priv->current_album,
					       NORMAL_COVER);
	g_free (album_cover->priv->uri);
	album_cover->priv->uri = cover_path;

	/* We refresh the small covers in the albums list */
	if (RB_IS_LIBRARY_SOURCE (album_cover->priv->source))
		rb_library_source_refresh_albumview (RB_LIBRARY_SOURCE (album_cover->priv->source));

	rb_album_cover_search_end (album_cover);
}

void
rb_album_cover_preferences_local_check_changed_cb (GtkCheckButton *butt,
						   RBAlbumCover *album_cover)
{
	eel_gconf_set_boolean (CONF_COVER_LOCAL, 
			       gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (album_cover->priv->local_check)));
}

void
rb_album_cover_preferences_amazon_check_changed_cb (GtkCheckButton *butt,
						    RBAlbumCover *album_cover)
{
	eel_gconf_set_boolean (CONF_COVER_AMAZON, 
			       gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (album_cover->priv->amazon_check)));
}

void
rb_album_cover_preferences_covertree_check_changed_cb (GtkCheckButton *butt,
						       RBAlbumCover *album_cover)
{
	eel_gconf_set_boolean (CONF_COVER_TREE_HIDDEN, 
			       !gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (album_cover->priv->covertree_check)));
}

void
rb_album_cover_preferences_amazon_country_changed_cb (GtkComboBoxEntry *combobox,
						      RBAlbumCover *album_cover)
{
	int i;

	i = gtk_combo_box_get_active (GTK_COMBO_BOX (album_cover->priv->amazon_country));

	eel_gconf_set_string (CONF_COVER_AMAZON_COUNTRY, 
			      amazon_countries[i]);
}

static void
rb_album_cover_preferences_sync (RBAlbumCover *album_cover)
{
	int i;
	char *current_country;

	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (album_cover->priv->local_check), 
				      eel_gconf_get_boolean (CONF_COVER_LOCAL));
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (album_cover->priv->amazon_check), 
				      eel_gconf_get_boolean (CONF_COVER_AMAZON));
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (album_cover->priv->covertree_check), 
				      !eel_gconf_get_boolean (CONF_COVER_TREE_HIDDEN));

	current_country = eel_gconf_get_string (CONF_COVER_AMAZON_COUNTRY);
	if (!current_country)
		current_country = "com";
	for (i = 0; amazon_countries[i] != NULL; i++) {
		if (!strcmp (amazon_countries[i], current_country)) {
			gtk_combo_box_set_active (GTK_COMBO_BOX (album_cover->priv->amazon_country), i);
			break;
		}
		gtk_combo_box_set_active (GTK_COMBO_BOX (album_cover->priv->amazon_country), 0);
	}

	g_free (current_country);
}

GtkWidget *
rb_album_cover_get_config_widget (RBAlbumCover *album_cover)
{
	GladeXML *xml;
	PangoAttrList *pattrlist;
	PangoAttribute *attr;
	GtkWidget *cover_label;
	GtkListStore *list_store;
	GtkCellRenderer *renderer;
	GtkTreeIter iter;
	int i;

	if (album_cover->priv->config_widget)
		return album_cover->priv->config_widget;

	xml = rb_glade_xml_new ("cover-prefs.glade", "covers_vbox", album_cover);
	album_cover->priv->config_widget =
		glade_xml_get_widget (xml, "covers_vbox");
	album_cover->priv->local_check =
		glade_xml_get_widget (xml, "local_checkbutton");
	album_cover->priv->amazon_check =
		glade_xml_get_widget (xml, "amazon_checkbutton");
	album_cover->priv->covertree_check =
		glade_xml_get_widget (xml, "covertree_checkbutton");
	album_cover->priv->amazon_country =
		glade_xml_get_widget (xml, "amazon_country_combobox");

	list_store = gtk_list_store_new (1, G_TYPE_STRING);

	for (i = 0; amazon_countries[i] != NULL; i++) {
		gtk_list_store_append (list_store, &iter);
		gtk_list_store_set (list_store, &iter,
				    0, amazon_countries[i],
				    -1);
	}

	gtk_combo_box_set_model (GTK_COMBO_BOX (album_cover->priv->amazon_country),
				 GTK_TREE_MODEL (list_store));
	g_object_unref (list_store);

	renderer = gtk_cell_renderer_text_new ();
	gtk_cell_layout_clear (GTK_CELL_LAYOUT (album_cover->priv->amazon_country));
	gtk_cell_layout_pack_start (GTK_CELL_LAYOUT (album_cover->priv->amazon_country), renderer, TRUE);
	gtk_cell_layout_set_attributes (GTK_CELL_LAYOUT (album_cover->priv->amazon_country), renderer,
					"text", 0, NULL);

	pattrlist = pango_attr_list_new ();
	attr = pango_attr_weight_new (PANGO_WEIGHT_BOLD);
	attr->start_index = 0;
	attr->end_index = G_MAXINT;
	pango_attr_list_insert (pattrlist, attr);
	cover_label = glade_xml_get_widget (xml, "cover_preferences_label");
	gtk_label_set_attributes (GTK_LABEL (cover_label), pattrlist);
	pango_attr_list_unref (pattrlist);

	g_object_unref (G_OBJECT (xml));
	
	rb_album_cover_preferences_sync (album_cover);
	return album_cover->priv->config_widget;
}
