/**
 * rt_com
 * ======
 *
 * RT-Linux kernel module for communication across serial lines.
 *
 * Copyright (C) 1997 Jens Michaelsen
 * Copyright (C) 1997-1999 Jochen Kpper
 * Copyright (C) 1999 Hua Mao <hmao@nmt.edu>
 *
 * POSIX IO and new init functions by Michael Barabanov
 */



#include <linux/config.h>
#include <linux/errno.h>
#include <linux/ioport.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/version.h>

#include <asm/system.h>
#include <asm/io.h>
#include <rtl_conf.h>
#include <rtl_core.h>
#include <rtl_sync.h>
#ifdef DEBUG_NEW_RT_COM
#include <rtl_debug.h>
#endif
#ifdef USE_TRACER
#include <rtl_tracer.h>
#endif



#include "rt_com.h"
#include "rt_comP.h"

#ifdef RTLINUX_V1
#error This file is no longer compatible with rtlinux version 1. Upgrade.
#endif
 
#define RT_COM_TABLE_SIZE 8
struct rt_com_struct rt_com_table[ RT_COM_TABLE_SIZE ];

#define MAX_IRQ 16

static int irq_counts[MAX_IRQ];
static int irq_coms[MAX_IRQ][RT_COM_TABLE_SIZE];

int NUM_SERIAL_PORTS=1;
MODULE_PARM(NUM_SERIAL_PORTS,"i");

int IOPORT[RT_COM_TABLE_SIZE] = {0x3F8, 0x2F8,0,0,0,0,0,0 };
MODULE_PARM(IOPORT,"1-8i");
int IRQ[RT_COM_TABLE_SIZE]={4,3,0,0,0,0,0,0};
MODULE_PARM(IRQ,"1-8i");
int BASE_BAUD[RT_COM_TABLE_SIZE]={STD_BASE_BAUD,STD_BASE_BAUD,
				  STD_BASE_BAUD,STD_BASE_BAUD,
				  STD_BASE_BAUD,STD_BASE_BAUD,
				  STD_BASE_BAUD,STD_BASE_BAUD};
MODULE_PARM(BASE_BAUD,"1-8i");
int FLAG[RT_COM_TABLE_SIZE]={STD_COM_FLAG,STD_COM_FLAG,
			     STD_COM_FLAG,STD_COM_FLAG,
			     STD_COM_FLAG,STD_COM_FLAG,
			     STD_COM_FLAG,STD_COM_FLAG};
MODULE_PARM(FLAG,"1-8i");

unsigned int rt_com_isr( unsigned int irq, struct pt_regs *r );

/**
 * Write data to a line.
 * @param com Port to use corresponding to internal numbering scheme.
 * @param ptr Address of data.
 * @param cnt Number of bytes to write.
 *
 * @author Jens Michaelsen, Jochen Kpper
 * @version 1999/07/20
 */
int rt_com_write( unsigned int com, char *ptr, int cnt )
{
    int bytes_put_in_output_buffer=0;
    if( com < NUM_SERIAL_PORTS ) {
	struct rt_com_struct *p = &( rt_com_table[ com ] );
	struct rt_buf_struct *b = &( p->obuf );
	long state;
	while( bytes_put_in_output_buffer < cnt ) 
	{
	    rt_com_irq_off( state );
	    if( ((b->head+1)&(RT_COM_BUF_SIZ-1)) != b->tail)
	    {
		/* put byte into buffer, 
		   move pointers to next elements */
		b->buf[ b->head++ ] = *ptr++;
		b->head &= ( RT_COM_BUF_SIZ - 1 );
		rt_com_irq_on( state );
		bytes_put_in_output_buffer++;
	    }
	    else 
	    {
		rt_com_irq_on( state );
		break;
	    }
	}
	if(bytes_put_in_output_buffer > 0)
	{
	    rt_com_irq_off( state );
	    p->ier |= 0x02;
	    outb( p->ier, p->port + RT_COM_IER );
	    rt_com_irq_on( state );
	}
    }
    return(bytes_put_in_output_buffer);
}



/**
 * Read data we got from a line.
 * @param com Port to use corresponding to internal numbering scheme.
 * @param ptr Address of data buffer. Needs to be of size > cnt !
 * @param cnt Number of bytes to read.
 * @return Number of bytes actually read.
 *
 * @author Jens Michaelsen, Jochen Kpper
 * @version 1999/07/20
 */
int rt_com_read( unsigned int com, char *ptr, int cnt )
{
    int done = 0;
    if( com < NUM_SERIAL_PORTS  && com >= 0 )
    {
	struct rt_com_struct *p = &( rt_com_table[ com ] );
	struct rt_buf_struct *b = &( p->ibuf );
	long state;
	rt_com_isr(p->irq,0);
	while(cnt > 0)
	{
	    rt_com_irq_off (state);
	    if( b->head != b->tail ) 
	    {
		done++;
		*ptr++ = b->buf[ b->tail++ ];
		b->tail &= ( RT_COM_BUF_SIZ - 1 );
		rt_com_irq_on (state);
	    }
	    else
	    {
		rt_com_irq_on (state);
		break;
	    }
	    cnt--;
	}
    }
    return( done );
}



/**
 * Get first byte from the write buffer.
 * @param p rt_com_struct of the line we are writing to.
 * @param c Address to put the char in.
 * @return  Number of characters we actually got.
 * 
 * @author Jens Michaelsen, Jochen Kpper
 * @version 1999/07/20
 */
static inline int rt_com_irq_get( struct rt_com_struct *p, unsigned char *c )
{
	struct rt_buf_struct *b = &( p->obuf );
	long state;
	rt_com_irq_off(state);
	if( b->head != b->tail ) {
		*c = b->buf[ b->tail++ ];
		b->tail &= ( RT_COM_BUF_SIZ - 1 );
		rt_com_irq_on(state);
		return( 1 );
	}
	rt_com_irq_on(state);
	return( 0 );
}



/**
 * Concatenate a byte to the read buffer.
 * @param p  rt_com_struct of the line we are writing to.
 * @param ch Byte to put into buffer.
 *
 * @author Jens Michaelsen, Jochen Kpper
 * @version 1999/07/20
 */
static inline void rt_com_irq_put( struct rt_com_struct *p, unsigned char ch )
{
	struct rt_buf_struct *b = &( p->ibuf );
	long state;
	rt_com_irq_off(state);
	b->buf[ b->head++ ] = ch;
	b->head &= ( RT_COM_BUF_SIZ - 1 );
	rt_com_irq_on(state);
}




static int rt_com_irq_count=0;
static int rt_com_read_count=0;
static int rt_com_write_count=0;

int get_rt_com_irq_count(void)
{
    return rt_com_irq_count;
}

int get_rt_com_read_count(void)
{
    return rt_com_read_count;
}
int get_rt_com_write_count(void)
{
    return rt_com_write_count;
}

/**
 * Real interrupt handler.
 * Called by the registered ISRs to do the actual work.
 * @param com Port to use corresponding to internal numbering scheme.
 *
 * @author Jens Michaelsen, Jochen Kpper, Hua Mao
 * @version 1999/07/26
 */
unsigned int rt_com_isr( unsigned int irq, struct pt_regs *r )
{
    struct rt_com_struct *p=0;
    unsigned int B;
    unsigned char data, sta;
    struct rt_buf_struct *ob;
    struct rt_buf_struct *ib;
    int com,com_index, irq_counts_max;
    long state;

    if(r != 0)
    {
	rt_com_irq_count++;
	rt_com_irq_count %= 0x40000000;
    }

    if(irq >= MAX_IRQ || irq < 0)
    {
	return -1;
    }
    
    irq_counts_max = irq_counts[irq];
    if(irq_counts_max < 0 || irq_counts_max >= RT_COM_TABLE_SIZE)
    {
	return -1;
    }
    
    for(com_index = 0 ; com_index < irq_counts_max; com_index++)
    {
	com = irq_coms[irq][com_index];
	if(com >= NUM_SERIAL_PORTS || com < 0 )
	{
	    break;
	}
	p = &(rt_com_table[com]);
	B= p->port;
	ob = &( p->obuf );
	ib = &( p->ibuf );

	/* get available data while the DATA_READY flag is set
	   and the input ring buffer is not full. */
	while(1)
	{
	    rt_com_irq_off(state);
	    sta = inb(B+RT_COM_LSR);
#ifdef USE_TRACER
	    rtl_trace2(RTL_TRACE_USER, 0x100 + sta);
#endif
	    if(sta & DATA_READY && 
	       ((ib->head+1)&(RT_COM_BUF_SIZ-1))  != ib->tail)
	    {
		data = inb(B+RT_COM_RXB);
		ib->buf[ ib->head++ ] = data;
		ib->head &= ( RT_COM_BUF_SIZ - 1 );
		rt_com_irq_on(state);
	    }
	    else
	    {
		rt_com_irq_on(state);
		break;
	    }
	    rt_com_read_count++;
	}
	
	/* write data while the Transmit Holding Register is empty and
	   output ring buffer is not empty */
	while(1)
	{
	    rt_com_irq_off(state);
	    sta = inb(B+RT_COM_LSR);
#ifdef USE_TRACER
	    rtl_trace2(RTL_TRACE_USER, 0x200 + sta);
#endif
	    if(sta & 0x20 && ob->head != ob->tail)
	    {
		data = ob->buf[ob->tail ];
		ob->tail++;
		ob->tail &= (RT_COM_BUF_SIZ -1);
		outb(data,B+RT_COM_TXB);
		rt_com_irq_on(state);
	    }
	    else
	    {
/* if there is no data in output buffer, 
   disable Transmitter Holding Register Empty Interrupt */
		if(ob->head == ob->tail)
		{
		    p->ier &= ~0x02;
		    outb( p->ier, B + RT_COM_IER );
		}
		rt_com_irq_on(state);				
		break;
	    }
	    rt_com_write_count++;
	}
    }
    if(0 != r)
    {
	rtl_hard_enable_irq(p->irq);
    }
    return 0;	
}



/**
 * Enable hardware fifo buffers. This requires an 16550A or better !
 * @param base    Port base address to work on.
 * @param trigger Bytecount to buffer before an interrupt.
 *
 * @author Jochen Kpper
 * @version 1999/07/20
 */
static inline void enable_fifo( int base, int trigger )
{
	switch ( trigger ) {
	case  0:
		outb( 0x00, base + RT_COM_FCR );
		break;
	case  1:
		outb( 0x01, base + RT_COM_FCR );
		break;
	case  4:
		outb( 0x41, base + RT_COM_FCR );
		break;
	case  8:
		outb( 0x81, base + RT_COM_FCR );
		break;
	case 14:
		outb( 0xC1, base + RT_COM_FCR );
		break;
	default:
		break;
	}
	return;
}



/**
 * Calls from init_module + cleanup_module have baud == 0; in these cases we
 * only do some cleanup.
 *
 * To allocate a port, give usefull setup parameter, to deallocate give negative
 * baud.
 * @param com        Number corresponding to internal port numbering scheme.
 *                   This is esp. the index of the rt_com_table to use.
 * @param baud       Data transmission rate to use [Byte/s].
 * @param parity     Parity for transmission protocol.
 *                   (RT_COM_PARITY_EVEN, RT_COM_PARITY_ODD or RT_COM_PARITY_NONE)
 * @param stopbits   Number of stopbits to use. 1 gives you one stopbit, 2
 *                   actually gives really two stopbits for wordlengths of
 *                   6 - 8 bit, but 1.5 stopbits for a wordlength of 5 bits.
 * @param wordlength Number of bits per word (5 - 8 bits).
 *
 * @author Jens Michaelsen, Jochen Kpper
 * @version 1999/07/20
 */
void rt_com_setup( unsigned int com, int baud, unsigned int parity,
				   unsigned int stopbits, unsigned int wordlength )
{
	struct rt_com_struct  *p = &( rt_com_table[ com ] );
	unsigned int base = p->port, divider, par = parity;
	struct rt_buf_struct *ob;
	struct rt_buf_struct *ib;
	int i;

	/* Stop everything, set DLAB */
	outb( 0x00, base + RT_COM_IER );
	outb( 0x80, base + RT_COM_LCR );

	

	/* clear irq */
	inb( base + RT_COM_IIR );
	inb( base + RT_COM_LSR );
	inb( base + RT_COM_RXB );
	inb( base + RT_COM_MSR );

	ob = &( p->obuf );
	ib = &( p->ibuf );
	for(i = 0; i < RT_COM_BUF_SIZ ; i++)
	{
	    ob->buf[i]=0;
	}
	for(i = 0; i < RT_COM_BUF_SIZ ; i++)
	{
	    ib->buf[i]=0;
	}
	
	ob->head = ob->tail = 0;
	ib->head = ib->tail = 0;
	
	if( 0 == baud ) {
		/* return */
	} else if( 0 > baud ) {
		MOD_DEC_USE_COUNT;
	} else {
		MOD_INC_USE_COUNT;
		divider = p->baud_base / baud;
		outb( divider % 256, base + RT_COM_DLL );
		outb( divider / 256, base + RT_COM_DLM );
		/* bits 3,4 + 5 determine parity */
		if( par & 0xC7 )
			par = RT_COM_PARITY_NONE;
		/* set transmission parameters and clear DLAB */
		outb( ( wordlength - 5 ) + ( ( stopbits - 1 ) << 2 ) + par, base + RT_COM_LCR );
		outb( RT_COM_DTR + RT_COM_RTS +  RT_COM_Out1 + RT_COM_Out2, base + RT_COM_MCR );
		p->ier = 0x01;
		outb( p->ier, base + RT_COM_IER );
		enable_fifo( base, FIFO_TRIGGER );
	}
	return;
}


#ifdef CONFIG_RTL_POSIX_IO

#define RT_COM_MAJOR 4

#include <rtl_posixio.h>

static int rtl_rt_com_open (struct rtl_file *filp)
{
	if (!(filp->f_flags & O_NONBLOCK)) {
		return -EACCES; /* TODO: implement blocking IO */
	}
	if ((unsigned) RTL_MINOR_FROM_FILEPTR(filp) >= NUM_SERIAL_PORTS) {
		return -ENODEV;
	}
	return 0;
}

static int rtl_rt_com_release (struct rtl_file *filp)
{
	return 0;
}

static ssize_t rtl_rt_com_write(struct rtl_file *filp, const char *buf, size_t count, loff_t* ppos)
{
/*	if (rt_com_table[RTL_MINOR_FROM_FILEPTR(filp)].type == 0) {
		return -ENODEV;
	} */
	rt_com_write(RTL_MINOR_FROM_FILEPTR(filp), (char *) buf, count);
	return count;
	
}

static ssize_t rtl_rt_com_read(struct rtl_file *filp, char *buf, size_t count, loff_t* ppos)
{
/*	if (rt_com_table[RTL_MINOR_FROM_FILEPTR(filp)].type == 0) {
		return -ENODEV;
	} */
	return rt_com_read(RTL_MINOR_FROM_FILEPTR(filp), buf, count);
	
}

static struct rtl_file_operations rtl_rt_com_fops = {
       	NULL,
	rtl_rt_com_read,
	rtl_rt_com_write,
	NULL,
	NULL,
	NULL,
	rtl_rt_com_open,
	rtl_rt_com_release
};
#endif


/**
 * Request port memory and register ISRs, if we cannot get the memory of all
 * ports, release all already requested ports and return an error.
 * @return Success status, zero on success.
 *
 * @author Jochen Kpper, Hua Mao
 * @version 1999/07/28
 */
int init_module( void )
{
	struct rt_com_struct *p=0;
	int errorcode = 0, i, j;
#ifdef DEBUG_NEW_RT_COM
	breakpoint();
#endif

	if(NUM_SERIAL_PORTS > RT_COM_TABLE_SIZE)
	{
	    printk( KERN_WARNING "rt_com: NUM_SERIAL_PORTS parameter set to %d but this module is compiled to only support %d\n", NUM_SERIAL_PORTS,RT_COM_TABLE_SIZE );
	    return(-1);
	}
	if(NUM_SERIAL_PORTS < 1)
	{
	    printk( KERN_WARNING "rt_com: NUM_SERIAL_PORTS parameter set to %d\n", NUM_SERIAL_PORTS );
	    return(-1);
	}

	for(i = 0; i < 32; i++)
	{
	    irq_counts[i]=0;
	    for(j=0; j < RT_COM_TABLE_SIZE; j++)
	    {
		irq_coms[i][j]=0;
	    }
	}
	for( i=0; i<NUM_SERIAL_PORTS; i++ ) 
	{
	    rt_com_table[i].baud_base = BASE_BAUD[i];
	    rt_com_table[i].port = IOPORT[i];
	    rt_com_table[i].irq = IRQ[i];
	    rt_com_table[i].flag = FLAG[i];
	    p = &( rt_com_table[ i ] );
	    if(p->port < 1 )
	    {
		printk(KERN_WARNING "Bad IO port address\n");
		errorcode=1;
		break;
	    }
	    if(-EBUSY == check_region( p->port, 8 ) ) {
		errorcode = 1;
		printk(KERN_WARNING "check_region(0x%X,8) failed.\n",p->port);
		break;
	    }
	    request_region( p->port, 8, "rt_com" );
	    irq_coms[p->irq][irq_counts[p->irq]]=i;
	    printk(KERN_INFO "irq_coms: %d %d %d %d %d %d %d %d\n",
		   irq_coms[p->irq][0],
		   irq_coms[p->irq][1],
		   irq_coms[p->irq][2],
		   irq_coms[p->irq][3],
		   irq_coms[p->irq][4],
		   irq_coms[p->irq][5],
		   irq_coms[p->irq][6],
		   irq_coms[p->irq][7]);   
	    irq_counts[p->irq]++;
	    printk(KERN_INFO "irq_counts[%d] = %d\n",
		   p->irq, irq_counts[p->irq]);
	    if(irq_counts[p->irq] == 1)
	    {
		rtl_request_irq( p->irq, rt_com_isr);
		rtl_hard_enable_irq( p->irq );
	    }
	    rt_com_setup( i, 0, 0, 0, 0 );
	    printk( KERN_INFO "rt_com Added port %d io:0x%X irq:%d\n" ,i, p->port, p->irq);
	}
	if( 0 != errorcode ) {
		printk( KERN_WARNING "rt_com: cannot request all port regions,\nrt_com: giving up.\n" );
		for( j=0; j<i; j++ ) {
			rtl_free_irq(p->irq);
			release_region( p->port, 8 );
		}
		return(-1);
	}
#ifdef CONFIG_RTL_POSIX_IO
	if (rtl_register_chrdev (RT_COM_MAJOR, "ttyS", &rtl_rt_com_fops)) {
		printk ("RT-COM: unable to get RTL major %d\n", RT_COM_MAJOR);
		return -EIO;
	}
#endif
	printk( KERN_INFO "rt_com sucessfully loaded.\n" );
	return( errorcode );
}



/**
 * Unregister ISR and releases memory for all ports
 *
 * @author Jochen Kpper, Hua Mao
 * @version 1999/07/26
 */ 
void cleanup_module( void )
{
	struct rt_com_struct *p;
	int i;
	printk(KERN_INFO "rt_com cleanup_module()\n");
#ifdef CONFIG_RTL_POSIX_IO
	rtl_unregister_chrdev(RT_COM_MAJOR, "ttyS");
#endif
	for( i=NUM_SERIAL_PORTS-1; i >= 0; i-- ) {
		p = &( rt_com_table[ i ] );
		printk(KERN_INFO "irq_counts[%d] = %d\n",
		       p->irq, irq_counts[p->irq]);
		if(irq_counts[p->irq]  == 1 )
		{
		    rtl_free_global_irq(p->irq);
		}
		irq_counts[p->irq]--;
		rt_com_setup( i, 0, 0, 0, 0 );
		release_region( p->port, 8 );
	}
	printk( KERN_INFO "rt_com unloaded.\n" );
	return;
}




/**
 * Local Variables:
 * mode: C
 * c-file-style: "Stroustrup"
 * End:
 */
