/*
ACKNOWLEDGMENT: 
- Part of this code is derived from that of the latency calibration example,
  which in turn was copied from a similar program distributed with NMT-RTL.
*/

#include "time.h"
#include "count.h"
#include "task.h"
#include "rtf.h"
#include "module.h"
#include "linux_wrapper.h"
#include "rtai_wrapper.h"

// #define OVERALL

#define ONE_SHOT 1

#define DEBUG_FIFO 0

#define TIMER_TO_CPU 3 // < 0  || > 1 to maintain a symmetric processed timer.

#define RUNNABLE_ON_CPUS 3  // 1: on cpu 0 only, 2: on cpu 1 only, 3: on any;

#define RUN_ON_CPUS ( __smp_num_cpus() > 1 ? RUNNABLE_ON_CPUS : 1)

#define SKIP 4000

#define PERIOD 100000

#define USE_FPU 0


static int cpu_used[RTAI_NR_CPUS];

struct sample 
{ 
	long min; 
	long max; 
	int index; 
};

class Thread 
:	public RTAI::Task 
{
public:
	Thread(int stack, bool use_fpu)
	:	RTAI::Task(stack,0,use_fpu,0,0)
	{
		m_fifo = new RTAI::fifoT<sample>(DEBUG_FIFO, 16000);
	}

	~Thread()
	{
		delete m_fifo;
	}

	int make_periodic( const RTAI::Count& start, const RTAI::Count& period)
	{
		m_Period = period;
		m_Expected = start;

		return RTAI::Task::make_periodic(start,period);
	}

	int run()
	{
		sample samp;
		int diff;
		int skip;
		int average;
		int min_diff;
		int max_diff;
#ifdef OVERALL
		min_diff =  1000000000;
		max_diff = -1000000000;
#endif
		while (1) {
#ifndef OVERALL
			min_diff =  1000000000;
			max_diff = -1000000000;
#endif
			average = 0;

			for (skip = 0; skip < SKIP; skip++) {
				cpu_used[ RTAI::hard_cpu_id() ]++;

				m_Expected += m_Period;
				wait_period();

				diff = (RTAI::Count::now() - m_Expected).to_time();
				if (diff < min_diff) {
					min_diff = diff;
				}
				if (diff > max_diff) {
					max_diff = diff;
				}
				average += diff;
			}
			samp.min = min_diff;
			samp.max = max_diff;
			samp.index = average/SKIP;

			m_fifo->put(&samp);
		}
	
		return -1;
	}
protected:
	RTAI::Count m_Period;
	RTAI::Count m_Expected;
	RTAI::fifoT<sample>* m_fifo;
};


class Module
:	public RTAI::Module
{
public:
	Module()
	{
		RTAI::Count start, period;
		RTAI::linux_use_fpu(USE_FPU);

		m_Thread = new Thread(3000,USE_FPU);
		m_Thread->set_runnable_on_cpus(RUN_ON_CPUS);
#ifdef ONE_SHOT
		RTAI::set_oneshot_mode();
#endif
		RTAI::assign_irq_to_cpu(RTAI::TIMER_8254_IRQ, TIMER_TO_CPU);
		period = RTAI::start_timer( RTAI::Count::from_time(PERIOD) );

		start = RTAI::Count::now() + (10 * period);

		m_Thread->make_periodic( start, period);
	}

	~Module()
	{
		int cpuid;
		RTAI::reset_irq_to_sym_mode(RTAI::TIMER_8254_IRQ);
		RTAI::stop_timer();

		delete m_Thread;
	
		__rt_printk("\n\nCPU USE SUMMARY\n");
		for (cpuid = 0; cpuid < RTAI_NR_CPUS; cpuid++) {
			__rt_printk("# %d -> %d\n", cpuid, cpu_used[cpuid]);
		}
		__rt_printk("END OF CPU USE SUMMARY\n\n");
	}
protected:
	Thread *m_Thread;
};

Module theModule;
