/*
 * Project: rtai_cpp - RTAI C++ Framework 
 *
 * File: $Id: $
 *
 * Copyright: (C) 2001 Erwin Rol <erwin@muffin.org>
 *
 * Licence:
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#include "rtai_wrapper.h"

#ifdef RTASK
#undef RTASK
#endif

#ifdef SEM
#undef SEM
#endif

#ifdef CND
#undef CND
#endif

#ifdef MBX
#undef MBX
#endif

#ifdef __KERNEL__
#include <rtai_sched.h>
#include <rtai.h>
#include <rtai_fifos.h>
#include <rtai_nam2num.h>
#else
#include <sched.h>
#include <stdio.h>
#include <unistd.h>
#include <sys/types.h> // libc5 does not need this file. libc6 does. 

#define KEEP_STATIC_INLINE
#include "../lxrt/rtai_lxrt_user.h"
#include <rtai_lxrt.h>
#endif

#ifdef NR_RT_CPUS
#if RTAI_NR_CPUS != NR_RT_CPUS
#error RTAI_NR_CPUS is not the same as NR_RT_CPUS, check rtai_wrapper.h
#endif
#endif

int cpp_key;


int __rt_printk(const char* fmt, ... ){
	va_list ap;
	int res;
	
	va_start( ap, fmt );
#ifdef __KERNEL__	
	res = rt_printk(fmt,ap);
#else
	res = rtai_print_to_screen(fmt,ap);
#endif
	va_end( ap );
	
	return res;
}


#ifdef __KERNEL__

void __rt_get_global_lock(void){
	rt_get_global_lock();
}

void __rt_release_global_lock(void){
	rt_release_global_lock();
}

int __hard_cpu_id(void){
	return hard_cpu_id();
}

int __rt_assign_irq_to_cpu(int irq, unsigned long cpus_mask){
	return rt_assign_irq_to_cpu(irq,cpus_mask);
}

int __rt_reset_irq_to_sym_mode(int irq){
	return rt_reset_irq_to_sym_mode(irq);
}

#endif

long long __nano2count(long long nano){
	return nano2count(nano);
}

long long __count2nano(long long count){
	return count2nano(count);
}

long long __rt_get_time(void){
	return rt_get_time();
}

long long __rt_get_time_ns(void){
	return rt_get_time_ns();
}

long long __rt_get_cpu_time_ns(void){
	return rt_get_cpu_time_ns();
}

void __rt_set_oneshot_mode(void){
	rt_set_oneshot_mode();
} 

void __rt_set_periodic_mode(void){
	rt_set_periodic_mode();
}

long long __start_rt_timer(int period){
	return start_rt_timer(period);
}

void __stop_rt_timer(void){
	return stop_rt_timer();
}

unsigned long __nam2num(const char *name){
	return nam2num(name);
}

void __num2nam(unsigned long num, char *name){
	num2nam(num,name);
}

// task functions

#ifdef __KERNEL__

RT_TASK * __rt_task_init(void (*rt_thread)(int), int data,
			int stack_size, int priority, int uses_fpu, void(*signal)(void))
{
	RT_TASK * task;

	task = rt_malloc( sizeof(RT_TASK) );

	if(task == 0)
		return 0;

	memset(task,0,sizeof(RT_TASK));
	
	rt_task_init(task,rt_thread,data,stack_size,priority,uses_fpu,signal);

	__rt_tld_set_data(task,cpp_key,(void*)data);
			
	return task;
}

RT_TASK * __rt_task_init_cpuid(void (*rt_thread)(int), int data,
				int stack_size, int priority, int uses_fpu,
				void(*signal)(void), unsigned int run_on_cpu)
{
	RT_TASK * task;
	
	task = rt_malloc(sizeof(RT_TASK));
	
	if(task == 0)
		return 0;

	memset(task,0,sizeof(RT_TASK));
		
	rt_task_init_cpuid(task,rt_thread,data,stack_size,priority,uses_fpu,signal,run_on_cpu);

	__rt_tld_set_data(task,cpp_key,(void*)data);
	
	return task;
}

#else

RT_TASK* __rt_task_init_schmod(int name, int priority, int stack_size, 
				int max_msg_size, int policy, int cpus_allowed)
{
	return rt_task_init_schmod(name,priority,stack_size,max_msg_size,policy,cpus_allowed);
}

unsigned long __rt_get_name(void *adr){
	return rt_get_name(adr);
}

void *__rt_get_adr(int name){
	return rt_get_adr(name);
}

#endif

void __rt_set_runnable_on_cpus(RT_TASK *task, unsigned int cpu_mask)
{
	rt_set_runnable_on_cpus(task,cpu_mask);
}

void __rt_set_runnable_on_cpuid(RT_TASK *task, unsigned int cpuid)
{
	rt_set_runnable_on_cpuid(task,cpuid);
}

int __rt_task_delete(RT_TASK *task)
{
	int result;
	__rt_printk("__rt_task_delete(%p)\n",task);

	if(task == 0)
		return -1;
	
	rt_task_suspend(task);
	
	result = rt_task_delete(task);
	
	rt_free(task);
	
	return result;
}

RT_TASK* __rt_whoami(){
#ifdef __KERNEL__
	return rt_whoami();
#else
	return rt_agent();
#endif
}

void __rt_task_yield(){
	rt_task_yield();
}

int __rt_task_suspend(RT_TASK* task){
	return rt_task_suspend(task);
}

int __rt_task_resume(RT_TASK* task){
	return rt_task_resume(task);
}


#ifdef __KERNEL__

// get a new key and mark it in tld_used_mask 
int __rt_tld_create_key(void){
#if 1
	return 0;
#else
	return rt_tld_create_key();
#endif
}

// free the bit in tld_used_mask
int __rt_tld_free_key(int key){
#if 1
	return 0;
#else	
	return rt_tld_free_key(key);
#endif
}

// set the data in the currents RT_TASKS tld_data array
int __rt_tld_set_data(RT_TASK *task,int key,void* data){
#if 1
	task->system_data_ptr = (void*)data;
	return 0;
#else
	return rt_tld_set_data(task,key,data);
#endif
}

// get the data from the current task
void* __rt_tld_get_data(RT_TASK *task,int key){
#if 1
	return (void*)task->system_data_ptr;
#else
	return rt_tld_get_data(task,key);
#endif
}

#endif

int __rt_task_make_periodic(RT_TASK* task, long long start_time, long long period){
	return rt_task_make_periodic(task,start_time,period);
}

int __rt_task_make_periodic_relative_ns(RT_TASK* task, long long start_delay, long long period){
	return rt_task_make_periodic_relative_ns(task,start_delay,period);
}

void __rt_task_wait_period(void){
	rt_task_wait_period();
}

void __rt_busy_sleep(int nanosecs){
	rt_busy_sleep(nanosecs);
}

void __rt_sleep(long long delay){
	rt_sleep(delay);
}

void __rt_sleep_until(long long time){
	rt_sleep_until(time);
}

void __rt_task_signal_handler(RT_TASK* task, void (*handler)(void)){
	rt_task_signal_handler(task,handler);
}

int __rt_task_use_fpu(RT_TASK* task, int use_fpu_flag){
	return rt_task_use_fpu(task,use_fpu_flag);
}

void __rt_linux_use_fpu(int use_fpu_flag){
	rt_linux_use_fpu(use_fpu_flag);
}

void __rt_preempt_always(int yes_no){
	rt_preempt_always(yes_no);
}

void __rt_preempt_always_cpuid(int yes_no, unsigned int cpu_id){
	rt_preempt_always_cpuid(yes_no, cpu_id);
} 

void __rt_make_soft_real_time(void){
#ifndef __KERNEL__
	rt_make_soft_real_time();
#endif
}

void __rt_make_hard_real_time(void){
#ifndef __KERNEL__
	rt_make_hard_real_time();
#endif
}

int long __rt_is_hard_real_time(RT_TASK *rt_task){
#ifdef __KERNEL__
	return 1;
#else
	return rt_is_hard_real_time(rt_task);
#endif
}

int long __rt_is_soft_real_time(RT_TASK *rt_task){
#ifdef __KERNEL__
	return 0;
#else
	return rt_is_soft_real_time(rt_task);     
#endif
}


/* semaphore */

#if BIN_SEM != __BIN_SEM
#error __BIN_SEM is not the same as BIN_SEM, check rtai_wrapper.h
#endif
 
#if CNT_SEM != __CNT_SEM
#error __CNT_SEM is not the same as CNT_SEM, check rtai_wrapper.h
#endif
 
#if RES_SEM != __RES_SEM
#error __RES_SEM is not the same as RES_SEM, check rtai_wrapper.h
#endif

SEM* __rt_typed_sem_init(int value, int type){
#ifdef __KERNEL__
	SEM * sem;
	
	sem = rt_malloc(sizeof(SEM));
	
	if(sem == 0)
		return 0;
		

	memset(sem,0,sizeof(SEM));

	rt_typed_sem_init(sem,value,type);

	return sem;
#else
	return 0;
#endif
}

int __rt_sem_delete(SEM* sem){
	int result;
	
	if(sem == 0)
		return -1;

	result =  rt_sem_delete(sem);

	rt_free(sem);

	return result;
}

int __rt_sem_signal(SEM* sem){
	return rt_sem_signal(sem);
}

int __rt_sem_broadcast(SEM* sem){
	return rt_sem_broadcast(sem);
}

int __rt_sem_wait(SEM* sem){
	return rt_sem_wait(sem);
}

int __rt_sem_wait_if(SEM* sem){
	return rt_sem_wait_if(sem);
}

int __rt_sem_wait_until(SEM* sem, long long time){
	return rt_sem_wait_until(sem,time);
}

int __rt_sem_wait_timed(SEM* sem, long long delay){
	return rt_sem_wait_timed(sem,delay);
}

/* mailbox functions */

MBX* __rt_mbx_init(int size){
#ifdef __KERNEL__
	MBX * mbx;
	
	mbx = rt_malloc(sizeof(MBX));
	
	if(mbx == 0)
		return 0;
		

	memset(mbx,0,sizeof(MBX));

	rt_mbx_init(mbx,size);

	return mbx;
#else
	return 0;
#endif
}

int __rt_mbx_delete(MBX* mbx){
	int result;
	
	if(mbx == 0)
		return -1;

	result =  rt_mbx_delete(mbx);

	rt_free(mbx);

	return result;
}

int __rt_mbx_send(MBX* mbx, void* msg, int msg_size){
	return rt_mbx_send(mbx,msg,msg_size);
}

int __rt_mbx_send_wp(MBX* mbx, void* msg, int msg_size){
	return rt_mbx_send_wp(mbx,msg,msg_size);
}

int __rt_mbx_send_if(MBX* mbx, void* msg, int msg_size){
	return rt_mbx_send_if(mbx,msg,msg_size);
}

int __rt_mbx_send_until(MBX* mbx, void* msg, int msg_size, long long time){
	return rt_mbx_send_until(mbx,msg,msg_size,time);
}

int __rt_mbx_send_timed(MBX* mbx, void* msg, int msg_size, long long delay){
	return rt_mbx_send_timed(mbx,msg,msg_size,delay);
}

int __rt_mbx_receive(MBX* mbx, void* msg, int msg_size){
	return rt_mbx_receive(mbx,msg,msg_size);
}

int __rt_mbx_receive_wp(MBX* mbx, void* msg, int msg_size){
	return rt_mbx_receive_wp(mbx,msg,msg_size);
}

int __rt_mbx_receive_if(MBX* mbx, void* msg, int msg_size){
	return rt_mbx_receive_if(mbx,msg,msg_size);
}

int __rt_mbx_receive_until(MBX* mbx, void* msg, int msg_size, long long time){
	return rt_mbx_receive_until(mbx,msg,msg_size,time);
}

int __rt_mbx_receive_timed(MBX* mbx, void* msg, int msg_size, long long delay){
	return rt_mbx_receive_timed(mbx,msg,msg_size,delay);
}

// RTF

#ifdef __KERNEL__

int __rtf_create(unsigned int fifo, int size){
	return rtf_create(fifo,size);
}

int __rtf_destroy(unsigned int fifo){
	return rtf_destroy(fifo);
}

int __rtf_reset(unsigned int fifo){
	return rtf_reset(fifo);
}

int __rtf_resize(unsigned int fifo, int size){
	return rtf_resize(fifo,size);
}

int __rtf_put(unsigned int fifo, void *buf, int count){
	return rtf_put(fifo,buf,count);
}

int __rtf_get(unsigned int fifo, void *buf, int count){
	return rtf_get(fifo,buf,count);
}

int __rtf_create_handler(unsigned int fifo, int (*handler)(unsigned int fifo)){
	return rtf_create_handler(fifo,handler);
}

#endif
