/*
 * Project: rtai_cpp - RTAI C++ Framework 
 *
 * File: $Id: $
 *
 * Copyright: (C) 2001 Erwin Rol <erwin@muffin.org>
 *
 * Licence:
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#include "task.h"

extern "C" {

#ifdef __KERNEL__
// OK, this is _ugly_, int casting to this* is even
// worse than casting a void* to a this* (like with pthreads)
void entry_point(int this_pointer){
	RTAI::Task* This = (RTAI::Task*)this_pointer;
	This->run();
}
#else
void* entry_point(void* this_pointer){
        RTAI::Task* This = (RTAI::Task*)this_pointer;
        
        if(This == 0)
                return 0;

	This->m_PID = getpid();
	int priority = 1;
	int stack_size = 1024*32;
	int max_msg_size = sizeof(int);
	int policy = SCHED_RR;
	int cpus_allowed = 1;

	This->m_Task = __rt_task_init_schmod(This->m_PID, 
					     	priority, stack_size,
						max_msg_size,policy,cpus_allowed);

        This->run();

        if( This->is_hard_real_time() ){
                This->make_soft_real_time();
        }

        RT_TASK* tmp = This->m_Task;
        This->m_Task = 0;
        if(tmp != 0){
                __rt_task_delete(tmp);
        }


        return 0;
}
#endif

}

void signal_handler(){
	RTAI::Task* This = RTAI::Task::self();
	This->signal_handler();
}

namespace RTAI {

void get_global_lock(void){
	__rt_get_global_lock();
}

void release_global_lock(void){
	__rt_release_global_lock();
}

int hard_cpu_id(void){
	return __hard_cpu_id();
}

int assign_irq_to_cpu(int irq, unsigned long cpus_mask){
	return __rt_assign_irq_to_cpu(irq,cpus_mask);
}

int reset_irq_to_sym_mode(int irq){
	return __rt_reset_irq_to_sym_mode(irq);
}

void set_oneshot_mode(void){
	__rt_set_oneshot_mode();
}

void set_periodic_mode(void){
	__rt_set_periodic_mode();
}

Count start_timer(void){
	return Count( __start_rt_timer(0) );
}

Count start_timer(const Count& period){
	return Count( __start_rt_timer( period ) );
}

void stop_timer(void){
	__stop_rt_timer();
}

void linux_use_fpu(bool use_fpu_flag){
	__rt_linux_use_fpu(use_fpu_flag);
}

void preempt_always(bool yes_no){
	__rt_preempt_always(yes_no);
}

void preempt_always_cpuid(bool yes_no, unsigned int cpu_id){
	__rt_preempt_always_cpuid(yes_no,cpu_id);
}

/////////////////////////////////////////////
// class RTAI::Task
//

Task::Task(){
	__rt_printk("Task::Task() %p\n",this);
	m_Task = 0;
}

Task::~Task(){
	__rt_printk("Task::~Task() %p\n",this);	

	if(m_Task != 0) {
		__rt_task_delete(m_Task);
	}
}

Task::Task(int stack_size,
           int priority,
           bool uses_fpu,
           bool use_signal,
           unsigned int cpuid)
{
	__rt_printk("Task::Task(...) %p\n",this);
	
	m_Task = 0;

	init(stack_size,priority,uses_fpu,use_signal,cpuid);
}

bool Task::init(int stack_size,
	        int priority, 
                bool uses_fpu, 
                bool use_signal,
                unsigned int cpuid)
{
	__rt_printk("Task::init(...) %p\n",this);

	if(m_Task != 0)
		return false;

	if(stack_size == 0)
		stack_size = 1024*4;

#ifdef __KERNEL__
	if(use_signal)
		m_Task = __rt_task_init(::entry_point,(int)this,
        	               stack_size,priority,uses_fpu,::signal_handler);
	else
		m_Task = __rt_task_init(::entry_point,(int)this,
        	               stack_size,priority,uses_fpu,0);
#else
	if (pthread_create(&m_PthreadID, NULL, ::entry_point,(void*)this)) {
		__rt_printk("PTHREAD_CREATE failed\n");
		return false;
	}

        while ( !is_valid() ) {
                sleep( RTAI::Count::from_time(1000000LL) );
	}
#endif
	return true;
}

Task* Task::self(){
	struct rt_task_struct* task = __rt_whoami();
	if(task == 0)
		return 0;

	return (Task*)__rt_tld_get_data(task,cpp_key);
}

void Task::suspend(){
	__rt_task_suspend(m_Task);
}
	
void Task::resume(){
	__rt_task_resume(m_Task);
}

void Task::yield(){
	__rt_task_yield();
}

int Task::make_periodic(const Count& start_time, const Count& period){
	return __rt_task_make_periodic(m_Task,start_time, period );
}

int Task::make_periodic_relative(const Time& start_delay, const Count& period){
	return __rt_task_make_periodic_relative_ns(m_Task,start_delay, period);
}

void Task::set_runnable_on_cpus(unsigned int cpu_mask){
	__rt_set_runnable_on_cpus(m_Task,cpu_mask);
}

void Task::set_runnable_on_cpuid(unsigned int cpuid){
	__rt_set_runnable_on_cpuid(m_Task,cpuid);
}

void Task::wait_period(){
	__rt_task_wait_period();
}

void Task::busy_sleep(const Time& time){
	__rt_busy_sleep((long long)time);
}

void Task::sleep(const Count& delay){
	__rt_sleep( delay );
}

void Task::sleep_until(const Count& count){
	__rt_sleep_until( count );
}

int Task::use_fpu(bool use_fpu_flag){
	return __rt_task_use_fpu(m_Task,use_fpu_flag);
}

bool Task::is_hard_real_time(){
        return __rt_is_hard_real_time(m_Task) ? true : false;
}

bool Task::is_soft_real_time(){
        return __rt_is_hard_real_time(m_Task) ? false : true;
}

bool Task::is_valid(){
#ifndef __KERNEL__
        RT_TASK* tmp = (RT_TASK*)__rt_get_adr((unsigned long)m_PID);

        if(tmp == 0)
                return false;

        return (tmp == m_Task);
#else
	return (m_Task != 0);
#endif
}

bool Task::make_hard_real_time(){
#ifndef __KERNEL__
	__rt_make_hard_real_time();
#endif
	return true;
}

bool Task::make_soft_real_time(){
#ifdef __KERNEL__
	return false;
#else
	__rt_make_soft_real_time();
	return true;
#endif
} 


}; // namespace RTAI