/***
 *
 *  recv_multiple.cpp
 *
 *  PF_PACKET / SOCK_RAW multiple receiver - Listens on multiple devices
 *  for raw ethernet frames to be received using select().  This program
 *  is intended to demonstrate select() functionality using RTnet.
 *
 *  Author(s): Rob Gubler
 *
 *  Copyright (C) 2006 Rob Gubler <rgubler@gmail.com>
 *
 *  RTnet - real-time networking example
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include <iostream>
#include <cstring>
#include <stdint.h>
#include <limits.h>
#include <errno.h>
#include <pthread.h>
#include <sys/mman.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <netpacket/packet.h>
#include <net/if.h>
#include <arpa/inet.h>
#include <rtnet.h>

enum RESULT
{
	OK = 0,
	ERROR_SOCKET_CREATION,
	ERROR_INTERFACE_INDEX,
	ERROR_BIND,
	ERROR_NONBLOCKING
};

static const uint16_t ETH_IP_PROTO = 0x0800;
static bool g_interrupted = false;

void kill_signal(int signal)
{
	switch(signal)
	{
		case SIGTERM:
		case SIGINT:
		case SIGHUP:
			g_interrupted = true;
			break;
	}
}

static RESULT initialize(int *sock, char *devName)
{
	using namespace std;

	int res;
	struct ifreq ifr;
	struct sockaddr_ll addr;
	int64_t nonBlockingVal = -1;


	if((*sock = socket(PF_PACKET, SOCK_RAW, htons(ETH_IP_PROTO))) < 0)
	{
		cout << "Error creating socket" << endl;

		return ERROR_SOCKET_CREATION;
	}

	cout << "Opened socket " << *sock << " on dev " << devName << endl;

	strncpy(ifr.ifr_name, devName, IFNAMSIZ);

	if((res = ioctl(*sock, SIOCGIFINDEX, &ifr)) < 0)
	{
		cout << "Error getting interface index for device \"" << ifr.ifr_name 
			<< "\"; res = " << res << "; errno = " << errno <<  "; " 
			<< strerror(errno) << endl;

		close(*sock);

		return ERROR_INTERFACE_INDEX;
	}

	addr.sll_family   = AF_PACKET;
	addr.sll_protocol = htons(ETH_IP_PROTO);
	addr.sll_ifindex  = ifr.ifr_ifindex;

	if(bind(*sock, (struct sockaddr *)&addr, sizeof(addr)) < 0)
	{
		cout << "Error binding" << endl;
		close(*sock);

		return ERROR_BIND;
	}

	if((res = ioctl(*sock, RTNET_RTIOC_TIMEOUT, reinterpret_cast<char*>(&nonBlockingVal))) < 0)
	{
		cout << "Error setting socket to non-blocking; " << res << ", " 
			<< strerror(errno) << endl; 

		close(*sock);

		return ERROR_NONBLOCKING;
	}

	return OK;
}

void shutdown(int *sockArr, int sz)
{
	if(!sockArr)
		return;

	for(int i = 0; i < sz; i++)
	{
		if(sockArr[i])
			close(sockArr[i]);
	}

	delete sockArr;
}


int main(int argc, char *argv[])
{
	using namespace std;

	struct sched_param param;
	int sockArrSz = argc - 1;
	int *sockArr;
	int minSockId = INT_MAX;
	int maxSockId = 0;
	fd_set masterSet;
	fd_set workingSet;

	if(argc < 2)
	{
		cout << "Not enough arguments specified" << endl << endl
			<< "Syntax:" << endl 
			<< "  " << *argv << " [Device Name 0] [Device Name 1] ... [Device Name n]" << endl << endl
			<< "Arguments:" << endl
			<< "  [Device Name 0]: The name of the device to open.  This argument is required" << endl << endl
			<< "Example: " << endl
			<< "  " << *argv << " rteth0 rteth1 rtlo" << endl << endl;

		return -1;
	}

	signal(SIGTERM, kill_signal);
	signal(SIGINT, kill_signal);
	signal(SIGHUP, kill_signal);

	mlockall(MCL_CURRENT | MCL_FUTURE);

	sockArr = new int[sockArrSz];
	memset(sockArr, 0, sizeof(int) * sockArrSz);
	FD_ZERO(&masterSet);

	for(int i = 1; i < argc; i++)
	{
		RESULT res = initialize(&sockArr[i - 1], argv[i]);

		if(res != OK)
		{
			cout << "Error initializing; res = " << res << endl;
			shutdown(sockArr, sockArrSz);

			return -1;
		}

		FD_SET(sockArr[i - 1], &masterSet);

		if(sockArr[i - 1] > maxSockId)
			maxSockId = sockArr[i - 1];

		if(sockArr[i - 1] < minSockId)
			minSockId = sockArr[i - 1];
	}

	param.sched_priority = sched_get_priority_max(SCHED_FIFO) - 5;
	pthread_setschedparam(pthread_self(), SCHED_FIFO, &param);

	cout << "Waiting for packets to be received..." << endl;

	do
	{
		int res;

		memcpy(&workingSet, &masterSet, sizeof(fd_set));

		res = select(maxSockId + 1, &workingSet, 0, 0, 0);

		if(res < 0 && errno == EINTR)
		{
			continue;	// Program interrupted by outside process... don't print an error message, 
						// just quit quietly
		}
		if(res < 0)
		{
			cout << "Error, select() failed; res = " << res << ", errno = " << errno << ", " 
				<< strerror(errno) << endl;

			continue;
		}
		else if(res == 0)
		{
			cout << "select() returned indicating timeout was reached (indefinite in this case) ... what?" 
			<< endl;

			continue;
		}
		

		for(int i = minSockId; i < maxSockId; i++)
		{
			const int maxSz = 1514;
			char data[maxSz];

			if(FD_ISSET(i, &workingSet))
			{
				res = recv(i, data, maxSz, 0);

				if(res < 0 && errno == EWOULDBLOCK)
				{
					cout << "recv(): res = " << res << " && errno == EWOULDBLOCK" << endl;
					continue;
				}
				else if(res < 0)
				{
					cout << "recv(): res = " << res << ", errno = " << errno << ", " << strerror(errno) 
						<< endl;

					continue;
				}

				cout << "Received " << res << " byte packet on socket " << i;
			}
		}


	} while(!g_interrupted);


	shutdown(sockArr, sockArrSz);


	return 0;
}

