#--
# Copyright (c) 2008 Andreas Gungl
# Copyright (c) 2008 Thomas Sachse
#
# Permission is hereby granted, free of charge, to any person obtaining
# a copy of this software and associated documentation files (the
# "Software"), to deal in the Software without restriction, including
# without limitation the rights to use, copy, modify, merge, publish,
# distribute, sublicense, and/or sell copies of the Software, and to
# permit persons to whom the Software is furnished to do so, subject to
# the following conditions:
#
# The above copyright notice and this permission notice shall be
# included in all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
# MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
# LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
# OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
# WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
#++

# The ActiveRecord implementation is extended by two methods which provide
# real iterator functionality for a set of enhanced database adapters.
#
# #find_each_by_sql is the iterating form of #find_by_sql and #find_each of 
# #find. Both take a block which gets passed an ActiveRecord object.
# 
# This implementation reduces significantly the amount of memory needed to
# process large result sets.
#
module ActiveRecord
  class Base
    class << self
      # Similar to +find_by_sql+ this method executes a custom sql query 
      # against your database and returns all the results. However the results 
      # will not be returned as an array with columns requested encapsulated 
      # as attributes of the model you call this method from.
      # 
      # Instead, the provided +block+ is called and the ActiveRecord object
      # representing the currently processed row is passed. The rows are 
      # processed in a real result set iteration.
      #
      # ==== Examples
      #   # A simple sql query spanning multiple tables
      #   Post.find_each_by_sql( "SELECT p.title, c.author FROM posts p, comments c WHERE p.id = c.post_id" ) do |post|
      #     post.to_s
      #   end
      #
      def find_each_by_sql(sql, &block)
        connection.select_each(sanitize_sql(sql), "#{name} Load") { |record| yield instantiate(record) }
      end

      def extract_options_from_args!(args) #:nodoc:
        args.last.is_a?(Hash) ? args.pop : {}
      end

      # #find_each supports all sets of arguments as +find+ does. Additionally
      # it expects a block which is used similar to #find_each_by_sql 
      # (ActiveRecord objects are passed to the block instead of returning
      # an array of them).
      #
      def find_each(*args)
        options =  extract_options_from_args!(args)
        validate_find_options(options)
        set_readonly_option!(options)
        if(block_given?)
          if scoped?(:find, :include) || options[:include] then
            find_each_with_associations(options) { |r| yield r}
          else
            find_each_by_sql(construct_finder_sql(options)) { |r| yield r }
          end
        end
      end
    end

    private
    def attributes_with_quotes(include_primary_key = true, include_readonly_attributes = true)
      quoted = {}
      @attributes.each_pair do |name, value|
        if column = column_for_attribute(name)
          quoted[name] = quote_value(read_attribute(name), column) unless !include_primary_key && column.primary
        end
      end
      include_readonly_attributes ? quoted : remove_readonly_attributes(quoted)
    end
  end

  module Associations
    module ClassMethods
      class JoinDependency # :nodoc:
        def instantiate_each(row, &block)
          if row != nil
            primary_id = join_base.record_id(row)
            unless @base_records_hash[primary_id]
              if @base_records_in_order.size > 0
                yield @base_records_in_order.first
                # Die Theorie ist hier ein primary_key der Haupttabelle
                # ist verarbeitet und nun kann der Satz entsorgt werden.
                @base_records_in_order.pop
                # instatiate_each nicht das gesamte Ergebnis durchsucht,
                # wird @base_record_hash nur fuer den Gruppenwechsel
                # verwendet.
                # Bei einem neuen primary_key wird der Hash geleert.
                @base_records_hash = {}
                # record cache leeren
                # join_base.cached_record = {}
                @joins.each {|j| j.cached_record = {} }
              end
              @base_records_in_order << (@base_records_hash[primary_id] = join_base.instantiate(row))
            end
            construct(@base_records_hash[primary_id], @associations, join_associations.dup, row)
          else
            yield @base_records_in_order.first
          end
        end

        class JoinBase
          attr_accessor :cached_record
          def extract_record(row)
            # if the :select option is set, only the selected field should be extracted
            # column_names_with_alias.inject({}){|record, (cn, an)| record[cn] = row[an] if row.has_key?(an); record}
            record = {}
            column_names_with_alias.each {|(cn, an)| record[cn] = row[an] if row.has_key?(an) }
            record
          end
        end
      end

      def find_each_with_associations(options = {}, &block)
        catch :invalid_query do
          join_dependency = JoinDependency.new(self, merge_includes(scope(:find, :include), options[:include]), options[:joins])
          select_each_rows(options, join_dependency) do |row|
            join_dependency.instantiate_each(row) do |record|
              yield record
            end
          end
          # pick up the last record
          join_dependency.instantiate_each(nil) { |record|  yield record }

        end
      end


      def select_each_rows(options, join_dependency, &block)
        connection.select_each(construct_finder_sql_with_included_associations_and_select(options, join_dependency),
          "#{name} Load Including Associations") { |row| yield row }
      end


      def construct_finder_sql_with_included_associations_and_select(options, join_dependency)
        scope = scope(:find)
        sql = "SELECT #{column_aliases_with_select(join_dependency,options[:select])} FROM #{(scope && scope[:from]) || options[:from] || table_name} "
        sql << join_dependency.join_associations.collect{|join| join.association_join }.join

        add_joins!(sql, options, scope)
        add_conditions!(sql, options[:conditions], scope)
        add_limited_ids_condition!(sql, options, join_dependency) if !using_limitable_reflections?(join_dependency.reflections) && ((scope && scope[:limit]) || options[:limit])

        sql << "GROUP BY #{options[:group]} " if options[:group]

        add_order!(sql, options[:order], scope)
        add_limit!(sql, options, scope) if using_limitable_reflections?(join_dependency.reflections)
        add_lock!(sql, options, scope)
        # puts sql
        return sanitize_sql(sql)
      end

      def column_aliases_with_select(join_dependency, select = nil)
        select_data = {}
        select.split(',').each { |field| select_data[field.strip] = 1 } if select != nil
        join_dependency.joins.collect{|join|
          fields = []
          # puts join.primary_key
          join.column_names_with_alias.each{|column_name, aliased_name|
            if select == nil
              fields << "#{join.aliased_table_name}.#{connection.quote_column_name column_name} AS #{aliased_name}"
            else
              # include columns from the select list and the parts of
              # the primray key
              if select_data.has_key?("#{join.aliased_table_name}.#{column_name}")  ||
                  select_data.has_key?("#{column_name}") ||
                  column_part_of_primary_key?(join, column_name)
                fields << "#{join.aliased_table_name}.#{connection.quote_column_name column_name} AS #{aliased_name}"
              end
            end
          }
          fields
        }.flatten.join(", ")
      end

      def column_part_of_primary_key?(join, column_name)
        # the primary key may be a composite key, therefore we
        # are using each to check all parts
        join.primary_key.each {|k| return true if k.to_s == column_name}
        false
      end
    end
  end

  # Here are the enhancements for the DatabaseStatements and the concrete
  # adapter implementations which support the iterator function.
  module ConnectionAdapters

    module DatabaseStatements
      # Each row hash provided by the #select_each_row method of the 
      # database adapter is handed over to the block.
      def select_each(sql, name = nil, &block)
        select_each_row(sql, name = nil) { |row_hash| yield row_hash }
        nil
      end
    end


    class JdbcAdapter < AbstractAdapter
      # Process each row of the result set of the query immediately after the
      # row has been fetched.
      def select_each_row(sql, name = nil)
        # If you have patched your JDBC adapter, you can uncomment the
        # following statement and comment the next one. Only this way you get
        # a real iteration through the result set.

        #@connection.execute_query_iterative(sql) { |row|
        select(sql, name).each { |row|
          yield row
        }
        nil
      end
    end


    class MysqlAdapter < AbstractAdapter
      # Process each row of the result set of the query immediately after the
      # row has been fetched.
      def select_each_row(sql, name = nil)
        @connection.query_with_result = true
        result = execute(sql, name)
        result.each_hash { |row|
          yield row
        }
        result.free
        nil
      end
    end


    class OracleEnhancedAdapter < AbstractAdapter
      # Process each row of the result set of the query immediately after the
      # row has been fetched.
      def select_each_row(sql, name = nil)
        cursor = execute(sql, name)
        cols = cursor.get_col_names.map { |x| oracle_downcase(x) }
        rows = []
        while row = cursor.fetch
          hash = Hash.new

          cols.each_with_index do |col, i|
            hash[col] =
              case row[i]
            when OCI8::LOB
              name == 'Writable Large Object' ? row[i]: row[i].read
            when OraDate
              (row[i].hour == 0 and row[i].minute == 0 and row[i].second == 0) ?
                row[i].to_date : row[i].to_time
            else row[i]
            end unless col == 'raw_rnum_'
          end
          yield hash
        end
        return nil
      ensure
        cursor.close if cursor
      end
    end


    class OracleAdapter < AbstractAdapter
      # Process each row of the result set of the query immediately after the
      # row has been fetched.
      def select_each_row(sql, name = nil)
        cursor = execute(sql, name)
        cols = cursor.get_col_names.map { |x| oracle_downcase(x) }
        rows = []
        while row = cursor.fetch
          hash = Hash.new
          cols.each_with_index do |col, i|
            hash[col] =
              case row[i]
            when OCI8::LOB
              name == 'Writable Large Object' ? row[i]: row[i].read
            when OraDate
              (row[i].hour == 0 and row[i].minute == 0 and row[i].second == 0) ?
                row[i].to_date : row[i].to_time
            else row[i]
            end unless col == 'raw_rnum_'
          end
          yield hash
        end
        return nil
      ensure
        cursor.close if cursor
      end
    end
  end
end