/**
 * 
 */
package com.poferries.common.rules.api;

import java.util.ArrayList;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.drools.RuleBase;
import org.drools.StatelessSession;
import org.drools.audit.WorkingMemoryFileLogger;
import org.drools.event.DebugAgendaEventListener;
import org.drools.event.DebugRuleFlowEventListener;
import org.drools.event.DebugWorkingMemoryEventListener;


/**
 * @author Sarath 
 * Wraps the Rule engine management and invocation of rules. This
 * is the interface to the rules library for the client. Client should
 * override the abstract methods, {@link RulesInvoker#getPackageName}
 * and {@link RulesInvoker#getFacts} and call {@link RulesInvoker#invokeRules}
 * 
 */
public abstract class RulesInvoker {

	/**
	 * Instance of RuleBaseManager that's used to create RuleBase.
	 */
	private RuleBaseManager manager;

	/**
	 * Log instance for application logging. Working memory logging is done to a
	 * separate file using {@link org.drools.audit.WorkingMemoryFileLogger}
	 */
	private static final Log LOG= LogFactory.getLog(RulesInvoker.class);

	/**
	 * Log directory for working memory logger.Logs the events generated by a
	 * working memory. Logs will be written to the file. 
	 * <b><i>&lt;workingMemoryLogDir&gt;/&lt;packageName&gt;_log-&lt;
	 * timeInMillis&gt;"</i></b><br/> If <b>null</b> working memory logging 
	 * will be disabled.
	 */

	private String workingMemoryLogDir = null;
	
	/**
	 * Name of the global variable used to store the Rule Violations
	 * when writing the rules. This variable should be declared as <b>global</b>
	 * in the Rule file. 
	 */
	private static final String RULES_GLOBAL_ERR_LIST = "errorList";

	/**
	 * RulesInvoker need a RuleBaseManager to get the instance of RuleBase.
	 * 
	 * @param manager
	 */
	public RulesInvoker(RuleBaseManager manager) {
		this.manager = manager;
	}

	/**
	 * This method will be used to find the package name of the Rules that this
	 * object is trying to invoke. Implemention should provide a valid package
	 * name that exists in the database.
	 * 
	 * @see {@link RuleBaseLineItem}
	 * @return
	 */
	public abstract String getPackageName();

	/**
	 * This method is used to get the Facts (Objects associated with the rules)
	 * As a List. Implementations should provide all necessary facts for the 
	 * rules in the package. 
	 * 
	 * @return List of objects
	 */
	public abstract List<Object> getFacts();

	/**
	 * This method will be invoked before firing the rules and <b>after</b>
	 * inserting the facts to the RuleEngine.
	 */
	public void preInvokeRule() {

	}

	/**
	 * This method will be invoked after firing all the rules.
	 */
	public void postInvokeRule() {

	}

	/**
	 * invoke the Rule engine by calling it's <i>fireAllRules</i> mehod.
	 * 
	 * @return List of RuleVuilation Objects. 
	 */
	public final List<RuleViolation> invokeRules() {

		LOG.debug("Creating RuleBase object for :" + getPackageName());
		// Get the instance of Rulebase with the package added
		final RuleBase ruleBase = manager.getRuleBase(getPackageName());
		final StatelessSession session = ruleBase.newStatelessSession();
		List<RuleViolation> errList = new ArrayList<RuleViolation>();

		WorkingMemoryFileLogger workingMemoryFileLogger = null;

		if (null != workingMemoryLogDir) {
			LOG.debug("Setting up working memory logging");
			// Logging for Rules internel events enabled. Setup the log file.
			workingMemoryFileLogger = new WorkingMemoryFileLogger(session);
			final String logFileName = workingMemoryLogDir + getPackageName()
					+ "_log-" + System.currentTimeMillis();
			workingMemoryFileLogger.setFileName(logFileName);

			session.addEventListener(new DebugAgendaEventListener());
			session.addEventListener(new DebugRuleFlowEventListener());
			session.addEventListener(new DebugWorkingMemoryEventListener());
			session.addEventListener(new DebugWorkingMemoryEventListener());
		} else {
			LOG.debug("Working memory logging disabled");
		}

		//Set global variable to store the Rule violations.
		session.setGlobal(RULES_GLOBAL_ERR_LIST, errList);

		LOG.debug("Excuting Rules ...");
		// Insert facts to the working memory and execute the rules.
		session.execute(getFacts());

		//write working memory log to disk
		if (workingMemoryFileLogger != null) {
			workingMemoryFileLogger.writeToDisk();
		}

		return errList;
	}

	/**
	 * @return the workingMemoryLogDir
	 */
	public String getWorkingMemoryLogDir() {
		return workingMemoryLogDir;
	}

	/**
	 * @param workingMemoryLogDir
	 *            the workingMemoryLogDir to set
	 */
	public void setWorkingMemoryLogDir(String workingMemoryLogDir) {
		this.workingMemoryLogDir = workingMemoryLogDir;
	}

	/**
	 * @return the manager
	 */
	public RuleBaseManager getManager() {
		return manager;
	}
	
	/**
	 * @param manager the manager to set
	 */
	public void setManager(RuleBaseManager manager) {
		this.manager = manager;
	}

}
