package com.sample;

import java.util.ArrayList;
import java.util.List;
import java.util.Random;

import org.drools.KnowledgeBase;
import org.drools.KnowledgeBaseFactory;
import org.drools.builder.KnowledgeBuilder;
import org.drools.builder.KnowledgeBuilderError;
import org.drools.builder.KnowledgeBuilderErrors;
import org.drools.builder.KnowledgeBuilderFactory;
import org.drools.builder.ResourceType;
import org.drools.io.ResourceFactory;
import org.drools.logger.KnowledgeRuntimeLogger;
import org.drools.logger.KnowledgeRuntimeLoggerFactory;
import org.drools.runtime.StatefulKnowledgeSession;

/**
 * This is a sample class to launch a rule.
 */
public class DroolsTest {

	public static final void main(String[] args) {
		try {
			// load up the knowledge base
			KnowledgeBase kbase = readKnowledgeBase();
			StatefulKnowledgeSession ksession = kbase.newStatefulKnowledgeSession();
			KnowledgeRuntimeLogger logger = KnowledgeRuntimeLoggerFactory.newFileLogger(ksession, "test");
			// go !
			generateData(ksession);
			ksession.fireAllRules();
			logger.close();
		} catch (Throwable t) {
			t.printStackTrace();
		}
	}
	
	private static final void generateData(StatefulKnowledgeSession session) {
		
		final Random rand = new Random();
		
		for(int i = 0; i < 1000; i++) {
			session.insert(
				new Sighting(
					i, 
					Direction.values()[rand.nextInt(Direction.values().length)],
					rand.nextInt(3),
					rand.nextInt(50)
				)
			);
		}
	}

	private static final KnowledgeBase readKnowledgeBase() throws Exception {
		KnowledgeBuilder kbuilder = KnowledgeBuilderFactory.newKnowledgeBuilder();
		kbuilder.add(ResourceFactory.newClassPathResource("Sample.drl"), ResourceType.DRL);
		KnowledgeBuilderErrors errors = kbuilder.getErrors();
		if (errors.size() > 0) {
			for (KnowledgeBuilderError error: errors) {
				System.err.println(error);
			}
			throw new IllegalArgumentException("Could not parse knowledge.");
		}
		KnowledgeBase kbase = KnowledgeBaseFactory.newKnowledgeBase();
		kbase.addKnowledgePackages(kbuilder.getKnowledgePackages());
		return kbase;
	}

	public static abstract class VehicleActivity {
		
		public final Direction direction;
		public final int lane;
		public final int position;
		
		public VehicleActivity(Direction direction, int lane, int position) {
			super();
			this.direction = direction;
			this.lane = lane;
			this.position = position;
		}
		
		public Direction getDirection() {
			return direction;
		}
		public int getLane() {
			return lane;
		}
		public int getPosition() {
			return position;
		}
		public String toString() {
			StringBuilder sb = new StringBuilder();
			sb.append(getClass().getSimpleName());
			sb.append(": ");
			sb.append(direction);
			sb.append(", ");
			sb.append(lane);
			sb.append(", ");
			sb.append(position);
			return sb.toString();
		}
	}
	
	public static class Sighting extends VehicleActivity {
		
		private final int vehicle;
		
		private Collision collision = null;
		
		public Sighting(int vehicle, Direction direction, int lane, int position) {
			super(direction, lane, position);
			this.vehicle = vehicle;
		}
		
		public int getVehicle() {
			return vehicle;
		}

		public Collision getCollision() {
			return collision;
		}

		public void setCollision(Collision collision) {
			this.collision = collision;
		}
	}
	
	public static class Collision extends VehicleActivity {

		private List<Sighting> sightings = new ArrayList<Sighting>();

		public Collision(Direction direction, int lane, int position) {
			super(direction, lane, position);
		}
		
		public List<Sighting> getSightings() {
			return sightings;
		}
	}
	
	public static enum Direction {
		NORTH, SOUTH, EAST, WEST;
	}

}