"""
===============================================================================

Created by    : carlos
Created on    : 04/10/2011
Modified on   : 05/10/2011

===============================================================================

flowSolver.py

This script solves 2D steady incompressible flows and simulates 2D unsteady
incompressible flows on a square domain. User-defined Dirichlet boundary
conditions defines the considered flow. The Finite Volume Method is employed
for spatial discretisations. Temporal solutions are obtained through time
marching schemes. For a correct execution, the following packages are
required:

1) ghost.py
2) meshGen.py
3) presMat.py
4) timeMarch.py
      a) bdf.py
      b) rk.py
5) fluxes.py
      a) centralDisc.py
      b) upwindDisc.py
6) plotSol.py

===============================================================================
"""

import time
tic = time.clock()

#==============================================================================
# IMPORT PACKAGES
#==============================================================================

from numpy import zeros, max, append
from numpy.linalg import norm
from matplotlib.pyplot import show

from BC import selectFlow
from ghost import updateGhost
from meshGen import uniMesh, nonUniMesh
from poisson import pressureMatrix, solvePoisson
from fluxes import conDiff
from timeMarch import dudt
from plotSol import plotConv, plotStream, plotQuiver, plotSurf

#==============================================================================
# PARAMETERS
#==============================================================================

""" User defined """

N       = 0 # Amount of grid cells in one direction
flow    = 0 # Predefined flow cases selection paramter
linSol  = 0 # Linear system solving method parameter
tol     = 0 # Linear system solver and solution residual tolerance
maxIter = 0 # Maximum amount of iterations

while (flow <= 0 or flow > 5):
    flowCases = 'Select flow case:\n\
1: Uniform parallel flow\n\
2: Poiseuille flow\n\
3: Lid-driven cavity flow\n\
4: Rotating flow\n'
    flow = input(flowCases)

    if (flow < 1 or flow > 5):
        print ' '
        print 'ERROR: Select one from 1 through 5 to specify a flow type'
        print ' '

print ' '

while (N <= 1):
    N = input('Input grid size:\n')   # Number of cells in the x- and y-direction [-]

    if N <= 1:
        print ' '
        print 'ERROR: Grid size must be larger than 1 '
        print ' '

print ' '

while (linSol < 1 or linSol > 3):
    solvCases = 'Select linear system solving method:\n\
1: Gaussian elimination\n\
2: Conjugate gradient (CG)\n\
3: Bi-conjugate gradient (BiCG)\n'
    linSol = input(solvCases)

print ' '

while (tol >=1 or tol <= 0):
    tol = input('Set linear system solver and solution residual tolerance:\n')
    tol = 10.**tol

print ' '

while (maxIter <=0):
    maxIter = input('Set maximum allowable iterations:\n')

print ' '

""" Flow properties  """

u   = zeros([N+2,N+1],float)    # Horizontal velocity [m/s]
v   = zeros([N+1,N+2],float)    # Vertical velocity [m/s]
p   = zeros([N*N,1],float)      # Pressure [Pa]

f   = zeros([N*N,1],float)

Re  = 1000.                     # Reynolds number [-]

""" Solver parameters """

uni   = 0       # 0 = uniform mesh
                # 1 = non-uniform mesh

disc  = 0       # 0 = Central discretisation in space
                # 1 = Upwind discretisation in space

march = 0       # 0 = FE time integration
                # 1 = BE time integration
                # 2 = BDF2 time integration
                # 3 = ESDIRK time integration

#==============================================================================
# MESH GENERATION
#==============================================================================

if ~uni:
    [nodes, dx, dy] = uniMesh(N)

else:
    [nodes, dx, dy] = nonUniMesh(N)

#==============================================================================
# TIME STEP
#==============================================================================

if dx[0,0] < 2./Re:
    dt = Re * dx[0,0]**2./2.
else:
    dt = 2. / Re

#==============================================================================
# BOUNDARY CONDITIONS
#==============================================================================

[u, v, uT, uB, vL, vR] = selectFlow(u, v, N, flow, dx[0,:])

#==============================================================================
# GHOST POINTS
#==============================================================================

[u, v] = updateGhost(u, v, uT, uB, vL, vR)

#==============================================================================
# PRESSURE MATRIX
#==============================================================================

A = zeros([N**2,N**2], float)
A = pressureMatrix(N, dx, dy, linSol)

#==============================================================================
# SIMULATION
#==============================================================================

it      = 1             # Iteration number
eps_u   = 1.            # Residual of u
eps_v   = 1.            # Residual of v

u_old   = u
v_old   = v

while (max([eps_u, eps_v]))>tol:

    [Rx, Ry, f]     = conDiff(u_old, v_old, N, dx, dy, Re, dt)          # Compute fluxes at t = n + 1
    p               = solvePoisson(A, f/dt, N, linSol, tol)             # Compute pressure at t = n + 1

    [u_new, v_new]  = dudt(u_old, v_old, Rx, Ry, p, dx, dy, dt)         # Compute velocities at t = n + 1
    [u_new, v_new]  = updateGhost(u_new, v_new, uT, uB, vL, vR)         # Compute ghost velocities at t = n + 1

    eps_u           = norm([ u_old[1:-2,0:-1] - u_new[1:-2,0:-1] ])     # Residuals of u at t = n + 1
    eps_v           = norm([ v_old[0:-1,1:-2] - v_new[0:-1,1:-2] ])     # Residuals of v at t = n + 1

    if it==1:
        conv_u = eps_u                                                  # Store residual of u
        conv_v = eps_v                                                  # Store residual of v
    else:
        conv_u = append(conv_u, eps_u)
        conv_v = append(conv_v, eps_v)

    u_old           = u_new                                             # Store u velocities at t = n
    v_old           = v_new                                             # Store v velocities at t = n

    print repr(it).rjust(4), repr(eps_u).rjust(4), repr(eps_v).rjust(4)

    it = it + 1

    if it==maxIter:
        break

#==============================================================================
# RESULTS
#==============================================================================

if (linSol != 1):
    A=A.todense()

toc = time.clock()

print ' '
print 'Elapsed time is ',toc-tic,' seconds.'

plotConv(conv_u, conv_v, it)
plotStream(nodes, u_new, v_new, N, dx, dy)
plotQuiver(nodes, u_new, v_new, N)
plotSurf(nodes, u_new, v_new, N, dx, dy)
show()